/**************************************************************************/
/* FILE NAME: tpu_oc.c                       COPYRIGHT (c) MOTOROLA 2002  */
/* VERSION: 1.0                                   All Rights Reserved     */
/*                                                                        */
/* DESCRIPTION: This file contains the TPU OC functions. These functions  */
/* allow you to completely control TPU channels running the DIO function. */
/* They provide a simple interface requiring the minimum amount of        */
/* configuration by the user.                                             */
/*========================================================================*/
/* HISTORY           ORIGINAL AUTHOR: Jeff Loeliger                       */
/* REV      AUTHOR      DATE       DESCRIPTION OF CHANGE                  */
/* ---   -----------  ---------    ---------------------                  */
/* 1.0   J. Honnold   19/Aug/02    Initial version of function.           */
/**************************************************************************/

#include "mpc500_util.h"
#include "tpu_oc.h"

/*******************************************************************************
FUNCTION      : tpu_oc_host_init_i
PURPOSE       : To initialize a channel to run the OC function in output mode 
				with immediate output.
INPUTS NOTES  : This function has 8 parameters:
                 *tpu - This is a pointer to the TPU3 module to use. It is of
                         type TPU3_tag which is defined in m_tpu3.h
                 channel - This is the channel number of the primary QDEC
                           channel.
                 offset - This is the TCR time count offset from start.
                 mode - Mode is used to select which host service request mode to run:
                 		0 = All code is executed
                 		1 = Only code that updates the TCR values is executed.
                 force - This is what the OC will force the output to at initialization.
                 		Force High = 1
                 		Force Low = 2
                 		Don't Force = 3
                 change - This is what the OC will do to the output after the OFFSET time.
                 		Change High = 1
                 		Change Low = 2
                 		Toggle = 3
                 match - This selects which TCR value to match.
                 		TCR1 = 1
                 		TCR2 = 2
                 priority - This is the priority to assign to both channels.
                            This parameter should be assigned a value of:
                            TPU_PRIORITY_HIGH, TPU_PRIORITY_MIDDLE or
                            TPU_PRIORITY_LOW.
                 level - This is the initial level of the pin.
RETURNS NOTES : none
WARNING       : The channels must be stopped before it is reconfigured. The
                function disables the channels but if they were currently
                being serviced it would continue. The delay for assigning the
                pram pointer may to enough but depends on system loading.
*******************************************************************************/
void tpu_oc_host_init_i(struct TPU3_tag *tpu, UINT8 channel, UINT16 offset, \
		UINT8 mode, UINT8 force, UINT8 change, UINT8 match_tcr, UINT8 priority )
{
    /* diable channel so it can be configured safely */
    tpu_disable(tpu, channel);
    
    /* select OC function for channel */
    tpu_func(tpu, channel, TPU_FUNCTION_OC);
    
    /* Initialize channel */
	tpu_hsr(tpu, channel, TPU_OC_INIT_PULSE);
    
    tpu->PARM.R[channel][TPU_OC_CHANNEL_CONTROL] = 0x80 + 0x20 * match_tcr + change + force ;
    tpu->PARM.R[channel][TPU_OC_OFFSET] = offset ;

	// If match = 0x0, set the REF_ADDR1 for TCR1
	if ( match_tcr == 0x0 )
	    tpu->PARM.R[channel][TPU_OC_REF_ADDR1] = 0xEC ;
	else
	    tpu->PARM.R[channel][TPU_OC_REF_ADDR1] = 0xEE ;

    /* Initialize HSQ for code execution */
    if( mode != 0x0 )
		tpu_hsq(tpu, channel, TPU_OC_TCR_ONLY);    
	else
		tpu_hsq(tpu, channel, TPU_OC_ALL_CODE);    
	
    /* Enable channel by assigning a priority. */
    tpu_enable(tpu, channel, priority);
}

/*******************************************************************************
FUNCTION      : tpu_oc_host_init_im
PURPOSE       : To initialize a channel to run the OC function in output mode 
				with immediate output not selected.
INPUTS NOTES  : This function has 9 parameters:
                 *tpu - This is a pointer to the TPU3 module to use. It is of
                         type TPU3_tag which is defined in m_tpu3.h
                 channel - This is the channel number of the primary QDEC
                           channel.
                 offset - This is the TCR time count offset from start.
                 ref_addr1 - This is the low byte of the address where a reference 
                 			 time resides, or the address of the current TCR value.
                 mode - Mode is used to select which host service request mode to run:
                 		0 = All code is executed
                 		1 = Only code that updates the TCR values is executed.
                 force - This is what the OC will force the output to at initialization.
                 		Force High = 1
                 		Force Low = 2
                 		Don't Force = 3
                 change - This is what the OC will do to the output after the OFFSET time.
                 		Change High = 1
                 		Change Low = 2
                 		Toggle = 3
                 match - This selects which TCR value to match.
                 		TCR1 = 1
                 		TCR2 = 2
                 priority - This is the priority to assign to both channels.
                            This parameter should be assigned a value of:
                            TPU_PRIORITY_HIGH, TPU_PRIORITY_MIDDLE or
                            TPU_PRIORITY_LOW.
RETURNS NOTES : none
WARNING       : The channels must be stopped before it is reconfigured. The
                function disables the channels but if they were currently
                being serviced it would continue. The delay for assigning the
                pram pointer may to enough but depends on system loading.
*******************************************************************************/
void tpu_oc_host_init_ni(struct TPU3_tag *tpu, UINT8 channel, \
		UINT16 offset, UINT8 ref_addr1, UINT8 mode, UINT8 force, \
		UINT8 change, UINT8 match_tcr, UINT8 priority )
{
    /* diable channel so it can be configured safely */
    tpu_disable( tpu, channel);
    
    /* select OC function for channel */
    tpu_func( tpu, channel, TPU_FUNCTION_OC);
    
    /* Initialize channel */
	tpu_hsr(tpu, channel, TPU_OC_INIT_PULSE);
    
    tpu->PARM.R[channel][TPU_OC_CHANNEL_CONTROL] = 0x80 + 0x20 * match_tcr + change + force ;
    tpu->PARM.R[channel][TPU_OC_OFFSET] = offset ;
    tpu->PARM.R[channel][TPU_OC_REF_ADDR1] = ref_addr1 ;

    /* Initialize HSQ for code execution */
    if( mode != 0x0 )
		tpu_hsq(tpu, channel, TPU_OC_TCR_ONLY);    
	else
		tpu_hsq(tpu, channel, TPU_OC_ALL_CODE);    
	
    /* Enable channel by assigning a priority. */
    tpu_enable(tpu, channel, priority);
}

/*******************************************************************************
FUNCTION      : tpu_oc_continuous
PURPOSE       : To initialize a channel to run the OC function in continuous mode.
INPUTS NOTES  : This function has 12 parameters:
                 *tpu - This is a pointer to the TPU3 module to use. It is of
                         type TPU3_tag which is defined in m_tpu3.h
                 channel - This is the channel number of the primary QDEC
                           channel.
                 offset - Contains the count value from a reference time value 
                 		  that indicates when the next match event will occur. In 
                 		  host-initiated pulse mode, the CPU writes this parameter. 
                 		  In continuous pulse mode, OFFSET determines the pulse 
                 		  width and is calculated by the TPU when a link is received
                 		  as OFFSET = (REF_ADDR2) * RATIO where () indicates the data
                 		  pointed to by the specified address.
                 ratio - An 8-bit fractional parameter (0.xxxxxxxx) that scales
                 		 the contents indicated by REF_ADDR2 in the calculation 
                 		 of the OFFSET parameter when operating in the continuous
                 		 pulse mode. The actual fraction used by the TPU is 
                 		 RATIO/$FF. Thus, a value of $80 in RATIO represents the 
                 		 fraction $80/$FF, which is approximately equal to 1/2.  
                 		 Similarly, a value of $FF in RATIO represents a ratio of 1/1.
                 ref_addr1 - This is the low byte of the address where a reference 
                 			 time resides, or the address of the current TCR value.
                 ref_addr2 - A pointer into the parameter RAM map.  It points to a 
                 			 reference value that is scaled by the RATIO parameter 
                 			 to form a new OFFSET value whenever a link is received.
                 ref_addr3 - A pointer into the parameter RAM map that points to a 
                 			 synchronization reference value used only once in 
                 			 continuous pulse mode when the first link service request 
                 			 is serviced after initialization.
                 mode - Mode is used to select which host service request mode to run:
                 		0 = All code is executed
                 		1 = Only code that updates the TCR values is executed.
                 force - This is what the OC will force the output to at initialization.
                 		Force High = 1
                 		Force Low = 2
                 		Don't Force = 3
                 change - This is what the OC will do to the output after the OFFSET time.
                 		Change High = 1
                 		Change Low = 2
                 		Toggle = 3
                 match_tcr - This selects which TCR value to match.
                 		TCR1 = 1
                 		TCR2 = 2
                 priority - This is the priority to assign to both channels.
                            This parameter should be assigned a value of:
                            TPU_PRIORITY_HIGH, TPU_PRIORITY_MIDDLE or
                            TPU_PRIORITY_LOW.
RETURNS NOTES : none
WARNING       : The channels must be stopped before it is reconfigured. The
                function disables the channels but if they were currently
                being serviced it would continue. The delay for assigning the
                pram pointer may to enough but depends on system loading.  This
                mode requires a link in order to be started.  The parameter offset
                must be less than 0x8001.
*******************************************************************************/
void tpu_oc_continuous(struct TPU3_tag *tpu, UINT8 channel, \
		UINT8 ratio, UINT8 ref_addr1, UINT8 ref_addr2, UINT8 ref_addr3, \
		UINT8 mode, UINT8 force, UINT8 change, UINT8 match_tcr, UINT8 priority )
{
    /* disable channel so it can be configured safely */
    tpu_disable( tpu, channel);
    
    /* select OC function for channel */
    tpu_func( tpu, channel, TPU_FUNCTION_OC);
    
    /* Initialize channel */
	tpu_hsr(tpu, channel, TPU_OC_CONTINUOUS);
    
    /* Initialize HSQ for code execution */
    if( mode != 0x0 )
		tpu_hsq(tpu, channel, TPU_OC_TCR_ONLY);    
	else
		tpu_hsq(tpu, channel, TPU_OC_ALL_CODE);    
	
    tpu->PARM.R[channel][TPU_OC_CHANNEL_CONTROL] = 0x80 + 0x20 * match_tcr + change + force ;
    tpu->PARM.R[channel][TPU_OC_PARAM2] = ratio * 0x100 + ref_addr1 ;
    tpu->PARM.R[channel][TPU_OC_PARAM3] = ref_addr2 * 0x100 + ref_addr3 ;

    /* Enable channel by assigning a priority. */
    tpu_enable(tpu, channel, priority);
}


/*******************************************************************************
FUNCTION      : tpu_oc_continuous_update
PURPOSE       : To initialize a channel to run the OC function in continuous 
				mode to update the function.
INPUTS NOTES  : This function has 8 parameters:
                 *tpu - This is a pointer to the TPU3 module to use. It is of
                         type TPU3_tag which is defined in m_tpu3.h
                 channel - This is the channel number of the primary QDEC
                           channel.
                 ratio - An 8-bit fractional parameter (0.xxxxxxxx) that scales
                 		 the contents indicated by REF_ADDR2 in the calculation 
                 		 of the OFFSET parameter when operating in the continuous
                 		 pulse mode. The actual fraction used by the TPU is 
                 		 RATIO/$FF. Thus, a value of $80 in RATIO represents the 
                 		 fraction $80/$FF, which is approximately equal to 1/2.  
                 		 Similarly, a value of $FF in RATIO represents a ratio of 1/1.
                 ref_addr1 - This is the low byte of the address where a reference 
                 			 time resides, or the address of the current TCR value.
                 ref_addr2 - A pointer into the parameter RAM map.  It points to a 
                 			 reference value that is scaled by the RATIO parameter 
                 			 to form a new OFFSET value whenever a link is received.
                 mode - Mode is used to select which host service request mode to run:
                 		0 = All code is executed
                 		1 = Only code that updates the TCR values is executed.
                 force - This is what the OC will force the output to at initialization.
                 		Force High = 1
                 		Force Low = 2
                 		Don't Force = 3
                 change - This is what the OC will do to the output after the OFFSET time.
                 		Change High = 1
                 		Change Low = 2
                 		Toggle = 3
RETURNS NOTES : none
WARNING       : none
*******************************************************************************/
void tpu_oc_continuous_update(struct TPU3_tag *tpu, UINT8 channel, UINT8 ratio, \
		UINT8 ref_addr1, UINT8 ref_addr2, UINT8 force, UINT8 change, UINT8 match_tcr ) 
{
    tpu->PARM.R[channel][TPU_OC_CHANNEL_CONTROL] = 0x80 + 0x20 * match_tcr + change + force ;
    tpu->PARM.R[channel][TPU_OC_PARAM2] = ratio * 0x100 + ref_addr1 ;
    tpu->PARM.R[channel][TPU_OC_PARAM3] = ref_addr2 * 0x100 ;
}


/*******************************************************************************
FUNCTION      : tpu_oc_read_ref_time
PURPOSE       : To read the reference time of the TCR value when the next match
				event will occur, or the TCR value of the most recent match event.
INPUTS NOTES  : This function has 2 parameters:
                 *tpu - This is a pointer to the TPU3 module to use. It is of
                         type TPU3_tag which is defined in m_tpu3.h
                 channel - This is the channel number of the primary QDEC
                           channel.
RETURNS NOTES : Function returns the reference time of the Output Compare function.
WARNING       : None
*******************************************************************************/
UINT16 tpu_oc_read_ref_time(struct TPU3_tag *tpu, UINT8 channel )
{	
    return(tpu->PARM.R[channel][TPU_OC_REF_TIME]);
}


/*******************************************************************************
FUNCTION      : tpu_oc_read_actual_match_time
PURPOSE       : ACTUAL_MATCH_TIME contains the TCR value captured at the time 
				of the last match event. This may be used by the host CPU to 
				estimate time available before the next match and to confirm 
				that a change in offset was effected as desired. The TBS field
				determines which TCR is captured and placed in this parameter. 
INPUTS NOTES  : This function has 2 parameters:
                 *tpu - This is a pointer to the TPU3 module to use. It is of
                         type TPU3_tag which is defined in m_tpu3.h
                 channel - This is the channel number of the primary QDEC
                           channel.
RETURNS NOTES : Function returns the actual match time of the function.
WARNING       : This parameter is used only in the host-initiated pulse mode.
*******************************************************************************/
UINT16 tpu_oc_read_actual_match_time(struct TPU3_tag *tpu, UINT8 channel )
{	
    return(tpu->PARM.R[channel][TPU_OC_MATCH_TIME]);
}

/*********************************************************************
 *
 * Copyright:
 *	MOTOROLA, INC. All Rights Reserved.  
 *  You are hereby granted a copyright license to use, modify, and
 *  distribute the SOFTWARE so long as this entire notice is
 *  retained without alteration in any modified and/or redistributed
 *  versions, and that such modified versions are clearly identified
 *  as such. No licenses are granted by implication, estoppel or
 *  otherwise under any patents or trademarks of Motorola, Inc. This 
 *  software is provided on an "AS IS" basis and without warranty.
 *
 *  To the maximum extent permitted by applicable law, MOTOROLA 
 *  DISCLAIMS ALL WARRANTIES WHETHER EXPRESS OR IMPLIED, INCLUDING 
 *  IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR
 *  PURPOSE AND ANY WARRANTY AGAINST INFRINGEMENT WITH REGARD TO THE 
 *  SOFTWARE (INCLUDING ANY MODIFIED VERSIONS THEREOF) AND ANY 
 *  ACCOMPANYING WRITTEN MATERIALS.
 * 
 *  To the maximum extent permitted by applicable law, IN NO EVENT
 *  SHALL MOTOROLA BE LIABLE FOR ANY DAMAGES WHATSOEVER (INCLUDING 
 *  WITHOUT LIMITATION, DAMAGES FOR LOSS OF BUSINESS PROFITS, BUSINESS 
 *  INTERRUPTION, LOSS OF BUSINESS INFORMATION, OR OTHER PECUNIARY
 *  LOSS) ARISING OF THE USE OR INABILITY TO USE THE SOFTWARE.   
 * 
 *  Motorola assumes no responsibility for the maintenance and support
 *  of this software
 ********************************************************************/
