/*--------------------------------------------------------------------------    
*
* File:  MPC8272.H
*
*
* Description:  
*
* Main Internal Memory Maps for the MPC8272 PowerQUICC 2.
*
* Note that different structures can be overlaid at the same offsets for the 
* different modes of operation.
*
*
* History:
*
* 13 JUL 98    ggh    initial version
* 15 JAN 99    ggh    updated MCC structures
* 19 Jul 05    rer    update for MPC8272 (no SEC)
*
*----------------------------------------------------------------------------*/

#ifndef _MPC8272_H 
#define _MPC8272_H

//#define _Packed __packed__
#define _Packed 
#define _PackedType

typedef unsigned long uword;
typedef unsigned char uchar;
typedef unsigned short ushort;
 

/******************************************************************************
*
*  PARAMETER RAM (PRAM) FOR EACH PERIPHERAL
*  
*  Each subsection contains protocol-specific PRAM for each peripheral,
*  followed by the PRAM common to all protocols for that peripheral.  These 
*  structs are used as needed in the main MPC8272 memory map structure.  Note 
*  that different modes of operation will require the use of different PRAM 
*  structs, and that certain structs may overlay and conflict with the use of 
*  other PRAM areas.  Consult the MPC8272 User Manual for details as to what
*  is unavailable when certain protocols are run on certain peripherals.
*
******************************************************************************/
                                  

#pragma pack(push,1)

/*---------------------------------------------------------------------------*/
/*                   SERIAL COMMUNICATION CONTROLLER (SCC)                   */
/*---------------------------------------------------------------------------*/
								
/*----------*/
/* SCC HDLC */
/*----------*/

typedef _Packed struct 
{
    uchar  reserved1[4]; 		/* Reserved area 				*/
    uword  c_mask;       		/* CRC constant 				*/
    uword  c_pres;       		/* CRC preset 					*/
    ushort disfc;       		/* discarded frame counter 		*/
    ushort crcec;      		    /* CRC error counter 			*/
    ushort abtsc;      			/* abort sequence counter 		*/
    ushort nmarc;  		        /* nonmatching address rx cnt 	*/
    ushort retrc;        		/* frame transmission counter	*/
                          		/* FCC, this area is reserved	*/
    ushort mflr;      		    /* maximum frame length reg 	*/
    ushort max_cnt;  		    /* maximum length counter 		*/
    ushort rfthr;     		    /* received frames threshold 	*/
    ushort rfcnt;     		    /* received frames count 		*/
    ushort hmask;     		    /* user defined frm addr mask 	*/
    ushort haddr1;    		    /* user defined frm address 1 	*/
    ushort haddr2;    		    /* user defined frm address 2 	*/
    ushort haddr3;    		    /* user defined frm address 3 	*/
    ushort haddr4;    		    /* user defined frm address 4 	*/
    ushort tmp;       		    /* temp 						*/
    ushort tmp_mb;     		    /* temp 						*/
}_PackedType t_HdlcScc_Pram;

 
/*--------------*/
/* SCC Ethernet */
/*--------------*/

typedef _Packed struct 
{
    uword  c_pres;      		/* CRC preset 					*/
    uword  c_mask;   		    /* CRC constant mask			*/
    uword  crcec;    		    /* CRC error counter 			*/
    uword  alec;     		    /* alignment error counter 		*/
    uword  disfc;   		    /* discarded frame counter 		*/
    ushort pads;    		    /* Short frame pad character. 	*/
    ushort ret_lim; 		    /* Retry limit threshold. 		*/
    ushort ret_cnt;  		    /* Retry limit counter. 		*/
    ushort mflr;      		    /* maximum frame length reg 	*/
    ushort minflr;    			/* minimum frame length reg 	*/
    ushort maxd1;     		    /* max DMA1 length register. 	*/
    ushort maxd2;     			/* max DMA2 length register. 	*/
    ushort maxd;       		    /* Rx max DMA. 					*/
    ushort dma_cnt;    			/* Rx DMA counter. 				*/
    ushort max_b;      			/* max BD byte count. 			*/
    ushort gaddr1;     		    /* group address filter 		*/
    ushort gaddr2;     			/* group address filter 		*/
    ushort gaddr3;    			/* group address filter 		*/
    ushort gaddr4;    		    /* group address filter 		*/
    uword  tbuf0_data0; 		/* Saved area 0, current frame. */
    uword  tbuf0_data1; 		/* Saved area 1, current frame. */
    uword  tbuf0_rba0;
    uword  tbuf0_crc;
    ushort tbuf0_bcnt;
    ushort paddr1_h;    		/* physical address (MSB) 		*/
    ushort paddr1_m;   			/* physical address 			*/
    ushort paddr1_l;   		    /* physical address (LSB) 		*/
    ushort p_per;      			/* persistence 					*/
    ushort rfbd_ptr;   			/* Rx first BD pointer. 		*/
    ushort tfbd_ptr;  		    /* Tx first BD pointer. 		*/
    ushort tlbd_ptr;    		/* Tx last BD pointer. 			*/
    uword  tbuf1_data0;			/* Saved area 0, next frame. 	*/
    uword  tbuf1_data1; 		/* Saved area 1, next frame. 	*/
    uword  tbuf1_rba0;
    uword  tbuf1_crc;
    ushort tbuf1_bcnt;
    ushort tx_len;      		/* tx frame length counter 		*/
    ushort iaddr1;      		/* individual address filter. 	*/
    ushort iaddr2;     			/* individual address filter.  	*/
    ushort iaddr3;      		/* individual address filter. 	*/
    ushort iaddr4;      		/* individual address filter.  	*/
    ushort boff_cnt;    		/* back-off counter 			*/
    ushort taddr_h;    			/* temp address (MSB) 			*/
    ushort taddr_m;   			/* temp address 				*/
    ushort taddr_l;   			/* temp address (LSB) 			*/
} _PackedType t_EnetScc_Pram;

/*----------*/
/* SCC VUART */
/*----------*/

typedef _Packed struct 
{
    uchar  reserved1[8];  		/* Reserved area 				*/
    ushort max_idl;       		/* maximum idle characters 		*/
    ushort idlc;          		/* rx idle counter (internal) 	*/
    ushort brkcr;         		/* break count register 		*/
    ushort parec;         		/* Rx parity error counter 		*/
    ushort frmec;        	    /* Rx framing error counter 	*/
    ushort nosec;        		/* Rx noise counter 			*/
    ushort brkec;        		/* Rx break character counter 	*/
    ushort brkln;        	    /* Receive break length 		*/
    ushort uaddr1;        		/* address character 1 			*/
    ushort uaddr2;       	    /* address character 2 			*/
    ushort rtemp;        		/* temp storage 				*/
    ushort toseq;          		/* Tx out of sequence char 		*/
    ushort cc[8];          		/* Rx control characters 		*/
    ushort rccm;           		/* Rx control char mask 		*/
    ushort rccr;          		/* Rx control char register 	*/
    ushort rlbc;         	    /* Receive last break char 		*/
} _PackedType t_UartScc_Pram;


/*-----------------*/
/* SCC Transparent */
/*-----------------*/

typedef _Packed struct  
{
    uword  c_mask;         		/* CRC constant 				*/
    uword  c_pres;         		/* CRC preset 					*/
} _PackedType t_TransScc_Pram;


/*------------*/
/* SCC Bisync */
/*------------*/

typedef _Packed struct  
{
    uchar  reserved1[4];  		/* Reserved area 				*/
    uword  crcc;          		/* CRC Constant Temp Value 		*/
    ushort prcrc;         		/* Preset Receiver CRC-16/LRC 	*/
    ushort ptcrc;          		/* Preset Transmitter CRC-16/LRC*/
    ushort parec;          		/* Receive Parity Error Counter */
    ushort bsync;          		/* BISYNC SYNC Character 		*/
    ushort bdle;           		/* BISYNC DLE Character 		*/
    ushort cc[8];          		/* Rx control characters 	    */
    ushort rccm;           		/* Rx Control Character Mask	*/
} _PackedType t_BisyncScc_Pram;


/*-----------------*/
/* SCC Common PRAM */
/*-----------------*/

typedef _Packed struct 
{
    ushort rbase;      			/* RX BD base address 			*/
    ushort tbase;      			/* TX BD base address 			*/
    uchar  rfcr;       			/* Rx function code 			*/
    uchar  tfcr;       			/* Tx function code 			*/
    ushort mrblr;      			/* Rx buffer length 			*/
    uword  rstate;     			/* Rx internal state 			*/
    uword  rptr;       			/* Rx internal data pointer 	*/
    ushort rbptr;      			/* rb BD Pointer 				*/
    ushort rcount;     			/* Rx internal byte count 		*/
    uword  rtemp;      			/* Rx temp 						*/
    uword  tstate;     			/* Tx internal state 			*/
    uword  tptr;       			/* Tx internal data pointer 	*/
    ushort tbptr;      			/* Tx BD pointer 				*/
    ushort tcount;     			/* Tx byte count 				*/
    uword  ttemp;      			/* Tx temp 						*/
    uword  rcrc;       			/* temp receive CRC 			*/
    uword  tcrc;       			/* temp transmit CRC 			*/
    union 
    {
        t_HdlcScc_Pram    h;
        t_EnetScc_Pram    e;
        t_UartScc_Pram    u;
        t_TransScc_Pram   t;
        t_BisyncScc_Pram  b;
    } SpecificProtocol;
    uchar COMPLETE_SIZE_OF_DPRAM_PAGE[0x5c];
} _PackedType t_Scc_Pram;



/*---------------------------------------------------------------------------*/
/*                      FAST COMMUNICATION CONTROLLER (FCC)                  */
/*---------------------------------------------------------------------------*/

/*----------*/
/* FCC HDLC */
/*----------*/

typedef _Packed struct 
{
    uchar  reserved1[8];   		/* Reserved area 				*/
    uword  c_mask;         		/* CRC constant 				*/
    uword  c_pres;         		/* CRC preset 					*/
    ushort disfc;          		/* discarded frame counter 		*/
    ushort crcec;          		/* CRC error counter 			*/
    ushort abtsc;          		/* abort sequence counter 		*/
    ushort nmarc;          		/* nonmatching address rx cnt 	*/
    uword  max_cnt;        		/* maximum length counter 		*/
    ushort mflr;           		/* maximum frame length reg 	*/
    ushort rfthr;          		/* received frames threshold 	*/
    ushort rfcnt;          		/* received frames count 		*/
    ushort hmask;          		/* user defined frm addr mask 	*/
    ushort haddr1;         		/* user defined frm address 1 	*/
    ushort haddr2;         		/* user defined frm address 2 	*/
    ushort haddr3;         		/* user defined frm address 3 	*/
    ushort haddr4;         		/* user defined frm address 4 	*/
    ushort tmp;            		/* temp 						*/
    ushort tmp_mb;         		/* temp 						*/
} _PackedType t_HdlcFcc_Pram;


/*--------------*/
/* FEC Ethernet */
/*--------------*/

typedef _Packed struct 
{
    uword  stat_buf;       		/* Internal use buffer. 		*/
    uword  cam_ptr;        		/* CAM address. 				*/
    uword  c_mask;         		/* CRC constant mask			*/
    uword  c_pres;         		/* CRC preset 					*/
    uword  crcec;          		/* CRC error counter 			*/
    uword  alec;           		/* alignment error counter 		*/
    uword  disfc;          		/* discarded frame counter 		*/
    ushort ret_lim;        		/* Retry limit threshold. 		*/
    ushort ret_cnt;        		/* Retry limit counter. 		*/
    ushort p_per;          		/* persistence 					*/
    ushort boff_cnt;       		/* back-off counter 			*/
    uword  gaddr_h;        		/* group address filter, high 	*/
    uword  gaddr_l;        		/* group address filter, low 	*/
    ushort tfcstat;        		/* out of sequece Tx BD staus. 	*/
    ushort tfclen;         		/* out of sequece Tx BD length. */
    uword  tfcptr;         		/* out of sequece Tx BD data pointer. */
    ushort mflr;           		/* maximum frame length reg 	*/
    ushort paddr1_h;       		/* physical address (MSB) 		*/
    ushort paddr1_m;       		/* physical address 			*/
    ushort paddr1_l;       		/* physical address (LSB) 		*/
    ushort ibd_cnt;        		/* internal BD counter. 		*/
    ushort ibd_start;      		/* internal BD start pointer. 	*/
    ushort ibd_end;        		/* internal BD end pointer. 	*/
    ushort tx_len;         		/* tx frame length counter 		*/
    uchar  ibd_base[0x20]; 		/* internal micro code usage. 	*/
    uword  iaddr_h;        		/* individual address filter, high 	*/
    uword  iaddr_l;        		/* individual address filter, low 	*/
    ushort minflr;         		/* minimum frame length reg 	*/
    ushort taddr_h;        		/* temp address (MSB) 			*/
    ushort taddr_m;        		/* temp address 				*/
    ushort taddr_l;        		/* temp address (LSB) 			*/
    ushort pad_ptr;        		/* pad_ptr. 					*/
    ushort cf_type;        		/* RESERVED (flow control frame type coding) */
    ushort cf_range;       		/* flow control frame range. 	*/
    ushort max_b;          		/* max buffer descriptor byte count. */
    ushort maxd1;          		/* max DMA1 length register. 	*/
    ushort maxd2;          		/* max DMA2 length register. 	*/
    ushort maxd;           		/* Rx max DMA. 					*/
    ushort dma_cnt;        		/* Rx DMA counter. 				*/
    
    /* counter: */
    uword  octc;           		/* received octets counter. 	*/
    uword  colc;           		/* estimated number of collisions */
    uword  broc;           		/* received good packets of broadcast address 	*/
    uword  mulc;           		/* received good packets of multicast address 	*/
    uword  uspc;           		/* received packets shorter then 64 octets. 	*/
    uword  frgc;           		/* as uspc + bad packets 		*/
    uword  ospc;           		/* received packets longer then 1518 octets. 	*/
    uword  jbrc;           		/* as ospc + bad packets  		*/
    uword  p64c;           		/* received packets of 64 octets*/
    uword  p65c;           		/* received packets of 65-128 octets. 	*/
    uword  p128c;          		/* received packets of 128-255 octets. 	*/
    uword  p256c;          		/* received packets of 256-511 octets. 	*/
    uword  p512c;          		/* received packets of 512-1023 octets. */
    uword  p1024c;         		/* received packets of 1024-1518 octets.*/
    uword  cam_buf;        		/* cam respond internal buffer. */
    uword  reserved0;	   		/* reserved, should be cleared  */
} _PackedType t_EnetFcc_Pram;


/*-----------------*/
/* FCC Common PRAM */
/*-----------------*/

typedef _Packed struct 
{
    ushort riptr;          		/* Rx internal temporary data pointer. */
    ushort tiptr;          		/* Tx internal temporary data pointer. */
    ushort reserved0;      		/* Reserved 					*/
    ushort mrblr;          		/* Rx buffer length 			*/
    uword  rstate;         		/* Rx internal state 			*/
    uword  rbase;          		/* RX BD base address 			*/
    ushort rbdstat;        		/* Rx BD status and control 	*/
    ushort rbdlen;         		/* Rx BD data length 			*/
    uword  rdptr;          		/* rx BD data pointer 			*/
    uword  tstate;         		/* Tx internal state 			*/
    uword  tbase;          		/* TX BD base address 			*/
    ushort tbdstat;        		/* Tx BD status and control 	*/
    ushort tbdlen;         		/* Tx BD data length 			*/
    uword  tdptr;          		/* Tx  data pointer 			*/
    uword  rbptr;          		/* rx BD pointer 				*/
    uword  tbptr;          		/* Tx BD pointer 				*/
    uword  rcrc;           		/* Temp receive CRC 			*/
    uword  reserved_1[0x1];
    uword  tcrc;           		/* Temp transmit CRC 			*/
    union                   	/* Protocol-Specific parameter ram */
    {
        t_HdlcFcc_Pram    h;
        t_EnetFcc_Pram    e;
    } SpecificProtocol;      
} _PackedType t_Fcc_Pram;



/*---------------------------------------------------------------------------*/
/*                  UNIVERSAL SERIAL BUS (USB)					             */
/*---------------------------------------------------------------------------*/


/*------------------------*/
/* USB Host/Function PRAM */
/*------------------------*/

typedef _Packed struct
{
    ushort  ep1ptr;         	/* Endpoint 1 pointer register 	*/ 
    ushort  ep2ptr;         	/* Endpoint 2 pointer register 	*/
    ushort  ep3ptr;         	/* Endpoint 3 pointer register 	*/
    ushort  ep4ptr;         	/* Endpoint 4 pointer register 	*/
    uword   rstate;         	/* Rx internal state           	*/
    uword   rptr;           	/* Rx internal data pointer    	*/
    ushort  frame_n;        	/* Frame number                	*/
    ushort  rbcnt;          	/* Rx internal byte count      	*/
    uword   rtemp;          	/* Rx temp 					  	*/
    uword   rxusb_data;     	/* Rx data temp 			   	*/
    ushort  rxuptr;				/* Rx ucode return address temp	*/            
    
} _PackedType t_USB_Pram;


/*------------------*/
/* Endpoint PRAM	*/
/*------------------*/

typedef _Packed struct
{
    ushort rbase;          		/* Rx base address          	*/
    ushort tbase;          		/* Tx base address          	*/
    uchar  rfcr;           		/* Rx function code         	*/
    uchar  tfcr;           		/* Tx function code         	*/
    ushort mrblr;		   		/* Max. Rx buffer length    	*/
    ushort rbptr;		   		/* RxBD pointer			   		*/
    ushort tbptr;		   		/* TxBD pointer 		       	*/
    uword  tstate;		   		/* Tx internal state        	*/
    uword  tptr; 		   		/* Tx internal data pointer 	*/
    ushort tcrc;		   		/* Tx temp CRC 		       		*/
    ushort tbcnt;		   		/* Tx internal byte count   	*/
    uword  ttemp;		   		/* Tx temp				   		*/
    ushort txusbu_ptr;	   		/* Tx ucode return add temp 	*/
    ushort himmr;		   		/* IMMR val (transaction-   	*/
    					   		/* level interface only)	   	*/
} _PackedType t_endpt_Pram;


/*----------------------------------------------------------------------*/
/* 					QUICC MULTICHANNEL CONTROLLER (QMC)					*/
/*----------------------------------------------------------------------*/

/******************************************************************************
* Note that each QMC uses multiple logical channels.  We first define the     *
* PRAM for a logical channel (which can be used in either HDLC or Transparent *
* mode;  wherever there are differences, it is specified), followed by the    *
* PRAM for an QMC itself.                                                     *
******************************************************************************/

/*---------------------*/
/* QMC Logical Channel */
/*---------------------*/

typedef _Packed struct
{
	ushort tbase;		    	/* TxBD base address 		     */
	ushort chamr;		    	/* Channel mode register		 */
	uword  tstate;				/* Tx internal state			 */
	uword  reserved0;			/* Reserved area				 */
	ushort tbptr;				/* TxBD pointer					 */
	ushort reserved1;			/* Reserved						 */
	uword  tupack;				/* Tx temp for unpacking		 */
	uword  zistate;				/* Zero-insertion state			 */
	uword  tcrc;				/* HDLC:  Temp Tx CRC			 */
								/* TRANS:  Reserved				 */
	ushort intmsk;				/* Channel's interrupt mask flag */
	ushort bdflags;				/* Temp							 */
	ushort rbase;				/* RxBD offset					 */
	ushort mflr;				/* HDLC:  Max frame length register*/
								/* TRANS:  Max Rx buffer length	 */
	uword  rstate;				/* Rx internal state  			 */
	uword  reserved2;			/* Reserved						 */
	ushort rbptr;				/* RxBD pointer					 */
	ushort reserved3;			/* Reserved						 */
	uword  rpack;				/* Rx temp for packing			 */
	uword  zdstate;				/* Zero deletion machine state	 */
	uword  rcrc;				/* HDLC:  Temp Rx CRC			 */
								/* TRANS:  Reserved				 */
	ushort max_cnt;				/* HDLC:  Max length counter	 */
								/* TRANS:  Transparent SYNC		 */
	ushort tmp_mp;				/* HDLC:  Temp					 */
								/* TRANS:  Resevered			 */
} _PackedType t_Qch_Pram;

/*-------------------*/
/* Global PRAM       */
/*-------------------*/
typedef _Packed struct
{
    uword  mcbase;         		/* A pointer to starting address of BD rings. 	*/
    ushort qmcstate;       		/* Controller state. 				*/
    ushort mrblr;          		/* Maximum receive buffer length. 	*/
    ushort tx_s_ptr;       		/* Tx time slot assignment table pointer 		*/
    ushort rxptr;          		/* Rx pointer 						*/
    ushort grfthr;         		/* Global Rx frame threshold 		*/                            
    ushort grfcnt;         		/* Global Rx frame count 			*/
    uword  intbase;        		/* Multichannel interrupt base pointer 			*/
    uword  intptr;         		/* Multichannel interrupt next pointer 			*/
    ushort rx_s_ptr;       		/* Rx time slot assignment table pointer 		*/ 
    ushort txptr;          		/* Points to current transmitter time slot 		*/
    uword  c_mask32;       		/* CRC constant 					*/
    ushort tsatrx[0x20];   		/* Time slot assignment table Rx 	*/
    ushort tsattx[0x20];   		/* Time slot assignment table Tx 	*/
    ushort c_mask16;       		/* CRC constant. 					*/
    uword  temp_rba;       		/* Temp Rx buffer address 			*/                            
    uword  temp_crc;       		/* Temp CRC 						*/
    ushort rx_frm_base;    		/* Temp location for Rx framer parameters 		*/
    ushort tx_frm_base;    		/* Temp location for Tx framer parameters 		*/
    
} _PackedType t_qmc_glob_Pram;

/*---------------------------------------------------------------------------*/
/*                              ATM PARAMETER RAM                            */
/*---------------------------------------------------------------------------*/


/*--------------------------------------*/
/* Address Compression parameters table */
/*--------------------------------------*/

_Packed struct AddressCompressionPram 
{
    uword  VptBase;        		/* VP-level addressing table base address 	*/
    uword  VctBase;        		/* VC-level addressing table base address 	*/
    uword  Vpt1Base;       		/* VP1-level addressing table base address 	*/ 
    uword  Vct1Base;       		/* VC1-level addressing table base address 	*/
    ushort VpMask;         		/* VP mask for address compression look-up 	*/
} _PackedType;


/*-------------------------------*/
/* External CAM parameters table */
/*-------------------------------*/

_Packed struct ExtCamPram 
{
    uword  ExtCamBase;     		/* Base address of the external CAM 		*/
    uchar  reserved00[4];  		/* Reserved 								*/
    uword  ExtCam1Base;    		/* Base address of the external CAM1 		*/
    uchar  reserved01[6];  		/* Reserved 								*/
} _PackedType;


/*---------------------------*/
/* ATM mode parameters table */
/*---------------------------*/

typedef _Packed struct AtmPram 
{
    uchar  reserved0[64];  		/* Reserved 								*/
    ushort RxCellTmpBase;  		/* Rx cell temporary base address 			*/
    ushort TxCellTmpBase;  		/* Tx cell temporary base address 			*/
    ushort VUdcTmpBase;    		/* VUDC temp base address (in VUDC mode only) */
    ushort IntRctBase;     		/* Internal RTC base address 				*/
    ushort IntTctBase;     		/* Internal TCT base address 				*/
    ushort IntTcteBase;    		/* Internal ACT base address 				*/
    uchar  reserved1[4];   		/* reserved four bytes 						*/
    uword  ExtRctBase;     		/* Extrnal RTC base address 				*/
    uword  ExtTctBase;     		/* Extrnal TCT base address 				*/
    uword  ExtTcteBase;    		/* Extrnal ACT base address 				*/
    ushort VUeadOffset;    		/* The offset in half-wordunits of the VUEAD
                               		entry in the VUDC extra header. Should be
                              		even address. If little-endian format is
                               		used, the VUeadOffset is of the little-endian
                               		format. 								*/
    uchar  reserved2[2];   		/* Reserved 								*/
    ushort PmtBase;        		/* Performance monitoring table base address*/
    ushort ApcParamBase;   		/* APC Parameters table base address 		*/
    ushort FbpParamBase;   		/* Free buffer pool parameters base address */
    ushort IntQParamBase;  		/* Interrupt queue parameters table base 	*/
    uchar  reserved3[2];
    ushort VUniStatTableBase; 	/* VUNI statistics table base 				*/
    uword  BdBaseExt;      		/* BD ring base address extension 			*/
    union 
    {
        struct AddressCompressionPram   AddrCompression;
        struct ExtCamPram               ExtCam;
    } AddrMapping;          	/* Address look-up mechanism 				*/
    ushort VciFiltering;   		/* VCI filtering enable bits. If bit i is set,
                               		the cell with VCI=i will be sent to the
                               		raw cell queue. The bits 0-2 and 5 should
                               		be zero. 								*/
    ushort Gmode;          		/* Global mode 								*/
    ushort CommInfo1;      		/* The information field associated with the*/
    uword  CommInfo2;      		/* last host command 						*/
    uchar  reserved4[4];   		/* Reserved 								*/
    uword  CRC32Preset;    		/* Preset for CRC32 						*/
    uword  CRC32Mask;      		/* Constant mask for CRC32 					*/
    ushort AAL1SnpTableBase; 	/* AAl1 SNP protection look-up table base 	*/
    ushort reserved5;      		/* Reserved 								*/
    uword  SrtsBase;       		/* External SRTS logic base address. For AAL1
                               		only. Should be 16 bytes aligned 		*/
    ushort IdleBase;       		/* Idle cell base address 					*/
    ushort IdleSize;       		/* Idle cell size: 52, 56, 60, 64 			*/
    uword  EmptyCellPayload; 	/* Empty cell payload (little-indian) 		*/
    
    /* ABR specific only */
    uword  Trm; 				/* VUpper bound on time between F-RM cells for active source 	*/                                 
    ushort Nrm; 				/* Controls the maximum data cells sent for each F-RM cell. 	*/                           
    ushort Mrm;					/* Controls bandwidth between F-RM, B-RM and user data cell 	*/
    ushort Tcr;            		/* Tag cell rate 							*/
    ushort AbrRxTcte;      		/* ABR reserved area address (2-UHWORD aligned)*/
    uchar  reserved7[76];  		/* Reserved 								*/
} _PackedType t_Atm_Pram;


/*----------------------------*/
/* AAL2						  */
/*----------------------------*/

typedef _Packed struct
{
	uchar  reserved0[64];		/* Reserved area								*/
	ushort rcell_tmp_base;		/* Rx cell temp address							*/
	ushort tcell_tmp_base;		/* Tx cell temp address							*/
	ushort udc_tmp_base;		/* UDC mode only								*/
	ushort int_rct_base;		/* Internal Rx connection table base			*/
	ushort int_tct_base;		/* Internal Tx connection table base			*/
	ushort int_tcte_base;		/* Internal Tx connection table extension base	*/
	uword  ras_timer_dur;		/* RAS timer duration in useconds				*/
	uword  ext_rct_base;		/* External Rx connection table	base			*/
	uword  ext_tct_base; 		/* External Tx connection table	base			*/
	uword  ext_tcte_base;		/* External Tx connection table extension base  */
	ushort uead_offest;			/* Offset of UEAD entry in UDC extra header		*/
	ushort rxqd_base;			/* Pointer to base address of internal RxQD 	*/
	ushort pmt_base;			/* Performance monitoring table					*/
	ushort apcp_base;			/* APC parameters table base 					*/
	ushort fbt_base;			/* Free buffer pool parameters table base		*/
	ushort intt_base;			/* Interrupt queue parameters table base		*/
	ushort reserved1;			/* Reserved; should be cleared during init		*/
	ushort uni_statt_base;		/* UNI statistics table base					*/
	uword  bd_base_ext;			/* BD table base address extension				*/
	uword  vpt_base;			/* VP table/external CAM base address			*/
	uword  vct_base;			/* VC table base address						*/
	uword  vpt1_base;			/* VP1/EXT CAM1 base address					*/
	uword  vct1_base;			/* VC1 base address								*/
	ushort vp_mask;				/* VP mask for address compression lookup		*/
	ushort vci_filter;			/* VCI filtering enable bits					*/
	ushort gmode;				/* Global mode									*/
	ushort comm_info[3];		/* Information field for last host command		*/
	uword  reserved2;			/* Reserved; clear during init					*/
	uword  crc32_pres;			/* Preset for CRC32								*/
	uword  crc32_mask;			/* Constant mask for CRC32						*/
	ushort aal1_snpt_base;		/* AAL1 SN protection lookup table base			*/
	ushort reserved3;			/* Reserved; clear during init					*/
	uword  srts_base;			/* External SRTS logic base						*/
	ushort idle_base;			/* Idle/unassign cell base						*/
	ushort idle_size;			/* Idle/unassign cell size						*/
	uword  epayload;			/* Reserved payload; init to 0x6A6A6A6A			*/
	uword  trm;					/* Upper bound on time between F-RM cells		*/
	ushort nrm;					/* Controls max cells source may transmit		*/
	ushort mrm;					/* Controls BW between F-RM, B-RM cells			*/
	ushort tcr;					/* Tag cell rate								*/
	ushort abr_rx_tcte;			/* Pointer to 16B reserved DPRAM area			*/
	uword  rxqd_base_ext;		/* Points to base address of ext RxQD table		*/
	uword  rx_udc_base;			/* Points to base of RxUDC header table			*/
	uword  tx_udc_base;			/* Points to base of TxUDC header table 		*/
	uchar  reserved4[16];		/* Reserved; clear at init						*/
	uword  tcell_tmp_ext;		/* Tx cell temp base							*/
	uchar  reserved5[12];		/* Reserved; clear at init						*/
	ushort pad_tmp_base;		/* PAD template base 							*/
	uchar  reserved6[28];		/* Reserved; clear at init						*/
		
}_PackedType t_AAL2_Pram;


/*---------------------------------------------------------------------------*/
/*                      SERIAL MANAGEMENT CHANNEL  (SMC)                     */
/*---------------------------------------------------------------------------*/

typedef _Packed struct  
{
    ushort rbase;          		/* Rx BD Base Address 							*/
    ushort tbase;          		/* Tx BD Base Address 							*/
    uchar  rfcr;           		/* Rx function code 							*/
    uchar  tfcr;           		/* Tx function code 							*/
    ushort mrblr;          		/* Rx buffer length 							*/
    uword  rstate;         		/* Rx internal state 							*/
    uword  rptr;           		/* Rx internal data pointer 					*/
    ushort rbptr;          		/* rb BD Pointer 								*/
    ushort rcount;        		/* Rx internal byte count 						*/
    uword  rtemp;          		/* Rx temp 										*/
    uword  tstate;         		/* Tx internal state 							*/
    uword  tptr;           		/* Tx internal data pointer 					*/
    ushort tbptr;          		/* Tx BD pointer 								*/
    ushort tcount;         		/* Tx byte count 								*/
    uword  ttemp;          		/* Tx temp 										*/
    
    /* SMC VUART-specific PRAM */
    ushort max_idl;        		/* Maximum IDLE Characters 						*/
    ushort idlc;           		/* Temporary IDLE Counter 						*/
    ushort brkln;          		/* Last Rx Break Length 						*/
    ushort brkec;          		/* Rx Break Condition Counter 					*/
    ushort brkcr;          		/* Break Count Register (Tx) 					*/
    ushort r_mask;         		/* Temporary bit mask 							*/  
    uword  reserved;	   		/* SDMA temp				  					*/
} _PackedType t_Smc_Pram;



/*---------------------------------------------------------------------------*/
/*                            IDMA PARAMETER RAM                             */
/*---------------------------------------------------------------------------*/

typedef _Packed struct  
{
    ushort ibase;          		/* IDMA BD Base Address 						*/
    ushort dcm;            		/* DMA channel mode register 					*/
    ushort ibdptr;         		/* next bd ptr 									*/
    ushort DPR_buf;        		/* ptr to internal 64 byte buffer 				*/
    ushort BUF_inv;        		/* The quantity of data in DPR_buf 				*/
    ushort SS_max;         		/* Steady State Max. transfer size 				*/
    ushort DPR_in_ptr;     		/* write ptr for the internal buffer 			*/
    ushort sts;           		/* Source Transfer Size 						*/
    ushort DPR_out_ptr;   		/* read ptr for the internal buffer 			*/
    ushort seob;          		/* Source end of burst 							*/
    ushort deob;          		/* Destination end of burst 					*/
    ushort dts;            		/* Destination Transfer Size 					*/
    ushort RetAdd;         		/* return address when ERM==1 					*/
    ushort Reserved;      		/* reserved 									*/
    uword  BD_cnt;        		/* Internal byte count 							*/
    uword  S_ptr;         		/* source internal data ptr 					*/
    uword  D_ptr;         		/* destination internal data ptr 				*/
    uword  istate;         		/* Internal state 								*/
} _PackedType t_Idma_Pram;



/*-------------------------------------------------------------------*/
/*                    INTER-INTEGRATED CIRCUIT  (I2C)                */
/*-------------------------------------------------------------------*/

typedef _Packed struct 
{
    ushort rbase;          		/* RX BD base address 							*/
    ushort tbase;          		/* TX BD base address 							*/
    uchar  rfcr;           		/* Rx function code 							*/
    uchar  tfcr;           		/* Tx function code 							*/
    ushort mrblr;          		/* Rx buffer length 							*/
    uword  rstate;         		/* Rx internal state 							*/
    uword  rptr;           		/* Rx internal data pointer 					*/
    ushort rbptr;          		/* rb BD Pointer 								*/
    ushort rcount;         		/* Rx internal byte count 						*/
    uword  rtemp;          		/* Rx temp 										*/
    uword  tstate;         		/* Tx internal state 							*/
    uword  tptr;           		/* Tx internal data pointer 					*/
    ushort tbptr;          		/* Tx BD pointer 								*/
    ushort tcount;         		/* Tx byte count 								*/
    uword  ttemp;          		/* Tx temp 										*/
    uword  sdmatmp;		   		/* SDMA temp 									*/
} _PackedType t_I2c_Pram;



/*---------------------------------------------------------------------------*/
/*                       SERIAL PERIPHERAL INTERFACE  (SPI)                  */
/*---------------------------------------------------------------------------*/

typedef _Packed struct  
{
    ushort rbase;          		/* Rx BD Base Address 							*/
    ushort tbase;          		/* Tx BD Base Address 							*/
    uchar  rfcr;           		/* Rx function code 							*/
    uchar  tfcr;           		/* Tx function code 							*/
    ushort mrblr;          		/* Rx buffer length 							*/
    uword  rstate;         		/* Rx internal state 							*/
    uword  rptr;           		/* Rx internal data pointer 					*/
    ushort rbptr;          		/* Rx BD Pointer 								*/
    ushort rcount;         		/* Rx internal byte count 						*/
    uword  rtemp;          		/* Rx temp 										*/
    uword  tstate;         		/* Tx internal state 							*/
    uword  tptr;           		/* Tx internal data pointer 					*/
    ushort tbptr;          		/* Tx BD pointer 								*/
    ushort tcount;         		/* Tx byte count 								*/
    uword  ttemp;          		/* Tx temp 										*/
    uword  reserved;	   		/* SDMA temp        							*/
} _PackedType t_Spi_Pram;



/*---------------------------------------------------------------------------*/
/*                      RISC TIMER PARAMETER RAM                             */
/*---------------------------------------------------------------------------*/

typedef _Packed struct  
{
 
    ushort tm_base;        		/* RISC timer table base adr 					*/
    ushort tm_ptr;         		/* RISC timer table pointer 					*/
    ushort r_tmr;          		/* RISC timer mode register 					*/
    ushort r_tmv;          		/* RISC timer valid register 					*/
    uword  tm_cmd;         		/* RISC timer cmd register 						*/
    uword  tm_cnt;         		/* RISC timer internal cnt 						*/
} _PackedType t_timer_pram;



/*--------------------------------------------------------------------------*/
/*                  ROM MICROCODE PARAMETER RAM AREA                        */
/*--------------------------------------------------------------------------*/

typedef _Packed struct  
{
    ushort rev_num;        		/* VUcode Revision Number 						*/
    ushort d_ptr;          		/* MISC Dump area pointer 						*/
} _PackedType t_ucode_pram;




/*--------------------------------------------------------------------------*/
/*                MAIN DEFINITION OF MPC8272 INTERNAL MEMORY MAP            */
/*--------------------------------------------------------------------------*/

typedef _Packed struct 
{

/* cpm_ram */
    t_Qch_Pram  qmc_pram[128];  	 /* QMC logical channels parameter ram 		*/
    uchar      reserved0[0x6000];    /* Reserved area 							*/
    
/* DPR_BASE+0x8000*/

/*---------------------------------------------------------------------*/
/* A note about the pram union:                                        */
/* The pram area has been broken out three ways for clean access into  */
/* certain peripherals' spaces.  This arrangement allows programmers   */
/* flexibility of usage in terms of being able to change which         */
/* peripheral is being accessed by simply changing an array value.     */
/* Given the interweaving of certain peripherals' pram areas, this     */
/* would not be possible with only one large pram structure.           */
/*                                                                     */
/* SERIALS  - For accessing SCC, non-ATM FCC, and MCC pram             */
/* ATM      - For accessing ATM FCC pram                               */
/* STANDARD - For accessing timers, revnum, d_ptr, RAND, and the pram  */
/*            base pointers of the SMCs, IDMAs, SPI, and I2C           */
/*---------------------------------------------------------------------*/
           
    union 
    {
    
    /*for access to the PRAM structs for SCCs, FCCs, and QMC */ 
        struct serials 
        {
            t_Scc_Pram scc_pram[4];		/* Note there is no SCC2 				*/
            t_Fcc_Pram fcc_pram[2];
            uchar reserved1[0x900];
        } serials;
        
    /* for access to ATM PRAM structs */
        struct atm
        {
            uchar reserved2[0x400];
            t_Atm_Pram atm_pram[2];
            uchar reserved3[0xa00];
        } atm;
        
    /* for access to the memory locations holding user-defined 
       base addresses of PRAM for SMCs, IDMA, SPI, and I2C. */     
       struct standard
       {
            uchar scc1[0x100];
            uchar reserved0[0x100];
            uchar scc3[0x100];
            uchar scc4[0x100];
            uchar fcc1[0x100];
            uchar fcc2[0x100];
            uchar reserved1[0x100];
            uchar reserved2[0x80];
            uchar reserved3[0x7c];
            uchar smc1[0x2];
            uchar reserved4[0x2];
            uchar reserved5[0x80];
            uchar reserved6[0x7c];
            uchar smc2[0x2];
            uchar idma2[0x2];
            uchar reserved7[0xfc];
            uchar spi[0x2];
            uchar idma3[0x2];
            uchar reserved8[0xe0];
            uchar timers[0x10];
            uchar Rev_num[0x2];
            uchar D_ptr[0x2];
            uchar reserved9[0x4];
            uchar rand[0x4];
            uchar i2c[0x2];
            uchar reserved10[0x2];
            uchar usb[0x100];
            uchar reserved11[0x400];
        } standard;
        
    } pram;
    
    uchar  cpm_ram_dpram_2[0x1000];
    uchar  reserved12[0x6000];      /* Reserved area 							*/
 
/* siu */
    uword  siu_siumcr;         		/* SIU Module Configuration Register 		*/
    uword  siu_sypcr;          		/* System Protection Control Register 		*/
    uchar  reserved13[0x6];    		/* Reserved area 							*/
    ushort siu_swsr;           		/* Software Service Register 				*/

/* buses */
    uchar  reserved14[0x14];   		/* Reserved area 							*/
    uword  bcr;                		/* Bus Configuration Register 				*/
    uchar  ppc_acr;            		/* Arbiter Configuration Register 			*/
    uchar  reserved15[0x3];    		/* Reserved area 							*/
    uword  ppc_alrh;           		/* Arbitration Level Reg. (First clients)	*/
    uword  ppc_alrl;           		/* Arbitration Level Reg. (Next clients) 	*/
    uchar  lcl_acr;            		/* LCL Arbiter Configuration Register 		*/
    uchar  reserved16[0x3];    		/* Reserved area 							*/
    uword  lcl_alrh;      	   		/* LCL Arbitration Level Reg. (First clients) 		*/
    uword  lcl_alrl;           		/* LCL Arbitration Level Register (Next clients) 	*/
    uword  tescr1;             		/* PPC bus transfer error status control reg. 1 	*/
    uword  tescr2;             		/* PPC bus transfer error status control reg. 2 	*/
    uword  reserved17[0x2];    		/* Reserved									*/
    uword  pdtea;              		/* PPC bus DMA Transfer Error Address 		*/
    uchar  pdtem;              		/* PPC bus DMA Transfer Error MSNUM  		*/
    uchar  reserved18[0xAB];   		/* Reserved area 							*/

/* memc */
    struct memc_regs 
    {
        uword br;              		/* Base Register 							*/
        uword or;              		/* Option Register 							*/
    } memc_regs[8];
    uchar  reserved19[0x28];   		/* Reserved area 							*/
    uword  memc_mar;           		/* Memory Address Register 					*/
    uchar  reserved20[0x4];    		/* Reserved area 							*/
    uword  memc_mamr;          		/* Machine A Mode Register 					*/
    uword  memc_mbmr;          		/* Machine B Mode Register 					*/
    uword  memc_mcmr;          		/* Machine C Mode Register 					*/
    uword  memc_mdmr;          		/* Machine D Mode Register 					*/
    uchar  reserved21[0x4];    		/* Reserved area 							*/
    ushort memc_mptpr;         		/* Memory Periodic Timer Prescaler 			*/
    uchar  reserved22[0x2];    		/* Reserved area 							*/
    uword  memc_mdr;           		/* Memory Data Register 					*/
    uchar  reserved23[0x4];    		/* Reserved area 							*/
    uword  memc_psdmr;         		/* PowerPC Bus SDRAM machine Mode Register 	*/
    uword  reserved24;         		/* Reserved								 	*/
    uchar  memc_purt;          		/* PowerPC Bus assigned VUPM Refresh Timer 	*/
    uchar  reserved25[0x3];    		/* Reserved area 							*/
    uchar  memc_psrt;          		/* PowerPC Bus assigned SDRAM Refresh Timer */
    uchar  reserved26[0xB];   		/* Reserved area 							*/
    uword  memc_immr;          		/* Internal Memory Map Register 			*/

/* pci */
    uword  pcibr0;             		/* Base address+valid for  window 1 		*/
    uword  pcibr1;             		/* Base address+valid for  window 2 		*/
    uword  secbr;			   		/* SEC address base register			 	*/
    uword  reserved27; 		   		/* Reserved area						 	*/
    uword  secmr;			   		/* SEC mask register					 	*/
    uword  reserved28;		   		/* Reserved 							 	*/
    uword  pcimsk0;            		/* Mask for  window 1 						*/
    uword  pcimsk1;            		/* Mask for  window 2 						*/
    uchar  reserved29[0x54];   		/* Reserved area 							*/
    

/* si_timers */
    ushort si_timers_tmcntsc;  		/* Time Counter Status and Control Register */
    uchar  reserved30[0x2];    		/* Reserved area 							*/
    uword  si_timers_tmcnt;    		/* Time Counter Register 					*/
    uword  si_timers_tmcntsec; 		/* Time Counter Seconds						*/
    uword  si_timers_tmcntal;  		/* Time Counter Alarm Register 				*/
    uchar  reserved31[0x10];   		/* Reserved area 							*/
    ushort si_timers_piscr;    		/* Periodic Interrupt Status and Control Reg*/
    uchar  reserved32[0x2];    		/* Reserved area 							*/
    uword  si_timers_pitc;     		/* Periodic Interrupt Count Register 		*/
    uword  si_timers_pitr;     		/* Periodic Interrupt Timer Register 		*/
    uchar  reserved33[0x1E4];  		/* Reserved area 							*/

/*  pci registers  */
	uword  omisr; 			   		/* Outbound interrupt status register  		*/
	uword  omimr;			   		/* Outbound interrupt mask register    		*/
	uword  reserved34[0x2];	   		/* Reserved area						  	*/
	uword  ifqpr;			  		/* Inbound FIFO queue port register    		*/
	uword  ofqpr;			   		/* Outbound FIFO queue port register   		*/
	uword  reserved35[0x2];	   		/* Reserved area 					  		*/
	uword  imr0;			   		/* Inbound message register 0 		  		*/
	uword  imr1; 			   		/* Inbound message register 1 		  		*/
	uword  omr0;			   		/* Outbound message register 0 		  		*/
	uword  omr1; 			   		/* Outbound message register 1 		  		*/
	uword  odr;				   		/* Outbound doorbell register		  		*/
	uword  reserved36;		   		/* Reserved area						  	*/
	uword  idr;				   		/* Inbound doorbell register			  	*/
	uchar  reserved37[0x14];	 	/* Reserved area					      	*/
	uword  imisr;			   		/* Inbound message interrupt status	  		*/
	uword  imimr;			   		/* Inbound message interrupt mask	  		*/
	uword  reserved38[0x6];	   		/* Reserved area						  	*/
	uword  ifhpr;			   		/* Inbound free_FIFO head pointer	  		*/
	uword  reserved39;		   		/* Reserved area						  	*/
	uword  iftpr;			   		/* Inbound free_FIFO tail pointer	  		*/
	uword  reserved40;		   		/* Reserved area 					  		*/
	uword  iphpr;			   		/* Inbound post_FIFO head pointer	  		*/
	uword  reserved41;		   		/* Reserved area 					  		*/
  	uword  iptpr;			   		/* Inbound post_FIFO tail pointer	  		*/
 	uword  reserved42;         		/* Reserved area 					  		*/
    uword  ofhpr;			   		/* Outbound free_FIFO head pointer  	  	*/
    uword  reserved43;         		/* Reserved area 					  		*/
    uword  oftpr;			   		/* Outbound free_FIFO tail pointer	  		*/
    uword  reserved44;         		/* Reserved area 					  		*/
    uword  ophpr;			   		/* Outbound post_FIFO head pointer	  		*/
    uword  reserved45;         		/* Reserved area 					  		*/
    uword  optpr;			   		/* Outbound post_FIFO tail pointer	  		*/
    uword  reserved46[2];      		/* Reserved area 					  		*/
    uword  mucr;			   		/* Message unit control register		 	*/
    uword  reserved47[2];      		/* Reserved area 					  		*/
    uword  qbar;			   		/* Queue base address register		  		*/
    uword  reserved48[3];      		/* Reserved area 					  		*/
    uword  dmamr0;			   		/* DMA 0 mode register			      		*/
    uword  dmasr0;			   		/* DMA 0 status register				  	*/
    uword  dmacdar0;		   		/* DMA 0 current descriptor address 	  	*/
    uword  reserved990;		   		/* Reserved area 					  		*/
    uword  dmasar0;			   		/* DMA 0 source address register		  	*/
    uword  reserved991;        		/* Reserved area 					  		*/
    uword  dmadar0;			   		/* DMA 0 destination address register  		*/
    uword  reserved992;        		/* Reserved area 					  		*/
    uword  dmabcr0;			   		/* DMA 0 byte count register			  	*/
    uword  dmandar0;		   		/* DMA 0 next descriptor address	      	*/
    uword  reserved49[22];     		/* Reserved area 					  		*/
    uword  dmamr1;			   		/* DMA 1 mode register			      		*/
    uword  dmasr1;			   		/* DMA 1 status register				  	*/
    uword  dmacdar1;		   		/* DMA 1 current descriptor address 	  	*/
    uword  reserved50;         		/* Reserved area 					  		*/
    uword  dmasar1;			   		/* DMA 1 source address register		  	*/
    uword  reserved51;         		/* Reserved area 					  		*/
    uword  dmadar1;			   		/* DMA 1 destination address register  		*/
    uword  reserved52;         		/* Reserved area 					  		*/ 
    uword  dmabcr1;			   		/* DMA 1 byte count register			  	*/
    uword  dmandar1;           		/* DMA 1 next descriptor address	      	*/
    uword  reserved53[22];     		/* Reserved area 					  		*/
    uword  dmamr2;			   		/* DMA 2 mode register			      		*/
    uword  dmasr2;             		/* DMA 2 status register				  	*/
    uword  dmacdar2;           		/* DMA 2 current descriptor address 	  	*/
    uword  reserved54;         		/* Reserved area 					  		*/
    uword  dmasar2;            		/* DMA 2 source address register		  	*/
    uword  reserved55;         		/* Reserved area 					  		*/
    uword  dmadar2;            		/* DMA 2 destination address register  		*/
    uword  reserved56;         		/* Reserved area 					  		*/
    uword  dmabcr2;            		/* DMA 2 byte count register			  	*/
    uword  dmandar2;           		/* DMA 2 next descriptor address	      	*/
    uword  reserved57[22];     		/* Reserved area 					  		*/
    uword  dmamr3;      	   		/* DMA 3 mode register				  		*/       
    uword  dmasr3;			   		/* DMA 3 status register				  	*/
    uword  dmacdar3;		   		/* DMA 3 current descriptor register	  	*/
    uword  reserved58;         		/* Reserved area 					  		*/
    uword  dmasar3;			   		/* DMA 3 source address register		  	*/
    uword  reserved59;         		/* Reserved area 					  		*/
    uword  dmadar3;			   		/* DMA 3 destination address register  		*/
    uword  reserved60;         		/* Reserved area 					  		*/
    uword  dmabcr3;			   		/* DMA 3 byte count register			  	*/
    uword  dmandar3;		   		/* DMA 3 next descriptor address		  	*/
    uword  reserved61[86];     		/* Reserved area 					  		*/
    uword  potar0;			   		/* PCI outbound translation address	  		*/
    uword  reserved62;         		/* Reserved area 					  		*/
    uword  pobar0;			   		/* PCI outbound base address register  		*/
    uword  reserved63;         		/* Reserved area 					  		*/
    uword  pocmr0;			   		/* PCI outbound comparison mask 		  	*/
    uword  reserved64;         		/* Reserved area 					  		*/
    uword  potar1;			   		/* PCI outbound translation address	  		*/
    uword  reserved65;         		/* Reserved area 					  		*/
    uword  pobar1;			  		/* PCI outbound base address register  		*/
    uword  reserved66;         		/* Reserved area 					  		*/
    uword  pocmr1;			   		/* PCI outbound comparison mask 		  	*/
    uword  reserved67;         		/* Reserved area 					  		*/
    uword  potar2;			   		/* PCI outbound translation address	  		*/
    uword  reserved68;         		/* Reserved area 					  		*/
    uword  pobar2;			   		/* PCI outbound base address register  		*/
    uword  reserved69;         		/* Reserved area 					  		*/
    uword  pocmr2;			   		/* PCI outbound comparison mask 		  	*/
    uword  reserved70[13];     		/* Reserved area 					  		*/
    uword  ptcr;			   		/* Discard timer control register      		*/
    uword  gpcr;			   		/* General purpose control register	  		*/
    uword  pci_gcr;			   		/* PCI general control register	      		*/
    uword  esr;			 	   		/* Error status register				  	*/
    uword  emr;				   		/* Error mask register				  		*/
    uword  ecr;				   		/* Error control register			  		*/
    uword  pci_eacr;		   		/* PCI error address capture register  		*/
    uword  reserved71;         		/* Reserved area 					  		*/
    uword  pci_edcr;		   		/* PCI error data capture register	  		*/
    uword  pci_eccr;		   		/* PCI error control capture register  		*/
    uword  reserved72[12];     		/* Reserved area 					  		*/
    uword  pitar1;			   		/* PCI inbound translation address     		*/
    uword  reserved73;         		/* Reserved area 					  		*/
    uword  pibar1;			   		/* PCI inbound base address register1  		*/
    uword  reserved74;        		/* Reserved area 					  		*/
    uword  picmr1;		       		/* PCI inbound comparison mask		  		*/
    uword  reserved75;         		/* Reserved area 					  		*/
    uword  pitar0;			   		/* PCI inbound translation address	  		*/
    uword  reserved76;         		/* Reserved area 					  		*/
    uword  pibar0;			   		/* PCI inbound base address register 0 		*/
    uword  reserved77;         		/* Reserved area 					  		*/
    uword  picmr0;			   		/* PCI inbound comparison mask		  		*/
    uword  reserved78;         		/* Reserved area 					  		*/
    uword  cfg_addr;		   		/* PCI cfg_addr						  		*/
    uword  cfg_data;	 	   		/* PCI cfg_data						  		*/
    uword  intack;			   		/* PCI int_ack						  		*/
    uword  reserved79[189];    		/* Reserved area 					  		*/	

/* ic */
    ushort ic_sicr;            		/* Interrupt Configuration Register 		*/
    uchar  reserved80[0x2];    		/* Reserved area 							*/
    uword  ic_sivec;           		/* CP Interrupt Vector Register 			*/
    uword  ic_sipnr_h;         		/* Interrupt Pending Register (HIGH)		*/
    uword  ic_sipnr_l;         		/* Interrupt Pending Register (LOW) 		*/
    uword  ic_siprr;           		/* SIU Interrupt Priority Register 			*/
    uword  ic_scprr_h;         		/* Interrupt Priority Register (HIGH) 		*/
    uword  ic_scprr_l;         		/* Interrupt Priority Register (LOW) 		*/
    uword  ic_simr_h;          		/* Interrupt Mask Register (HIGH) 			*/
    uword  ic_simr_l;          		/* Interrupt Mask Register (LOW) 			*/
    uword  ic_siexr;           		/* External Interrupt Control Register 		*/
    uchar  reserved81[0x58];   		/* Reserved area 							*/

/* clocks */
    uword  clocks_sccr;        		/* System Clock Control Register 			*/
    uchar  reserved82[0x4];    		/* Reserved area 							*/
    uword  clocks_scmr;        		/* System Clock Mode Register 				*/
    uchar  reserved83[0x4];    		/* Reserved area 							*/
    uword  clocks_rsr;         		/* Reset Status Register 					*/
    uword  clocks_rmr;         		/* Reset Moode Register  					*/
    uchar  reserved84[0x68];   		/* Reserved area 							*/

/* io_ports */
    struct io_regs 
    {
        uword  pdir;           		/* Port A-D Data Direction Register 		*/
        uword  ppar;           		/* Port A-D Pin Assignment Register 		*/
        uword  psor;           		/* Port A-D Special Operation Register 		*/
        uword  podr;           		/* Port A-D Open Drain Register 			*/
        uword  pdat;           		/* Port A-D Data Register 					*/
        uchar reserved85[0xc]; 		/* Reserved area 							*/
    } io_regs[4];

/* cpm_timers */
    uchar  cpm_timers_tgcr1;   		/* Timer Global Configuration Register 		*/
    uchar  reserved86[0x3];    		/* Reserved area 							*/
    uchar  cpm_timers_tgcr2;   		/* Timer Global Configuration Register 		*/
    uchar  reserved87[0xb];    		/* Reserved area 							*/
    ushort cpm_timers_tmr1;    		/* Timer Mode Register 						*/
    ushort cpm_timers_tmr2;    		/* Timer Mode Register 						*/
    ushort cpm_timers_trr1;    		/* Timer Reference Register 				*/
    ushort cpm_timers_trr2;    		/* Timer Reference Register 				*/
    ushort cpm_timers_tcr1;    		/* Timer Capture Register 					*/
    ushort cpm_timers_tcr2;    		/* Timer Capture Register 					*/
    ushort cpm_timers_tcn1;    		/* Timer Counter 							*/
    ushort cpm_timers_tcn2;    		/* Timer Counter 							*/
    ushort cpm_timers_tmr3;    		/* Timer Mode Register 						*/
    ushort cpm_timers_tmr4;    		/* Timer Mode Register 						*/
    ushort cpm_timers_trr3;    		/* Timer Reference Register 				*/
    ushort cpm_timers_trr4;    		/* Timer Reference Register 				*/
    ushort cpm_timers_tcr3;    		/* Timer Capture Register 					*/
    ushort cpm_timers_tcr4;    		/* Timer Capture Register 					*/
    ushort cpm_timers_tcn3;    		/* Timer Counter 							*/
    ushort cpm_timers_tcn4;    		/* Timer Counter 							*/
    ushort cpm_timers_ter[4];  		/* Timer Event Register 					*/
    uchar reserved88[0x260];   		/* Reserved area 							*/

/* sdma general */
    uchar  sdma_sdsr;          		/* SDMA Status Register 					*/
    uchar  reserved89[0x3];    		/* Reserved area 							*/
    uchar  sdma_sdmr;          		/* SDMA Mask Register 						*/
    uchar  reserved90[0x3];    		/* Reserved area 							*/

/* idma */
    uchar  reserved91[0x8];    		/* Reserved area 							*/
    uchar  idma_idsr2;         		/* IDMA Status Register 					*/
    uchar  reserved93[0x3];    		/* Reserved area 							*/
    uchar  idma_idmr2;         		/* IDMA Mask Register 						*/
    uchar  reserved94[0x3];    		/* Reserved area 							*/
    uchar  idma_idsr3;         		/* IDMA Status Register 					*/
    uchar  reserved95[0x3];    		/* Reserved area 							*/
    uchar  idma_idmr3;         		/* IDMA Mask Register 						*/
    uchar  reserved96[0x2cb];  		/* Reserved area 							*/
    
/* fcc1 */
    uword  gfmr1;           	   		/* FCC General Mode Register 				*/
    uword  psmr1;               		/* FCC Protocol Specific Mode Register 		*/
    ushort todr1;               		/* FCC Transmit On Demand Register 			*/
    uchar  reserved99[0x2];    		/* Reserved area 							*/
    ushort dsr1;                		/* FCC Data Sync. Register 					*/
    uchar  reserved100[0x2];   		/* Reserved area 							*/
    uword  fcce1;               		/* FCC Event Register 						*/
    uword  fccm1;               		/* FCC Mask Register 						*/
    uchar  fccs1;               		/* FCC Status Register 						*/
    uchar  reserved101[0x3];   		/* Reserved area 							*/
    uchar  ftprr0;             		/* FCC Transmit Partial Rate Register 		*/
    uchar  ftprr1;		       		/* FCC Transmit Partial Rate Register 		*/
    uchar  ftprr2;		       		/* FCC Transmit Partial Rate Register 		*/
    uchar  ftprr3; 		       		/* FCC Transmit Partial Rate Register 		*/
     
/* fcc2 */
    uword  gfmr2;               		/* FCC General Mode Register 				*/
    uword  psmr2;               		/* FCC Protocol Specific Mode Register 		*/
    ushort todr2;               		/* FCC Transmit On Demand Register 			*/
    uchar  reserved102[0x2];   		/* Reserved area 							*/
    ushort dsr2;                		/* FCC Data Sync. Register 					*/
    uchar  reserved103[0x2];   		/* Reserved area 							*/
    uword  fcce2;               		/* FCC Event Register 						*/
    uword  fccm2;               		/* FCC Mask Register 						*/
    uchar  fccs2;               		/* FCC Status Register 						*/
    uchar  reserved104[0x47];  		/* Reserved area 							*/

/* fcc1 extended registers */
    uword  firper1;		       		/* FCC1 internal rate port enable			*/
	uword  firer1;		       		/* FCC1 internal rate even 					*/
	uword  firsr1_hi;	       		/* FCC1 internal rate select high			*/
	uword  firsr1_lo;	       		/* FCC1 internal rate select low			*/
	uchar  gfemr1;		       		/* FCC1 general extended mode 	`			*/
	uchar  reserved105[0x1f];  		/* Reserved area					    	*/
	
/* fcc2 extended registers */
	uchar  gfemr2;		       		/* FCC2 general extended mode 				*/
	uchar  reserved106[0x23F]; 		/* Reserved area				    		*/
	
/* brgs 5-8 */
    uword  brgs_brgc5;         		/* BRG Configuration Register 				*/
    uword  brgs_brgc6;         		/* BRG Configuration Register 				*/
    uword  brgs_brgc7;         		/* BRG Configuration Register 				*/
    uword  brgs_brgc8;         		/* BRG Configuration Register 				*/
    uchar  reserved107[0x260]; 		/* Reserved area 							*/
    
/* i2c */
    uchar  i2c_i2mod;          		/* IC Mode Register 						*/
    uchar  reserved108[0x3];   		/* Reserved area 							*/
    uchar  i2c_i2add;          		/* IC Address Register 						*/
    uchar  reserved109[0x3];  		/* Reserved area 							*/
    uchar  i2c_i2brg;          		/* IC BRG Register 							*/
    uchar  reserved110[0x3];   		/* Reserved area 							*/
    uchar  i2c_i2com;          		/* IC Command Register 						*/
    uchar  reserved111[0x3];   		/* Reserved area 							*/
    uchar  i2c_i2cer;          		/* IC Event Register 						*/
    uchar  reserved112[0x3];   		/* Reserved area 							*/
    uchar  i2c_i2cmr;          		/* IC Mask Register 						*/
    uchar  reserved113[0x14b]; 		/* Reserved area 							*/
    
/* cpm */
    uword  cpm_cpcr;           		/* Communication Processor Command Register */
    uword  cpm_rccr;           		/* RISC Configuration Register 				*/
    uword  cpm_rmdr;           		/* RISC Microcode Dev. Support Control Reg. */
    ushort cpm_rctr1;          		/* RISC Controller Trap Register 			*/
    ushort cpm_rctr2;          		/* RISC Controller Trap Register 			*/
    ushort cpm_rctr3;          		/* RISC Controller Trap Register 			*/
    ushort cpm_rctr4;          		/* RISC Controller Trap Register 			*/
    uchar  reserved114[0x2];   		/* Reserved area 							*/
    ushort cpm_rter;           		/* RISC Timers Event Register 				*/
    uchar  reserved115[0x2];   		/* Reserved area 							*/
    ushort cpm_rtmr;           		/* RISC Timers Mask Register 				*/
    ushort cpm_rtscr;          		/* RISC Time-Stamp Timer Control Register 	*/
    ushort cpm_rmds;           		/* RISC Development Support Status Register */
    uword  cpm_rtsr;           		/* RISC Time-Stamp Register 				*/
    uchar  reserved116[0xc];   		/* Reserved area 							*/

/* brgs 1-4 */
    uword  brgs_brgc1;         		/* BRG Configuration Register 				*/
    uword  brgs_brgc2;         		/* BRG Configuration Register 				*/
    uword  brgs_brgc3;         		/* BRG Configuration Register 				*/
    uword  brgs_brgc4;         		/* BRG Configuration Register 				*/
    
/* scc */
	uword  sgsmr_l1;       			/* SCC General Mode Register 				*/
    uword  sgsmr_h1;       			/* SCC General Mode Register 				*/
    ushort spsmr1;          		/* SCC Protocol Specific Mode Register 		*/
    uchar  reserved117[0x2];		/* Reserved area 							*/
    ushort stodr1;          		/* SCC Transmit-On-Demand Register 			*/
    ushort sdsr1;           		/* SCC Data Synchronization Register 		*/
    ushort scce1;           		/* SCC Event Register 						*/
    uchar  reserved118[0x2];		/* Reserved area 							*/
    ushort sccm1;          			/* SCC Mask Register 						*/
    uchar  reserved119;     		/* Reserved area 							*/
    uchar  sccs1;          			/* SCC Status Register 						*/
    uchar  reserved120[0x28];		/* Reserved area 							*/
	
    struct scc_regs
    {
        uword  sgsmr_l;         	/* SCC General Mode Register 				*/
        uword  sgsmr_h;         	/* SCC General Mode Register 				*/
        ushort spsmr;           	/* SCC Protocol Specific Mode Register 		*/
        uchar  reserved121[0x2];	/* Reserved area 							*/
        ushort stodr;           	/* SCC Transmit-On-Demand Register 			*/
        ushort sdsr;            	/* SCC Data Synchronization Register 		*/
        ushort scce;           		/* SCC Event Register 						*/
        uchar  reserved122[0x2];	/* Reserved area 							*/
        ushort sccm;           		/* SCC Mask Register 						*/
        uchar  reserved123;     	/* Reserved area 							*/
        uchar  sccs;           		/* SCC Status Register 						*/
        uchar  reserved124[0x8];	/* Reserved area 							*/
    } scc_regs[2];
    
/* smc */
    struct smc_regs
    {
        uchar  reserved125[0x2];	/* Reserved area 							*/
        ushort smcmr;          		/* SMC Mode Register 						*/
        uchar  reserved126[0x2];	/* Reserved area 							*/
        uchar  smce;           		/* SMC Event Register 						*/
        uchar  reserved127[0x3];	/* Reserved area 							*/
        uchar  smcm;           		/* SMC Mask Register 						*/
        uchar  reserved128[0x5];	/* Reserved area 							*/
    } smc_regs[2];
    
/* spi */
    ushort spi_spmode;         		/* SPI Mode Register 						*/
    uchar  reserved129[0x4];    	/* Reserved area 							*/
    uchar  spi_spie;           		/* SPI Event Register 						*/
    uchar  reserved130[0x3];    	/* Reserved area 							*/
    uchar  spi_spim;           		/* SPI Mask Register 						*/
    uchar  reserved131[0x2];    	/* Reserved area 							*/
    uchar  spi_spcom;          		/* SPI Command Register 					*/
    uchar  reserved132[0x52];   	/* Reserved area 							*/
    
/* cpm_mux */
    uchar  cpm_mux_cmxsi1cr;   		/* CPM MUX SI Clock Route Register 			*/
    uchar  reserved133;         	/* Reserved area 							*/
    uchar  cpm_mux_cmxsi2cr;   		/* CPM MUX SI Clock Route Register 			*/
    uchar  reserved134;         	/* Reserved area 							*/
    uword  cpm_mux_cmxfcr;     		/* CPM MUX FCC Clock Route Register 		*/
    uword  cpm_mux_cmxscr;     		/* CPM MUX SCC Clock Route Register 		*/
    uchar  cpm_mux_cmxsmr;     		/* CPM MUX SMC Clock Route Register 		*/
    uchar  reserved135;         	/* Reserved area 							*/
    ushort cpm_mux_cmxuar;     		/* CPM MUX VUTOPIA Address Register 		*/
    uchar  reserved136[0x30];   	/* Reserved area 							*/
    
/* si */
    ushort sixmr[2];       			/* SI TDM Mode Registers A & B 				*/
    uchar  reserved137[0x3]; 		/* Reserved area			  				*/
    uchar  sigmr;          			/* SI Global Mode Register 					*/
    uchar  reserved138;     		/* Reserved area 							*/
    uchar  sicmdr;         			/* SI Command Register 						*/
    uchar  reserved139;     		/* Reserved area 							*/
    uchar  sistr;          			/* SI Status Register 						*/
    uchar  reserved140[0x2];     	/* Reserved area 							*/
    ushort sirsr;          			/* SI RAM Shadow Address Register 			*/
    uchar  reserved144[0x10];   	/* Reserved area 							*/

/* usb   */
	uchar  usmod;			 		/* USB mode register						*/
	uchar  usadr;					/* USB address register						*/
	uchar  uscom;					/* USB command register						*/
	uchar  reserved145[0x1];
	ushort usep1;					/* USB endpoint 0 register					*/
	ushort usep2; 					/* USB endpoint 1 register					*/
	ushort usep3; 					/* USB endpoint 2 register					*/
	ushort usep4; 					/* USB endpoint 3 register					*/
	uchar  reserved146[0x4];		/* Reserved area							*/
	ushort usber;					/* USB event register						*/
	uchar  reserved147[0x2];		/* Reserved area							*/
	ushort usbmr;					/* USB mask register						*/
	uchar  reserved148[0x1];		/* Reserved area							*/
	uchar  usbs;					/* USB status register						*/
	uchar  reserved149;				/* Reserved area							*/
	uchar  reserved150[0xc87];		/* Reserved area							*/
	
	
   
/* si_ram */
    ushort tx_siram[0x100];			/* SI Transmit Routing RAM 					*/
    uchar  reserved151[0x200];		/* Reserved area 							*/
    ushort rx_siram[0x100];			/* SI Receive Routing RAM 					*/
    uchar  reserved152[0x200];		/* Reserved area 							*/
    uchar  reserved153[0xD000]; 	/* Reserved area 							*/
    
/* cpm instruction ram */
	uchar  iram[0x1000];			/* CPM instruction ram 						*/

} _PackedType EPPC;


#pragma pack(pop)


/***************************************************************************/
/*                   General Global Definitions                            */
/***************************************************************************/

#define PORTA   0           		/* Index into Parallel I/O Regs Array 		*/              
#define PORTB   1           		/* Index into Parallel I/O Regs Array 		*/              
#define PORTC   2           		/* Index into Parallel I/O Regs Array 		*/              
#define PORTD   3           		/* Index into Parallel I/O Regs Array 		*/
                                 
#define PAGE1   0           		/* Index 1 into PRAM Array 					*/
#define PAGE2   1           		/* Index 2 into PRAM Array 					*/
#define PAGE3   2           		/* Index 3 into PRAM Array 					*/
#define PAGE4   3           		/* Index 4 into PRAM Array 					*/

#define FCC1    0           		/* FCC1 Index into FCC PRAM Array 			*/
#define FCC2    1           		/* FCC2 Index into FCC PRAM Array 			*/
#define FCC3    2           		/* FCC3 Index into FCC PRAM Array 			*/

#define MCC1    0           		/* MCC1 Index into MCC PRAM Array 			*/
#define MCC2    1           		/* MCC2 Index into MCC PRAM Array 			*/

#define SCC1    0           		/* SCC1 Index into SCC PRAM Array 			*/             
#define SCC2    1           		/* SCC2 Index into SCC PRAM Array 			*/             
#define SCC3    2           		/* SCC3 Index into SCC PRAM Array 			*/             
#define SCC4    3           		/* SCC4 Index into SCC PRAM Array 			*/

#define TDMA    0           		/* Index into TDM Mode Register Array 		*/
#define TDMB    1           		/* Index into TDM Mode Register Array 		*/
#define TDMC    2           		/* Index into TDM Mode Register Array 		*/
#define TDMD    3           		/* Index into TDM Mode Register Array 		*/

#define SI1     0           		/* Index into SI Register Array 			*/
#define SI2     1           		/* Index into SI Register Array 			*/             

#endif
