#include "sci.h"



/*
** ===================================================================
**     Method      :  AS1_RecvChar (bean AsynchroSerial)
**
**     Description :
**         If any data received, this method returns one character,
**         otherwise it returns error code (it does not wait for
**         data). This method is enabled only if the receiver
**         property is enabled. 
**         DMA mode:
**         If DMA controller is available on selected CPU and
**         receiver is configured to use DMA controller then this
**         method only sets selected DMA channel. Status of the DMA
**         transfer can be then checked using method GetCharsInRxBuf.
**         See typical usage for details about communication using
**         DMA.
**     Parameters  :
**         NAME            - DESCRIPTION
**       * Chr             - Pointer to received character
**     Returns     :
**         ---             - Error code, possible codes:
**                           ERR_OK - OK
**                           ERR_SPEED - This device does not work in
**                           the active speed mode
**                           ERR_RXEMPTY - No data in receiver
**                           ERR_OVERRUN - Overrun error is detected
**                           ERR_FRAMING - Framing error is detected
**                           ERR_PARITY - Parity error is detected
**                           ERR_BREAK - Break character is detected
**                           DMA mode:
**                           If DMA controller is available on
**                           selected CPU and receiver is configured
**                           to use DMA controller then only ERR_OK,
**                           ERR_RXEMPTY and ERR_SPEED error code can
**                           be returned from this method.
** ===================================================================
*/
byte AS1_RecvChar(byte *Chr)
{
  byte Result = ERR_OK;                /* Return error code */

  if (SCIS1 & 31) {                   /* Is an error set? */
    if (SCIS1_IDLE)
      SerFlag |= IDLE_ERR;             /* Idle character flag */
    if (SCIS1_PF)
      SerFlag |= PARITY_ERR;           /* Parity error flag */
    if (SCIS1_NF)
      SerFlag |= NOISE_ERR;            /* Noise error flag */
    if (SCIS1_OR)
      SerFlag |= OVERRUN_ERR;          /* Overrun error flag */
    if (SCIS1_FE)
      SerFlag |= FRAMING_ERR;          /* Framming error flag */
  }
  if (!(SCIS1_RDRF) && !(SCIS1 & 15)) /* Is the reciver empty and no error is set? */
    return ERR_RXEMPTY;                /* If yes then error */
  *Chr = SCID;                        /* Read data from the receiver */
  if ((SerFlag & (OVERRUN_ERR|FRAMING_ERR|PARITY_ERR|NOISE_ERR|IDLE_ERR)) != 0) {
    if((SerFlag & OVERRUN_ERR) != 0)   /* Is the overrun error detected? */
      Result = ERR_OVERRUN;            /* If yes then set the overrun error flag */
    else if((SerFlag & PARITY_ERR) != 0) /* Is the parity error detected? */
      Result = ERR_PARITY;             /* If yes then set the parity error flag */
    else if((SerFlag & FRAMING_ERR) != 0) /* Is the framing error detected? */
      Result = ERR_FRAMING;            /* If yes then set the framing error flag */
    else if((SerFlag & NOISE_ERR) != 0) /* Is the noise error detected? */
      Result = ERR_NOISE;              /* If yes then set the noise error flag */
  }
  SerFlag &= ~(OVERRUN_ERR|FRAMING_ERR|PARITY_ERR|NOISE_ERR|IDLE_ERR); /* Clear all errors in the status variable */
  return Result;                       /* Return error code */
}

/*
** ===================================================================
**     Method      :  AS1_SendChar (bean AsynchroSerial)
**
**     Description :
**         Send one character to the channel. This method is
**         available only if the transmitter property is enabled.
**         DMA mode:
**         If DMA controller is available on selected CPU and
**         transmitter is configured to use DMA controller then this
**         method only sets selected DMA channel. Status of the DMA
**         transfer can be then checked using method GetCharsInTxBuf.
**         See typical usage for details about communication using
**         DMA.
**     Parameters  :
**         NAME            - DESCRIPTION
**         Chr             - Character to send
**     Returns     :
**         ---             - Error code, possible codes:
**                           ERR_OK - OK
**                           ERR_SPEED - This device does not work in
**                           the active speed mode
**                           ERR_TXFULL - Transmitter is full
** ===================================================================
*/
byte AS1_SendChar(byte Chr)
{
  if(!SCIS1_TDRE)                     /* Is the transmitter empty? */
    return ERR_TXFULL;                 /* If no then error */
  SCID = (byte)Chr;
  return ERR_OK;                       /* OK */
}


/*
** ===================================================================
**     Method      :  AS1_Init (bean AsynchroSerial)
**
**     Description :
**         This method is internal. It is used by Processor Expert
**         only.
** ===================================================================
*/
void AS1_Init(void)
{
  SerFlag = 0;                         /* Reset flags */
  SCIC1 = 0;
  /* SCI1C3: R8=0,T8=0,TXDIR=0,??=0,ORIE=0,NEIE=0,FEIE=0,PEIE=0 */
  SCIC3 = 0;          
  SCIBD = SCIDIVIDER; //Set in sci.h file
  SCIC2_TE = 1;                     /* Enable transmitter */
  SCIC2_RE = 1;                     /* Enable receiver */
}
