#define LINMSG_C
/******************************************************************************
*                                                       
*       Copyright (C) 2005 Freescale Semiconductor, Inc.
*       All Rights Reserved								              
*														                            
* Filename:     linmsg.c                
*														                            
* Revision:      										                    
*														                            
* Functions:    Message processing subsystem 
*												                            
* Description:  
*												                            
******************************************************************************/

#include <linbase.h>

#if !defined(LINAPI_1_0)

/****************************************************************************
 * Function prototypes 
 ***************************************************************************/
static void LinMemCopy( LIN_BYTE * dst, LIN_BYTE * src, LIN_BYTE len);


/****************************************************************************
 * Functions 
 ***************************************************************************/


 /***************************************************************************
 * Function :   LinMemCopy
 *
 * Description: Copy data from src to dst
 *
 * Returns:     none
 *
 * Notes:       
 *
 **************************************************************************/
void LinMemCopy( LIN_BYTE * dst, LIN_BYTE * src, LIN_BYTE len)
{
    while( len > 0 )
    {
        len-- ;
        dst[len] = src[len];
    }
    return;
}


/***************************************************************************
 * Function :   LIN_GetMsg
 *
 * Description: Find message and copy the message data to the buffer
 *
 * Returns:
 *  LIN_NO_ID       - msg is not configured for this node
 *  LIN_INVALID_ID  - msg is received by this node (but data is copied)
 *  LIN_MSG_NODATA  - msg data has not been received so far
 *  LIN_OK          - msg data is copied to the buffer
 *
 * Notes:       API function, OPTIM
 *
 **************************************************************************/
LINStatusType   LIN_GetMsg( LINMsgIdType msgId, LINMsgRefType msgData )
{
    LIN_BYTE    idx;
    LIN_BYTE    intMask;
    LIN_BYTE    retStatus;

    /* check requested Id */
    if ( msgId > (LIN_BYTE) LIN_FRAME_MAX_ID )
    {
        /* Id is not valid */
        return LIN_NO_ID;
    }

    /* Search message */
    idx = LinLookupTable[msgId];

    if ( (idx & LIN_MSG_NOT_IGNORED) == 0 )
    {
        /* message is ignored by this node */
        return LIN_NO_ID;
    }

    LIN_DBG_SET_PORT_7;
    intMask = LIN_DisableInt();         /* disable interrupts */

    /* If we transmit this message return error */
    if ( idx & LIN_MSG_SEND ) 
    {
        /* we send this message, but copy anyway */
        /* NB: not initialized data can be copied ! */

#if defined(SLAVE)
        /* Message's index is least 6 bits. */
        idx &= LIN_MSG_INDEX_MASK;

        LinMemCopy( msgData, LinMsgBuf[idx], LinMsgLen[idx]);
#endif /* defined(SLAVE) */

        /* don't change msg status */

        retStatus = LIN_INVALID_ID;
    }
    else
    {
        /* we receive this message */

        /* Message's index is least 6 bits. */
        idx &= LIN_MSG_INDEX_MASK;

                                 /* LinMsgStatus can be change from interrupt only to LIN_MSG_OK or */
                                 /* LIN_MSG_OVERRUN so we didn`t need close interrupt early  */
                                 /* ??? paranoia: LinMsgStatus can be changed to LIN_MSG_NODATA only from */ 
                                 /* LIN_Init if call it from interrupt */

        /* Check if there is data */
        if (( LinMsgStatus[idx] & LIN_MSG_NODATA ) != 0 )
        {
            /* no data -> don't copy */
            retStatus = LIN_MSG_NODATA;
        }
        else
        {    
            /* If message was found copy the data */

#if defined(SLAVE)
            LinMemCopy( msgData, LinMsgBuf[idx], LinMsgLen[idx]);
#endif /* defined(SLAVE) */

            /* change message status */
            LinMsgStatus[idx] = LIN_MSG_NOCHANGE;

            retStatus   = LIN_OK;
        }    
    }
    
    LIN_EnableInt(intMask);             /* enable interrupts */
    LIN_DBG_CLR_PORT_7;

    return retStatus;
}

/***************************************************************************
 * Function :   LIN_PutMsg
 *
 * Description: find and write message 
 *
 * Returns:
 *  LIN_NO_ID       - msg is not configured for this node
 *  LIN_INVALID_ID  - msg is transmitted by this node
 *  LIN_OK          - data is copied to the msg buffer
 *
 * Notes:        API function, OPTIM
 *
 **************************************************************************/
LINStatusType   LIN_PutMsg( LINMsgIdType msgId, LINMsgRefType msgData )
{
    LIN_BYTE    idx;
    LIN_BYTE    intMask;
    
    /* check requested Id */
    if ( msgId > (LIN_BYTE) LIN_FRAME_MAX_ID )
    {
        /* Id is not valid */
        return LIN_NO_ID;
    }

    /* Search message */
    idx = LinLookupTable[msgId];

    if ( (idx & LIN_MSG_NOT_IGNORED) == 0 )
    {
        /* message is ignored by this node */
        return LIN_NO_ID;
    }

    /* If we receive this message return error */
    if ( ( idx & LIN_MSG_SEND ) == 0 )
    {
        return LIN_INVALID_ID;
    }

    /* Node process this message. Message's index is least 6 bits. */
    idx &= LIN_MSG_INDEX_MASK;

    LIN_DBG_SET_PORT_7;
    intMask = LIN_DisableInt();     /* disable interrupts */

    /* If message was found then copy the data */

#if defined(SLAVE)
    LinMemCopy( LinMsgBuf[idx], msgData, LinMsgLen[idx]);
#endif /* defined(SLAVE) */

    /* change the status */
    LinMsgStatus[idx] = LIN_MSG_NOCHANGE | LIN_MSG_UPDATED;
    
    LIN_EnableInt(intMask);         /* enable interrupts */
    LIN_DBG_CLR_PORT_7;

    return LIN_OK;
}

/***************************************************************************
 * Function :   LIN_MsgStatus
 *
 * Description: Return current message status 
 *
 * Returns:    
 * When message is specified as received:
 *  LIN_NO_ID           - msg is not configured for this node
 *  LIN_OK              - msg data changed since last LIN_GetMsg call;
 *  LIN_MSG_NOCHANGE    - msg data not received since last LIN_GetMsg call;
 *  LIN_MSG_NODATA      - msg data not received since driver initialization;
 *  LIN_MSG_OVERRUN     - msg data was overwritten by driver before read.
 *
 * When message is specified as transmitted:
 *  LIN_NO_ID           - msg is not configured for this node
 *  LIN_OK              - msg data transmitted since last LIN_PutMsg call;
 *  LIN_MSG_NOCHANGE    - msg data not transmitted since last LIN_PutMsg call;
 *  LIN_MSG_NODATA      - msg data not updated since driver initialization.
 * 
 * Notes:               API function, OPTIM
 *
 **************************************************************************/
LINStatusType   LIN_MsgStatus( LINMsgIdType msgId )
{
    LIN_BYTE    idx;

    /* check requested Id */
    if ( msgId > (LIN_BYTE) LIN_FRAME_MAX_ID )
    {
        /* Id is not valid */
        return LIN_NO_ID;
    }

    /* Search message */
    idx = LinLookupTable[msgId];

    if ( (idx & LIN_MSG_NOT_IGNORED) == 0 )
    {
        /* message is ignored by this node */
        return LIN_NO_ID;
    }

    /* Node process this message. Message's index is least 6 bits. */
    return ( LinMsgStatus[idx & LIN_MSG_INDEX_MASK] & LIN_MSG_STATUS_MASK );
}

#endif /* !defined(LINAPI_1_0) */
