/**************************************************************************
 * FILE NAME: $RCSfile: etpu_decimate.c,v $  COPYRIGHT (c) FREESCALE 2006 *
 * DESCRIPTION:                                     All Rights Reserved   *
 * This file contains the eTPU Decimate API                               *
 *========================================================================*
 * ORIGINAL AUTHOR: Geoff Emerson (r47354)                                *
 * $Log: etpu_decimate.c,v $
 * Revision 1.1  2006/06/14 10:45:32  r47354
 * Initial revision.
 *
 *........................................................................*
 **************************************************************************/
#include "etpu_util.h"		    /* Utility routines for working with the eTPU */
#include "etpu_decimate.h"		  	/* eTPU DECIMATE API defines */
extern uint32_t fs_etpu_data_ram_start;

/******************************************************************************
FUNCTION     : fs_etpu_decimate_init
PURPOSE      : To initialize an eTPU channel to run the DECIMATE function.
INPUTS NOTES : This function has the following parameters:
               channel - This is the channel number.
                         0-31 for FS_ETPU_A and 64-95 for FS_ETPU_B.
               Tcd_Daddr - Transfer Control Descriptor Destination Address.
               TCD_Doff -  Transfer Control Descriptor Destination Offset.
               *Ping_pong_Address - The address of the ping pong output buffer.
               Ping_Pong_Length - The lenght in words of the ping pong buffer.

RETURNS NOTES: Error code if channel could not be initialized. Error code that
               can be returned are: FS_ETPU_ERROR_MALLOC ,FS_ETPU_ERROR_FREQ.
WARNING      : *This function does not configure the pin only the eTPU. In a
               system a pin may need to be configured to select the eTPU.
******************************************************************************/
int32_t fs_etpu_decimate_init ( uint8_t channel, uint8_t priority,
                                       vuint32_t Tcd_Daddr, vuint16_t Tcd_Doff ,
                                       vuint16_t *Ping_Pong_Address,
                                       uint16_t Ping_Pong_Length )
{
    uint32_t *pba;	/* parameter base address for channel */
    uint8_t *pba8, y;
    uint32_t *x;
    

	/* Disable channel to assign function safely */
	fs_etpu_disable( channel );

    if (eTPU->CHAN[channel].CR.B.CPBA == 0 )
    {
    	/* get parameter RAM for
    	number of parameters passed from eTPU C code */
    	pba = fs_etpu_malloc(FS_ETPU_DECIMATE_NUM_PARMS );
			
			if (pba == 0)
    	{
    		return (FS_ETPU_ERROR_MALLOC);
    	}
    }
    else /*set pba to what is in the CR register*/
    {
    	pba=fs_etpu_data_ram(channel);
    }
    pba8 = (uint8_t *) pba;
    
	eTPU->CHAN[channel].CR.B.CPBA = (((uint32_t)pba - fs_etpu_data_ram_start)>>3);

  /* populate the Data Memory  */
	*(pba + ((FS_ETPU_DECIMATE_TCD_DADDR)>>2)) = Tcd_Daddr;
	*(pba + ((FS_ETPU_DECIMATE_TCD_DOFF)>>2)) = Tcd_Doff;
	*(pba + ((FS_ETPU_DECIMATE_PING_PONG_ADDRESS)>>2)) = (uint32_t)Ping_Pong_Address;
	*(pba + ((FS_ETPU_DECIMATE_PING_PONG_SIZE)>>2)) = Ping_Pong_Length;

 /* write channel configuration register */
	eTPU->CHAN[channel].CR.R = 
				(priority << 28) + (FS_ETPU_DECIMATE_TABLE_SELECT << 24) +
	      (FS_ETPU_DECIMATE_FUNCTION_NUMBER << 16) +
	      (((uint32_t)pba - fs_etpu_data_ram_start)>>3);

	return(0);
}


/*********************************************************************
 *
 * Copyright:
 *	Freescale Semiconductor, INC. All Rights Reserved.
 *  You are hereby granted a copyright license to use, modify, and
 *  distribute the SOFTWARE so long as this entire notice is
 *  retained without alteration in any modified and/or redistributed
 *  versions, and that such modified versions are clearly identified
 *  as such. No licenses are granted by implication, estoppel or
 *  otherwise under any patents or trademarks of Freescale
 *  Semiconductor, Inc. This software is provided on an "AS IS"
 *  basis and without warranty.
 *
 *  To the maximum extent permitted by applicable law, Freescale
 *  Semiconductor DISCLAIMS ALL WARRANTIES WHETHER EXPRESS OR IMPLIED,
 *  INCLUDING IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A
 *  PARTICULAR PURPOSE AND ANY WARRANTY AGAINST INFRINGEMENT WITH
 *  REGARD TO THE SOFTWARE (INCLUDING ANY MODIFIED VERSIONS THEREOF)
 *  AND ANY ACCOMPANYING WRITTEN MATERIALS.
 *
 *  To the maximum extent permitted by applicable law, IN NO EVENT
 *  SHALL Freescale Semiconductor BE LIABLE FOR ANY DAMAGES WHATSOEVER
 *  (INCLUDING WITHOUT LIMITATION, DAMAGES FOR LOSS OF BUSINESS PROFITS,
 *  BUSINESS INTERRUPTION, LOSS OF BUSINESS INFORMATION, OR OTHER
 *  PECUNIARY LOSS) ARISING OF THE USE OR INABILITY TO USE THE SOFTWARE.
 *
 *  Freescale Semiconductor assumes no responsibility for the
 *  maintenance and support of this software
 ********************************************************************/

