/*
 * File:    sysinit.c
 * Purpose: Reset configuration of the M5213EVB
 *
 * Notes:   
 */

#pragma cplusplus off

#include "common.h"
#include "uart.h"
#include "clock.h"
#include "mcf5213.h"

/********************************************************************/

/* Actual system clock frequency */
int sys_clk_khz = SYS_CLK_KHZ;

/********************************************************************/

void mcf5213_pll_init(void) {
    /* Initialize the PLL */
    clock_pll(REF_CLK_KHZ, SYS_CLK_KHZ, 0) / clock_lpd(1);
}

/********************************************************************/

void mcf5213_uart_init(void) {

    /* Enable the proper UART pins */
    switch (TERMINAL_PORT)
    {
        case 2:
            MCF_GPIO_PUCPAR = 0
                | MCF_GPIO_PUCPAR_RXD2_RXD2
                | MCF_GPIO_PUCPAR_TXD2_TXD2;
            break;
        case 1:
            MCF_GPIO_PUBPAR = 0
                | MCF_GPIO_PUBPAR_RXD1_RXD1
                | MCF_GPIO_PUBPAR_TXD1_TXD1;
            break;
        case 0:
        default:
            MCF_GPIO_PUAPAR = 0
                | MCF_GPIO_PUAPAR_RXD0_RXD0
                | MCF_GPIO_PUAPAR_TXD0_TXD0;
    }

    /* Enable the default UART terminal port */
    uart_init(TERMINAL_PORT, sys_clk_khz, TERMINAL_BAUD, 0);
}

void mcf5213_allow_interrupts(void) 
{
    /* 
     * Allow interrupts from ABORT, SW1, and SW2 (IRQ[4,5,7]) 
     */
     
    /* Enable IRQ signals on the port */
    MCF_GPIO_PNQPAR = 0
        | MCF_GPIO_PNQPAR_IRQ4_IRQ4
        | MCF_GPIO_PNQPAR_IRQ5_IRQ5
        | MCF_GPIO_PNQPAR_IRQ7_IRQ7;
    
    /* Set EPORT to look for rising edges */
    MCF_EPORT_EPPAR = 0
        | MCF_EPORT_EPPAR_EPPA4_RISING
        | MCF_EPORT_EPPAR_EPPA5_RISING
        | MCF_EPORT_EPPAR_EPPA7_RISING;
        
    /* Clear any currently triggered events on the EPORT */
    MCF_EPORT_EPIER = 0
        | MCF_EPORT_EPIER_EPIE4
        | MCF_EPORT_EPIER_EPIE5
        | MCF_EPORT_EPIER_EPIE7;
        
    /* Enable interrupts in the interrupt controller */
    MCF_INTC_IMRL &= ~(0
        | MCF_INTC_IMRL_MASK4 
        | MCF_INTC_IMRL_MASK5 
        | MCF_INTC_IMRL_MASK7 
        | MCF_INTC_IMRL_MASKALL);
}

/********************************************************************/
/* Watchdog Initialization                                             */
/********************************************************************/

void 
mcf5213_wtm_init(void)
{
}

/********************************************************************/
/* Stand Alone initialization                                       */
/********************************************************************/

void
mcf5213_init(void)
{
/********************************************************************/
/*  RAMBAR & FLASHBAR already initialized in asm_startmeup	        */
/********************************************************************/
	extern char __DATA_ROM[];
	extern char __DATA_RAM[];
	extern char __DATA_END[];
	extern char __BSS_START[];
	extern char __BSS_END[];
	extern uint32 VECTOR_TABLE[];
	extern uint32 __VECTOR_RAM[];
	register uint32 n;
	register uint8 *dp, *sp;

	mcf5213_wtm_init();
	mcf5213_pll_init();
	mcf5213_uart_init();
	mcf5213_allow_interrupts();
		
	/* Copy the vector table to RAM */
	if (__VECTOR_RAM != VECTOR_TABLE)
	{
		for (n = 0; n < 256; n++)
			__VECTOR_RAM[n] = VECTOR_TABLE[n];
	}
    mcf5xxx_wr_vbr((uint32)__VECTOR_RAM);

	/* 
	 * Move initialized data from ROM to RAM. 
	 */
	if (__DATA_ROM != __DATA_RAM)
	{
		dp = (uint8 *)__DATA_RAM;
		sp = (uint8 *)__DATA_ROM;
		n = __DATA_END - __DATA_RAM;
		while (n--)
			*dp++ = *sp++;
	}
 
	/* 
	 * Zero uninitialized data 
	 */
	if (__BSS_START != __BSS_END)
	{
		sp = (uint8 *)__BSS_START;
		n = __BSS_END - __BSS_START;
		while (n--)
			*sp++ = 0;
	}
}

/********************************************************************/
/* Initialize                                                       */
/********************************************************************/

void __initialize_hardware(void)
{
  mcf5213_pll_init();
  mcf5213_uart_init();
  mcf5213_allow_interrupts();
}
