;*******************************************************************
;* This stationery serves as the framework for a user application. *
;* For a more comprehensive program that demonstrates the more     *
;* advanced functionality of this processor, please see the        *
;* demonstration applications, located in the examples             *
;* subdirectory of the "Freescale CodeWarrior for HC08" program    *
;* directory.                                                      *
;*******************************************************************

; Include derivative-specific definitions
            INCLUDE 'derivative.inc'
;
; export symbols
;
            XDEF _Startup
            ABSENTRY _Startup

;*****************************************************************************
;*                            User Constants                                 *
;*****************************************************************************

Table_Data EQU $3E00
         
; Variable declarations
ACMP_ENABLE      SET   $92
ACMP_DISABLED    SET   $20
MTIM_INIT        SET   $50
MTIM_ENABLE      SET   $60
MTIM_STOP_RESET  SET   $30
MTIM_128_DIV     SET   $07
FREE_RUN         SET   $00
DEBUG_MODE       SET   $00
RUN_MODE         SET   $01

MODE:            EQU   DEBUG_MODE                																              

;*****************************************************************************
;*                            User Variables                                 *
;*****************************************************************************
;
; variable/data section
;
            ORG    RAMStart         ; Insert your data definition here

ConvertedValue: DS.B   1            ; This varible store converted value
Complement      DS.B   1
Temp_Page       DS.B   1            ; Temporal backup Page 
SensorReading   DS.B   1
PcBuffer        DS.B   2            ; Temporal backup SPC
            
            ORG    ROMStart
            
            
;**************************************************************
;*                      MACRO DEFINITIONS                     *
;**************************************************************
TRIM_ICS: MACRO                   ; Macro used to configure the ICS with TRIM
  MOV   #$FF,PAGESEL              ; change to last page
  LDX   #$FA                      ; load the content which TRIM value is store
  LDA   ,x                        ; read D[X]
  CBEQA #$FF,No_Trim              ; Omit the 0xFF value if $3FFA location content the value
  STA   ICSTRM                    ; Store TRIM value into ICSTRM register
No_Trim:                          
  ENDM

ENTRY_SUB: MACRO			            ; Macro for "stacking" SPC
	SHA
	STA PcBuffer + 2*(\1)
	SHA
	SLA
	STA PcBuffer + 2*(\1) +1
	SLA
  ENDM
      
  NOP                             ; Needs to separate MACROS

EXIT_SUB: MACRO				            ; Macro for restore SPC
	SHA
	LDA PcBuffer + 2*(\1)
	SHA
	SLA
	LDA PcBuffer + 2*(\1) +1
	SLA
  ENDM
  
                
;**************************************************************
;*                      CONFIGURES PORT A                     *
;**************************************************************
PortA:       
  MOV #HIGH_6_13(PTAPE), PAGESEL
  MOV #$FE, MAP_ADDR_6(PTAPE)				; Enables internal Pulling device
         
  MOV #HIGH_6_13(PTAPUD), PAGESEL
  MOV #$04,MAP_ADDR_6(PTAPUD)				; Configures Internal pull up device in PTA  5
  
  MOV #$32, PTADD        				    ; PTA5, PTA4 and PTA3 as outputs    
  MOV #$00, PTAD

  RTS        

;**************************************************************
;*                    CONFIGURES SYSTEM CONTROL               *
;**************************************************************
Init_Config: 
  IFNE  MODE
  MOV #HIGH_6_13(SOPT), PAGESEL  
  MOV #$01, MAP_ADDR_6(SOPT)      ; Disables COP and enables RESET (PTA2) pin                                
  ELSE 
  MOV #HIGH_6_13(SOPT), PAGESEL  
  MOV #$03, MAP_ADDR_6(SOPT)      ; Disables COP, enables BKGD (PTA3) and RESET (PTA2) pins                                  
  ENDIF
  CLR  ICSC1                      ; FLL is selected as Bus Clock
  TRIM_ICS
  CLR  ICSC2
  RTS


;**************************************************************
;*          Analog Comparator Initial Configuration           *
;**************************************************************
ACMP_Conf: 
      MOV #ACMP_ENABLE,ACMPSC           ; ACMP Enabled, ACMP+ pin active, Interrupt enabled, Rising edges detections
      RTS

;**************************************************************
;*            Modulus Timer Configuration for ADC             *
;**************************************************************
 MTIM_ADC_Init:
      MOV #MTIM_128_DIV,MTIMCLK         ; Select bus clock as reference, Set prescaler with 64
      MOV #FREE_RUN,MTIMMOD             ; Configure Timer as free running
      MOV #MTIM_STOP_RESET,MTIMSC
      RTS


;**************************************************************
;*                   Discharge Capacitor                      *
;**************************************************************
Discharge_Cap: 
      BSET  1,PTADD                     ; Configure PTA1 as Output
      BCLR  1,PTAD                      ; Start Capacitor discharging
      LDA   #$FE                        ; Set delay time
Waste_time:   
      DBNZA Waste_time                  ; Wait until Delay = 0
      RTS


_Startup:

  BSR Init_Config
  BSR PortA
NextValue:
  BSR MTIM_ADC_Init
  BSR Discharge_Cap
  BSR ACMP_Conf                     ; Configure ACMP+ and ACMP-
  MOV #MTIM_ENABLE,MTIMSC           ; Timer Counter Enabled
  
mainLoop:
  WAIT                              ; Wait for ACMP interrupt
  BSET 4,MTIMSC
  LDA MTIMCNT                         
  STA SensorReading                 ; Store counter value
  MOV #HIGH_6_13(SIP1), PAGESEL     
  BRSET 3, MAP_ADDR_6(SIP1),PWM     ; Branch if ACMP interrupt arrives
  BCLR 7,ACMPSC
  BRA NextValue        
  
  
PWM:

  MOV #MTIM_STOP_RESET,MTIMSC       ; Stop and reset counter
  MOV #ACMP_DISABLED, ACMPSC        ; ACMP Disabled, Clear Interrupt flag

  LDA SensorReading
  CMP #75
  BHS NextValue
  JSR LookupTable


;**************************************************************
;*                     Configure MTIM                         *
;**************************************************************
Activecounter:
  MOV #$00,MTIMCLK          ; Enables interrupt, stops and resets timer counter
  MOV #$01,MTIMMOD          ; MTIM modulo with 1 counts before interrupt. 
  MOV #$70,MTIMSC           ; Selects internal clock as reference bus clock (4 MHz) with prescaler 1  
  
  MOV #MTIM_STOP_RESET,MTIMSC       ; Stop and reset counter
  MOV #ACMP_DISABLED, ACMPSC        ; ACMP Disabled, Clear Interrupt flag
  
  LDA #$FF
  SUB ConvertedValue
  STA Complement

  BCLR 4,MTIMSC								; MTIM counter is Active
  BRA PWMLoopOn

  
;**************************************************************
;*                           PWM                              *
;**************************************************************

PWMLoopOn:  
  BRSET 7,MTIMSC,PWM_Isr_D    ; Branch if timer interrupt pending
	BRA PWMLoopOn
PWM_Isr_D:       
  BSET 5,MTIMSC   						; Reset MTIM Counter, Clear  overflow flag
  BRA PWM_Set_D
PWM_Set_D:
  BSET 4,PTAD                 ; Turn on led 4
  DBNZ ConvertedValue,PWMLoopOn
  
PWMLoopOff:   
  BRSET 7,MTIMSC,PWM_Isr_D1   ; Branch if timer interrupt pending
	BRA PWMLoopOff
PWM_Isr_D1:
  BSET 5,MTIMSC   						; Reset MTIM Counter, Clear  overflow flag
  BRA PWM_Clear
PWM_Clear:
  BCLR 4,PTAD                 ; Turn off led 4
  DBNZ Complement,PWMLoopOff
  

  BRA NextValue  
  
  
LookupTable:
      LDA SensorReading 
      ROLA                              ; Getting 2 MSB
      ROLA 
      ROLA 
      AND #$03                                                  
      ADD #(Table_Data>>6)              ; Page Calculating
      MOV #PAGESEL,Temp_Page            ; Backup actual page
      STA PAGESEL                       ; Page Change
      LDA SensorReading                    
      AND #$3F                          ; Extract 6 LSB
      ADD #$C0                          ; Index to paging window
      TAX                                 
      LDA ,x                            ; Load table result 
      STA ConvertedValue                ; Store result
      MOV #Temp_Page, PAGESEL           ; Back Page	
      RTS 
  
;**************************************************************
;*                 Startup Vector                             *
;**************************************************************
            ORG   $3FFD

            JMP _Startup          ; Reset
            
            
;**************************************************************
;*                    Data Table                              *
;**************************************************************         
 ORG Table_Data 
 dc.b 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
 dc.b 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
 dc.b 0,0,0,0,0,0,0,0,255,248,241,234,227,220,213,206
 dc.b 199,192,185,178,171,164,157,150,143,136,129,122,115,108,101,94
 dc.b 87,80,73,66,59,52,45,38,31,24,17,0,0,0,0,0
 dc.b 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
 dc.b 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
 dc.b 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
 dc.b 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
 dc.b 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
 dc.b 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
 dc.b 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
 dc.b 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
 dc.b 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
 dc.b 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
 dc.b 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0


