// ============================================================================
//  Copyright (c)2007 Freescale Semiconductor Inc.  -- All Rights Reserved --
//
//  The code is the property of Freescale GmbH Munich
//  C&I GTM Operation
//
//  The copyright notice above does not evidence any actual or intended 
//  publication of such source code.
//
//  This software is provided by Freescale "AS IS" and any expressed or implied
//  warrenties, including, but not limited to, the implied warrenties of 
//  merchantability and fitness for a particular purpose are disclaimed.
//  In no event shall Freescale or its contributors be liable for any direct, 
//  indirect, incidental, special, examplary, or consequential damages 
//  (including, but not limited to, procurement of substitue goods or services;
//  loss of use, data, or profits; or business interruption). However caused 
//  and on any theory of liability, whether in contract, strict liability, or
//  tort (including negligence or otherwise) arising in any way out of the use 
//  of this software, even if advised of the possibility of such damage.
// 
//  ---------------------------------------------------------------------------
//
//  Filename:     $Source: $
//  Author:       $Author: r51406 $
//  Locker:       $Locker:  $
//  State:        $State: $
//  Revision:     $Revision: $
//
//  Functions:    
//
//  History:      Use the CVS command log to display revision history information.
//
//  Description:  OLED functions 
//
//  Notes:        
//                
//
//
// ============================================================================
#include "OLED.h"


//-----------------------------------------------------------------------------
// OLED Init Sequence for Pictiva 128x64 2.7" Yellow   OS128064PK27MY0B00
// is used by OLED_Display_Init()
//-----------------------------------------------------------------------------
#if defined(OS128064PK27MY0B00)

const UINT8 _Display_Init_Seq[] = {
  OLED_SETCONTRAST,0x5D,                                    // for yellow
  OLED_SETCURRENTRANGE_100,

#if defined (_UP_SIDE_DOWN)
  OLED_SETREMAP,0x52,
  OLED_SETDISPLAYOFFSET,0x4C,                               // mapping of RAM to display
#else 
  OLED_SETREMAP,0x41,
  OLED_SETDISPLAYOFFSET,0x44,                               // mapping of RAM to display
#endif

  OLED_SETDISPLAYSTARTLINE,0,                               // TOP
  OLED_SETMULTIPLEXRATIO,63,                                // 64 MUX
  OLED_SETDISPLAYMODE_NORM, 
  OLED_SETPHASELENGTH,0x22,
  OLED_SETROWPERIOD,0x46,
  OLED_SETDISPLAYCLOCKDIVIDE,0x41,
  OLED_SETSEGMENTLOWVOLT,0x0D,
  OLED_SETVCOMH,0x00,
  OLED_SETPRECHAGEVOLT,0x10,
  OLED_SETGREYSCALETABLE,0x01,0x11,0x22,0x32,0x43,0x54,0x65,0x76,
  OLED_SETMASTERCONFIG,0x02,                                // DC-DC 0x02 disabled, 0x03 enabled
};

#endif

//-----------------------------------------------------------------------------
// OLED Init Sequence for Pictiva 128x64 1.6" Yellow   OS128064PK16MY0A01
// is used by OLED_Display_Init()
//-----------------------------------------------------------------------------
#if defined(OS128064PK16MY0A01)

const UINT8 _Display_Init_Seq[] = {
  OLED_SETCONTRAST,0x33,                                    // for yellow
  OLED_SETCURRENTRANGE_100,

#if defined (_UP_SIDE_DOWN)
  OLED_SETREMAP,0x41,
  OLED_SETDISPLAYOFFSET,0x50,                               // mapping of RAM to display
#else 
  OLED_SETREMAP,0x52,
  OLED_SETDISPLAYOFFSET,0x40,                               // mapping of RAM to display
#endif

  OLED_SETDISPLAYSTARTLINE,0,                               // TOP
  OLED_SETMULTIPLEXRATIO,63,                                // 64 MUX
  OLED_SETDISPLAYMODE_NORM, 
  OLED_SETPHASELENGTH,0x22,
  OLED_SETROWPERIOD,0x46,
  OLED_SETDISPLAYCLOCKDIVIDE,0x41,
  OLED_SETSEGMENTLOWVOLT,0x0D,
  OLED_SETVCOMH,0x00,
  OLED_SETPRECHAGEVOLT,0x0B,
  OLED_SETGREYSCALETABLE,0x01,0x11,0x22,0x32,0x43,0x54,0x65,0x76,
  OLED_SETMASTERCONFIG,0x02,                                // DC-DC 0x02 disabled, 0x03 enabled
};

#endif

//-----------------------------------------------------------------------------
// Function:    OLED_Interface_Init
// Parameters:  none
// Returns:     nothing
// Description: MCU <> OLED Hardware-Interface initialization 
//-----------------------------------------------------------------------------
void OLED_Interface_Init(void)  {

  _SPI_Init();
  _12V_Init();
  _RST_Init();
  _DC_Init();
}
//-----------------------------------------------------------------------------
// Function:    OLED_VCC_On
// Parameters:  none
// Returns:     nothing
// Description: turn on VCC (12V) OLED power 
//-----------------------------------------------------------------------------
void OLED_VCC_On(void)  {
  
  _12V_Enable();                                            
}
//-----------------------------------------------------------------------------
// Function:    OLED_VCC_Off
// Parameters:  none
// Returns:     nothing
// Description: turn off VCC (12V) OLED power 
//-----------------------------------------------------------------------------
void OLED_VCC_Off(void)  {

  _12V_Disable();                                            
}

//-----------------------------------------------------------------------------
// Function:    OLED_Reset
// Parameters:  none
// Returns:     nothing
// Description: OLED display (hardware) reset  
// Notes:       no specification found on minimum low time???
//-----------------------------------------------------------------------------
void OLED_Reset(void)  {
UINT8 n;
  
  _RST_Assert();
  for(n=0;n<30;n++)  {;}
  _RST_Release();
}
//-----------------------------------------------------------------------------
// Function:    OLED_WriteCmd
// Parameters:  cmd  one byte 
// Returns:     nothing
// Description: transfers one byte of command data
//-----------------------------------------------------------------------------
void OLED_WriteCmd(UINT8 cmd)  {

  _DC_Clr();
  _SPI_Send(cmd);
}
//-----------------------------------------------------------------------------
// Function:    OLED_WriteData
// Parameters:  cmd  one byte 
// Returns:     nothing
// Description: transfers one byte of data
//-----------------------------------------------------------------------------
void OLED_WriteData(UINT8 data)  {

  _DC_Set();
  _SPI_Send(data);
}
//-----------------------------------------------------------------------------
// Function:    OLED_Display_Init
// Parameters:  none
// Returns:     nothing
// Description: initialization sequenz for OLED display 
//              as specified in array _Display_Init_Seq
//-----------------------------------------------------------------------------
void OLED_Display_Init(void)  {
UINT8 i;

  for(i=0;i<sizeof(_Display_Init_Seq);i++)  {
    OLED_WriteCmd(_Display_Init_Seq[i]);
  }
}
//-----------------------------------------------------------------------------
// Function:    OLED_Fill
// Parameters:  left/top are the x/y coordinates of the top/left corner 
//              (for 4bpp left has to be an even number)
//              width/height gives the size of the area to be filled
//              (for 4bpp width has to be an even number)
//              fill is the color (pattern) to fill (high/low nibble specifying the color) (4bpp)
// Returns:     nothing
// Description: fills a specified area with a color 
//-----------------------------------------------------------------------------
void OLED_Fill(UINT8 left, UINT8 top, UINT8 width, UINT8 height, UINT8 fill)  {
UINT8 x,y,xe,ye;

  x = left>>1;
  xe = x+width/2-1;


  y = top;
  ye = y+height-1;

  OLED_SetCanvas(left,top,width,height);
  
  while(y<=ye)  {
    x = left>>1;
    while(x<=xe)  {
      OLED_WriteData(fill);
      x++;
    }
    y++;
  }
}
//-----------------------------------------------------------------------------
// Function:    OLED_SetCanvas
// Parameters:  left/top are the x/y coordinates of the top/left corner 
//              (for 4bpp x has to be an even number)
//              width/height of image (has to match image properties)
//              (for 4bpp width has to be an even number)
// Returns:     nothing
// Description: sets a canvas (an sub-set of the display area) to be active
//-----------------------------------------------------------------------------
void OLED_SetCanvas(UINT8 left, UINT8 top, UINT8 width, UINT8 height)  {
UINT8 x,y,xe,ye;

  x = left>>1;
  xe = x+(width>>1)-1;


  y = top;
  ye = y+height-1;

  if(xe>63) xe=63;     //clipping
  if(ye>63) ye=63;     //clipping
  
  
  OLED_WriteCmd(OLED_SETXADD);
  OLED_WriteCmd(x);
  OLED_WriteCmd(xe);                 // 0..63  (128/2)
  
  OLED_WriteCmd(OLED_SETYADD);
  OLED_WriteCmd(y);
  OLED_WriteCmd(ye);                 // 0..63 (visual) 79 total
}

//-----------------------------------------------------------------------------
// Function:    OLED_DrawImage
// Parameters:  left/top are the x/y coordinates of the top/left corner 
//              (for 4bpp x has to be an even number)
//              width/height of image (has to match image properties)
//              (for 4bpp width has to be an even number)
// Returns:     nothing
// Description: sets a canvas the size of width/heigth and fills it with the
//              data of the image 
//-----------------------------------------------------------------------------
/*
version without mask

void OLED_DrawImage(UINT8 left, UINT8 top, UINT8* image)  {
UINT8* ptr;
UINT8 x,y,xe,ye;
UINT8 width,height;

  ptr   = image;                                            // pointer to image data

  width = *ptr++;
  height= *ptr++;
  OLED_SetCanvas(left,top,width,height);                    // set canvas


  x = left>>1;
  xe = x+(width>>1);
  y = top;
  ye = y+height;
  if(ye>63) ye=63;                                          //clipping

  while(y<ye)  {
    x = left>>1;
    while(x<xe)  {

      if(x>63)  {                                           //clipping     
        ptr++;
      }else{
        OLED_WriteData(*ptr++);
      }
      x++;
    }
    y++;
  }
}
*/
//-----------------------------------------------------------------------------
// Function:    OLED_DrawImage
// Parameters:  left/top are the x/y coordinates of the top/left corner 
//              (for 4bpp x has to be an even number)
//              width/height of image (has to match image properties)
//              (for 4bpp width has to be an even number)
//              mask allows to shift the gray scale levels (e.g. to darken a picture)
// Returns:     nothing
// Description: sets a canvas the size of width/heigth and fills it with the
//              data of the image 
//-----------------------------------------------------------------------------
void OLED_DrawImage(UINT8 left, UINT8 top, UINT8 mask, UINT8* image)  {
UINT8* ptr;
UINT8 x,y,xe,ye;
UINT8 width,height;
UINT8 hn,ln;

  ptr   = image;                                            // pointer to image data

  width = *ptr++;
  height= *ptr++;
  OLED_SetCanvas(left,top,width,height);                    // set canvas


  x = left>>1;
  xe = x+(width>>1);
  y = top;
  ye = y+height;
  if(ye>63) ye=63;                                          //clipping

  while(y<ye)  {
    x = left>>1;
    while(x<xe)  {

      if(x>63)  {                                           //clipping     

        ptr++;
      }else{
        if(mask==0) {
          OLED_WriteData(*ptr++);
          
        }else{
          
          hn = *ptr++;
          ln = (hn&0x0F);                                   // low nibble
          if(ln<mask)
            ln = 0;
          else 
            ln = ln-mask;
          
          
          
          hn = (hn&0xF0);                                   // high nibble

          if(hn<mask<<4)
            hn = 0; 
          else 
            hn = hn-(mask<<4);


          OLED_WriteData((hn&0xF0)|(ln&0x0F));
        }
      }
      x++;
    }
    y++;
  }
}

//-----------------------------------------------------------------------------
// Function:    OLED_WriteString5x7
// Parameters:  left/top are the x/y coordinates of the top/left corner 
//              (for 4bpp x has to be an even number)
//              cc is the pattern to fill (high/low nibble specifying the color) (4bpp)
// Returns:     nothing
// Description: writes the text to the display
//-----------------------------------------------------------------------------
void OLED_WriteString5x7(UINT8 left,UINT8 top,UINT8 cc, char* text)  {
  UINT8 b,i,xstep,ystep;                                    //current position
  UINT16 a;

  i=0;
  while (text[i]!=0)  {
	  a=(text[i]-32);                                         //index of symbol into font array
	  i++;
	  for(xstep=0;xstep<3;xstep++)  {
	    OLED_SetCanvas(left,top,2,8);
	    left+= 2;
	    ystep=0x01;
	    while(ystep)  {
  			b=cc;
  			if(!(FONT5X7[a][xstep*2]&ystep)) b&=0x0F;
  			if(!((FONT5X7[a][xstep*2+1]&ystep)&&(xstep<2))) b&=0xF0;
  			ystep<<=1;
		    OLED_WriteData(b);
 			}
    }
  }	
}
//-----------------------------------------------------------------------------
// Function:    OLED_WriteString 8x15
// Parameters:  left/top are the x/y coordinates of the top/left corner 
//              (for 4bpp x has to be an even number)
//              cc is the pattern to fill (high/low nibble specifying the color) (4bpp)
// Returns:     nothing
// Description: writes the text to the display
//-----------------------------------------------------------------------------
void OLED_WriteString8x15(UINT8 left,UINT8 top,UINT8 cc, char* text)  {

  UINT8 b,i,xstep,ystep;                                    //current position
  UINT16 a;
  UINT8 y;

  i=0;
  while (text[i]!=0) 	{
    a=text[i]-32;
    i++;
    for (xstep=0;xstep<4;xstep++)  {
	    OLED_SetCanvas(left,top,2,16);
      y = top;
	    left += 2;
      ystep = 0x01;
      while(ystep)  {
   			b = cc;
   			if(!(FONT8X15[a][xstep*2][0]&ystep)) b&=0x0F;
   			if(!((FONT8X15[a][xstep*2+1][0]&ystep))) b&=0xF0;
   			ystep <<= 1;
		    OLED_WriteData(b);
   			y++;
 			}
      ystep = 0x01;
      while(ystep)  {
  			b = cc;
  			if(!(FONT8X15[a][xstep*2][1]&ystep))     b &= 0x0F;
  			if(!((FONT8X15[a][xstep*2+1][1]&ystep))) b &= 0xF0;
  			ystep <<= 1;
        if(!(y>63))                                         //clipping
		      OLED_WriteData(b);
  			y++;
 			}
    } 
  }	
}