/******************************************************************************
 Copyright  1995-2005 Freescale Semiconductor Inc.
 All Rights Reserved

 This is unpublished proprietary source code of Freescale Semiconductor Inc.
 The copyright notice above does not evidence any actual or intended
 publication of such source code.

 $Date: 2006/12/25 08:31:37 $
 $Id: net_demo.c,v 1.3 2006/12/25 08:31:37 hilela Exp $
 $Source: /home/swproj/sw/REPOSITORY/dsp/SmartDSP/demos/starcore/msc814x/net_multicore/net_demo.c,v $
 $Revision: 1.3 $

**************************************************************************//*

 @File         net_demo.c

 @Description   Pacsun NET program based off of the SDOS Net demo for  
 				use with the MSC8144 Ethernet Performance Application Note.

                This application enables NET module for up to 4 cores and 2 UECs, and
                performs echo operation on ports higher then 9999 for both .
 
                Refer to demo documenation for details

 @Cautions      None.
 
*//***************************************************************************/

/* control for net & frame module */
#define FRAME_SINGLE_BUFFER_ONLY

#include "smartdsp_os.h"
#include "os_config.h"

#include "os_mmu.h"
#include "os_cache.h"
#include "msc814x_mmu.h"

#include "msc814x_uec.h"
#include "net_if_ethernet.h"
#include "net_udp.h"
#include "net_arp.h"
#include "app_config.h"
#include "msc814x_hwi.h"


/* network configuration */
#define UDP_PORT                    10000

#define DEFAULT_GATEWAY_IP_ADDRESS	0x0A00008A  /* 10.0.0.138 */
#define BASE_IP_ADDRESS	            0x0A000001  /* 10.0.0.1   */

#define BASE_MAC_ADDR_HIGH32  0xc2c3c4c5        /* c2-c3-c4-c5-c6-c7 */
#define BASE_MAC_ADDR_LOW16   0xc6c7

/* this core offset for channel/mac/ip */
uint8_t        core_channel_offset;

//****************************************************************************
// Variables
//****************************************************************************
bio_dev_handle   uec_handle[TEST_NUM_UECS];          /**< UEC device handler              */    

os_frames_pool_t    *sb_frames_pool;
os_mem_part_t       *buffers_pool;
uint8_t             mem_manager[MEM_PART_SIZE(TEST_NUM_OF_BUFS)];

struct ethercom     ifn[TEST_NUM_OF_CHANNELS];
os_socket_handler_t test_sockets[NUM_OF_SOCKETS_IN_TEST];

typedef struct
{
	bio_channel_t	channel;			/**< BIO channel */
	uint8_t			device_num; 		/**< UEC number */
	uint8_t			channel_num; 		/**< Local Channel number */
} test_channel_t;

test_channel_t tx_channel[TEST_NUM_OF_CHANNELS];
test_channel_t rx_channel[TEST_NUM_OF_CHANNELS];


/* Even channels go to UEC0 and Odd channels go to UEC1 */

uint32_t  mac_addr[TEST_NUM_OF_CHANNELS][2];
uint32_t  ip_addr [TEST_NUM_OF_CHANNELS];

/* buffers */
extern uint8_t buffers_space[];
#define PERF
unsigned int connectonce[TEST_NUM_OF_CHANNELS];
/**************************************************************************//*
 @Function      receiveCallBack

 @Description   This function is called on any frame recieved
 				by IP/UDP. It sends back all packets with source port
 				numbers greater then 9999.
 

 @Return        OS_SUCCES for port numbers > 9999 and OS_FAIL for other

*//***************************************************************************/
static os_status udpReceiveCallBack(void* channel_num, os_frame_t* frame)
{
	struct sockaddr_in dest;
	os_status	status;
	uint32_t	channel = (uint32_t)channel_num;
	
	udpGetSourceAddr(frame, &dest);

	status = udpSendTo(test_sockets[channel], frame, &dest);
	if(status != OS_SUCCESS)
		osFrameFree(sb_frames_pool, frame);
	return OS_SUCCESS;

	if (dest.sin_port == 10001)
	{
#ifdef PERF	
		if (connectonce[0] == 0)
		{
			status = udpConnect(test_sockets[channel], &dest);
			connectonce[0]++;
		}
#endif		
		dest.sin_port = dest.sin_port + 1000;
#ifdef PERF
		status = udpSend(test_sockets[channel], frame);
#else
		status = udpSendTo(test_sockets[channel], frame, &dest);
#endif
		if(status != OS_SUCCESS)
			osFrameFree(sb_frames_pool, frame);
		return OS_SUCCESS;
	}


	else if (dest.sin_port == 10002)
	{
#ifdef PERF	
		if (connectonce[1] == 0)
		{
			status = udpConnect(test_sockets[channel], &dest);
			connectonce[1]++;
		}
#endif		
		dest.sin_port = dest.sin_port + 1000;
#ifdef PERF
		status = udpSend(test_sockets[channel], frame);
#else
		status = udpSendTo(test_sockets[channel], frame, &dest);
#endif
		if(status != OS_SUCCESS)
			osFrameFree(sb_frames_pool, frame);
		return OS_SUCCESS;
	}

	return OS_FAIL;	// Send ICMP Message
}

/***************************************************************************/
static void initChannelMacIpAddress()
{
	uint8_t channel_num;

	/* setup this core offset */
    core_channel_offset = NUM_OF_CHANNELS_PER_UEC * osGetCoreID();

	/* setup mac/ip address for each channel there being 2 channels for each core*/
	for (channel_num = 0; channel_num < TEST_NUM_OF_CHANNELS; channel_num++)
	{
		/* setup channel mac address */
		mac_addr[channel_num][0] = BASE_MAC_ADDR_HIGH32;
		mac_addr[channel_num][1] = (BASE_MAC_ADDR_LOW16 + channel_num + core_channel_offset) << 16;

		/* setup channel IP address */
		ip_addr[channel_num] = BASE_IP_ADDRESS + channel_num + core_channel_offset;
	}
}

/***************************************************************************/

static void	initArpProxy()
{
	int8_t     core;
	uint8_t    channel_num;
	os_status  status;
	uint32_t   ift_mac_addr[2];
	uint32_t   ift_ip_addr;

	for(core=0;core < OS_NUM_OF_CORES; core++)
		for (channel_num = 0; channel_num < TEST_NUM_OF_CHANNELS; channel_num++)
		{
			ift_mac_addr[0] = mac_addr[channel_num][0];
			ift_mac_addr[1] = mac_addr[channel_num][1] + (core*TEST_NUM_OF_CHANNELS << 16);
			ift_ip_addr     = ip_addr[channel_num] + core*TEST_NUM_OF_CHANNELS;
	
			status = arpProxyAdd ((struct ifnet*)&(ifn[0]),
			                      (uint8_t*)ift_mac_addr,
			                      (uint8_t*)&ift_ip_addr, 6, 4);
			OS_ASSERT_COND(status == OS_SUCCESS);
		}
}


/***************************************************************************/

static void cleanBufferSpace(void)
{
	buffers_space[0] = 0;
}


/***************************************************************************/

static os_status appInit()
{
    uec_channel_params_t    txChanParams, rxChanParams;
    bio_dev_open_params_t   dev_open_params;
    bio_ch_open_params_t    ch_open_params;
    os_status               status;
    struct sockaddr_in      sin[TEST_NUM_OF_CHANNELS];
    uint8_t					channel_num = 0, device_num = 0;


    /* create the frames pool */
    sb_frames_pool = osFramePoolCreate(TEST_FRAMES_IN_POOL, 1, 0, 0, OS_MEM_LOCAL);
    OS_ASSERT_COND(sb_frames_pool != NULL);

	/* prepare buffer's pool */
    cleanBufferSpace();

    buffers_pool = osMemPartCreate(TEST_DATA_SIZE,
                                   TEST_NUM_OF_BUFS,
                                   buffers_space,
                                   ALIGNED_64_BYTES,
                                   OFFSET_0_BYTES,
                                   (os_mem_part_t *)&mem_manager,
                                   0);


	/* allocate mac/ip addresses */
	initChannelMacIpAddress();
 
	/* open the UEC device, with default parameters */
	dev_open_params.common_pool = NULL;
	dev_open_params.lld_params = NULL;
	
	
	
	for(device_num = 0; device_num < TEST_NUM_UECS; device_num++)
	{
		connectonce[device_num] = 0;

		if(device_num == 0)
		{
			uec_handle[0] = osBioDeviceOpen(UEC0_DEVICE_NAME, &dev_open_params);
		    OS_ASSERT_COND(uec_handle[0] != NULL);				
		}
		else if (device_num == 1)
		{
			uec_handle[1] = osBioDeviceOpen(UEC1_DEVICE_NAME, &dev_open_params);
    		OS_ASSERT_COND(uec_handle[1] != NULL);			
		}
		else 
			OS_ASSERT;


    	/* open channels */
    	/* We put first channel to UEC0, second to UEC1 etc. */
    	for (channel_num = device_num; channel_num < TEST_NUM_OF_CHANNELS; channel_num += TEST_NUM_UECS)
		{
		    /* open the receive channel */
		    rxChanParams.bd_ring_len = 8;
		    rxChanParams.addr_high32 = mac_addr[channel_num/TEST_NUM_UECS][0];
		    rxChanParams.addr_low16  = mac_addr[channel_num/TEST_NUM_UECS][1] >> 16;
   			rxChanParams.coherency_en = DCACHE_ENABLE;
		    rxChanParams.interrupt_en = TRUE;
		
			rx_channel[channel_num].device_num = device_num;
			rx_channel[channel_num].channel_num = channel_num;
		
		    ch_open_params.channel_num = channel_num/TEST_NUM_UECS + core_channel_offset;
		    ch_open_params.frames_pool = sb_frames_pool;
		    ch_open_params.callback = etherInput;
		    ch_open_params.cb_parameter = (void *)&(ifn[channel_num]);
		    ch_open_params.buffers_pool = buffers_pool;
			ch_open_params.lld_params = &rxChanParams;
			
   			status = osBioChannelOpen(uec_handle[device_num], &(rx_channel[channel_num].channel), 
   	                      BIO_READ | BIO_ACTIVE, &ch_open_params);

		    OS_ASSERT_COND(status == OS_SUCCESS);

			/* open the transmit channel */
			/* Note: the MAC address is not used when opening a Tx channel */
		    txChanParams.bd_ring_len = 32;
   		    txChanParams.coherency_en = DCACHE_ENABLE;
		    txChanParams.interrupt_en = FALSE;
	
	    	tx_channel[channel_num].device_num = device_num;
	    	tx_channel[channel_num].channel_num = channel_num;
		
		    ch_open_params.channel_num = channel_num/TEST_NUM_UECS + core_channel_offset;
		    ch_open_params.callback = NULL;
		    ch_open_params.cb_parameter = NULL;
			ch_open_params.buffers_pool = NULL;
			ch_open_params.lld_params = &txChanParams;
	
	
			status = osBioChannelOpen(uec_handle[device_num], &(tx_channel[channel_num].channel), 
	    	                      BIO_WRITE | BIO_ACTIVE, &ch_open_params);

		}
	
		/* Note: this must be called after all the channels have been opened ! */
		osWaitForAllCores();
		if (osGetCoreID() == osGetMasterCore())
		{
		status= 	osBioDeviceCtrl(uec_handle[device_num], UEC_CMD_ENABLE, NULL);

		}
	}
	
	udpInit(NUM_OF_SOCKETS_IN_TEST);

	/* open UDP sockets */
	/* Do not need to know device name - it is already embedded in channel */
	for (channel_num = 0; channel_num < TEST_NUM_OF_CHANNELS; channel_num++)
	{
		/* Create the Ethernet Interface and set the IP addresses */
		status = ifCreate((struct ifnet*)&(ifn[channel_num]), buffers_pool, sb_frames_pool,
		                  &(tx_channel[channel_num].channel), &(rx_channel[channel_num].channel),
		                  (uint8_t*)mac_addr[channel_num/TEST_NUM_UECS], IFT_ETHER);
		OS_ASSERT_COND(status == OS_SUCCESS);
	
		sin[channel_num].sin_addr.s_addr = DEFAULT_GATEWAY_IP_ADDRESS;
		inSetDefGateway((struct ifnet*)&(ifn[channel_num]), (struct sockaddr*)&(sin[channel_num]));
	
		sin[channel_num].sin_addr.s_addr = ip_addr[channel_num/TEST_NUM_UECS];
		inSetAddr((struct ifnet*)&(ifn[channel_num]), (struct sockaddr*)&(sin[channel_num]));

		status = udpSocket(&test_sockets[channel_num]);
		OS_ASSERT_COND(status == OS_SUCCESS);
	
		inSetUdpCallBack((struct ifnet*)&(ifn[channel_num]), udpReceiveCallBack, (void*)channel_num);
	
		sin[channel_num].sin_addr.s_addr = ip_addr[channel_num/TEST_NUM_UECS];
		sin[channel_num].sin_port = UDP_PORT;
		status = udpBind(test_sockets[channel_num], &(sin[channel_num]));
		OS_ASSERT_COND(status == OS_SUCCESS);
	}
	
	/* setup ARP protocol (ALL on channel 0 of core 0) */
	if (osGetCoreID() == osGetMasterCore())
	{
		initArpProxy();
	}

    return OS_SUCCESS;
}

/***************************************************************************/

static void appBackground(void)
{

	while(1)
	{
		;
	}
}

/***************************************************************************/

void main()
{
    os_status status;   // The only local variable in main()

    /* OS Initialization - call before any other OS API calls. */
    status = osInitialize();
    if (status != OS_SUCCESS) OS_ASSERT;
 
    /* Interrupts are disabled until osStart() is called.
       You must not enable interrupts at this point !!! */
 
    /* Place any required initialization code within appInit().
       Using local variables inside main() is NOT recommended, because
       its stack is never restored - osStart() never returns !!! */
    status = appInit();
    if (status != OS_SUCCESS) OS_ASSERT;

    /* Start the OS with the background task. OS ticks now start.
       appBackground() should not return unless there is an error. */
    status = osStart(appBackground);
    if (status != OS_SUCCESS) OS_ASSERT;
 
    /* Execution reaches this point only if an error occurs. */
}

/***************************************************************************/
