/******************************************************************************
 Copyright  1995-2006 Freescale Semiconductor Inc.
 All Rights Reserved

 This is unpublished proprietary source code of Freescale Semiconductor Inc.
 The copyright notice above does not evidence any actual or intended
 publication of such source code.
 
 $Date: 2007/02/12 15:13:08 $
 $Id: uec_demo.c,v 1.17 2007/02/12 15:13:08 alexg Exp $
 $Source: /home/swproj/sw/REPOSITORY/dsp/SmartDSP/demos/starcore/msc814x/uec_multicore/uec_demo.c,v $
 $Revision: 1.17 $
 **************************************************************************//*

 @File          uec_demo.c

 @Description   MSC814X UEC (Ethernet) demo - Altered for MSC8144 Ethernet Performance
 				Application Note. Example for UEC echo with 4 cores and 2 UECs..

                This is a demo application that performs Ethernet loopback
                through the QE UCC Ethernet Controller (UEC) of the
                MSC814X. The UEC is a BIO type driver, and therefore
                accessed through the BIO API.
 
                Requirements:
                -------------
                Make sure that the board is configured for Ethernet operation.
                Pin-Mux has to be 6, RJ45-D has to be connected to external loopback.
 
                The demo requires an external loopback connection (where the
                PHY Tx is connected to the PHY Rx).
 
                Initialization Phase:
                ---------------------
                appInit() initializes the pools of frames and buffers, gets a
                handle to the UEC device(s) and then opens one transmit channel
                and one receive channel. The receive channel is assigned a
                MAC address as the filter for incoming messages.
                appTxCallback() and appRxCallback() are assigned as the
                corresponding callback routines for the channels.
 
                The UEC configuration parameters can be found in the
                msc814x_config.c file. Note that changing the data size or
                the number of transmitted frames may require adjustments of
                those configuration parameters.
 
                Runtime Phase:
                --------------
                appBackground() is the runtime background (idle) task; first
                it calls appTransmit(), which transmits the pre-defined number
                of frames, and then it waits in a loop until all frames are
                received correctly. appRxCallback() checks that each received
                frame contains the expected data (and asserts in case a data
                mismatch is identified). appTxCallback() only frees the
                transmitted frames.
 
                The data inside the frames is constant, except for a sequence
                number and the core ID which are set in each frame.
 
                After all the frames were received, the UEC transmitter is
                gracefully stopped (by the master core) and the demo ends.

 @Cautions      None.
 
*//***************************************************************************/
#include <stdio.h>
#include "smartdsp_os.h"
#include "os_config.h"

#include "os_mmu.h"
#include "msc814x_uec.h"
#include "msc814x_hwi.h"
#include "app_config.h"
//andrew
#include "net_utils.h"
///////////


// disable prints
// #define printf(...) ;

// Define For Rx Polling
//#define RX_POLLING

//****************************************************************************
// Variables
//****************************************************************************
bio_dev_handle   uec_handle[TEST_NUM_UECS];          /**< UEC device handler              */    
bio_dev_handle   uec_handle_1;          /**< UEC device handler              */
uint8_t          core_channel_offset;   /**< UEC channel offset of this core */

typedef struct
{
	bio_channel_t	channel;			/**< BIO channel          */
	uint8_t			device_num; 		/**< UEC number           */
	uint8_t			channel_num; 		/**< Local Channel number */
} test_channel_t;

test_channel_t tx_channel[TEST_NUM_CHANNELS];
test_channel_t rx_channel[TEST_NUM_CHANNELS];

volatile uint32_t rx_counter[TEST_NUM_CHANNELS]={0};
volatile uint32_t tx_counter[TEST_NUM_CHANNELS]={0};
volatile bool     tx_stop_called = FALSE;

os_frames_pool_t    *sb_frames_pool;
os_mem_part_t       *buffers_pool;
uint8_t             mem_manager[MEM_PART_SIZE(TEST_NUM_OF_BUFS)];

extern uint8_t buffers_space[];

/* The frame below contains the individual type (unicast) destination address,
   that was set for the Rx channel. */

uint32_t frame_data[TEST_DATA_SIZE] =
{
    0x02608c87, 0x65400000, 0x00006364, 0x01eeffff, 0x11111111, 0xffffffff,
    0x22222222, 0xffffffff, 0x33333333, 0xffffffff, 0x44444444, 0xffffffff,
    0x55555555, 0xffffffff, 0x66666666, 0xffffffff, 0x77777777, 0xffffffff,
    0x88888888, 0xffffffff, 0x99999999, 0xffffffff, 0xaaaaaaaa, 0xffffffff,
    0xbbbbbbbb, 0xffffffff, 0xcccccccc, 0xffffffff, 0xdddddddd, 0xffffffff,
 
    0xbbbbbbbb, 0xffffffff, 0xcccccccc, 0xffffffff, 0xdddddddd, 0xffffffff,
    0xbbbbbbbb, 0xffffffff, 0xcccccccc, 0xffffffff, 0xdddddddd, 0xffffffff,
    0x11111111, 0x22222222, 0x33333333, 0x44444444, 0x12345678, 0x11223344,
    0x11111111, 0x22222222, 0x33333333, 0x44444444, 0x12345678, 0x11223344,
    0x11111111, 0x22222222, 0x33333333, 0x44444444, 0x12345678, 0x11223344,
 
    0x11111111, 0x22222222, 0x33333333, 0x44444444, 0x12345678, 0x11223344,
    0x11111111, 0x22222222, 0x33333333, 0x44444444, 0x12345678, 0x11223344,
    0x11111111, 0x22222222, 0x33333333, 0x44444444, 0x12345678, 0x99999999,
    0x02608c87, 0x65400000, 0x00006364, 0x012affff, 0x11111111, 0xffffffff,
    0x22222222, 0xffffffff, 0x33333333, 0xffffffff, 0x44444444, 0xffffffff,
 
    0x55555555, 0xffffffff, 0x66666666, 0xffffffff, 0x77777777, 0xffffffff,
    0x88888888, 0xffffffff, 0x99999999, 0xffffffff, 0xaaaaaaaa, 0xffffffff,
    0xbbbbbbbb, 0xffffffff, 0xcccccccc, 0xffffffff, 0xdddddddd, 0xffffffff,
    0xbbbbbbbb, 0xffffffff, 0xcccccccc, 0xffffffff, 0xdddddddd, 0xffffffff,
    0xbbbbbbbb, 0xffffffff, 0xcccccccc, 0xffffffff, 0xdddddddd, 0xffffffff,
 
    0x11111111, 0x22222222, 0x33333333, 0x44444444, 0x12345678, 0x11223344,
    0x11111111 };

/*****************************************************************************/

static void appTxCallback(void *param, uint32_t data, uint32_t error_status)
{
	test_channel_t 	*test_channel = (test_channel_t *)param;
	uint16_t		device_num = test_channel->device_num;
    uint16_t		channel_num = test_channel->channel_num;
    VAR_UNUSED(error_status);
 
    /* possibly do something with the frame, or check errors */
 
    /* delete the frame */
    osFrameFree(sb_frames_pool, (os_frame_t *)data);
    tx_counter[channel_num]++;
}


#define MY_MAC_ADDRESS mac_address
unsigned char mac_address[] = {0xC0,0xC3,0xC4,0xC5,0xC6,0xC7};
/*****************************************************************************/
static void appRxCallback(void *param, uint32_t unused, uint32_t error_status)
{
    os_frame_t  *frame;
    uint8_t     *data;
    uint16_t    length;
    int         j;
    test_channel_t 	*test_channel = (test_channel_t *)param;
    uint16_t		device_num = test_channel->device_num;
    uint16_t		channel_num = test_channel->channel_num;
 

    VAR_UNUSED(unused);
 
    OS_ASSERT_COND(error_status == 0);
 
    /* get the received frame from BIO */
    frame = osBioChannelRx(&(test_channel->channel));
 
    data = osFrameSingleBufferGet(frame, &length);
 
    /* transmit the frame */
	while (osBioChannelTx(&(tx_channel[channel_num].channel), frame) != OS_SUCCESS) ;

}

/*****************************************************************************/
static void appTxStop(uint32_t param)
{
    VAR_UNUSED(param);
    printf("Core %d: Tx stopped.\n", osGetCoreID());
    tx_stop_called = TRUE;
}

/*****************************************************************************/
static void appTransmitAndReceive()
{
    os_frame_t  *frame;
    uint8_t     *data;
    int         i, j;
    uint8_t		channel_num;
 
	/* prepare and transmit all frames */
	for(channel_num = 0; channel_num < TEST_NUM_CHANNELS; channel_num++)
		for (i=0; i < TEST_NUM_OF_TX_FRAMES; i++)
		{
		    frame = osFrameGet(sb_frames_pool, buffers_pool);
		  	OS_ASSERT_COND(frame != NULL);
	
		    data = osFrameBufferNew(frame);
		    OS_ASSERT_COND(data != NULL);

		    for (j=0; j < (((TEST_DATA_SIZE-CRC_SIZE) >> 2) - 1); j++)
		    {
		        *(uint32_t *)(data + (j << 2)) = frame_data[j];
		    }
	
		    *(uint32_t *)(data + (j << 2)) = i;
	
			/* set correct source address according to channel number and device number */
			*(uint32_t *)(data + 8) += channel_num+core_channel_offset;

		    /* set correct destination address according to channel number and device number */
			*(uint32_t *)(data + 4) += ((channel_num  + core_channel_offset) << 16);
	
		    /* set the buffer to the frame */
		    osFrameSingleBufferSet(frame, data, TEST_DATA_SIZE-CRC_SIZE);

		    /* transmit the frame */
			while (osBioChannelTx(&(tx_channel[channel_num].channel), frame) != OS_SUCCESS) ;

#ifdef RX_POLLING
			/* Poll Rx channel: */
			osBioChannelCtrl(&(rx_channel[channel_num].channel), UEC_CMD_RX_POLL, NULL);
	#endif /* RX_POLLING */
		}
}

/*****************************************************************************/

static void cleanBufferSpace(void)
{
	buffers_space[0] = 0;
}

/*****************************************************************************/

static os_status appInit()
{
    uec_channel_params_t	txChanParams, rxChanParams;
    bio_dev_open_params_t   dev_open_params;
    bio_ch_open_params_t    ch_open_params;
 
    os_status               status;
    uint8_t					channel_num = 0, device_num = 0;
	uint32_t				mac_addr_high32 = 0xc2c3c4c5;
    uint16_t				mac_addr_low16  = 0xc6c7;
 
    /* create the frames pool */
    sb_frames_pool = osFramePoolCreate(TEST_FRAMES_IN_POOL, 1, 0, 0, OS_MEM_LOCAL);
    OS_ASSERT_COND(sb_frames_pool != NULL);
 
	cleanBufferSpace();

    buffers_pool = osMemPartCreate(TEST_DATA_SIZE,
                                   TEST_NUM_OF_BUFS,
                                   buffers_space,
                                   ALIGNED_256_BYTES,
                                   OFFSET_0_BYTES,
                                   (os_mem_part_t *)&mem_manager,
                                   0);

    /* open the UEC device, with default parameters */
	dev_open_params.common_pool = NULL;
	dev_open_params.lld_params = NULL;
 
	/* channels allocated starting from channel_offset */
	core_channel_offset = NUM_OF_CHANNELS_PER_UEC * osGetCoreID(); //changed from TEST_NUM_CHANNELS

	for(device_num = 0; device_num < TEST_NUM_UECS; device_num++)
	{
		if(device_num == 0)
		{
			uec_handle[0] = osBioDeviceOpen(UEC0_DEVICE_NAME, &dev_open_params);
		    OS_ASSERT_COND(uec_handle[0] != NULL);				
		}
		else if (device_num == 1)
		{
			uec_handle[1] = osBioDeviceOpen(UEC1_DEVICE_NAME, &dev_open_params);
    		OS_ASSERT_COND(uec_handle[1] != NULL);			
		}
		else 
			OS_ASSERT;


	    for (channel_num = device_num; channel_num < TEST_NUM_CHANNELS; channel_num += TEST_NUM_UECS)
		{
			/* open the receive channel */
			rxChanParams.addr_high32 = mac_addr_high32;
			rxChanParams.addr_low16 = mac_addr_low16+ core_channel_offset;
			rxChanParams.bd_ring_len = 16;
			rxChanParams.coherency_en = DCACHE_ENABLE; //andrew test 1 DCACHE_ENABLE;
	#ifdef RX_POLLING
			rxChanParams.interrupt_en = FALSE;
		#else
			rxChanParams.interrupt_en = TRUE;
		#endif /* RX_POLLIMG */
		
			rx_channel[channel_num].device_num = device_num;
			rx_channel[channel_num].channel_num = channel_num;
	
			ch_open_params.channel_num = channel_num/TEST_NUM_UECS +core_channel_offset;
			ch_open_params.frames_pool = sb_frames_pool;
			ch_open_params.callback = appRxCallback;
			ch_open_params.cb_parameter = (void *)&(rx_channel[channel_num]);
			ch_open_params.buffers_pool = buffers_pool;
			ch_open_params.lld_params = &rxChanParams;
		
	   		status = osBioChannelOpen(uec_handle[device_num], &(rx_channel[channel_num].channel), 
	   	                      BIO_READ | BIO_ACTIVE, &ch_open_params);
	
		    /* open the transmit channel */
		    /* Note: the MAC address is not used when opening a Tx channel */
		    txChanParams.addr_high32 = mac_addr_high32;
			txChanParams.addr_low16 = mac_addr_low16 + (channel_num/TEST_NUM_UECS) + core_channel_offset;	
		    txChanParams.bd_ring_len = 32;
		    txChanParams.interrupt_en = FALSE;
			txChanParams.coherency_en = DCACHE_ENABLE; //andrew test 2 DCACHE_ENABLE;
		
			tx_channel[channel_num].device_num = device_num;
			tx_channel[channel_num].channel_num = channel_num;
	
		    ch_open_params.channel_num = channel_num/TEST_NUM_UECS + core_channel_offset;
		   	ch_open_params.callback = appTxCallback;
			ch_open_params.cb_parameter = (void *)&(tx_channel[channel_num]);
			ch_open_params.buffers_pool = NULL;
			ch_open_params.lld_params = &txChanParams;
		
			status = osBioChannelOpen(uec_handle[device_num], &(tx_channel[channel_num].channel), 
		    	                      BIO_WRITE | BIO_ACTIVE, &ch_open_params);
		}
	
	    /* Note: this must be called after all the channels have been opened ! */
		osWaitForAllCores();
	
		if (osGetCoreID() == osGetMasterCore())
		{
			status= 	osBioDeviceCtrl(uec_handle[device_num], UEC_CMD_ENABLE, NULL);
		}
	}
    return OS_SUCCESS;
}

/*****************************************************************************/
static void appBackground(void)
{
    uec_stop_params_t  stop_params;
	uec_statistics_t *statistics;
	uec_additional_tx_statistics_t *ext_tx_stats;
	uec_additional_rx_statistics_t *ext_rx_stats;
	uint8_t channel_num;
	uint32_t num_of_loops = 1;

    osWaitForAllCores();
 
    while(num_of_loops > 0)
    {
    	num_of_loops--;
    }
 
	while(1) 
	{

#ifdef RX_POLLING
			/* Poll Rx channel: */
			osBioChannelCtrl(&(rx_channel[channel_num].channel), UEC_CMD_RX_POLL, NULL);
	#endif /* RX_POLLING */
	;
	}
}



/*****************************************************************************/

void  main()
{
    os_status status;   /* The only local variable in main() */

#ifdef INTERNAL_LOOPBACK
    printf("Internal Loopback Demo on UEC1\n");
    printf("Please make sure PIN-MUX = 6\n");
#else
    /* marvell errata for 1G external loopback */
   	extern bool bsp_marvell_external_loopback;
    bsp_marvell_external_loopback = FALSE;
#endif
 
    /* OS Initialization - call before any other OS API calls. */
    status = osInitialize();
    if (status != OS_SUCCESS) OS_ASSERT;
 
    /* Interrupts are disabled until osStart() is called.
       You must not enable interrupts at this point !!! */
 
    /* Place any required initialization code within appInit().
       Using local variables inside main() is NOT recommended, because
       its stack is never restored - osStart() never returns !!! */
    status = appInit();
    if (status != OS_SUCCESS) OS_ASSERT;
 
    /* Start the OS with the background task. OS ticks now start.
       appBackground() should not return unless there is an error. */
    status = osStart(appBackground);
    if (status != OS_SUCCESS) OS_ASSERT;

    /* Execution reaches this point only if an error occurs. */
}

/*****************************************************************************/
