//////////////////////////
//	Project Stationery  //
//////////////////////////

#include <stdio.h>


/*
   System Memory Base Address.
*/
#define     MBAR_ADDR                   0xE0000000



/*
   Registers' offsets.
*/
#define     ERR_INJECT_OFFSET           0x2E08
#define     ERR_DISABLE_OFFSET          0x2E44
#define     ERR_INT_EN_OFFSET           0x2E48
#define     ERR_SBE_OFFSET              0x2E58
#define     SICFR_OFFSET                0x700
#define     SIMSR_L_OFFSET              0x724



/*   
   This value defines the number of single bit errors to be detected before
   raising an exception. For additional detail see the Reference Manual.
*/
#define     ERR_SBE_THRESHOLD      0x00A00000



/*
   Size of the array to be written. This value determines the number of
   accesses to be performed to the memory area with single-bit errors.
   The number of exceptions that will be generated is given by the
   approximate formula:

                          ARRAY_SIZE
       NumExceptions = -----------------
                       ERR_SBE_THRESHOLD

   With ARRAY_SIZE = 1024 and ERR_SBE_THRESHOLD = 0xA0, the number of
   exceptions to be generated during program execution will be 6.
*/
#define     ARRAY_SIZE                  1024



int main ()
{
    
    /* Local variables */
    unsigned char *charPtr;
    unsigned char charArray[ARRAY_SIZE];
    unsigned int i;

    
    /*
       Set the pointer to some harmless addresses in memory. Make sure this
       addresses exist and are free on your system.
     */
    charPtr = (unsigned char*)0x000F0000;



    /*
       Set DDR as the source for the Highest Priority Interrupt (HPI) and
       select Critical Interrupt (cint) as the output for HPI.
       
       SICFR: System Global Interrupt Configuration Register
     
       SICFR[HPI]  = 0x4C (DDR)
       SICFR[HPIT] = 0x2  (cint)
     */
    *((unsigned int*)(MBAR_ADDR + SICFR_OFFSET)) = 0x4C000200;



    /*
       Unmask interrupts originated by DDR.
       
       SIMSR_L: System Internal Interrupt Mask Register (Low)
     
       SIMSR_L = 0x00080000 <=> SIMSR_L[12] = 1 <=> DDR interrupts not masked.
     */
    *((unsigned int*)(MBAR_ADDR + SIMSR_L_OFFSET)) = 0x00080000;



    /* 
       Enable External and Critical interrupts in the Machine State Register.
       
       MSR[EE]=1 (External interrupt enable)
       MSR[CE]=1 (Critical interrupt enable)
     */
    asm("mfmsr r5");
    asm("ori r4, r5, 0x8080");
    asm("mtmsr r4");
    asm("isync");
    asm("sync");



    /*
       Enable interrupt generation when ECC single-bit errors are detected.
       
       ERR_INT_EN: Memory Error Interrupt Enable
    
       ERR_INT_EN[SBEE]=1;
     */
    *((unsigned int*)(MBAR_ADDR + ERR_INT_EN_OFFSET)) = 0x00000004;



    /*
       Set the number of single-bit errors to be detected before an exception
       is generated.
       
       ERR_SBE: Single-Bit ECC Memory Error Management 
    
       Example: ERR_SBE[SBET]=0xA0 (other values can be chosen)
     */
    *((unsigned int*)(MBAR_ADDR + ERR_SBE_OFFSET)) = ERR_SBE_THRESHOLD;



    /*
       Temporarily enable single-bit error injection to generate single-bit
       errors in memory.

       ERR_INJECT: Memory Data Path Error Injection Mask ECC 
           
       ERR_INJECT[EIEN]=1
       ERR_INJECT[EEIM]=0x01
     */
    *((unsigned int*)(MBAR_ADDR + ERR_INJECT_OFFSET)) = 0x00000110;



    /*
       Inject a single-bit ECC error in the charPtr variable.
     */
    *charPtr = 0x12;



    /*
       Single-bit errors already generated and stored in memory, therefore
       disable single-sit error injection.

       ERR_INJECT: Memory Data Path Error Injection Mask ECC

       ERR_INJECT[EIEN]=0
       ERR_INJECT[EEIM]=0
     */
    *((unsigned int*)(MBAR_ADDR + ERR_INJECT_OFFSET)) = 0x00000000;


    
   /*
      This line assigns charPtr's contents to itself. This implies a
      read from memory followed by a write. While the read has no affect
      on the existing single-bit error, during the write the DDR
      controller's single-bit error correction capabilities will correct
      the single-bit error. Therefore, uncommenting this line causes the
      program to run without the exception being generated.

      *charPtr = *charPtr;
    */



    /*
       Enable single-bit error detection (SBED). All other errors are not
       detected nor reported.
       
       ERR_DISABLE: Memory Error Disable
       
       ERR_DISABLE[ACED, MBED, MSED] = 1 (disabled)
       ERR_DISABLE[SBED] = 0 (enabled)
       
     */
    *((unsigned int*)(MBAR_ADDR + ERR_DISABLE_OFFSET)) = 0x000000089;



    /*
       In each iteration of this "for" loop there is one read access to
       the single-bit-errored variable, charPtr. When the number of
       single-bit errors exceeds the threshold set in ERR_SBE[SBET],
       the exception is generated and the PC jumps to the corresponding
       interrupt vector address.
       
       IMPORTANT NOTE: When single-stepping through the code with JTAG,
       the program is unable to return from the exception handler routine
       (not tested with other target debuggers). It is advised to set a
       breakpoint before the end of main(), run the program and then check
       the number of times the exception-handling routine printed on the
       console.

     */
    for (i = 0; i < ARRAY_SIZE; i++)
    {
        charArray[i] = *charPtr;
    }



    /*
       Place a break point here to make sure the program returns from the
       interrupt-handling routine.     
     */
    return 0;

}

