/******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2004-2005 Freescale Semiconductor, Inc.
* (c) Copyright 2001-2004 Motorola, Inc.
* ALL RIGHTS RESERVED.
*
***************************************************************************//*!
*
* @file      controller.c
*
* @author    r79251
* 
* @version   1.1.0.0
* 
* @date      Sep-20-2007
* 
* @brief     PI regulator and PI regulator with a saturation  
*
******************************************************************************/
#include <hidef.h> /* for EnableInterrupts macro */

#include "math.h"
#include "controller.h"

#pragma MESSAGE DISABLE C3604   /* "Static <Internal Object> was not referenced" */


/*****************************************************************************
*
* Function: int regPI(..)
*
* Description:
* PI Controller 8 bit inputs, 16 bit outputs without limitation
*                           ---          -----------
*             w(k)  -------| + |  e(k)  |           |  u(k)
*                          |   |--------|    PI     |---------
*             y(k)  -------| - |        |           |
*                           ---          -----------
*
*      w(t) - desiredTemperature in continuous time domain
*      y(t) - measuredValue (feedback) in continuous time domain
*      e(t) - input error in continuous time domain
*      u(t) - controller output in continuous time domain
*
*      w(k) - desiredTemperature in step k - discrete time domain
*      y(k) - measuredValue (feedback) in step k - discrete time domain
*      e(k) - input error in step k - discrete time domain
*      u(k) - controller output in step k - discrete time domain
*
*
*      The PI controller algorithm in continuous time domain:
*                             /t
*      u(t) = K[e(t) + 1/Ti * | e(t)*dt]                               (1)
*                             /0
*
*      K  - controller gain
*      Ti - integral time constant
*
*      e(t) = w(t) - y(t)                                              (2)
*
*      PI controller expressed in signed byte:
*
*      u_si(k) = KP * e_sb(k) + ui_si(k - 1) + KI * e_sb(k)            (3)
*
*      e_sb(k) = w_sb(k) - y_sb(k)                                     (4)
*
*      ui_si = ui_si(k - 1) +  KI * e_sb(k)                            (5)
*
*      T - sampling time
*     
* Returns: int = ui_si
*
* Global Data:None
*
* Arguments:
*   desiredTemperature      byte        - Desired value (input)
*   measuredValue           byte        - Measured value (input)
*   sPIparams               sPIparams   - controller parameters (input/output)
*
* Range Issues: None
*
* Special Issues:
*   Output, Integral Portion and Control Difference are saturated
*   Set proper value to IntegralPortionK_1 before first calling is recommended
*
*****************************************************************************/

                                                                   
int regPI(BYTE regPI_ControlDifference, REG_PIparamsType *regPI_Params)
{                                    
    
    int proportionalPortion, integralPortion, outputReg;
    
    if (regPI_ControlDifference >= 0)
    {
        /* Positive difference */
        proportionalPortion = (int)((UBYTE)regPI_ControlDifference * regPI_Params->proportionalGain);
        integralPortion     = (int)((UBYTE)regPI_ControlDifference * regPI_Params->integralGain);
        
    }
    else  
    {
        /* Negative difference */
        regPI_ControlDifference   = NEG8 (regPI_ControlDifference);
        proportionalPortion = - (int)((UBYTE)regPI_ControlDifference * regPI_Params->proportionalGain);
        integralPortion     = - (int)((UBYTE)regPI_ControlDifference * regPI_Params->integralGain);

    }
    
    
    regPI_Params->integral_K1 = ADD16(regPI_Params->integral_K1, integralPortion);

    outputReg = ADD16(regPI_Params->integral_K1, proportionalPortion);
    
    return outputReg;
}


/*****************************************************************************
*
* Function: int regPI(..)
*
* Description:
* PI Controller 8 bit inputs, 16 bit outputs without limitation
*                           ---          -----------
*             w(k)  -------| + |  e(k)  |           |  u(k)
*                          |   |--------|    PI     |---------
*             y(k)  -------| - |        |           |
*                           ---          -----------
*
*      w(t) - desiredTemperature in continuous time domain
*      y(t) - measuredValue (feedback) in continuous time domain
*      e(t) - input error in continuous time domain
*      u(t) - controller output in continuous time domain
*
*      w(k) - desiredTemperature in step k - discrete time domain
*      y(k) - measuredValue (feedback) in step k - discrete time domain
*      e(k) - input error in step k - discrete time domain
*      u(k) - controller output in step k - discrete time domain
*
*
*      The PI controller algorithm in continuous time domain:
*                             /t
*      u(t) = K[e(t) + 1/Ti * | e(t)*dt]                               (1)
*                             /0
*
*      K  - controller gain
*      Ti - integral time constant
*
*      e(t) = w(t) - y(t)                                              (2)
*
*      PI controller expressed in signed byte:
*
*      u_si(k) = KP * e_sb(k) + ui_si(k - 1) + KI * e_sb(k)            (3)
*
*      e_sb(k) = w_sb(k) - y_sb(k)                                     (4)
*
*      ui_si = ui_si(k - 1) +  KI * e_sb(k)                            (5)
*
*      T - sampling time
*     
* Returns: int = ui_si
*
* Global Data:None
*
* Arguments:
*   desiredTemperature      byte        - Desired value (input)
*   measuredValue           byte        - Measured value (input)
*   sPIparams               sPIparams   - controller parameters (input/output)
*
* Range Issues: None
*
* Special Issues:
*   Output, Integral Portion and Control Difference are saturated
*   Set proper value to IntegralPortionK_1 before first calling is recommended
*
*****************************************************************************/


int regPIsat(BYTE regPI_ControlDifference, REG_PIparamsType *regPI_Params, UBYTE bSatFlag )
{                                    
    
       int proportionalPortion, integralPortion, outputReg;
    
    if (regPI_ControlDifference >= 0)
    {
        /* Positive difference */
        proportionalPortion = (int)((UBYTE)regPI_ControlDifference * regPI_Params->proportionalGain);
        integralPortion     = (int)((UBYTE)regPI_ControlDifference * regPI_Params->integralGain);
        
    }
    else  
    {
        /* Negative difference */
        regPI_ControlDifference   = NEG8 (regPI_ControlDifference);
        proportionalPortion = - (int)((UBYTE)regPI_ControlDifference * regPI_Params->proportionalGain);
        integralPortion     = - (int)((UBYTE)regPI_ControlDifference * regPI_Params->integralGain);

    }
    
    if (bSatFlag == 0) {
        regPI_Params->integral_K1 = ADD16(regPI_Params->integral_K1, integralPortion);
    }
    
    if (regPI_Params->integral_K1 < 0) regPI_Params->integral_K1=0;
    
    outputReg = ADD16(regPI_Params->integral_K1, proportionalPortion);
   
    return outputReg;
}