/******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2004-2005 Freescale Semiconductor, Inc.
* (c) Copyright 2001-2004 Motorola, Inc.
* ALL RIGHTS RESERVED.
*
***************************************************************************//*!
*
* @file      math.c
*
* @brief     mathematical functions
*
* @author    R79251
* 
* @version   1.1.0.0
* 
* @date      Sep-2007
* 
******************************************************************************/

#include <hidef.h> /* for EnableInterrupts macro */
#include "math.h"



#pragma MESSAGE DISABLE C5703   /* Parametr 'operand_X' declared in function but not referenced */
#pragma MESSAGE DISABLE C1404   /* Return expected */
#pragma MESSAGE DISABLE C5661   /* "Not all control paths return a value" */



/******************************************************************************
* Global functions
******************************************************************************/

/***************************************************************************//*!
*
* @brief  Negation with saturation for neg -128 (0x80) to 127 (0x7F)
*
* @param  operand_X      operand (byte) that will be negated
*
* @return This function returns negated operand_X
*
* @remarks for neg -128 (0x80) returns 127 (0x7F)
*
****************************************************************************/


BYTE NEG8 (BYTE operand_X)
{   
    __asm 
    {

       CBEQA #$80, SATURATION                     /* ; if X is max negative value, saturation to max pozitive value */
       COMA                                       /* ; Complement accumulator to acumulator */
       ADD #$1                                    /* ; add one to correct of shift negative value against pozitive value */
       RTS
  SATURATION:
       LDA #$7F                                   /* ; returns the saturation value  */
    }
}  

/***************************************************************************//*!
*
* @brief  Subtraction with saturation to max +127 and -128
*         Operand Y is negated and further a addition with saturation is done 
*
* @param  operand_X   a first operand (byte) for the substraction   
*
* @param  operand_Y   a second operand (byte) that will be negated
*
* @return This function returns result of the subtraction (byte -content of accumulator)
*
* @remarks results are saturated to max +127 and min -128
*
****************************************************************************/


#pragma NO_RETURN
BYTE SUB8 (BYTE operand_X, BYTE operand_Y)
{   

    __asm 
    {
                                             
        COMA                                       /* ; complement of accumulator                                     */
        ADD #$1                                    /* ; addition of 1 to accu to correct shift across zero            */
        BRSET 7,operand_X, TEST_NEGATIVE_Y         /* ; Test if X is negative, jump TEST_NEGATIVE_Y                   */
        BRCLR 7,operand_Y, SIMPLE_ADD              /* ; When X was positive, now Y is tested. If also Y is pozitive   */
    ADD_SAT:                                       /* ; jump SIMPLE_ADD                                               */
        ADD operand_X                              /* ; operand_Y is already in accu, X is added                      */
        STA operand_X                              /* ; operand_X is used as scratch variable                         */
        BRCLR 7,operand_X, RETURN                  /* ; test if result is positive, YES -  */
        LDA #$7F                                   /* ; saturation to max positive value of byte - 0x7F               */
        BRA RETURN        
    
    TEST_NEGATIVE_Y:
        BRSET 7,operand_Y, SIMPLE_ADD
        ADD operand_X
        STA operand_X                              /* ; operand_X is used as scratch variable */
        BRSET 7,operand_X, RETURN                  /* ; test of sign inversion, if false > to RETURN */
        LDA #$80                                   /* ; test of sign inversion, if true > negative saturation  */
      
    RETURN:
        RTS
    SIMPLE_ADD:
        ADD operand_X
        RTS    
    }
}  

/***************************************************************************//*!
*
* @brief  Unsigned subtraction with saturation to max +127 and -128
*         resolution is BYTE, but both operants are UBYTE
*         (BYTE = UBYTE - UBYTE)
*
* @param  operand_X   a first operand (ubyte) for the substraction   
*
* @param  operand_Y   a second operand (ubyte) that will be negated
*
* @return This function returns result of the subtraction (byte - content of accumulator)
*
* @remarks results are saturated to max +127 and min -128
*
****************************************************************************/

#pragma NO_RETURN
BYTE uSUB8 (UBYTE operand_X, UBYTE operand_Y)
    
{   
    __asm 
    {
        LDA operand_X                             
        SUB operand_Y;                           ; /*  operand_X - operand_Y                                      */
        STA operand_X                            ; /*  operand_X is used as scratch variable                      */
        BCS TEST_OVER                            ; /*  test for zerocrossing                                      */
        BRCLR 7,operand_X, RETURN                ; /*  if resolution is not under zero, test for plus BYTE range  */
        LDA #$7F                                 ; /*  saturation to max plus value for a BYTE                    */
        RTS                                      ; /*  return a saturated value 0x7F                              */ 
    TEST_OVER:
        BRSET 7,operand_X, RETURN                ; /*  if resolution is under zero, test for minus BYTE range     */
        LDA #$80                                 ; /*  saturation to max negative value for a BYTE - 0x80         */
    RETURN:                                                                                                      
        RTS                                      ; /*  return a actual value of accumulator                       */       
            
    }
 }  

/***************************************************************************//*!
*
* @brief  16 bits addition with saturation to min 0x8000 and max 0x7FFF
*         
*
* @param  operand_X   a first operand (integer) for the addition   
*
* @param  operand_Y   a second operand (integer) for the addition
*
* @return This function returns result of the addition stored in variable'temp' 
*
* @remarks results are saturated to max 0x7FFF and min 0x8000
*
****************************************************************************/
 

int ADD16 (int operand_X, int operand_Y)
{   
    int temp;
    
    __asm 
    {
    
        BRSET 7,operand_X, TEST_NEGATIVE_Y          ; /* Test if X is negative, if YES jump TEST_NEGATIVE_Y         */
        BRSET 7,operand_Y, SIMPLE_ADD               ; /* X was positive and now Y is tested.                        */
    ADD_SAT:                                        ; /* If also Y positive, the operands are added with saturation */
        LDA operand_X:1                             ; /* The addition of LSB X is preparing; (LSB X to accu)        */
        ADD operand_Y:1                             ; /* The addition of LSB, bytes is done in accumulator          */
        STA temp:1                                  ; /* The result of LSBs  addition is stored to temp:1           */
        LDA operand_X                               ; /* The addition of MSB bytes is preparing; MSB X to accu      */
        ADC operand_Y                               ; /* The addition with carry of MSB bytes is done               */
        STA temp                                    ; /* The result of MSB addition is stored to temp               */
        BRCLR 7,temp, RETURN                        ; /* Test of result sign bit, negative result is returned       */
        MOV #$7F, temp                              ; /* If result is positive, the max negative value is returned  */
        MOV #$FF, temp:1                            ; /* If result is positive, the max negative value is returned  */
        BRA RETURN                                  ; /* END, returning of the 0x7FFF as result                     */
    
    TEST_NEGATIVE_Y:                                
        BRCLR 7,operand_Y, SIMPLE_ADD               ; /* Test if Y is not negative, if isn't >> simple addition     */
        LDA operand_X:1                             ; /* The addition of LSBs is preparing; (LSB X to accu)         */
        ADD operand_Y:1                             ; /* The addition of LSBs is done in accumulator                */
        STA temp:1                                  ; /* The result of LSBs addition is stored to temp:1            */
        LDA operand_X                               ; /* The addition of MSBs is preparing; high X to accu          */
        ADC operand_Y                               ; /* The addition with carry of MSBs is done                    */
        STA temp                                    ; /* The result of MSBs addition is stored to temp              */
        BRSET 7,temp, RETURN                        ; /* Test if result is neg., if isn't >> saturation to max neg. */
        MOV #$80, temp                              ; /* if isn't >> temp is saturated to max negative value      . */
        MOV #$00, temp:1

        BRA RETURN                                  ; /* END, returning of the 0x8000 as result                     */
    
    SIMPLE_ADD:
        LDA operand_X:1                             ; /* The addition of LSBs is preparing;(LSB X to accu)          */
        ADD operand_Y:1                             ; /* The addition of LSBs is done in accumulator                */
        STA temp:1                                  ; /* The result of LSBs addition is stored to temp:1            */
        LDA operand_X                               ; /* The addition of MSBs is preparing; high X to accu          */
        ADC operand_Y                               ; /* The addition with carry of MSBs is done                    */
        STA temp                                    ; /* The result of MSBs addition is stored to temp              */
    RETURN:          
    }
    return temp;                                    ; /* END, returning of the actual temp value as result          */
}  

/***************************************************************************//*!
*
* @brief  Multiplication together two UBYTE numbers with saturation to min and max integer value
*         
*
* @param  operand_X   a first operand (ubyte) for the multiplication   
*
* @param  operand_Y   a second operand (ubyte) for the multiplication
*
* @return This function returns 16-bit result of the multiplication stored in variable'RESULT' 
*         (RESULT:RESULT+1)
*
* @remarks results are saturated to max 0x7FFF and min 0x8000
*
****************************************************************************/


int MULTI8 (UBYTE operand_X, UBYTE operand_Y)
{   
  UBYTE INDEX; 
  int TEMP;
  unsigned int RESULT;
  
  
  __asm 
  {

UMULT8: MOV operand_X,TEMP:1        ;store multiplicand in partial product LSB
        MOV #8,INDEX                ;set number of loops
        CLR TEMP                    ;zero partial product MSB
        CLR RESULT:1                ;zero result LSB
        CLR RESULT                  ;zero partial product MSB

MULT:   BRCLR 0,operand_Y,NO_MULT   ;branch if operand_y[x] is zero
        LDA TEMP+1                  ;load TEMP LSB
        ADD RESULT:1                ;add TEMP LSB to result LSB
        STA RESULT:1                ;store result LSB
        LDA TEMP                    ;load TEMP MSB
        ADC RESULT                  ;add w/carry TEMP MSB to result MSB
        STA RESULT                  ;store result MSB

NO_MULT: LDA TEMP:1                 ;load TEMP LSB
         LSLA                       ;shift left (x2)
         STA TEMP:1                 ;store TEMP LSB
         LDA TEMP                   ;load TEMP MSB
         ROLA                       ;rotate left with carry
         STA TEMP                   ;store TEMP MSB
         LDA operand_Y              ;load multiplier
         RORA                       ;rotate right
         STA operand_Y              ;store multiplier
         DBNZ INDEX,MULT            ;branch to mult if INDEX not zero
   }
   
   return RESULT;      
  }