/*
Copyright (c) 2008, Freescale Semiconductor Inc.
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.
Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.
Neither the name of Freescale Semiconductor Inc. nor the names of its contributors
may be used to endorse or promote products derived from this software
without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*Version 5*/

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
/*Include library for handling mmap function*/
#include <sys/mman.h>
#include <dirent.h>

typedef signed char s8;
typedef unsigned char u8;
typedef signed short s16;
typedef unsigned short u16;
typedef signed int s32;
typedef unsigned int u32;

/* LAWAR size definitions*/

#define LAWAR_SIZE_BASE         0xa
#define LAWAR_SIZE_4K           (LAWAR_SIZE_BASE+1)
#define LAWAR_SIZE_8K           (LAWAR_SIZE_BASE+2)
#define LAWAR_SIZE_16K          (LAWAR_SIZE_BASE+3)
#define LAWAR_SIZE_32K          (LAWAR_SIZE_BASE+4)
#define LAWAR_SIZE_64K          (LAWAR_SIZE_BASE+5)
#define LAWAR_SIZE_128K         (LAWAR_SIZE_BASE+6)
#define LAWAR_SIZE_256K         (LAWAR_SIZE_BASE+7)
#define LAWAR_SIZE_512K         (LAWAR_SIZE_BASE+8)
#define LAWAR_SIZE_1M           (LAWAR_SIZE_BASE+9)
#define LAWAR_SIZE_2M           (LAWAR_SIZE_BASE+10)
#define LAWAR_SIZE_4M           (LAWAR_SIZE_BASE+11)
#define LAWAR_SIZE_8M           (LAWAR_SIZE_BASE+12)
#define LAWAR_SIZE_16M          (LAWAR_SIZE_BASE+13)
#define LAWAR_SIZE_32M          (LAWAR_SIZE_BASE+14)
#define LAWAR_SIZE_64M          (LAWAR_SIZE_BASE+15)
#define LAWAR_SIZE_128M         (LAWAR_SIZE_BASE+16)
#define LAWAR_SIZE_256M         (LAWAR_SIZE_BASE+17)
#define LAWAR_SIZE_512M         (LAWAR_SIZE_BASE+18)
#define LAWAR_SIZE_1G           (LAWAR_SIZE_BASE+19)
#define LAWAR_SIZE_2G           (LAWAR_SIZE_BASE+20)

#define	ROWAR_MAINT	0x80077000
#define ROWAR_MEM	0x80045000
#define	RIWAR_MEM	0x80f44000

#define LCSBA1CSR	0x5c

#define DSP_RIWTAR4	0x90d60

/***************************************
 Structure for ATMU Outbound Windows
/***************************************/

struct rio_atmu_row {
	u32 rowtar;
	u32 rowtear;
	u32 rowbar;
	u32 pad2;
	u32 rowar;
	u32 pad3[3];
};

/***************************************
 Structure for ATMU Inbound Windows
 ***************************************/

struct rio_atmu_riw {
	u32 riwtar;
	u32 pad1;
	u32 riwbar;
	u32 pad2;
	u32 riwar;
	u32 pad3[3];
};


/* Memory configuration to be used for the code
 *
 * We will use
 * M ROWBAR7 	- 0xdc000000 -> 0xdc3fffff 4M
 * M2/3 ROWBAR8	- 0xd0000000 -> 0xd7ffffff 128M
 *
 * And  on the DSP
 * M2 RIWBAR5  	- 0xd7000000 -> 0xc0000000 1M
 * M3 RIWBAR5  	- 0xd0000000 -> 0xd0000000 16M
 */

#define	RIO_ADDR	0xffec0000	/* Must be on a 4k boundary for linux */
#define	RIO_SIZE	0x20000		/*131k*/

/*Define location of SRIO configuration registers on MPC8572*/#
define	ATMU_OFFSET	0x10c00

#define	MAINT_ADDR	0xdc000000
#define MAINT_SIZE	0x00400000	/*4M*/
#define	M2_ADDR		0xd7000000
#define	M2_SIZE		0x00100000
#define DSP_M2_ADDR	0xc0000000

#define SWITCH_ADDR	0xd0000000
#define SWITCH_SIZE	0x00100000

#define DSP_ADDR		0xd0200000
#define DSP_MAINT_SIZE  0x00100000

#define M3_ADDR		0xd0000000
#define	M3_SIZE		0x00a00000

#define DSP_BOOT_ADDR		0x7b010
#define DSP_BOOT_MAGIC_ADDR	0x7b000
#define DSP_BOOT_MAGIC_1	0x17171717
#define DSP_BOOT_MAGIC_2	0xa5a5a5a5

u32 download_srec(FILE * fd, u32 m2, u32 m3, u8 write);
u8 hex_value(FILE * fd);

void read_filesystem(int , u32 *, int);
void reset_ackIDs();
void synch();
int port_check(u32 switch_regs, u32 port_num);
void dsp_reset(int dsp_no, int switch_count, u32 port_config[][16], int endpoint_count, u32 rio_addr)
void enum_device(u32 switch_device, u32 current_device_mem, u32 current_device_id, int hop_count, int port_num, u32 rio_addr)
void enum_network(int switch_count, u32 port_config[][16], int endpoint_count, u32 rio_addr, int hop_count)
void read_filesystem(int *switch_count, u32 port_config[][16], int *endpoint_count, u32 *immr_base)


extern int errno;
char *datafile;
/*Declare file descriptor*/
int mem_fd;


/***********************************************************************************/
void usage()
{
	printf("riodsp reset <device_id>\n"
	       "riodsp download <device_id> <filename>\n");
}


/***********************************************************************************/

/*Function to check the status of the ESCR register for the relevant port*/

int port_check(u32 switch_regs, u32 port_num)
{

u32 init_status;
u32 err_reg;
u32 port_error;
int enum_status = 0;

/*Read ESCSR register.*/
err_reg = (*(u32 *) (switch_regs + (0x158 + (port_num * 0x20))));
/*printf("Error reg = 0x%x\n", err_reg);*/
init_status = err_reg & 0x00000001;
port_error = err_reg & 0x00000004;

	if (init_status == 1) {
		/*printf("Link is not initialised.  Please check configuration\n");*/
		enum_status = 2;
		}
	else
		{
		/* If there has been an error on the link, reset the error reg and ackIDs*/
		if (port_error == 4) {
			/* Reset error register*/
			*(u32 *) (switch_regs + (0x158 + (port_num * 0x20))) = 0xffffffff;
			/*Reset ackIDs*/
			reset_ackIDs(switch_regs, port_num);
			enum_status = 1;
		}
	}

return enum_status;
}


/***********************************************************************************/

void reset_ackIDs(u32 switch_regs, u32 port_num)
{

*(u32 *) (switch_regs + (0x148 + (port_num * 0x20))) = 0x00000000;

}

/***********************************************************************************/

void enum_device(u32 switch_device, u32 current_device_mem, u32 current_device_id, int hop_count, int port_num, u32 rio_addr)
{

u32 deviceid_read;
volatile static struct rio_atmu_row *dsp_maint;
volatile static struct rio_atmu_row *atmu_regs;

	if ((mem_fd = open("/dev/mem", O_RDWR)) == -1) {
		fprintf(stderr, "dsp: Error: Cannot open /dev/mem\n");
		exit(EXIT_FAILURE);
	}

/*Open access to ATMU registers*/
	atmu_regs =
	    (struct rio_atmu_row *)mmap(0, RIO_SIZE, (PROT_READ | PROT_WRITE),
					MAP_SHARED, mem_fd, rio_addr);

	if ((s32) atmu_regs == -1) {
		printf("Error: mmap failed.\n");
		exit(EXIT_FAILURE);
	}

atmu_regs = (struct rio_atmu_row *)(((u32) atmu_regs) + ATMU_OFFSET);

dsp_maint = atmu_regs + 5;

dsp_maint->rowbar = DSP_ADDR >> 12;
dsp_maint->rowar = ROWAR_MAINT | LAWAR_SIZE_1M;
dsp_maint->rowtar = (0xffc << 20) | ((hop_count +1) << 12);
dsp_maint->rowtear = 0x3f;
synch();

/*Apply new device ID*/
*(u32 *) (current_device_mem + 0x60) = current_device_id;
synch();
printf("Device found & programmed with device ID 0x%x\n", current_device_id);


/*Reprogram the ports on the switch and the DSP window.*/
/*Update DSP window*/
dsp_maint->rowtar = ( current_device_id << 22) | ((hop_count +1) << 12);
dsp_maint->rowtear = 0;
*(u32 *) (switch_device + 0x70) = current_device_id;
*(u32 *) (switch_device + 0x74) = port_num;
*(u32 *) (switch_device + 0x78) = port_num;
synch();

/*Readback and verify*/
deviceid_read = *(u32 *) (current_device_mem + 0x60);
synch();

printf("Readback device ID = 0x%x\n",deviceid_read);

/*Reset DSP window ID back to 0xff*/
dsp_maint->rowtar = (0xffc << 20) | ((hop_count +1) << 12);
dsp_maint->rowtear = 0x3f;

}

/*********************************************************************************/

void enum_network(int switch_count, u32 port_config[][16], int endpoint_count, u32 rio_addr, int hop_count)
{
	volatile static struct rio_atmu_row *atmu_regs;
	volatile static struct rio_atmu_row *switch_maint;
	volatile static struct rio_atmu_row *dsp_maint;
	volatile u32 switch_maintenance, dsp_mem;
	u32 host_port;
	u32 port_num;
	u32 pefcar;
	u32 deviceid_read;
	int switch_num = 0;
	int endpoint_num = 0;
	u32 device_id;
	int enum_status;


	if ((mem_fd = open("/dev/mem", O_RDWR)) == -1) {
		fprintf(stderr, "dsp: Error: Cannot open /dev/mem\n");
		exit(EXIT_FAILURE);
	}

/*Open access to ATMU registers*/
	atmu_regs =
	    (struct rio_atmu_row *)mmap(0, RIO_SIZE, (PROT_READ | PROT_WRITE),
					MAP_SHARED, mem_fd, rio_addr);

	if ((s32) atmu_regs == -1) {
		printf("Error: mmap failed.\n");
		exit(EXIT_FAILURE);
	}

atmu_regs = (struct rio_atmu_row *)(((u32) atmu_regs) + ATMU_OFFSET);

switch_maint = atmu_regs + 2;
dsp_maint = atmu_regs + 5;

switch_maint->rowbar = SWITCH_ADDR >> 12;
switch_maint->rowar = ROWAR_MAINT | LAWAR_SIZE_1M;
switch_maint->rowtar = (0xffc << 20) | (hop_count << 12);
switch_maint->rowtear = 0x3f;

dsp_maint->rowbar = DSP_ADDR >> 12;
dsp_maint->rowar = ROWAR_MAINT | LAWAR_SIZE_1M;
dsp_maint->rowtar = (0xffc << 20) | ((hop_count +1) << 12);
dsp_maint->rowtear = 0x3f;
synch();


switch_maintenance = (u32) mmap(0, SWITCH_SIZE, (PROT_READ | PROT_WRITE),
			   MAP_SHARED, mem_fd, SWITCH_ADDR);

	if ((s32) switch_maintenance == -1) {
		printf("Error: maint mmap failed.\n");
		exit(EXIT_FAILURE);
	}

dsp_mem = (u32) mmap(0, DSP_MAINT_SIZE, (PROT_READ | PROT_WRITE),
			   MAP_SHARED, mem_fd, DSP_ADDR);

	if ((s32) dsp_mem == -1) {
		printf("Error: maint mmap failed.\n");
		exit(EXIT_FAILURE);
	}


	switch_num = switch_count - 1;

	host_port = (*(u32 *) (switch_maintenance + 0x14)) & 0x000000ff;
	/*Set-up routing*/
	/*printf("Setting up default routing for host on this switch\n");*/
	*(u32 *) (switch_maintenance + 0x70) = 0;
	*(u32 *) (switch_maintenance + 0x74) = host_port;
	*(u32 *) (switch_maintenance + 0x78) = host_port;


	//Program each switch port based on the port_config array
	for (endpoint_num = 0; endpoint_num < endpoint_count; endpoint_num++)
		{

		port_num = port_config[switch_num][endpoint_num] & 0x0000ff;
		device_id = port_config[switch_num][endpoint_num] >> 8;

		/*Check the port status*/
		enum_status = port_check(switch_maintenance, port_num);

		if (enum_status == 2) {
			printf("Skipping port %d as it is not initialised\n", port_num);
			continue;
			}

		if (enum_status == 0) {
			/*Set-up switch routing for DSP*/
			*(u32 *) (switch_maintenance + 0x70) = 0xffff;
			*(u32 *) (switch_maintenance + 0x74) = port_num;
			*(u32 *) (switch_maintenance + 0x78) = port_num;
			dsp_maint->rowtar = (0xffc << 20) | ((hop_count + 1) << 12);
			dsp_maint->rowtear = 0x3f;
			synch();

			/*Read device ID*/
			deviceid_read = *(u32 *) (dsp_mem + 0x60);
			/*Read PEFCAR of next device*/
			pefcar = *(u32 *) (dsp_mem + 0x10);
			synch();


			if (((pefcar >> 28) | 0xb) == 0xf) 	/*If device is not an endpoint then skip enumeration*/
		  		{
				enum_device(switch_maintenance, dsp_mem, device_id, hop_count, port_num, rio_addr);
				}

		}
	}

printf("Network enumeration complete\n");
}

/***********************************************************************************/

void read_filesystem(int *switch_count, u32 port_config[][16], int *endpoint_count, u32 *immr_base)
{
#define MAX_STRING  10
FILE * rapidio_info;
char string[MAX_STRING];
u32 dev_id, port_id;
int switch_num = 0;
int endpoint_num = 0;
char processor_info[10];
u32 endpoint_check[10];
u32 endpoint_val;
char parse_text[10];

DIR             *dip;
struct dirent   *dit;
int i = 0;
int j = 0;

/*Open directory & read no of switches and endpoints*/

if ((dip = opendir("/sys/devices/")) == NULL)
    {
     printf("failed to read directory\n");
	 return;
    }

while ((dit = readdir(dip)) != NULL)
  {
    /*printf("\n%s\n", dit->d_name);*/
		if (strstr(dit->d_name, ":s:")!= NULL)     /* If 'processor' found in string */
		{
		/*	printf("\nSwitch found\n");*/
			switch_num = switch_num +1;
		/*	printf("\nSwitch count = %d\n", switch_num);*/
		}

		if (strstr(dit->d_name, ":e:")!= NULL)
		{
	/*		printf("\nEndpoint found\n");*/
			parse_text[0] = dit->d_name[5];
			parse_text[1] = dit->d_name[6];
			parse_text[2] = dit->d_name[7];
			parse_text[3] = dit->d_name[8];

			endpoint_check[endpoint_num] = atoi(parse_text);
			endpoint_num = endpoint_num+1;
		}

		if (strstr(dit->d_name, "soc")!= NULL)
		{

			sscanf(dit->d_name, "%x%s", immr_base, &processor_info);
		}

  }


/*Close the directory*/
if (closedir(dip) == -1)
   {
    printf("failed to close directory\n");
	return;
   }

*switch_count = switch_num;
*endpoint_count = endpoint_num;

for(i=0; i < switch_num; i++)
{
	switch(i) {
		case 0 :
			rapidio_info = fopen("/sys/devices/00:s:0000/routes", "r");
			break;
		case 1 :
			rapidio_info = fopen("/sys/devices/00:s:0001/routes", "r");
			break;
		case 2:
			rapidio_info = fopen("/sys/devices/00:s:0002/routes", "r");
			break;
		case 3:
			rapidio_info = fopen("/sys/devices/00:s:0003/routes", "r");
			break;
		case 4:
			rapidio_info = fopen("/sys/devices/00:s:0004/routes", "r");
			break;
		case 5:
			rapidio_info = fopen("/sys/devices/00:s:0005/routes", "r");
			break;
		case 6:
			rapidio_info = fopen("/sys/devices/00:s:0006/routes", "r");
			break;
		case 7:
			rapidio_info = fopen("/sys/devices/00:s:0007/routes", "r");
			break;
		case 8:
			rapidio_info = fopen("/sys/devices/00:s:0008/routes", "r");
			break;
		case 9:
			rapidio_info = fopen("/sys/devices/00:s:0009/routes", "r");
			break;
		case 10:
			rapidio_info = fopen("/sys/devices/00:s:000a/routes", "r");
			break;
		default:
			printf("Switch limit exceeded\n");
			break;
		}

	if (!rapidio_info){
		printf("failed to read file\n");
		return;
		}
	j=0;
	while ( fgets(string,MAX_STRING,rapidio_info)!= NULL)
		{

		sscanf(string, "%x%x", &dev_id, &port_id);
		if (dev_id != endpoint_check[j])
			continue;
		port_config[i][j] = (dev_id << 8) | port_id;
		j++;
		}

	fclose(rapidio_info);
	}

}

/***********************************************************************************/


void dsp_reset(int dsp_no, int switch_count, u32 port_config[][16], int endpoint_count, u32 rio_addr)
{
	volatile static struct rio_atmu_row *atmu_regs;
	volatile static struct rio_atmu_row *switch_maint;
	volatile static struct rio_atmu_row *dsp_maint;
	volatile u32 switch_maintenance, dsp_mem;
	u32 dev_id;
	u32 pefcar0;
	u32 hdid_lock;
	u32 p0_route;
	u32 reset_link;
	u32 port_num;
	int dsp_found = 0;
	int hop_count = 0x00;
	u32 link_resp;
	volatile u32 config_regs;


	printf("Device %d to be reset\n", dsp_no);

	if ((mem_fd = open("/dev/mem", O_RDWR)) == -1) {
		fprintf(stderr, "dsp: Error: Cannot open /dev/mem\n");
		exit(EXIT_FAILURE);
	}

	/*Open access to config registers*/
	config_regs =
	    (u32)mmap(0, RIO_SIZE, (PROT_READ | PROT_WRITE),
					MAP_SHARED, mem_fd, rio_addr);

	if ((s32) config_regs == -1) {
		printf("Error: mmap failed.\n");
		exit(EXIT_FAILURE);
	}


/*Adjust port link timer*/
*(u32 *) (config_regs + 0x120) = 0x0000ff00;
*(u32 *) (config_regs + 0x124) = 0xff000000;

/*Send link response request to check see if SRIO link is alive*/
*(u32 *) (config_regs + 0x140) = 0x00000004;
synch();

link_resp = (*(u32 *) (config_regs + 0x144)) >> 31;

if (link_resp == 0)
	{
	/*If no response try resetting ackIDs to see if response comes back.*/
	*(u32 *) (config_regs + 0x148) = 0x00000000;
	*(u32 *) (config_regs + 0x140) = 0x00000004;
	synch();


	link_resp = (*(u32 *) (config_regs + 0x144)) >> 31;
	if (link_resp == 0) {
		printf("Link failed to detect any SRIO devices connected.  Please check configuration\n");
		exit(EXIT_FAILURE);
		}
	else {
		printf("Network enumeration required! \n");
		enum_network(switch_count, port_config, endpoint_count, rio_addr, hop_count);
		return;
		}
	}


/*Open access to ATMU registers*/
	atmu_regs =
	    (struct rio_atmu_row *)mmap(0, RIO_SIZE, (PROT_READ | PROT_WRITE),
					MAP_SHARED, mem_fd, rio_addr);

	if ((s32) atmu_regs == -1) {
		printf("Error: mmap failed.\n");
		exit(EXIT_FAILURE);
	}


atmu_regs = (struct rio_atmu_row *)(((u32) atmu_regs) + ATMU_OFFSET);

switch_maint = atmu_regs + 2;
dsp_maint = atmu_regs + 5;

switch_maint->rowbar = SWITCH_ADDR >> 12;
switch_maint->rowar = ROWAR_MAINT | LAWAR_SIZE_1M;
switch_maint->rowtar = (0xffc << 20) | (hop_count << 12);
switch_maint->rowtear = 0x3f;

dsp_maint->rowbar = DSP_ADDR >> 12;
dsp_maint->rowar = ROWAR_MAINT | LAWAR_SIZE_1M;

/*Large target ID system being used, so additional bits in ROWTAR & ROWTEAR are set*/
dsp_maint->rowtar = (dsp_no << 22) | ((hop_count + 1) << 12);
dsp_maint->rowtear = 0;

synch();

switch_maintenance = (u32) mmap(0, SWITCH_SIZE, (PROT_READ | PROT_WRITE),
			   MAP_SHARED, mem_fd, SWITCH_ADDR);

	if ((s32) switch_maintenance == -1) {
		printf("Error: maint mmap failed.\n");
		exit(EXIT_FAILURE);
	}


dsp_mem = (u32) mmap(0, DSP_MAINT_SIZE, (PROT_READ | PROT_WRITE),
			   MAP_SHARED, mem_fd, DSP_ADDR);

	if ((s32) dsp_mem == -1) {
		printf("Error: maint mmap failed.\n");
		exit(EXIT_FAILURE);
	}


while (dsp_found != 1) {

	*(u32 *) (switch_maintenance + 0x70) = dsp_no;
	synch();

	port_num = *(u32 *) (switch_maintenance + 0x74);
	synch();

	printf("Device %d is connected to port %d\n", dsp_no, port_num);


	/*Check the port status*/
	enum_status = port_check(switch_maintenance, port_num);

	if (enum_status == 2) {
			printf("Port %d is not initialised.  Please check configuration\n", port_num);
			exit(EXIT_FAILURE);
			}

	if (enum_status == 1) {
		/*Network enumeration required*/
		printf("Network enumeration required.  Enumerating network\n");

		/*Set-up switch routing for DSP*/
		*(u32 *) (switch_maintenance + 0x70) = 0xffff;
		*(u32 *) (switch_maintenance + 0x74) = port_num;
		*(u32 *) (switch_maintenance + 0x78) = port_num;

		hop_count++;

		enum_network(switch_count, port_config, endpoint_count, rio_addr, hop_count);
		return;
		}

	/*Read device ID of next device*/

	dev_id = *(u32 *) (dsp_mem + 0x60);
	synch();

	/*Read processing element feature register to check if device is a switch or an endpoint*/
	pefcar0 = *(u32 *) (dsp_mem + 0x10);
	/*printf("PEFCAR register = 0x%x\n", pefcar0);*/

	if (((pefcar0 >> 28) | 0xe) == 0xf)
	  {
		printf("Next device is another switch\n");
		hop_count++;
		/*Increment hop count and update both switch and DSP windows*/
		switch_maint->rowtar = (0xffc << 20) | (hop_count << 12);
		dsp_maint->rowtar = (dsp_no << 22) | ((hop_count + 1) << 12);
		synch();

	  }
	else {	/*Assume that an endpoint has been found*/
			/* Check that device ID matches with target ID*/
			if (dev_id == dsp_no) {
				dsp_found = 1;
				printf("Target device %d found\n", dev_id);
			}
			else {
				printf("Device id of detected device = %d\n", dev_id );
				printf("Error: Device ID does not map to correct DSP. Check window and port settings\n");
				exit(EXIT_FAILURE);
			}
	}

}


/*Adjust link timers on switch*/
*(u32 *) (switch_maintenance + 0x120) = 0x0000ff00;
*(u32 *) (switch_maintenance + 0x124) = 0x00ff0000;


//Determine which port of the switch needs to be reset
*(u32 *) (switch_maintenance + 0x70) = dsp_no;
synch();
p0_route = *(u32 *) (switch_maintenance + 0x74);
/*printf("Port used on device %d is port 0x%x\n", dsp_no, p0_route);*/
//reset port
reset_link = 0x140 + (p0_route * 0x20);

/*Send resets*/
*(u32 *) (switch_maintenance + reset_link) = 0x00000003;
synch();
/*Reset ackIDs*/
reset_ackIDs(switch_maintenance, p0_route);


/*Updating switch LUT*/
*(u32 *) (switch_maintenance + 0x70) = 0x0000ffff;
*(u32 *) (switch_maintenance + 0x74) = p0_route;
*(u32 *) (switch_maintenance + 0x78) = p0_route;

/*Update DSP window*/

dsp_maint->rowtar = (0xffc << 20) | ((hop_count + 1) << 12);
dsp_maint->rowtear = 0x0000003f;

synch();

dev_id = *(u32 *) (dsp_mem + 0x60);
printf("Reset device!\nDevice id after reset = 0x%x\n", dev_id );

/*Lock HBDID with required ID*/
*(u32 *) (dsp_mem + 0x68) = dsp_no;

/*Update device ID*/
*(u32 *) (dsp_mem + 0x60) = dsp_no;

/*Update port settings*/
*(u32 *) (switch_maintenance + 0x70) = dsp_no;
*(u32 *) (switch_maintenance + 0x74) = p0_route;
*(u32 *) (switch_maintenance + 0x78) = p0_route;

/*Update window*/
dsp_maint->rowtar = (dsp_no << 22) | ((hop_count + 1) << 12);
dsp_maint->rowtear = 0;

synch();

/*Check device ID*/
dev_id = *(u32 *) (dsp_mem + 0x60);
printf("Updated device id = %d\n", dev_id );

}

/***********************************************************************************/

void synch()
{
	asm("isync");
	asm("msync");
	asm("sync");
}

/***********************************************************************************/

int main(int argc, char **argv)
{
	char cmd;
	u32 device_id;
	FILE *fd;
	volatile  u32 maint, m2, m3;
	u32 start, i;
	u32 test;
	u8  write = 1;
	u8  verify = 0;

	/* Declare structure for outbound windows*/
	volatile static struct rio_atmu_row *atmu_regs;
	volatile static struct rio_atmu_row *maintenance;
	volatile static struct rio_atmu_row *memory;

	static struct rio_atmu_riw *dsp_ib_mem;
	u32 port_config [10][16];
	int switch_count=0;
	int endpoint_count=0;
	u32 immr_base;
	u32 rio_addr;


	cmd = argv[1][0];
	device_id = atoi(argv[2]);

	/*printf("Reading filesystem\n");*/
	read_filesystem(&switch_count, port_config, &endpoint_count, &immr_base);
	rio_addr = immr_base + 0xc0000;

	if (argc < 3) {
		usage();
		exit(EXIT_FAILURE);
	}


	if (cmd == 'r') {
		dsp_reset(device_id, switch_count, port_config, endpoint_count, rio_addr);
		return 0;
	}

	else if (cmd != 'd') {
		usage();
		exit(EXIT_FAILURE);
	}

/***************************************
 Open file descriptor to the dev/mem driver
*****************************************/


	if ((mem_fd = open("/dev/mem", O_RDWR)) == -1) {
		fprintf(stderr, "dsp: Error: Cannot open /dev/mem\n");
		exit(EXIT_FAILURE);
	}

	datafile = argv[3];
	if ((fd = fopen(datafile, "rb")) == NULL) {
		fprintf(stderr, "dsp: Can't open %s: %s\n",
			datafile, strerror(errno));
		exit(EXIT_FAILURE);
	}

/* Map to ATMU register location on the device*/
	atmu_regs =
	    (struct rio_atmu_row *)mmap(0, RIO_SIZE, (PROT_READ | PROT_WRITE),
					MAP_SHARED, mem_fd, RIO_ADDR);

	if ((s32) atmu_regs == -1) {
		printf("Error: mmap failed.\n");
		exit(EXIT_FAILURE);
	}

	*(u32 *)((u32) atmu_regs + 0x60c) = 0xffffffff;
	*(u32 *)((u32) atmu_regs + 0x644) = 0xffffffff;


	atmu_regs = (struct rio_atmu_row *)(((u32) atmu_regs) + ATMU_OFFSET);

	printf("Rev 1.0 - MSC8144 RapidIO Bootloader for 8572.\n");

/*Map to outbound window 7*/
	maintenance = atmu_regs + 7;
/*Map to outbound window 8*/
	memory = atmu_regs + 8;

	maintenance->rowbar = MAINT_ADDR >> 12;
	maintenance->rowar = ROWAR_MAINT | LAWAR_SIZE_4M;
	maintenance->rowtar = (device_id << 22) | (0xff << 12);
	maintenance->rowtear = 0;

	memory->rowbar = M3_ADDR >> 12;
	memory->rowar = ROWAR_MEM | LAWAR_SIZE_128M;
	memory->rowtar = (device_id << 22) | (M3_ADDR >> 12);
	memory->rowtear = 0;
	synch();
	maint = (u32) mmap(0, MAINT_SIZE, (PROT_READ | PROT_WRITE),
			   MAP_SHARED, mem_fd, MAINT_ADDR);

	if ((s32) maint == -1) {
		printf("Error: maint mmap failed.\n");
		exit(EXIT_FAILURE);
	}

	/* Move the DSP IMMR to the M2_ADDR as maintenace
	 * transactions only allow us access to low 64K
	 * RapidIO memory space. Memory transactions gives
	 * access to the entire Internal Memory map
	 * m2 will now access the DSP CCSR.
	 */
/* Access the LCSBA1CSR register*/
	*(u32 *) (maint + LCSBA1CSR) = M2_ADDR >> 3;
	usleep(10);

/*Map m2 to memory reserved for m2 transctions from outbound window.*/
	m2 = (u32) mmap(0, M2_SIZE, (PROT_READ | PROT_WRITE),
			MAP_SHARED, mem_fd, M2_ADDR);

	if ((s32) m2 == -1) {
		printf("Error: M2 mmap failed.\n");
		exit(EXIT_FAILURE);
	}
/*Map m3 memory to user program space*/
	m3 = (u32) mmap(0, M3_SIZE, (PROT_READ | PROT_WRITE),
			MAP_SHARED, mem_fd, M3_ADDR);

	if ((s32) m3 == -1) {
		printf("Error: M3 mmap failed.\n");
		exit(EXIT_FAILURE);
	}


/*Create outbound window for M3 memory on DSP*/
	dsp_ib_mem = (struct rio_atmu_riw *)((u32) m2 + DSP_RIWTAR4);

	dsp_ib_mem->riwbar = M3_ADDR >> 12;
	dsp_ib_mem->riwtar = M3_ADDR >> 12;
	dsp_ib_mem->riwar = 0x08000000 | RIWAR_MEM | LAWAR_SIZE_16M;

	dsp_ib_mem += 1;
/*Create outbound window for M2 memory on DSP*/
	dsp_ib_mem->riwbar = M2_ADDR >> 12;
	dsp_ib_mem->riwtar = DSP_M2_ADDR >> 12;
	dsp_ib_mem->riwar = 0x08000000 | RIWAR_MEM | LAWAR_SIZE_1M;
	synch();

	/* Move the DSP Internal Memory map out of the way.
	 * m2 now will access the DSP M2 memory via the rapidio
	 * inbound memory windows.
	 */


	*(u32 *) (maint + LCSBA1CSR) = 0xfff00000 >> 3;
	usleep(10);
	synch();

	if (*(u32 *) (m2 + DSP_BOOT_MAGIC_ADDR) != DSP_BOOT_MAGIC_1) {
		printf("Error: DSP Boot Program 0x17 Error.\n");
		exit(EXIT_FAILURE);
	}


	start = download_srec(fd, m2, m3, write);
	start = download_srec(fd, m2, m3, verify);

	printf("Boot Address 0x%08x\n", start);


	*(u32 *) ((u32) m2 + DSP_BOOT_ADDR) = start;
	*(u32 *) ((u32) m2 + DSP_BOOT_MAGIC_ADDR) = DSP_BOOT_MAGIC_2;

	printf("Download Complete.\n");
	return 0;
}

/* Should probably re-use existing linux libs for srec/elf handling */
u32 download_srec(FILE * fd, u32 m2, u32 m3, u8 write)
{
	u8 c0, c1, length;
	u32 i, dest;
	u8 data;

	fseek(fd,0, SEEK_SET);
	c0 = getc(fd);
	c1 = getc(fd);

	if ((c0 != 'S') || (c1 != '0')) {
		fprintf(stderr,
			"dsp: %s does not start with a valid srec header \n",
			datafile);
		exit(EXIT_FAILURE);
	}

	length = hex_value(fd);

	for (i = 0; i < length * 2; i++)
		getc(fd);

	if (fgetc(fd) == 0x0d) {
		getc(fd);
	}

	c0 = getc(fd);
	c1 = getc(fd);

	while (1) {
		if ((c0 != 'S') || (c1 != '3')) {
			fprintf(stderr,
				"dsp: %s does not start with a header \n",
				datafile);
			exit(EXIT_FAILURE);
		}
		length = hex_value(fd) - 5;

		dest = 0;
		for (i = 0; i < 4; i++) {
			dest = dest << 8;
			dest = dest | hex_value(fd);
		}

		if ((dest >= 0xc0000000) && (dest < 0xc0080000)) {
			for (i = 0; i < length; i++) {
				data = hex_value(fd);
				if (write) {
					*(u8 *) ((dest & ~0xfff00000) + m2) = data;
				} else {
					if (*(u8 *) ((dest & ~0xfff00000) + m2) != data ) {
						printf("Verify error addr 0x%08x, data 0x%08x\n",((dest & ~0xfff00000) + m2),data);
					}

				}
				dest++;
			}
		} else if ((dest >= 0xd0000000) && (dest < 0xd0a00000)) {
			for (i = 0; i < length; i++) {
				data = hex_value(fd);
				if (write) {
					*(u8 *) ((dest & ~0xff000000) + m3) = data;
				} else {
					if (*(u8 *) ((dest & ~0xff000000) + m3) != data ) {
						printf("Verify error addr 0x%08x, data 0x%08x\n",((dest & ~0xff000000) + m3),data);
					}

				}
				dest++;
			}
		} else {
			fprintf(stderr,
				"dsp: %s Invalid memory address.\n", datafile);
			exit(EXIT_FAILURE);
		}

		/* Skip checksum */
		getc(fd);
		getc(fd);

		/* Check for CR/LF */
		if (fgetc(fd) == 0x0d) {
			getc(fd);
		}
		/* Get next srec header */
		c0 = getc(fd);
		c1 = getc(fd);

		if ((c0 == 'S') && (c1 == '7')) {
			/* skip count */
			getc(fd);
			getc(fd);
			dest = 0;
			for (i = 0; i < 4; i++) {
				dest = dest << 8;
				dest = dest | hex_value(fd);
			}
			return dest;
		}
	}
}

u8 hex_value(FILE * fd)
{
	u32 i;
	u8 hex, temp;

	for (i = 0; i < 2; i++) {
		hex = hex << 4;
		temp = getc(fd);
		if (temp == '0')
			hex |= 0x0;
		if (temp == '1')
			hex |= 0x1;
		if (temp == '2')
			hex |= 0x2;
		if (temp == '3')
			hex |= 0x3;
		if (temp == '4')
			hex |= 0x4;
		if (temp == '5')
			hex |= 0x5;
		if (temp == '6')
			hex |= 0x6;
		if (temp == '7')
			hex |= 0x7;
		if (temp == '8')
			hex |= 0x8;
		if (temp == '9')
			hex |= 0x9;
		if (temp == 'A')
			hex |= 0xA;
		if (temp == 'B')
			hex |= 0xB;
		if (temp == 'C')
			hex |= 0xC;
		if (temp == 'D')
			hex |= 0xD;
		if (temp == 'E')
			hex |= 0xE;
		if (temp == 'F')
			hex |= 0xF;
	}
	return hex;
}
