/*
 * File:		Int_handlers.c
 * Purpose:		contains all interrupt processes
 *
 * IMPORTANT. Read the following Freescale Semiconductor Software License 
 * Agreement:
 * http://www.freescale.com/files/disclaimer/LA_OPT1.html
 * 
 */

#include "support_common.h" /* include peripheral declarations and more */

#define __interrupt__   __declspec(interrupt)

/***********************************************************************************************************/

__interrupt__ void /* general interrupt handler */ 
irq_handler (void)
{
	printf("general interrupt handler\n");
}


/***********************************************************************************************************/

__interrupt__ void
EPORT_SW1_button_pressing (void)
{
    /* Wait for the switch to deassert */
    while (!(MCF_EPORT0_EPPDR & MCF_EPORT_EPPDR_EPPD4)) {};
    /* Clear the interrupt event */
    MCF_EPORT0_EPFR = (uint8)(MCF_EPORT0_EPFR | MCF_EPORT_EPFR_EPF4);
    	
	
	if (!(MCF_ADC_CTRL1 & 0x0006) || (MCF_ADC_CTRL1 & MCF_ADC_CTRL1_STOP0))
	{
		// this way in case once mode or in case ADC is in STOP mode
		MCF_ADC_CTRL1 &= ~MCF_ADC_CTRL1_STOP0;
		MCF_ADC_CTRL1 |= MCF_ADC_CTRL1_START0; 		// start conversion	
	}
	else 
	{
		MCF_ADC_CTRL1 |= MCF_ADC_CTRL1_STOP0; 		// stop conversion
	}	
}

/***********************************************************************************************************/

__interrupt__ void
EPORT_SW2_button_pressing (void)
{
    /* Wait for the switch to deassert */
    while (!(MCF_EPORT0_EPPDR & MCF_EPORT_EPPDR_EPPD7)) {};
    /* Clear the interrupt event */
    MCF_EPORT0_EPFR = (uint8)(MCF_EPORT0_EPFR | MCF_EPORT_EPFR_EPF7);

	
	if ((MCF_ADC_CTRL1 & 0x0001) && !(MCF_ADC_CTRL2 & MCF_ADC_CTRL2_SIMULT))
	{
		// independent (non-simultaneous) mode
		if (MCF_ADC_CTRL2 & MCF_ADC_CTRL2_STOP1)
		{
			MCF_ADC_CTRL2 &= ~MCF_ADC_CTRL2_STOP1;
			MCF_ADC_CTRL2 |= MCF_ADC_CTRL2_START1; 		// start conversion	of ADCB
		}
		else 
		{
			MCF_ADC_CTRL2 |= MCF_ADC_CTRL2_STOP1; 		// stop conversion of ADCB
		}
	}

}

/***********************************************************************************************************/

__interrupt__ void  /* Sequential or parallel simultaneous mode conversion complete
                      or ADCA conversion complete (EOSI0) if parallel independent mode is used */
ADCA_conversion_complete (void) 
{
	uint16 ADC_status;
	uint16 i; // counter
	
	MCF_ADC_ADSTAT |= MCF_ADC_ADSTAT_EOSI0; // clear interrupt flag 
	
	ADC_status = MCF_ADC_ADSTAT; // read status register
	
	if (!(MCF_ADC_CTRL1 & 0x0001) || ((MCF_ADC_CTRL1 & 0x0001) && (MCF_ADC_CTRL2 & MCF_ADC_CTRL2_SIMULT)))
	{
		/* sequential or parallel simultaneous mode - it is needed to test all sample slots */
		for (i=0; i<8; i++)
		{
			if ((ADC_status >> i) & 0x0001)
			{
				printf("%d",i);
				printf(":");
				printf("%6d",(int16) MCF_ADC_ADRSLT(i));
				printf("  ");
			}
		}
		printf("\n\r");
	}
	
	else
	{
		/* parallel independent mode - it is needed to test only lower half of sample slots */
		for (i=0; i<4; i++)
		{
			if ((ADC_status >> i) & 0x0001)
			{
				printf("%d",i);
				printf(":");
				printf("%6d",(int16) MCF_ADC_ADRSLT(i));
				printf("  ");
			}
		}
		printf("\n\r");	
	}	
}

/***********************************************************************************************************/

__interrupt__ void /* ADCB conversion complete (from EOSI1) - only if parallel independent mode is used */
ADCB_conversion_complete (void) 
{	
	uint16 ADC_status;
	uint16 i; // counter
	
	MCF_ADC_ADSTAT |= MCF_ADC_ADSTAT_EOSI1; // clear interrupt flag 
	
	ADC_status = (uint16)((uint32)MCF_ADC_ADSTAT >> 4); // read status register (shift for test RDY7-RDY4)
	
	/* test only higher half of sample slots */
	for (i=0; i<4; i++)
	{
		if ((ADC_status >> i) & 0x0001)
		{
			printf("%d",i+4);
			printf(":");
			printf("%6d",(int16) MCF_ADC_ADRSLT(i+4));
			printf("  ");
		}
	}
	printf("\n\r");
	
}

/***********************************************************************************************************/

__interrupt__ void /*  ADC zero crossing or limit error (from ZCI, LLMTI, HLMTI) */
ADC_zero_crossing_or_limit_error (void) 
{
	uint16 ADC_status, ADC_limit_status, ADC_zero_crossing_status;
	uint16 i; // counter
	
	ADC_status = MCF_ADC_ADSTAT; // read status register
	
	if (ADC_status | MCF_ADC_ADSTAT_HLMTI) // high limit
	{
		ADC_limit_status = (uint16)((uint32)MCF_ADC_ADLSTAT >> 8); // read limit status register (MSB part)
		for (i=0; i<8; i++)
		{
			if ((ADC_limit_status >> i) & 0x0001)
			{
				MCF_ADC_ADLSTAT |= 0x0001 << 8+i; // clear appropriate high limit flag
				printf("%d", i);
				printf(": more than ");
				printf("%d",(int16) MCF_ADC_ADHLMT(i) - MCF_ADC_ADOFS(i));
				printf(" \n\r");	
			}
		}
	}
	
	if (ADC_status | MCF_ADC_ADSTAT_LLMTI) // low limit
	{
		ADC_limit_status = MCF_ADC_ADLSTAT; // read limit status register (LSB part)
		for (i=0; i<8; i++)
		{
			if ((ADC_limit_status >> i) & 0x0001)
			{
				MCF_ADC_ADLSTAT |= 0x0001 << i; // clear appropriate low limit flag
				printf("%d", i);
				printf(": less than ");
				printf("%d",(int16) MCF_ADC_ADLLMT(i) - MCF_ADC_ADOFS(i));
				printf(" \n\r");	
			}
		}
	}
	
	if (ADC_status | MCF_ADC_ADSTAT_ZCI) // zero crossing
	{
		ADC_zero_crossing_status = MCF_ADC_ADZCSTAT; // read zero crossing status register
		for (i=0; i<8; i++)
		{
			if ((ADC_zero_crossing_status >> i) & 0x0001)
			{
				MCF_ADC_ADZCSTAT |= 0x0001 << i; // clear appropriate zero crossing flag
				printf("%d", i);
				printf(": ZERO CROSSING\n\r");	
			}				
		}
	}	

}


/***********************************************************************************************************/