#include "hcc_types.h"
#include "mcf5222x_reg.h"
#include "driver_reg.h"
#include "qspi.h"

hcc_u16  receivedData[32];

hcc_u16  transmitData[6];

hcc_u8   transferSize;
hcc_u8   receivedPtr;
hcc_u8   sendStatus;

/********************** SPI MEMORY COMMANDS *********************************/

void WRENcommand()
{
	transmitData[0] = QSPI_MEMORY_WREN;
	QSPISendData(0,transmitData,1);
	
	return;
}

void WRDIcommand()
{
	transmitData[0] = QSPI_MEMORY_WRDI;
	QSPISendData(0,transmitData,1);
	
	return;
}

void WRSRcommand(hcc_u8 data)
{
	transmitData[0] = QSPI_MEMORY_WRSR;
	transmitData[1] = data;
	QSPISendData(0,transmitData,2);
	
	return;
}

hcc_u8 RDSRcommand()
{
	transmitData[0] = QSPI_MEMORY_RDSR;
	QSPISendData(0,transmitData,2);
	
	return (hcc_u8)receivedData[1];
}

hcc_u8 READcommand(hcc_u8 u8Address)
{
	transmitData[0] = QSPI_MEMORY_READ;
	transmitData[1] = u8Address;
	QSPISendData(0,transmitData,3);
	
	return (hcc_u8)receivedData[2];
}

void WRITEcommand(hcc_u8 u8Address, hcc_u8 u8Data)
{
	transmitData[0] = QSPI_MEMORY_WRITE;
	transmitData[1] = u8Address;
	transmitData[2] = u8Data;
	QSPISendData(0,transmitData,3);
	
	return;
}

/****************************************************************************/

/*****************************************************************************
 * vfnDriver: Configures the Interrupt controller and SR to recive           *
 *			   interrupt request from peripheral modules                     *
 *                                                                           *
 * Parameters: 																 *
 *                                                                           *
 * Return : none.                                                            *
 *****************************************************************************/
void QSPIinit(hcc_u8 u8TransferBits, hcc_u8 u8BaudRate)
{
	hcc_u8 u8Baud;	
	
	/* setting core interrupt level to zero */
	asm(move.w #0x2000,SR);
	
	/* Unmasking interrupt in interrupt controller */
	MCF_INTC_ICR18(0) = (0
					  | MCF_INTC_ICR_IP(INT_PRI) 
					  | MCF_INTC_ICR_IL(INT_LVL)
					  );
	
	MCF_INTC0_IMRL &= ~(0
					| MCF_INTC_IMRL_MASK18 
					| MCF_INTC_IMRL_MASKALL
					);
	
	/*calculates the value for the baud bits*/
	u8Baud = (BUS_CLOCK)/(u8BaudRate);
	/* Setting the QSPI pins for QSPI purposes (by default they are GPIO's)*/
    MCF_GPIO_PQSPAR = (0
    				| MCF_GPIO_PQSPAR_PQSPAR0(1)	/* DOUT */
    				| MCF_GPIO_PQSPAR_PQSPAR1(1)	/* DIN  */
    				| MCF_GPIO_PQSPAR_PQSPAR2(1)	/* CLK  */
    				| MCF_GPIO_PQSPAR_PQSPAR3(1)    /* CS0  */		
    				);
    
    if(u8TransferBits == 16)/*if the data size is 16 bits*/
    {
    	/*configure QSPI operation mode*/
    	MCF_QSPI_QMR = (0
    				  | MCF_QSPI_QMR_MSTR 
    				  //| MCF_QSPI_QMR_DOHIE
                      | MCF_QSPI_QMR_BITS(0) //16 bits
                      //| MCF_QSPI_QMR_CPHA
                      | MCF_QSPI_QMR_BAUD(u8Baud)
                      );
    }
    else 
    {
       /*configure QSPI operation mode*/
       MCF_QSPI_QMR = (0
       				  | MCF_QSPI_QMR_MSTR
       				  //| MCF_QSPI_QMR_DOHIE
                      | MCF_QSPI_QMR_BITS(u8TransferBits) 
                      //| MCF_QSPI_QMR_CPHA
                      | MCF_QSPI_QMR_BAUD(u8Baud)
                      );
	}
	
    /* set the clk delay and transfer delay for the QSPI */
    MCF_QSPI_QDLYR = (0
    				 | MCF_QSPI_QDLYR_QCD(8) 
                     | MCF_QSPI_QDLYR_DTL(1)
                     );
    
    /*enable the error flags and the interrupt of the QSPI*/                 
    MCF_QSPI_QIR = (0
    			   //| MCF_QSPI_QIR_WCEFB 
                   //| MCF_QSPI_QIR_ABRTB 
                   //| MCF_QSPI_QIR_ABRTL
                   | MCF_QSPI_QIR_SPIFE 
                   | MCF_QSPI_QIR_WCEF 
                   | MCF_QSPI_QIR_ABRT 
                   | MCF_QSPI_QIR_SPIF
                   );
                   
    return;   
}

/******************************************************************************
 * QSPISendData: transmits data using the QSPI and stores the data            *
 *               recieved from the fisical loop.                              *
 *                                                                            *
 * Parameters: u8cs:chip select signal                                        *
 *             message:data array to send                                     *
 *             u8MessageSize:number of packages to send                       *
 *                                                                            *            
 * Return : none.                                                             * 
 ******************************************************************************/
void QSPISendData(hcc_u8 u8cs, hcc_u16* message, hcc_u8 u8MessageSize)
{
    hcc_u8 u8DataCounter;
    static hcc_u8 transmitPtr;
    receivedPtr = 0;
    transmitPtr = 0;    
    if(u8MessageSize > 16){
        do{
            /*point to the QSPI command RAM space*/
            MCF_QSPI_QAR = QSPI_COMMAND_ADDRESS;
            transferSize = 16;
            for(u8DataCounter=0; u8DataCounter<16;u8DataCounter++){
                MCF_QSPI_QDR = (0
                			   | MCF_QSPI_QDR_CONT
                			   | MCF_QSPI_QCR_CS(u8cs)
                               | MCF_QSPI_QDR_DSCK
                               | MCF_QSPI_QDR_DT
                               | MCF_QSPI_QDR_BITSE
                               );        
            }
            MCF_QSPI_QAR = QSPI_TRANSMIT_ADDRESS;
    	    
    	    for(u8DataCounter=0; u8DataCounter<16;u8DataCounter++)
    	    {
                MCF_QSPI_QDR = message[transmitPtr];
                transmitPtr++;        
            }
            /*configure QSPI queue for transmition*/
            /*set the chip select inactive u8Level*/
            MCF_QSPI_QWR = (0               
    						| MCF_QSPI_QWR_CSIV
            				| MCF_QSPI_QWR_ENDQP(0x0F)
                            | MCF_QSPI_QWR_NEWQP(0x00)
                            );
            /*begin QSPI transmition*/               
            MCF_QSPI_QDLYR |= MCF_QSPI_QDLYR_SPE;
            while (!sendStatus)
            ; /* waiting for transmission end */
            u8MessageSize = u8MessageSize-16;
            sendStatus = WAIT;
        }while (u8MessageSize > 16);
    }
    transferSize = u8MessageSize;
    /*point to the QSPI command RAM space*/
    MCF_QSPI_QAR = QSPI_COMMAND_ADDRESS;
        for(u8DataCounter=0; u8DataCounter<u8MessageSize;u8DataCounter++)
        {
            MCF_QSPI_QDR = (0
            			   | MCF_QSPI_QDR_CONT
            			   | MCF_QSPI_QCR_CS(u8cs)
                           | MCF_QSPI_QDR_DSCK
                           | MCF_QSPI_QDR_DT
                           | MCF_QSPI_QDR_BITSE
                           );
        }
    /*point to the QSPI transmit RAM space*/
    MCF_QSPI_QAR = QSPI_TRANSMIT_ADDRESS;
  	    for(u8DataCounter=0; u8DataCounter<u8MessageSize;u8DataCounter++)
  	    {
            MCF_QSPI_QDR = message[transmitPtr];
            transmitPtr++;        
        }
    /*configure QSPI queue for transmission*/
    /*set the chip select inactive u8Level*/
    MCF_QSPI_QWR = (0
    				| MCF_QSPI_QWR_CSIV
    				| MCF_QSPI_QWR_ENDQP(u8MessageSize-1)
                    | MCF_QSPI_QWR_NEWQP(0x00)
                    );
    /*begin QSPI transmition*/               
    MCF_QSPI_QDLYR |= MCF_QSPI_QDLYR_SPE;
    while (!sendStatus)
    ; /* waiting for transmission end */
    	sendStatus = WAIT;
    return;
}

/*****************************************************************************
 * QSPIIsr: QSPI Interrupt Service Routine                                   *                               
 *          saves the received data into sram                                *                              
 *                                                                           *
 * Parameters: none.                                                         *
 *                                                                           *
 *                                                                           *
 * Return : none.                                                            *
 *****************************************************************************/
__declspec(interrupt:0) void QSPIIsr()
{ 
    hcc_u8 u8DataCounter;
    /*point to the QSPI receive RAM space*/
    MCF_QSPI_QAR = QSPI_RECEIVE_ADDRESS;
    for(u8DataCounter=0;u8DataCounter<transferSize;u8DataCounter++)
    {
        receivedData[receivedPtr] = MCF_QSPI_QDR;
        receivedPtr++;
    }
    sendStatus = SENDED;
    MCF_QSPI_QIR |= MCF_QSPI_QIR_SPIF;
}
