/**
 * Copyright (c) 2006, Freescale Semiconductor
 * Freescale Confidential Proprietary
 *
 * File name   : main.c
 *
 * Author      : Paolo Alcantara
 * Department  : RTAC Americas
 *
 * Description : RTC api
 */

#include "mcf5222x_reg.h"
#include "rtc.h"
#include "driver_reg.h"

/* Functions */

/*
 * RTCinit: starts RTC controller
 *
 * Parameters: none
 *
 * Return : none.
 */
void RTCinit()
{
	/* enabling RTC module */
	MCF_GPIO_PUAPAR = 0
					| MCF_GPIO_PTCPAR_PTCPAR0(2)
					| MCF_GPIO_PTCPAR_PTCPAR1(2)
					;

	/* reset RTC module */
	MCF_RTC_RTCCTL = 0 
				   | MCF_RTC_RTCCTL_SWR
				   ;

	/* Set real-time clock freq to 1 Hz using 32.768 KHz crystal */
	/* RTC crystal/4/RTCGOC = 1 Hz */
	MCF_RTC_RTCGOCU = 0;	 				/* Device-dependent */
	MCF_RTC_RTCGOCL = 0x2000; 				/* Device-dependent */
	
	/* enabling RTC clock */
	MCF_CLOCK_RTCCR = 0
	                | MCF_CLOCK_RTCCR_RTCSEL
					| MCF_CLOCK_RTCCR_LPEN  
					| MCF_CLOCK_RTCCR_REFS
					| MCF_CLOCK_RTCCR_OSCEN
					| (1<<6)				//EXTALEN
					;
#ifdef RTC_INTERRUPT					
	/* enabling RTC interrupts */
	/* Set IMRH to enable interrupt request */
	MCF_INTC0_IMRH &= ~(0
				   | MCF_INTC_IMRH_MASK63
				   );
    MCF_INTC0_IMRL &= ~(0
        		   | MCF_INTC_IMRL_MASKALL
        		   );

	/* Levels of interruptions */
	/* Level x, Priority x */
	MCF_INTC0_ICR(63) = 0
					  | MCF_INTC_ICR_IL(5) 
					  | MCF_INTC_ICR_IP(5)
					  ;
	
	/* Enable All Interrupts */
    asm (move.w	#0x2000,sr);
#endif

    return;		
}

/*
 * RTCstart: turn on RTC controller
 *
 * Parameters: none
 *
 * Return : none.
 */
void RTCstart()
{
	/* enabling RTC module */
	MCF_RTC_RTCCTL = 0 
				   | MCF_RTC_RTCCTL_EN
				   ;
	return;
}

/*
 * RTCsetTime: set RTC day, hour, minute and second
 *
 * Parameters: rtc_state: RTC state pointer
 *
 * Return : none.
 */
void RTCsetTime(rtc_state_t *rtc_state)
{
	/* setting time */
	MCF_RTC_DAYS = MCF_RTC_DAYS_DAYS(rtc_state->days);
	
	MCF_RTC_HOURMIN = 0
					| MCF_RTC_HOURMIN_HOURS(rtc_state->hours)
					| MCF_RTC_HOURMIN_MINUTES(rtc_state->minutes)
					;

	MCF_RTC_SECONDS = MCF_RTC_SECONDS_SECONDS(rtc_state->seconds);
	
	return;	
}

/*
 * RTCgetTime: return RTC day, hour, minute and second
 *
 * Parameters: rtc_state: RTC state pointer
 *
 * Return : none.
 */
void RTCgetTime(rtc_state_t *rtc_state)
{
	rtc_state->days    = MCF_RTC_DAYS_DAYS(MCF_RTC_DAYS);
	rtc_state->hours   = (MCF_RTC_HOURMIN)>>8;
	rtc_state->minutes = MCF_RTC_HOURMIN_MINUTES(MCF_RTC_HOURMIN);
	rtc_state->seconds = MCF_RTC_SECONDS_SECONDS(MCF_RTC_SECONDS);
	
	return;
}

/*
 * RTCsetAlarm: set RTC alarm day, hour, minute and second
 *
 * Parameters: rtc_state: RTC state pointer
 *
 * Return : none.
 */
void RTCsetAlarm(rtc_state_t *rtc_state)
{
	/* setting alarm */
	MCF_RTC_ALRM_DAY = MCF_RTC_ALRM_DAY_DAYSAL(rtc_state->days);
	
	MCF_RTC_ALRM_HM = 0
					| MCF_RTC_ALRM_HM_MINUTES(rtc_state->minutes)
					| MCF_RTC_ALRM_HM_HOURS(rtc_state->hours)
					;
					
	MCF_RTC_ALRM_SEC = MCF_RTC_ALRM_SEC_SECONDS(rtc_state->seconds);

	/* setting alarm interrupt */
	MCF_RTC_RTCIENR = MCF_RTC_RTCIENR_ALM;
	
	return;
}

/*
 * RTCgetAlarm: get RTC alarm day, hour, minute and second
 *
 * Parameters: rtc_state: RTC state pointer
 *
 * Return : none.
 */
void RTCgetAlarm(rtc_state_t *rtc_state)
{
	rtc_state->days    = MCF_RTC_ALRM_DAY_DAYSAL(MCF_RTC_ALRM_DAY);
	rtc_state->hours   = (MCF_RTC_ALRM_HM)>>8;
	rtc_state->minutes = MCF_RTC_ALRM_HM_HOURS(MCF_RTC_ALRM_HM);
	rtc_state->seconds = MCF_RTC_ALRM_SEC_SECONDS(MCF_RTC_ALRM_SEC);
	
	return;	
}

/*
 * rtc_ISR: RTC ISR
 *
 * Parameters: none
 *
 * Return : none.
 */
__declspec(interrupt:0) void rtc_ISR(void)
{
	/* insert code here */
	
	/* cleaning interrupt */
	MCF_RTC_RTCISR = MCF_RTC_RTCISR_ALM;
	
	return;
}