/**
  Copyright (c) 2008 Freescale Semiconductor
  Freescale Confidential Proprietary
  \file  		emsci.c
  \brief		Emulated Sci initialization routine and Tx & Rx function algorithms 
  \author		Freescale Semiconductor
  \author		R01160
  \author		Guadalajara Applications Laboratory RTAC Americas
  \version		0.1
  \date  		9/15/2008 11:14:00 AM
  
  * History:
  
*/
#include	 "emsci.h"
#include	 "Derivative.h"

const			 Emsci_ConfigType * TempConfigPtr = NULL_PTR;
	   volatile  temsci_data	   emsci[ 4 ];


/*----------------------------------------------------------------------------------------------------------------*/
/**
 * \brief	 Initialize Emsci units described in the Emsci Configuration file \n
			 (Emsci_Cfg.c file).
 * \param	 void 
 * \return	 void
 * \warning
/*-------------------------------------------------------------------------------*/
void vfnemsci_init( const Emsci_ConfigType * sEmsciPtr )
{
	uint8_t u8SciCntr = 0;

#if (  EMSCI_ERROR_DETECT  ==  ON  )							/*If Pwm Error Layer Enabled validate errors	  */
	if( sEmsciPtr == NULL_PTR )  								/*	Pwm module already initilized?				  */
	{
		vfnEmsci_Report_Error( EMSCI_INIT_ID, EMSCI_E_UNINIT ); /*	  Report None initilized module  			  */
	}
	else
	{
#endif

		/* Initialize temporal pointer with predefined configuration pointer */
		TempConfigPtr = ( Emsci_ConfigType * )sEmsciPtr;

		vfnEmiosConfig_General_Clocks( );						/*	  Initialize general Emios clocks			  */

		for( u8SciCntr = EMSCI_CHANNELS; u8SciCntr > 0U; u8SciCntr-- )
		{

			SIU.PCR[ ( &TempConfigPtr[ u8SciCntr - 1U ] )->u32Rx_InputPin ].R  = INPUT | ( &TempConfigPtr[ u8SciCntr - 1U ] )->u32Rx_InputPinFnc;
			SIU.PCR[ ( &TempConfigPtr[ u8SciCntr - 1U ] )->u32Tx_OutputPin ].R = OUTPUT | ( &TempConfigPtr[ u8SciCntr - 1U ] )->u32Tx_OutputPinFnc;

			vfnEmios_Init_Channels( ( &TempConfigPtr[ u8SciCntr - 1U ] ) );

			emsci[ ( &TempConfigPtr[ u8SciCntr - 1U ] )->u32Emsci_Unit ].tx_buffer	= EMSCI_TX_EMPTY;									/*		mark the Tx buffer as empty  			  */
			emsci[ ( &TempConfigPtr[ u8SciCntr - 1U ] )->u32Emsci_Unit ].rx_buffer	= EMSCI_RX_EMPTY;									/*		mark the Rx buffer as empty  			  */
			emsci[ ( &TempConfigPtr[ u8SciCntr - 1U ] )->u32Emsci_Unit ].tx_shift = 0U;
			emsci[ ( &TempConfigPtr[ u8SciCntr - 1U ] )->u32Emsci_Unit ].rx_shift = 0U;
			emsci[ ( &TempConfigPtr[ u8SciCntr - 1U ] )->u32Emsci_Unit ].tx_in_progress = 0U;
																/*		mark the transmitter as idle			  */
			emsci[ ( &TempConfigPtr[ u8SciCntr - 1U ] )->u32Emsci_Unit ].rx_in_progress = 0U;
																/*		mark the receiver as idle				  */
			emsci[ ( &TempConfigPtr[ u8SciCntr - 1U ] )->u32Emsci_Unit ].rx_last_edge = 1U;
																/*		idle state of the pin is high			  */

		}

#if (  EMSCI_ERROR_DETECT  ==  ON  )
	}
#endif

}

/******************************************************************************************************************/
/*								   RX FUNCTIONS  								 */
/*																				 */
/******************************************************************************************************************/
#if (  EMSCI_CHANNELS  ==  1  )
/*----------------------------------------------------------------------------------------------------------------*/
/**
 * \brief	 This function detects any changes on Rx line using the input capture\n
			 function of the MPC551x.
 * \param	 void 
 * \return	 void
 * \warning  Function used ONLY by EMSCI channel 1
/*-------------------------------------------------------------------------------*/
void vfn_Rx_Fnc_1( void  )
{
		   int16_t			 rx_edge_delta = 0U;
		   uint16_t  		 u16CounterVal = 0U;
		   temsci_data		* pemsci;
	const  Emsci_ConfigType * pemsci_cfg;

	pemsci	   = ( temsci_data * )&emsci[ EMSCI0 ];
	pemsci_cfg = ( Emsci_ConfigType * )&Emsci_ChannelConfig[ EMSCI0 ];

	/* SECTION 1: check for glitches and calculate the time from the last edge */
	/* check if the current state corresponds to the expected state */
	if( EMIOS.CH[ pemsci_cfg->u32Rx_EmiosChannel ].CSR.B.UCIN == pemsci->rx_last_edge )
	{
		/* last known state is the same as the current state - it was a short glitch */
		/* clear the interrupt flag */
		eMIOS_STATUS_FLAG( pemsci_cfg->u32Rx_EmiosChannel ) = 0x01U;
		/*---- MISRA Rule 14.7 violation: A function shall have a single point of exit */
		return;
	}															/*	check if this is the start bit of a new  	  */
																/*		character								  */
	if( pemsci->rx_in_progress == 0U )
	{
		/* it is a new character */
		/* check if start bit is low */
		if( pemsci->rx_last_edge == 0U )
		{
			/* start bit seems to be high - misalignement! */
			/* clear the interrupt flag */
			eMIOS_STATUS_FLAG( pemsci_cfg->u32Rx_EmiosChannel ) = 0x01U;
			/*---- MISRA Rule 14.7 violation: A function shall have a single point of exit */
			return;
		}														/*	  reception of the character is now in		  */
																/*		  progress								  */
		pemsci->rx_in_progress								= 1U;
		/* assign the bit counter with the number of bit times till the end of the character +1 to make sure we get past the stop bit */
		pemsci->rx_bit_counter = EMSCI_BIT_COUNT + 3U;
		/* initialise the shift register */
		pemsci->rx_shift									= 0xFFFFU;
		/* clear the interrupt flag */
		eMIOS_STATUS_FLAG( pemsci_cfg->u32Rx_EmiosChannel ) = 0x01U;
		/* the Rx pin is now low (start bit) */
		pemsci->rx_last_edge								= 0U;
		/* update the time of the last edge */
		pemsci->rx_last_edge_time = eMIOS_TIMER_COUNT( 1U );
		/* check state of the transmitter */
		if( pemsci->tx_in_progress == 0U )
		{
			/* transmitter is currently idle */
			/* lock the semaphore first */
			//while((_ssem(EMSCI_SEMAPHORE),_ssem(EMSCI_SEMAPHORE))==0);
			/* enable the Tx interrupts to guarantee correct identification of the end of the character */
			//Timer.tie.byte |= (0x02<<(emsci_no<<1));
			/* release the semaphore */
			//_csem(EMSCI_SEMAPHORE);
			u16CounterVal = eMIOS_TIMER_COUNT( pemsci_cfg->u32Tx_EmiosChannel );
			eMIOS_SET_MODE( 0U, EMIOS_GPIO_MODE );
			/* arrange for the first interrupt to come one bit time from now */
			eMIOS_TIMER_COUNT(0U) = (uint16_t)((uint16_t)u16CounterVal + (uint16_t)EMSCI_BIT_TIME);
			eMIOS_SET_MODE( 0U, EMIOS_SAOC_MODE );
		}														/*	  ---- MISRA Rule 14.7 violation: A function  */
																/*		  shall have a single point of exit  	  */
		return;
	}
	else
	{
		/* calculate the time difference between the last and the current edge */
		rx_edge_delta = eMIOS_TIMER_COUNT( pemsci_cfg->u32Rx_EmiosChannel ) - pemsci->rx_last_edge_time;
	}															/*	check for a glitch (pulse shorter than half   */
																/*		of bit time)							  */
	if( rx_edge_delta <= ( EMSCI_BIT_TIME>>1U ) )
	{
		/* clear the interrupt flag */
		eMIOS_STATUS_FLAG( pemsci_cfg->u32Rx_EmiosChannel ) = 0x01U;
		/*---- MISRA Rule 14.7 violation: A function shall have a single point of exit */
		return;
	}															/*	update the time of the last edge			  */
	pemsci->rx_last_edge_time = (uint16_t)((uint16_t)pemsci->rx_last_edge_time + (uint16_t)rx_edge_delta);
	/* SECTION 2: update the shift register based on the time of the captured edge */
	/* check the state of the receive line prior to the current edge */
	if( pemsci->rx_last_edge == 1U )
	{
		/* the Rx pin was high up to now - i.e. the edge could be the start bit of the next character */
		do
		{
			/* the time delta is larger than half of bit time - at least one bit to shift */
			pemsci->rx_shift = pemsci->rx_shift>>1U;
			/* set the bit just shifted in (Rx line was high) */
			pemsci->rx_shift = pemsci->rx_shift | 0x8000U;
			/* subtract one bit time from the time recorded between the edges */
			rx_edge_delta	 = (int16_t)((int16_t)rx_edge_delta - (int16_t)EMSCI_BIT_TIME);
			/* check if the reception is now complete */
			if( ( pemsci->rx_shift & ( 0x8000U>>( EMSCI_BIT_COUNT + 1U ) ) ) == 0U )
			{
				/* there is a 0 in the shift register where we expect the start bit */
				/* a complete character was received and the current edge is start of the next character */
				/* receive the character into the Rx buffer */
				if( ( pemsci->rx_shift & ( 0xffffU>>( EMSCI_BIT_COUNT + 2U ) ) )
					 == ( 0xffffU>>( EMSCI_BIT_COUNT + 2 ) ) )
				{
					/* it is a standard character of EMSCI_BIT_COUNT bits in it */
					pemsci->rx_buffer = ( ( pemsci->rx_shift )>>( 16U - EMSCI_BIT_COUNT - 1U ) ) & 0x00ffU;
				}
				else
				{
					/* it is a longer than normal character */
					/* shift the start bit into the LSB position */
					while( ( pemsci->rx_shift & 0x0001U ) == 1U )
					{
					   pemsci->rx_shift = pemsci->rx_shift>>1U;
					   /* get rid of the start bit and mark it as a special character */
					   pemsci->rx_buffer = ( pemsci->rx_shift>>1U ) | 0x8000U;
					}
						
				}												/*		  mark reception as complete			  */
				pemsci->rx_in_progress = 0U;
 /* interrupt the CPU (the Rx buffer is full) */
#if (  EMSCI_USE_INTERRUPTS  )
				pemsci_cfg->RxEndNotification( );
#endif															/*leave the interrupt flag set so the start of	  */
																/*	  the next character is processed in next	  */
																/*	  iteration of the thread					  */

				return;
			}
		} while( rx_edge_delta > ( EMSCI_BIT_TIME>>1U ) );
	}
	else
	{
		/* the Rx pin was low up to now, shift the low bits into the shift register */
		while( rx_edge_delta > ( EMSCI_BIT_TIME>>1U ) )
		{
			/* the time delta is larger than half of bit time - at least one bit to shift */
			pemsci->rx_shift = pemsci->rx_shift>>1U;
			/* subtract one bit time from the time recorded between the edges */
			rx_edge_delta	 = (int16_t)((int16_t)rx_edge_delta - (int16_t)EMSCI_BIT_TIME);
		}
	}															/*	toggle current state of the Rx signal		  */
	pemsci->rx_last_edge = pemsci->rx_last_edge ^ 1U;

}
#endif

#if (  EMSCI_CHANNELS  >  1  )
/*----------------------------------------------------------------------------------------------------------------*/
/**
 * \brief	 This function detects any changes on Rx line using the input capture\n
			 function of the MPC551x.
 * \param	 void 
 * \return	 void
 * \warning  Function used ONLY by EMSCI channel 4.
/*-------------------------------------------------------------------------------*/
void vfn_Rx_Fnc_2( void  )
{

		   int16_t			 rx_edge_delta = 0;
		   uint32_t  		 u16CounterVal = 0;
		   temsci_data		* pemsci;
	const  Emsci_ConfigType * pemsci_cfg;

	pemsci	   = ( temsci_data * )&emsci[ EMSCI1 ];
	pemsci_cfg = ( Emsci_ConfigType * )&Emsci_ChannelConfig[ EMSCI1 ];

	/* SECTION 1: check for glitches and calculate the time from the last edge */
	/* check if the current state corresponds to the expected state */
	if( EMIOS.CH[ pemsci_cfg->u32Rx_EmiosChannel ].CSR.B.UCIN == pemsci->rx_last_edge )
	{
		/* last known state is the same as the current state - it was a short glitch */
		/* clear the interrupt flag */
		eMIOS_STATUS_FLAG( pemsci_cfg->u32Rx_EmiosChannel ) = 0x01U;
		/*---- MISRA Rule 14.7 violation: A function shall have a single point of exit */
		return;
	}															/*	check if this is the start bit of a new  	  */
																/*		character								  */
	if( pemsci->rx_in_progress == 0U )
	{
		/* it is a new character */
		/* check if start bit is low */
		if( pemsci->rx_last_edge == 0U )
		{
			/* start bit seems to be high - misalignement! */
			/* clear the interrupt flag */
			eMIOS_STATUS_FLAG( pemsci_cfg->u32Rx_EmiosChannel ) = 0x01U;
			/*---- MISRA Rule 14.7 violation: A function shall have a single point of exit */
			return;
		}														/*	  reception of the character is now in		  */
																/*		  progress								  */
		pemsci->rx_in_progress								= 1U;
		/* assign the bit counter with the number of bit times till the end of the character +1 to make sure we get past the stop bit */
		pemsci->rx_bit_counter = EMSCI_BIT_COUNT + 3U;
		/* initialise the shift register */
		pemsci->rx_shift									= 0xFFFFU;
		/* clear the interrupt flag */
		eMIOS_STATUS_FLAG( pemsci_cfg->u32Rx_EmiosChannel ) = 0x01U;
		/* the Rx pin is now low (start bit) */
		pemsci->rx_last_edge								= 0U;
		/* update the time of the last edge */
		pemsci->rx_last_edge_time = eMIOS_TIMER_COUNT( 1U );
		/* check state of the transmitter */
		if( pemsci->tx_in_progress == 0U )
		{
			/* transmitter is currently idle */
			/* lock the semaphore first */
			//while((_ssem(EMSCI_SEMAPHORE),_ssem(EMSCI_SEMAPHORE))==0);
			/* enable the Tx interrupts to guarantee correct identification of the end of the character */
			//Timer.tie.byte |= (0x02<<(emsci_no<<1));
			/* release the semaphore */
			//_csem(EMSCI_SEMAPHORE);
			u16CounterVal = eMIOS_TIMER_COUNT( pemsci_cfg->u32Tx_EmiosChannel );
			eMIOS_SET_MODE( 0U, EMIOS_GPIO_MODE );
			/* arrange for the first interrupt to come one bit time from now */
			eMIOS_TIMER_COUNT( 0U ) = u16CounterVal + EMSCI_BIT_TIME;
			eMIOS_SET_MODE( 0U, EMIOS_SAOC_MODE );
		}														/*	  ---- MISRA Rule 14.7 violation: A function  */
																/*		  shall have a single point of exit  	  */
		return;
	}
	else
	{
		/* calculate the time difference between the last and the current edge */
		rx_edge_delta
		= eMIOS_TIMER_COUNT( pemsci_cfg->u32Rx_EmiosChannel ) - pemsci->rx_last_edge_time;
	}															/*	check for a glitch (pulse shorter than half   */
																/*		of bit time)							  */
	if( rx_edge_delta <= ( EMSCI_BIT_TIME>>1U ) )
	{
		/* clear the interrupt flag */
		eMIOS_STATUS_FLAG( pemsci_cfg->u32Rx_EmiosChannel ) = 0x01U;
		/*---- MISRA Rule 14.7 violation: A function shall have a single point of exit */
		return;
	}															/*	update the time of the last edge			  */
	pemsci->rx_last_edge_time = pemsci->rx_last_edge_time + rx_edge_delta;
	/* SECTION 2: update the shift register based on the time of the captured edge */
	/* check the state of the receive line prior to the current edge */
	if( pemsci->rx_last_edge == 1U )
	{
		/* the Rx pin was high up to now - i.e. the edge could be the start bit of the next character */
		do
		{
			/* the time delta is larger than half of bit time - at least one bit to shift */
			pemsci->rx_shift = pemsci->rx_shift>>1U;
			/* set the bit just shifted in (Rx line was high) */
			pemsci->rx_shift = pemsci->rx_shift | 0x8000U;
			/* subtract one bit time from the time recorded between the edges */
			rx_edge_delta	 = rx_edge_delta - EMSCI_BIT_TIME;
			/* check if the reception is now complete */
			if( ( pemsci->rx_shift & ( 0x8000U>>( EMSCI_BIT_COUNT + 1U ) ) ) == 0U )
			{
				/* there is a 0 in the shift register where we expect the start bit */
				/* a complete character was received and the current edge is start of the next character */
				/* receive the character into the Rx buffer */
				if( ( pemsci->rx_shift & ( 0xffffU>>( EMSCI_BIT_COUNT + 2U ) ) )
					 == ( 0xffffU>>( EMSCI_BIT_COUNT + 2 ) ) )
				{
					/* it is a standard character of EMSCI_BIT_COUNT bits in it */
					pemsci->rx_buffer
					= ( ( pemsci->rx_shift )>>( 16U - EMSCI_BIT_COUNT - 1U ) ) & 0x00ffU;
				}
				else
				{
						/* it is a longer than normal character */
					/* shift the start bit into the LSB position */
					while( ( pemsci->rx_shift & 0x0001U ) == 1U )
					{
					   pemsci->rx_shift = pemsci->rx_shift>>1U;
					   /* get rid of the start bit and mark it as a special character */
					   pemsci->rx_buffer = ( pemsci->rx_shift>>1U ) | 0x8000U;
					}
				}												/*		  mark reception as complete			  */
				pemsci->rx_in_progress = 0U;
 /* interrupt the CPU (the Rx buffer is full) */
#if (  EMSCI_USE_INTERRUPTS  )
				pemsci_cfg->RxEndNotification( );
#endif															/*leave the interrupt flag set so the start of	  */
																/*	  the next character is processed in next	  */
																/*	  iteration of the thread					  */

				return;
			}
		} while( rx_edge_delta > ( EMSCI_BIT_TIME>>1U ) );
	}
	else
	{
		/* the Rx pin was low up to now, shift the low bits into the shift register */
		while( rx_edge_delta > ( EMSCI_BIT_TIME>>1U ) )
		{
			/* the time delta is larger than half of bit time - at least one bit to shift */
			pemsci->rx_shift = pemsci->rx_shift>>1U;
			/* subtract one bit time from the time recorded between the edges */
			rx_edge_delta	 = rx_edge_delta - EMSCI_BIT_TIME;
		}
	}															/*	toggle current state of the Rx signal		  */
	pemsci->rx_last_edge = pemsci->rx_last_edge ^ 1U;
}
#endif

#if (  EMSCI_CHANNELS  >  2  )
/*----------------------------------------------------------------------------------------------------------------*/
/**
 * \brief	 This function detects any changes on Rx line using the input capture\n
			 function of the MPC551x.
 * \param	 void 
 * \return	 void
 * \warning  Function used ONLY by EMSCI channel 3
/*-------------------------------------------------------------------------------*/
void vfn_Rx_Fnc_3( void  )
{

		   int16_t			 rx_edge_delta = 0;
		   uint32_t  		 u16CounterVal = 0;
		   temsci_data		* pemsci;
	const  Emsci_ConfigType * pemsci_cfg;

	pemsci	   = ( temsci_data * )&emsci[ EMSCI0 ];
	pemsci_cfg = ( Emsci_ConfigType * )&Emsci_ChannelConfig[ EMSCI2 ];

	/* SECTION 1: check for glitches and calculate the time from the last edge */
	/* check if the current state corresponds to the expected state */
	if( EMIOS.CH[ pemsci_cfg->u32Rx_EmiosChannel ].CSR.B.UCIN == pemsci->rx_last_edge )
	{
		/* last known state is the same as the current state - it was a short glitch */
		/* clear the interrupt flag */
		eMIOS_STATUS_FLAG( pemsci_cfg->u32Rx_EmiosChannel ) = 0x01U;
		/*---- MISRA Rule 14.7 violation: A function shall have a single point of exit */
		return;
	}															/*	check if this is the start bit of a new  	  */
																/*		character								  */
	if( pemsci->rx_in_progress == 0U )
	{
		/* it is a new character */
		/* check if start bit is low */
		if( pemsci->rx_last_edge == 0U )
		{
			/* start bit seems to be high - misalignement! */
			/* clear the interrupt flag */
			eMIOS_STATUS_FLAG( pemsci_cfg->u32Rx_EmiosChannel ) = 0x01U;
			/*---- MISRA Rule 14.7 violation: A function shall have a single point of exit */
			return;
		}														/*	  reception of the character is now in		  */
																/*		  progress								  */
		pemsci->rx_in_progress								= 1U;
		/* assign the bit counter with the number of bit times till the end of the character +1 to make sure we get past the stop bit */
		pemsci->rx_bit_counter = EMSCI_BIT_COUNT + 3U;
		/* initialise the shift register */
		pemsci->rx_shift									= 0xFFFFU;
		/* clear the interrupt flag */
		eMIOS_STATUS_FLAG( pemsci_cfg->u32Rx_EmiosChannel ) = 0x01U;
		/* the Rx pin is now low (start bit) */
		pemsci->rx_last_edge								= 0U;
		/* update the time of the last edge */
		pemsci->rx_last_edge_time = eMIOS_TIMER_COUNT( 1U );
		/* check state of the transmitter */
		if( pemsci->tx_in_progress == 0U )
		{
			/* transmitter is currently idle */
			/* lock the semaphore first */
			//while((_ssem(EMSCI_SEMAPHORE),_ssem(EMSCI_SEMAPHORE))==0);
			/* enable the Tx interrupts to guarantee correct identification of the end of the character */
			//Timer.tie.byte |= (0x02<<(emsci_no<<1));
			/* release the semaphore */
			//_csem(EMSCI_SEMAPHORE);
			u16CounterVal = eMIOS_TIMER_COUNT( pemsci_cfg->u32Tx_EmiosChannel );
			eMIOS_SET_MODE( 0U, EMIOS_GPIO_MODE );
			/* arrange for the first interrupt to come one bit time from now */
			eMIOS_TIMER_COUNT( 0U ) = u16CounterVal + EMSCI_BIT_TIME;
			eMIOS_SET_MODE( 0U, EMIOS_SAOC_MODE );
		}														/*	  ---- MISRA Rule 14.7 violation: A function  */
																/*		  shall have a single point of exit  	  */
		return;
	}
	else
	{
		/* calculate the time difference between the last and the current edge */
		rx_edge_delta
		= eMIOS_TIMER_COUNT( pemsci_cfg->u32Rx_EmiosChannel ) - pemsci->rx_last_edge_time;
	}															/*	check for a glitch (pulse shorter than half   */
																/*		of bit time)							  */
	if( rx_edge_delta <= ( EMSCI_BIT_TIME>>1U ) )
	{
		/* clear the interrupt flag */
		eMIOS_STATUS_FLAG( pemsci_cfg->u32Rx_EmiosChannel ) = 0x01U;
		/*---- MISRA Rule 14.7 violation: A function shall have a single point of exit */
		return;
	}															/*	update the time of the last edge			  */
	pemsci->rx_last_edge_time = pemsci->rx_last_edge_time + rx_edge_delta;
	/* SECTION 2: update the shift register based on the time of the captured edge */
	/* check the state of the receive line prior to the current edge */
	if( pemsci->rx_last_edge == 1U )
	{
		/* the Rx pin was high up to now - i.e. the edge could be the start bit of the next character */
		do
		{
			/* the time delta is larger than half of bit time - at least one bit to shift */
			pemsci->rx_shift = pemsci->rx_shift>>1U;
			/* set the bit just shifted in (Rx line was high) */
			pemsci->rx_shift = pemsci->rx_shift | 0x8000U;
			/* subtract one bit time from the time recorded between the edges */
			rx_edge_delta	 = rx_edge_delta - EMSCI_BIT_TIME;
			/* check if the reception is now complete */
			if( ( pemsci->rx_shift & ( 0x8000U>>( EMSCI_BIT_COUNT + 1U ) ) ) == 0U )
			{
				/* there is a 0 in the shift register where we expect the start bit */
				/* a complete character was received and the current edge is start of the next character */
				/* receive the character into the Rx buffer */
				if( ( pemsci->rx_shift & ( 0xffffU>>( EMSCI_BIT_COUNT + 2U ) ) )
					 == ( 0xffffU>>( EMSCI_BIT_COUNT + 2 ) ) )
				{
					/* it is a standard character of EMSCI_BIT_COUNT bits in it */
					pemsci->rx_buffer
					= ( ( pemsci->rx_shift )>>( 16U - EMSCI_BIT_COUNT - 1U ) ) & 0x00ffU;
				}
				else
				{
					/* it is a longer than normal character */
					/* shift the start bit into the LSB position */
					while( ( pemsci->rx_shift & 0x0001U ) == 1U )
					{
					   pemsci->rx_shift = pemsci->rx_shift>>1U;
					   /* get rid of the start bit and mark it as a special character */
					   pemsci->rx_buffer = ( pemsci->rx_shift>>1U ) | 0x8000U;
					}
				}												/*		  mark reception as complete			  */
				pemsci->rx_in_progress = 0U;
			 /* interrupt the CPU (the Rx buffer is full) */
			#if (  EMSCI_USE_INTERRUPTS  )
			  pemsci_cfg->RxEndNotification( );
			#endif															/*leave the interrupt flag set so the start of	  */
																/*	  the next character is processed in next	  */
																/*	  iteration of the thread					  */

				return;
			}
		} while( rx_edge_delta > ( EMSCI_BIT_TIME>>1U ) );
	}
	else
	{
		/* the Rx pin was low up to now, shift the low bits into the shift register */
		while( rx_edge_delta > ( EMSCI_BIT_TIME>>1U ) )
		{
			/* the time delta is larger than half of bit time - at least one bit to shift */
			pemsci->rx_shift = pemsci->rx_shift>>1U;
			/* subtract one bit time from the time recorded between the edges */
			rx_edge_delta	 = rx_edge_delta - EMSCI_BIT_TIME;
		}
	}															/*	toggle current state of the Rx signal		  */
	pemsci->rx_last_edge = pemsci->rx_last_edge ^ 1U;

}
#endif

#if (  EMSCI_CHANNELS  >  3  )
/*----------------------------------------------------------------------------------------------------------------*/
/**
 * \brief	 This function detects any changes on Rx line using the input capture\n
			 function of the MPC551x.
 * \param	 void 
 * \return	 void
 * \warning  Function used ONLY by EMSCI channel 4.
/*-------------------------------------------------------------------------------*/
void vfn_Rx_Fnc_4( void  )
{

	int16_t  		  rx_edge_delta = 0;
	uint32_t		  u16CounterVal = 0;
	temsci_data  	 * pemsci;
	Emsci_ConfigType * pemsci_cfg;

	pemsci	   = ( temsci_data * )&emsci[ EMSCI3 ];
	pemsci_cfg = ( Emsci_ConfigType )&Emsci_ChannelConfig[ EMSCI3 ];

	/* SECTION 1: check for glitches and calculate the time from the last edge */
	/* check if the current state corresponds to the expected state */
	if( EMIOS.CH[ pemsci_cfg->u32Rx_EmiosChannel ].CSR.B.UCIN == pemsci->rx_last_edge )
	{
		/* last known state is the same as the current state - it was a short glitch */
		/* clear the interrupt flag */
		eMIOS_STATUS_FLAG( pemsci_cfg->u32Rx_EmiosChannel ) = 0x01U;
		/*---- MISRA Rule 14.7 violation: A function shall have a single point of exit */
		return;
	}															/*	check if this is the start bit of a new  	  */
																/*		character								  */
	if( pemsci->rx_in_progress == 0U )
	{
		/* it is a new character */
		/* check if start bit is low */
		if( pemsci->rx_last_edge == 0U )
		{
			/* start bit seems to be high - misalignement! */
			/* clear the interrupt flag */
			eMIOS_STATUS_FLAG( pemsci_cfg->u32Rx_EmiosChannel ) = 0x01U;
			/*---- MISRA Rule 14.7 violation: A function shall have a single point of exit */
			return;
		}														/*	  reception of the character is now in		  */
																/*		  progress								  */
		pemsci->rx_in_progress								= 1U;
		/* assign the bit counter with the number of bit times till the end of the character +1 to make sure we get past the stop bit */
		pemsci->rx_bit_counter = EMSCI_BIT_COUNT + 3U;
		/* initialise the shift register */
		pemsci->rx_shift									= 0xFFFFU;
		/* clear the interrupt flag */
		eMIOS_STATUS_FLAG( pemsci_cfg->u32Rx_EmiosChannel ) = 0x01U;
		/* the Rx pin is now low (start bit) */
		pemsci->rx_last_edge								= 0U;
		/* update the time of the last edge */
		pemsci->rx_last_edge_time = eMIOS_TIMER_COUNT( 1U );
		/* check state of the transmitter */
		if( pemsci->tx_in_progress == 0U )
		{
			/* transmitter is currently idle */
			/* lock the semaphore first */
			//while((_ssem(EMSCI_SEMAPHORE),_ssem(EMSCI_SEMAPHORE))==0);
			/* enable the Tx interrupts to guarantee correct identification of the end of the character */
			//Timer.tie.byte |= (0x02<<(emsci_no<<1));
			/* release the semaphore */
			//_csem(EMSCI_SEMAPHORE);
			u16CounterVal = eMIOS_TIMER_COUNT( pemsci_cfg->u32Tx_EmiosChannel );
			eMIOS_SET_MODE( 0U, EMIOS_GPIO_MODE );
			/* arrange for the first interrupt to come one bit time from now */
			eMIOS_TIMER_COUNT( 0U ) = u16CounterVal + EMSCI_BIT_TIME;
			eMIOS_SET_MODE( 0U, EMIOS_SAOC_MODE );
		}														/*	  ---- MISRA Rule 14.7 violation: A function  */
																/*		  shall have a single point of exit  	  */
		return;
	}
	else
	{
		/* calculate the time difference between the last and the current edge */
		rx_edge_delta
		= eMIOS_TIMER_COUNT( pemsci_cfg->u32Rx_EmiosChannel ) - pemsci->rx_last_edge_time;
	}															/*	check for a glitch (pulse shorter than half   */
																/*		of bit time)							  */
	if( rx_edge_delta <= ( EMSCI_BIT_TIME>>1U ) )
	{
		/* clear the interrupt flag */
		eMIOS_STATUS_FLAG( pemsci_cfg->u32Rx_EmiosChannel ) = 0x01U;
		/*---- MISRA Rule 14.7 violation: A function shall have a single point of exit */
		return;
	}															/*	update the time of the last edge			  */
	pemsci->rx_last_edge_time = pemsci->rx_last_edge_time + rx_edge_delta;
	/* SECTION 2: update the shift register based on the time of the captured edge */
	/* check the state of the receive line prior to the current edge */
	if( pemsci->rx_last_edge == 1U )
	{
		/* the Rx pin was high up to now - i.e. the edge could be the start bit of the next character */
		do
		{
			/* the time delta is larger than half of bit time - at least one bit to shift */
			pemsci->rx_shift = pemsci->rx_shift>>1U;
			/* set the bit just shifted in (Rx line was high) */
			pemsci->rx_shift = pemsci->rx_shift | 0x8000U;
			/* subtract one bit time from the time recorded between the edges */
			rx_edge_delta	 = rx_edge_delta - EMSCI_BIT_TIME;
			/* check if the reception is now complete */
			if( ( pemsci->rx_shift & ( 0x8000U>>( EMSCI_BIT_COUNT + 1U ) ) ) == 0U )
			{
				/* there is a 0 in the shift register where we expect the start bit */
				/* a complete character was received and the current edge is start of the next character */
				/* receive the character into the Rx buffer */
				if( ( pemsci->rx_shift & ( 0xffffU>>( EMSCI_BIT_COUNT + 2U ) ) )
					 == ( 0xffffU>>( EMSCI_BIT_COUNT + 2 ) ) )
				{
					/* it is a standard character of EMSCI_BIT_COUNT bits in it */
					pemsci->rx_buffer
					= ( ( pemsci->rx_shift )>>( 16U - EMSCI_BIT_COUNT - 1U ) ) & 0x00ffU;
				}
				else
				{
					/* it is a longer than normal character */
					/* shift the start bit into the LSB position */
					while( ( pemsci->rx_shift & 0x0001U ) == 1U )
					{
					   pemsci->rx_shift = pemsci->rx_shift>>1U;
					   /* get rid of the start bit and mark it as a special character */
					   pemsci->rx_buffer = ( pemsci->rx_shift>>1U ) | 0x8000U;
					}
				}												/*		  mark reception as complete			  */
				pemsci->rx_in_progress = 0U;
 /* interrupt the CPU (the Rx buffer is full) */
#if (  EMSCI_USE_INTERRUPTS  )
				pemsci_cfg->RxEndNotification( );
#endif															/*leave the interrupt flag set so the start of	  */
																/*	  the next character is processed in next	  */
																/*	  iteration of the thread					  */

				return;
			}
		} while( rx_edge_delta > ( EMSCI_BIT_TIME>>1U ) );
	}
	else
	{
		/* the Rx pin was low up to now, shift the low bits into the shift register */
		while( rx_edge_delta > ( EMSCI_BIT_TIME>>1U ) )
		{
			/* the time delta is larger than half of bit time - at least one bit to shift */
			pemsci->rx_shift = pemsci->rx_shift>>1U;
			/* subtract one bit time from the time recorded between the edges */
			rx_edge_delta	 = rx_edge_delta - EMSCI_BIT_TIME;
		}
	}															/*	toggle current state of the Rx signal		  */
	pemsci->rx_last_edge = pemsci->rx_last_edge ^ 1U;

}
#endif
/******************************************************************************************************************/
/*								   TX FUNCTIONS  								 */
/*																				 */
/******************************************************************************************************************/
#if (  EMSCI_CHANNELS  ==  1  )
/*----------------------------------------------------------------------------------------------------------------*/
/**
 * \brief	 This function generates SCI frames using the output compare function\n
			 of the MPC551x.
 * \param	 void 
 * \return	 void
 * \warning  This function is used only to generate Tx frames from EMSCI channel 0
/*-------------------------------------------------------------------------------*/

void vfn_Tx_Fnc_1( void  )
{
	uint32_t		  u32CntrVal = 0U;
	uint32_t		  u32Match_1 = 0U;
	temsci_data  	 * pemsci;
	Emsci_ConfigType * pemsci_cfg;

	pemsci	   = ( temsci_data * )&emsci[ EMSCI0 ];
	pemsci_cfg = ( Emsci_ConfigType * )&Emsci_ChannelConfig[ EMSCI0 ];
	/* SECTION 1: check whether the receiver needs any help */
	if( pemsci->rx_in_progress == 1U )
	{
		/* receiver is active */
		/* decrement the count of bit times until end of the character */
		if( pemsci->rx_bit_counter > 0U )
		{
		  	pemsci->rx_bit_counter--;	
		}
		
		/* check if the Rx line currently high */
		if( pemsci->rx_last_edge == 1U )
		{
			/* Rx line is high, the receiver might have gone past the stop bit without realising */
			if( pemsci->rx_bit_counter == 0U )
			{
				/* the receiver has gone past the stop bit and does not know about it */
				/* receive '1' bits until the start bit appears in the expected position */
				/* at least one bit must be received in any case */
				do
				{
					/* shift the register by 1 bit */
					pemsci->rx_shift >>= 1U;
					/* set the bit just shifted in (Rx line was high) */
					pemsci->rx_shift   = pemsci->rx_shift | 0x8000U;
				} while( ( pemsci->rx_shift & ( 0x8000U>>( EMSCI_BIT_COUNT + 1U ) ) )
							== 1U );
				/* receive the character into the Rx buffer */
				if( ( pemsci->rx_shift & ( 0xffffU>>( EMSCI_BIT_COUNT + 2U ) ) )
					 == ( 0xffffU>>( EMSCI_BIT_COUNT + 2U ) ) )
				{
					/* it is a standard character of EMSCI_BIT_COUNT bits in it */
					pemsci->rx_buffer
					= ( ( pemsci->rx_shift )>>( 16U - EMSCI_BIT_COUNT - 1U ) ) & 0x00ffU;
				}
				else
				{
					/* it is a longer than normal character */
					/* shift the start bit into the LSB position */
					while( ( pemsci->rx_shift & 0x0001U ) == 1U )
					{
						pemsci->rx_shift >>= 1U;
					    /* get rid of the start bit and mark it as a special character */
					    pemsci->rx_buffer = ( pemsci->rx_shift>>1U ) | 0x8000U;
					}
						
				}												/*		  mark reception as complete			  */
				pemsci->rx_in_progress = 0U;
 /* interrupt the CPU (the Rx buffer is full) */
#if (  EMSCI_USE_INTERRUPTS  )									/*make it the corresponding Rx interrupt channel  */
				pemsci_cfg->RxEndNotification( );
#endif
			}
		}
	}															/*	SECTION 2: update the time of the next bit	  */
																/*		edge, if this is the first character	  */
																/*		after some period of idle time then  	  */
																/*		section 2 will correct the edge time	  */
	u32CntrVal = eMIOS_TIMER_COUNT( pemsci_cfg->u32Rx_EmiosChannel );
	eMIOS_SET_MODE( 1U, EMIOS_GPIO_MODE );
	/* arrange for the first interrupt to come one bit time from now */
	eMIOS_TIMER_COUNT( 1 ) = (uint16_t)((uint16_t)u32CntrVal + (uint16_t)EMSCI_BIT_TIME);
	eMIOS_SET_MODE( 1U, EMIOS_SAIC_MODE );

	/* SECTION 3: this code handles start of a transmission when the transmitter was previously idle */
	if( pemsci->tx_in_progress == 0U )
	{
		/* the transmitter is currently idle */
		/* now try to see whether there is a new character to transmit in the buffer */
		if( pemsci->tx_buffer != EMSCI_TX_EMPTY )
		{
			/* Tx buffer is not empty */
			/* copy the character across, create a start bit in LSB and get rid of the MSB */
			pemsci->tx_shift = ( ( pemsci->tx_buffer )<<1U ) | 0x0001U;

			/* Check for NULL characters to be transmitted */
			if( ( pemsci->tx_buffer & 0x8000U ) == 0x0000U )
			{
				/* the character is a plain char to be transmitted - need to supply the stop bit */
				pemsci->tx_shift = pemsci->tx_shift | ( 0x0002U<<EMSCI_BIT_COUNT );
				/* mask-out any unused bits */
				pemsci->tx_shift
				= pemsci->tx_shift & ( 0xffffU>>( 16U - EMSCI_BIT_COUNT - 2U ) );
			}													/*		invalidate the buffer contents			  */
			pemsci->tx_buffer	   = EMSCI_TX_EMPTY;
			/* flag the transmitter as busy */
			pemsci->tx_in_progress = 1U;

			/* start bit will get transmitted one bit time from now */
			u32CntrVal = eMIOS_TIMER_COUNT( pemsci_cfg->u32Tx_EmiosChannel );
																/*		EMIOS.CH[1].CCNTR.R;					  */
			//eMIOS_SET_MODE(1,EMIOS_GPIO_MODE); pemsci_cfg->u32Tx_EmiosChannel
			eMIOS_SET_MODE( pemsci_cfg->u32Tx_EmiosChannel, EMIOS_GPIO_MODE );
			/* arrange for the first interrupt to come one bit time from now */
			//eMIOS_TIMER_COUNT(1) = u32CntrVal + EMSCI_BIT_TIME; pemsci_cfg->u32Tx_EmiosChannel
			eMIOS_TIMER_COUNT( pemsci_cfg->u32Tx_EmiosChannel ) = (uint16_t)((uint16_t)u32CntrVal + (uint16_t)EMSCI_BIT_TIME);
			//eMIOS_SET_MODE(1,EMIOS_SAOC_MODE);pemsci_cfg->u32Tx_EmiosChannel
			eMIOS_SET_MODE( pemsci_cfg->u32Tx_EmiosChannel, EMIOS_SAOC_MODE );
 /* interrupt the CPU (the Tx buffer is empty) */
#if (  EMSCI_USE_INTERRUPTS  )
			pemsci_cfg->TxEndNotification( );
#endif
		}
	}															/*	SECTION 4: this section handles transmission  */
																/*		out of the Tx pin						  */
	if( pemsci->tx_shift != 0U )
	{
		/* the shift register is not empty */
		/* set-up transmission of the LSB bit on next output compare action */
		if( ( pemsci->tx_shift & 0x0001U ) == 0x0001U )
		{
			/* the LSB of the shift register is 1, make sure the timer sets the pin on next compare */
			eMIOS_SET_EDGE_POLARITY( pemsci_cfg->u32Tx_EmiosChannel, 1U );
		}
		else
		{
			/* the LSB of the shift register is 0, make sure the timer clears the pin on next compare */
			eMIOS_SET_EDGE_POLARITY( pemsci_cfg->u32Tx_EmiosChannel, 0U );
		}														/*	  shift the register down, LSB is not needed  */
																/*		  anymore								  */
		pemsci->tx_shift = pemsci->tx_shift>>1U;
		/* there a at least one bit to be transmitted, so clear the interrupt flag */
		eMIOS_STATUS_FLAG( pemsci_cfg->u32Tx_EmiosChannel ) = 0x01U;
	}															/*	SECTION 5: this section tries to reload the   */
																/*		shift register							  */
	if( pemsci->tx_shift == 0U )
	{
		/* the character has completely transmitted, try reload it from the buffer */
		if( pemsci->tx_buffer != EMSCI_TX_EMPTY )
		{
			/* Tx buffer is not empty */
			/* copy the character across, create a start bit in LSB and get rid of the MSB */
			pemsci->tx_shift = pemsci->tx_buffer<<1U;

			if( ( pemsci->tx_buffer & 0x8000U ) == 0x0000U )
			{
				/* the character is was a plain char to be transmitted - need to supply the stop bit */
				pemsci->tx_shift |= ( 0x0002U<<EMSCI_BIT_COUNT );
				/* mask-out any unused bits */
				pemsci->tx_shift
				= pemsci->tx_shift & ( 0xffffU>>( 16U - EMSCI_BIT_COUNT - 2U ) );
			}													/*		invalidate the buffer contents			  */
			pemsci->tx_buffer = EMSCI_TX_EMPTY;
 /* interrupt the CPU (the Tx buffer is empty) */
#if (  EMSCI_USE_INTERRUPTS  )
			pemsci_cfg->TxEndNotification( );
#endif
		}
		else
		{
			/* the shift register is empty & there is nothing to transmit */
			/* mark the transmitter as idle */
			pemsci->tx_in_progress = 0U;
			/* check if reception is in progrees at this time */
			if( pemsci->rx_in_progress == 0U )
			{
				/* receiver is idle */
				/* lock the semaphore first */
				//while((_ssem(EMSCI_SEMAPHORE),_ssem(EMSCI_SEMAPHORE))==0);
				/* disable the output compare interrupt */
				eMIOS_DISABLE_CHANNEL_INTERRUPT( pemsci_cfg->u32Tx_EmiosChannel );
		    	/* release the semaphore */
		    	//_csem(EMSCI_SEMAPHORE);
			}
			else
			{
				/* receiver is active -> the interrupt is left enabled */
				/* clear the interrupt flag instead */
				eMIOS_STATUS_FLAG( pemsci_cfg->u32Tx_EmiosChannel ) = 0x01U;
			}
		}
	}

	u32Match_1	= ( EMIOS.CH[ pemsci_cfg->u32Tx_EmiosChannel ].CADR.R  + EMSCI_BIT_TIME );
	/* Reload Output Compare Counter */
	eMIOS_SET_COUNT_A( pemsci_cfg->u32Tx_EmiosChannel, u32Match_1 );
	/* Clear interrupt flag */
	eMIOS_MATCH_EVENT_FLAG( pemsci_cfg->u32Tx_EmiosChannel ) = 1U;
	/* Clear counter overflow flag */
	eMIOS_CLEAR_OVERFLOW_FLAG( pemsci_cfg->u32Tx_EmiosChannel );
}

#endif

#if (  EMSCI_CHANNELS  >  1  )
/*----------------------------------------------------------------------------------------------------------------*/
/**
 * \brief	 This function generates SCI frames using the output compare function\n
			 of the MPC551x.
 * \param	 void 
 * \return	 void
 * \warning  This function is used only to generate Tx frames from EMSCI channel 2
/*-------------------------------------------------------------------------------*/

void vfn_Tx_Fnc_2( void  )
{
	uint32_t		  u32CntrVal = 0;
	uint32_t		  u32Match_1 = 0;
	temsci_data  	 * pemsci;
	Emsci_ConfigType * pemsci_cfg;

	pemsci	   = ( temsci_data * )&emsci[ EMSCI1 ];
	pemsci_cfg = ( Emsci_ConfigType * )&Emsci_ChannelConfig[ EMSCI1 ];
	/* SECTION 1: check whether the receiver needs any help */
	if( pemsci->rx_in_progress == 1U )
	{
		/* receiver is active */
		/* decrement the count of bit times until end of the character */
		if( pemsci->rx_bit_counter > 0U )
		{
		  pemsci->rx_bit_counter--;	
		}
		
		/* check if the Rx line currently high */
		if( pemsci->rx_last_edge == 1U )
		{
			/* Rx line is high, the receiver might have gone past the stop bit without realising */
			if( pemsci->rx_bit_counter == 0U )
			{
				/* the receiver has gone past the stop bit and does not know about it */
				/* receive '1' bits until the start bit appears in the expected position */
				/* at least one bit must be received in any case */
				do
				{
					/* shift the register by 1 bit */
					pemsci->rx_shift >>= 1U;
					/* set the bit just shifted in (Rx line was high) */
					pemsci->rx_shift   = pemsci->rx_shift | 0x8000U;
				} while( ( pemsci->rx_shift & ( 0x8000U>>( EMSCI_BIT_COUNT + 1U ) ) )
							== 1U );
				/* receive the character into the Rx buffer */
				if( ( pemsci->rx_shift & ( 0xffffU>>( EMSCI_BIT_COUNT + 2U ) ) )
					 == ( 0xffffU>>( EMSCI_BIT_COUNT + 2U ) ) )
				{
					/* it is a standard character of EMSCI_BIT_COUNT bits in it */
					pemsci->rx_buffer
					= ( ( pemsci->rx_shift )>>( 16U - EMSCI_BIT_COUNT - 1U ) ) & 0x00ffU;
				}
				else
				{
					
					/* it is a longer than normal character */
					/* shift the start bit into the LSB position */
					while( ( pemsci->rx_shift & 0x0001U ) == 1U )
					{
						pemsci->rx_shift >>= 1U;
					    /* get rid of the start bit and mark it as a special character */
					    pemsci->rx_buffer = ( pemsci->rx_shift>>1U ) | 0x8000U;
					}
				}												/*		  mark reception as complete			  */
				pemsci->rx_in_progress = 0U;
 /* interrupt the CPU (the Rx buffer is full) */
#if (  EMSCI_USE_INTERRUPTS  )									/*make it the corresponding Rx interrupt channel  */
				pemsci_cfg->RxEndNotification( );
#endif
			}
		}
	}															/*	SECTION 2: update the time of the next bit	  */
																/*		edge, if this is the first character	  */
																/*		after some period of idle time then  	  */
																/*		section 2 will correct the edge time	  */
	u32CntrVal = eMIOS_TIMER_COUNT( pemsci_cfg->u32Rx_EmiosChannel );
	eMIOS_SET_MODE( 1U, EMIOS_GPIO_MODE );
	/* arrange for the first interrupt to come one bit time from now */
	eMIOS_TIMER_COUNT( 1 ) = u32CntrVal + EMSCI_BIT_TIME;
	eMIOS_SET_MODE( 1U, EMIOS_SAIC_MODE );

	/* SECTION 3: this code handles start of a transmission when the transmitter was previously idle */
	if( pemsci->tx_in_progress == 0U )
	{
		/* the transmitter is currently idle */
		/* now try to see whether there is a new character to transmit in the buffer */
		if( pemsci->tx_buffer != EMSCI_TX_EMPTY )
		{
			/* Tx buffer is not empty */
			/* copy the character across, create a start bit in LSB and get rid of the MSB */
			pemsci->tx_shift = ( ( pemsci->tx_buffer )<<1U ) | 0x0001U;

			/* Check for NULL characters to be transmitted */
			if( ( pemsci->tx_buffer & 0x8000U ) == 0x0000U )
			{
				/* the character is a plain char to be transmitted - need to supply the stop bit */
				pemsci->tx_shift = pemsci->tx_shift | ( 0x0002U<<EMSCI_BIT_COUNT );
				/* mask-out any unused bits */
				pemsci->tx_shift
				= pemsci->tx_shift & ( 0xffffU>>( 16U - EMSCI_BIT_COUNT - 2U ) );
			}													/*		invalidate the buffer contents			  */
			pemsci->tx_buffer	   = EMSCI_TX_EMPTY;
			/* flag the transmitter as busy */
			pemsci->tx_in_progress = 1U;

			/* start bit will get transmitted one bit time from now */
			u32CntrVal = eMIOS_TIMER_COUNT( pemsci_cfg->u32Tx_EmiosChannel );
																/*		EMIOS.CH[1].CCNTR.R;					  */
			//eMIOS_SET_MODE(1,EMIOS_GPIO_MODE); pemsci_cfg->u32Tx_EmiosChannel
			eMIOS_SET_MODE( pemsci_cfg->u32Tx_EmiosChannel, EMIOS_GPIO_MODE );
			/* arrange for the first interrupt to come one bit time from now */
			//eMIOS_TIMER_COUNT(1) = u32CntrVal + EMSCI_BIT_TIME; pemsci_cfg->u32Tx_EmiosChannel
			eMIOS_TIMER_COUNT( pemsci_cfg->u32Tx_EmiosChannel ) = u32CntrVal + EMSCI_BIT_TIME;
			//eMIOS_SET_MODE(1,EMIOS_SAOC_MODE);pemsci_cfg->u32Tx_EmiosChannel
			eMIOS_SET_MODE( pemsci_cfg->u32Tx_EmiosChannel, EMIOS_SAOC_MODE );
 /* interrupt the CPU (the Tx buffer is empty) */
#if (  EMSCI_USE_INTERRUPTS  )
			pemsci_cfg->TxEndNotification( );
#endif
		}
	}															/*	SECTION 4: this section handles transmission  */
																/*		out of the Tx pin						  */
	if( pemsci->tx_shift != 0U )
	{
		/* the shift register is not empty */
		/* set-up transmission of the LSB bit on next output compare action */
		if( ( pemsci->tx_shift & 0x0001U ) == 0x0001U )
		{
			/* the LSB of the shift register is 1, make sure the timer sets the pin on next compare */
			eMIOS_SET_EDGE_POLARITY( pemsci_cfg->u32Tx_EmiosChannel, 1U );
		}
		else
		{
			/* the LSB of the shift register is 0, make sure the timer clears the pin on next compare */
			eMIOS_SET_EDGE_POLARITY( pemsci_cfg->u32Tx_EmiosChannel, 0U );
		}														/*	  shift the register down, LSB is not needed  */
																/*		  anymore								  */
		pemsci->tx_shift = pemsci->tx_shift>>1U;
		/* there a at least one bit to be transmitted, so clear the interrupt flag */
		eMIOS_STATUS_FLAG( pemsci_cfg->u32Tx_EmiosChannel ) = 0x01U;
	}															/*	SECTION 5: this section tries to reload the   */
																/*		shift register							  */
	if( pemsci->tx_shift == 0U )
	{
		/* the character has completely transmitted, try reload it from the buffer */
		if( pemsci->tx_buffer != EMSCI_TX_EMPTY )
		{
			/* Tx buffer is not empty */
			/* copy the character across, create a start bit in LSB and get rid of the MSB */
			pemsci->tx_shift = pemsci->tx_buffer<<1U;

			if( ( pemsci->tx_buffer & 0x8000U ) == 0x0000U )
			{
				/* the character is was a plain char to be transmitted - need to supply the stop bit */
				pemsci->tx_shift |= ( 0x0002U<<EMSCI_BIT_COUNT );
				/* mask-out any unused bits */
				pemsci->tx_shift
				= pemsci->tx_shift & ( 0xffffU>>( 16U - EMSCI_BIT_COUNT - 2U ) );
			}													/*		invalidate the buffer contents			  */
			pemsci->tx_buffer = EMSCI_TX_EMPTY;
 /* interrupt the CPU (the Tx buffer is empty) */
#if (  EMSCI_USE_INTERRUPTS  )
			pemsci_cfg->TxEndNotification( );
#endif
		}
		else
		{
			/* the shift register is empty & there is nothing to transmit */
			/* mark the transmitter as idle */
			pemsci->tx_in_progress = 0U;
			/* check if reception is in progrees at this time */
			if( pemsci->rx_in_progress == 0U )
			{
				/* receiver is idle */
				/* lock the semaphore first */
				//while((_ssem(EMSCI_SEMAPHORE),_ssem(EMSCI_SEMAPHORE))==0);
				/* disable the output compare interrupt */
				eMIOS_DISABLE_CHANNEL_INTERRUPT( pemsci_cfg->u32Tx_EmiosChannel );
			/* release the semaphore */
			//_csem(EMSCI_SEMAPHORE);
			}
			else
			{
				/* receiver is active -> the interrupt is left enabled */
				/* clear the interrupt flag instead */
				eMIOS_STATUS_FLAG( pemsci_cfg->u32Tx_EmiosChannel ) = 0x01U;
			}
		}
	}

	u32Match_1
	= ( uint32_t )
		( ( uint32_t )EMIOS.CH[ pemsci_cfg->u32Tx_EmiosChannel ].CADR.R
		   + ( uint32_t )EMSCI_BIT_TIME );
	/* Reload Output Compare Counter */
	eMIOS_SET_COUNT_A( pemsci_cfg->u32Tx_EmiosChannel, u32Match_1 );
	/* Clear interrupt flag */
	eMIOS_MATCH_EVENT_FLAG( pemsci_cfg->u32Tx_EmiosChannel ) = 1U;
	/* Clear counter overflow flag */
	eMIOS_CLEAR_OVERFLOW_FLAG( pemsci_cfg->u32Tx_EmiosChannel );
}

#endif

#if (  EMSCI_CHANNELS  >  2  )
/*----------------------------------------------------------------------------------------------------------------*/
/**
 * \brief	 This function generates SCI frames using the output compare function\n
			 of the MPC551x.
 * \param	 void 
 * \return	 void
 * \warning  This function is used only to generate Tx frames from EMSCI channel 3
/*-------------------------------------------------------------------------------*/

void vfn_Tx_Fnc_3( void  )
{
	uint32_t		  u32CntrVal = 0;
	uint32_t		  u32Match_1 = 0;
	temsci_data  	 * pemsci;
	Emsci_ConfigType * pemsci_cfg;

	pemsci	   = ( temsci_data * )&emsci[ EMSCI2 ];
	pemsci_cfg = ( Emsci_ConfigType * )&Emsci_ChannelConfig[ EMSCI2 ];
	/* SECTION 1: check whether the receiver needs any help */
	if( pemsci->rx_in_progress == 1U )
	{
		/* receiver is active */
		/* decrement the count of bit times until end of the character */
		if( pemsci->rx_bit_counter > 0U )
		{
		   pemsci->rx_bit_counter--;	
		}
			
		/* check if the Rx line currently high */
		if( pemsci->rx_last_edge == 1U )
		{
			/* Rx line is high, the receiver might have gone past the stop bit without realising */
			if( pemsci->rx_bit_counter == 0U )
			{
				/* the receiver has gone past the stop bit and does not know about it */
				/* receive '1' bits until the start bit appears in the expected position */
				/* at least one bit must be received in any case */
				do
				{
					/* shift the register by 1 bit */
					pemsci->rx_shift >>= 1U;
					/* set the bit just shifted in (Rx line was high) */
					pemsci->rx_shift   = pemsci->rx_shift | 0x8000U;
				} while( ( pemsci->rx_shift & ( 0x8000U>>( EMSCI_BIT_COUNT + 1U ) ) )
							== 1U );
				/* receive the character into the Rx buffer */
				if( ( pemsci->rx_shift & ( 0xffffU>>( EMSCI_BIT_COUNT + 2U ) ) )
					 == ( 0xffffU>>( EMSCI_BIT_COUNT + 2U ) ) )
				{
					/* it is a standard character of EMSCI_BIT_COUNT bits in it */
					pemsci->rx_buffer
					= ( ( pemsci->rx_shift )>>( 16U - EMSCI_BIT_COUNT - 1U ) ) & 0x00ffU;
				}
				else
				{
			    	/* it is a longer than normal character */
					/* shift the start bit into the LSB position */
					while( ( pemsci->rx_shift & 0x0001U ) == 1U )
					{
						pemsci->rx_shift >>= 1U;
					    /* get rid of the start bit and mark it as a special character */
					    pemsci->rx_buffer = ( pemsci->rx_shift>>1U ) | 0x8000U;
					}
				}												/*		  mark reception as complete			  */
				pemsci->rx_in_progress = 0U;
 /* interrupt the CPU (the Rx buffer is full) */
#if (  EMSCI_USE_INTERRUPTS  )									/*make it the corresponding Rx interrupt channel  */
				pemsci_cfg->RxEndNotification( );
#endif
			}
		}
	}															/*	SECTION 2: update the time of the next bit	  */
																/*		edge, if this is the first character	  */
																/*		after some period of idle time then  	  */
																/*		section 2 will correct the edge time	  */
	u32CntrVal = eMIOS_TIMER_COUNT( pemsci_cfg->u32Rx_EmiosChannel );
	eMIOS_SET_MODE( 1U, EMIOS_GPIO_MODE );
	/* arrange for the first interrupt to come one bit time from now */
	eMIOS_TIMER_COUNT( 1 ) = u32CntrVal + EMSCI_BIT_TIME;
	eMIOS_SET_MODE( 1U, EMIOS_SAIC_MODE );

	/* SECTION 3: this code handles start of a transmission when the transmitter was previously idle */
	if( pemsci->tx_in_progress == 0U )
	{
		/* the transmitter is currently idle */
		/* now try to see whether there is a new character to transmit in the buffer */
		if( pemsci->tx_buffer != EMSCI_TX_EMPTY )
		{
			/* Tx buffer is not empty */
			/* copy the character across, create a start bit in LSB and get rid of the MSB */
			pemsci->tx_shift = ( ( pemsci->tx_buffer )<<1U ) | 0x0001U;

			/* Check for NULL characters to be transmitted */
			if( ( pemsci->tx_buffer & 0x8000U ) == 0x0000U )
			{
				/* the character is a plain char to be transmitted - need to supply the stop bit */
				pemsci->tx_shift = pemsci->tx_shift | ( 0x0002U<<EMSCI_BIT_COUNT );
				/* mask-out any unused bits */
				pemsci->tx_shift
				= pemsci->tx_shift & ( 0xffffU>>( 16U - EMSCI_BIT_COUNT - 2U ) );
			}													/*		invalidate the buffer contents			  */
			pemsci->tx_buffer	   = EMSCI_TX_EMPTY;
			/* flag the transmitter as busy */
			pemsci->tx_in_progress = 1U;

			/* start bit will get transmitted one bit time from now */
			u32CntrVal = eMIOS_TIMER_COUNT( pemsci_cfg->u32Tx_EmiosChannel );
																/*		EMIOS.CH[1].CCNTR.R;					  */
			//eMIOS_SET_MODE(1,EMIOS_GPIO_MODE); pemsci_cfg->u32Tx_EmiosChannel
			eMIOS_SET_MODE( pemsci_cfg->u32Tx_EmiosChannel, EMIOS_GPIO_MODE );
			/* arrange for the first interrupt to come one bit time from now */
			//eMIOS_TIMER_COUNT(1) = u32CntrVal + EMSCI_BIT_TIME; pemsci_cfg->u32Tx_EmiosChannel
			eMIOS_TIMER_COUNT( pemsci_cfg->u32Tx_EmiosChannel ) = u32CntrVal + EMSCI_BIT_TIME;
			//eMIOS_SET_MODE(1,EMIOS_SAOC_MODE);pemsci_cfg->u32Tx_EmiosChannel
			eMIOS_SET_MODE( pemsci_cfg->u32Tx_EmiosChannel, EMIOS_SAOC_MODE );
 /* interrupt the CPU (the Tx buffer is empty) */
#if (  EMSCI_USE_INTERRUPTS  )
			pemsci_cfg->TxEndNotification( );
#endif
		}
	}															/*	SECTION 4: this section handles transmission  */
																/*		out of the Tx pin						  */
	if( pemsci->tx_shift != 0U )
	{
		/* the shift register is not empty */
		/* set-up transmission of the LSB bit on next output compare action */
		if( ( pemsci->tx_shift & 0x0001U ) == 0x0001U )
		{
			/* the LSB of the shift register is 1, make sure the timer sets the pin on next compare */
			eMIOS_SET_EDGE_POLARITY( pemsci_cfg->u32Tx_EmiosChannel, 1U );
		}
		else
		{
			/* the LSB of the shift register is 0, make sure the timer clears the pin on next compare */
			eMIOS_SET_EDGE_POLARITY( pemsci_cfg->u32Tx_EmiosChannel, 0U );
		}														/*	  shift the register down, LSB is not needed  */
																/*		  anymore								  */
		pemsci->tx_shift = pemsci->tx_shift>>1U;
		/* there a at least one bit to be transmitted, so clear the interrupt flag */
		eMIOS_STATUS_FLAG( pemsci_cfg->u32Tx_EmiosChannel ) = 0x01U;
	}															/*	SECTION 5: this section tries to reload the   */
																/*		shift register							  */
	if( pemsci->tx_shift == 0U )
	{
		/* the character has completely transmitted, try reload it from the buffer */
		if( pemsci->tx_buffer != EMSCI_TX_EMPTY )
		{
			/* Tx buffer is not empty */
			/* copy the character across, create a start bit in LSB and get rid of the MSB */
			pemsci->tx_shift = pemsci->tx_buffer<<1U;

			if( ( pemsci->tx_buffer & 0x8000U ) == 0x0000U )
			{
				/* the character is was a plain char to be transmitted - need to supply the stop bit */
				pemsci->tx_shift |= ( 0x0002U<<EMSCI_BIT_COUNT );
				/* mask-out any unused bits */
				pemsci->tx_shift
				= pemsci->tx_shift & ( 0xffffU>>( 16U - EMSCI_BIT_COUNT - 2U ) );
			}													/*		invalidate the buffer contents			  */
			pemsci->tx_buffer = EMSCI_TX_EMPTY;
 /* interrupt the CPU (the Tx buffer is empty) */
#if (  EMSCI_USE_INTERRUPTS  )
			pemsci_cfg->TxEndNotification( );
#endif
		}
		else
		{
			/* the shift register is empty & there is nothing to transmit */
			/* mark the transmitter as idle */
			pemsci->tx_in_progress = 0U;
			/* check if reception is in progrees at this time */
			if( pemsci->rx_in_progress == 0U )
			{
				/* receiver is idle */
				/* lock the semaphore first */
				//while((_ssem(EMSCI_SEMAPHORE),_ssem(EMSCI_SEMAPHORE))==0);
				/* disable the output compare interrupt */
				eMIOS_DISABLE_CHANNEL_INTERRUPT( pemsci_cfg->u32Tx_EmiosChannel );
			/* release the semaphore */
			//_csem(EMSCI_SEMAPHORE);
			}
			else
			{
				/* receiver is active -> the interrupt is left enabled */
				/* clear the interrupt flag instead */
				eMIOS_STATUS_FLAG( pemsci_cfg->u32Tx_EmiosChannel ) = 0x01U;
			}
		}
	}

	u32Match_1
	= ( uint32_t )
		( ( uint32_t )EMIOS.CH[ pemsci_cfg->u32Tx_EmiosChannel ].CADR.R
		   + ( uint32_t )EMSCI_BIT_TIME );
	/* Reload Output Compare Counter */
	eMIOS_SET_COUNT_A( pemsci_cfg->u32Tx_EmiosChannel, u32Match_1 );
	/* Clear interrupt flag */
	eMIOS_MATCH_EVENT_FLAG( pemsci_cfg->u32Tx_EmiosChannel ) = 1U;
	/* Clear counter overflow flag */
	eMIOS_CLEAR_OVERFLOW_FLAG( pemsci_cfg->u32Tx_EmiosChannel );
}

#endif

#if (  EMSCI_CHANNELS  >  3  )
/*----------------------------------------------------------------------------------------------------------------*/
/**
 * \brief	 This function generates SCI frames using the output compare function\n
			 of the MPC551x.
 * \param	 void 
 * \return	 void
 * \warning  This function is used only to generate Tx frames from EMSCI channel 4
/*-------------------------------------------------------------------------------*/

void vfn_Tx_Fnc_4( void  )
{
	uint32_t		  u32CntrVal = 0;
	uint32_t		  u32Match_1 = 0;
	temsci_data  	 * pemsci;
	Emsci_ConfigType * pemsci_cfg;

	pemsci	   = ( temsci_data * )&emsci[ EMSCI3 ];
	pemsci_cfg = ( Emsci_ConfigType * )&Emsci_ChannelConfig[ EMSCI3 ];
	/* SECTION 1: check whether the receiver needs any help */
	if( pemsci->rx_in_progress == 1U )
	{
		/* receiver is active */
		/* decrement the count of bit times until end of the character */
		if( pemsci->rx_bit_counter > 0U )
		{
		  pemsci->rx_bit_counter--;
		}
			
		/* check if the Rx line currently high */
		if( pemsci->rx_last_edge == 1U )
		{
			/* Rx line is high, the receiver might have gone past the stop bit without realising */
			if( pemsci->rx_bit_counter == 0U )
			{
				/* the receiver has gone past the stop bit and does not know about it */
				/* receive '1' bits until the start bit appears in the expected position */
				/* at least one bit must be received in any case */
				do
				{
					/* shift the register by 1 bit */
					pemsci->rx_shift >>= 1U;
					/* set the bit just shifted in (Rx line was high) */
					pemsci->rx_shift   = pemsci->rx_shift | 0x8000U;
				} while( ( pemsci->rx_shift & ( 0x8000U>>( EMSCI_BIT_COUNT + 1U ) ) )
							== 1U );
				/* receive the character into the Rx buffer */
				if( ( pemsci->rx_shift & ( 0xffffU>>( EMSCI_BIT_COUNT + 2U ) ) )
					 == ( 0xffffU>>( EMSCI_BIT_COUNT + 2U ) ) )
				{
					/* it is a standard character of EMSCI_BIT_COUNT bits in it */
					pemsci->rx_buffer
					= ( ( pemsci->rx_shift )>>( 16U - EMSCI_BIT_COUNT - 1U ) ) & 0x00ffU;
				}
				else
				{
					/* it is a longer than normal character */
					/* shift the start bit into the LSB position */
					while( ( pemsci->rx_shift & 0x0001U ) == 1U )
					{
						pemsci->rx_shift >>= 1U;
					    /* get rid of the start bit and mark it as a special character */
					    pemsci->rx_buffer = ( pemsci->rx_shift>>1U ) | 0x8000U;
					}
				}												/*		  mark reception as complete			  */
				pemsci->rx_in_progress = 0U;
 /* interrupt the CPU (the Rx buffer is full) */
#if (  EMSCI_USE_INTERRUPTS  )									/*make it the corresponding Rx interrupt channel  */
				pemsci_cfg->RxEndNotification( );
#endif
			}
		}
	}															/*	SECTION 2: update the time of the next bit	  */
																/*		edge, if this is the first character	  */
																/*		after some period of idle time then  	  */
																/*		section 2 will correct the edge time	  */
	u32CntrVal = eMIOS_TIMER_COUNT( pemsci_cfg->u32Rx_EmiosChannel );
	eMIOS_SET_MODE( 1U, EMIOS_GPIO_MODE );
	/* arrange for the first interrupt to come one bit time from now */
	eMIOS_TIMER_COUNT( 1U ) = u32CntrVal + EMSCI_BIT_TIME;
	eMIOS_SET_MODE( 1U, EMIOS_SAIC_MODE );

	/* SECTION 3: this code handles start of a transmission when the transmitter was previously idle */
	if( pemsci->tx_in_progress == 0U )
	{
		/* the transmitter is currently idle */
		/* now try to see whether there is a new character to transmit in the buffer */
		if( pemsci->tx_buffer != EMSCI_TX_EMPTY )
		{
			/* Tx buffer is not empty */
			/* copy the character across, create a start bit in LSB and get rid of the MSB */
			pemsci->tx_shift = ( ( pemsci->tx_buffer )<<1U ) | 0x0001U;

			/* Check for NULL characters to be transmitted */
			if( ( pemsci->tx_buffer & 0x8000U ) == 0x0000U )
			{
				/* the character is a plain char to be transmitted - need to supply the stop bit */
				pemsci->tx_shift = pemsci->tx_shift | ( 0x0002U<<EMSCI_BIT_COUNT );
				/* mask-out any unused bits */
				pemsci->tx_shift
				= pemsci->tx_shift & ( 0xffffU>>( 16U - EMSCI_BIT_COUNT - 2U ) );
			}													/*		invalidate the buffer contents			  */
			pemsci->tx_buffer	   = EMSCI_TX_EMPTY;
			/* flag the transmitter as busy */
			pemsci->tx_in_progress = 1U;

			/* start bit will get transmitted one bit time from now */
			u32CntrVal = eMIOS_TIMER_COUNT( pemsci_cfg->u32Tx_EmiosChannel );
																/*		EMIOS.CH[1].CCNTR.R;					  */
			//eMIOS_SET_MODE(1,EMIOS_GPIO_MODE); pemsci_cfg->u32Tx_EmiosChannel
			eMIOS_SET_MODE( pemsci_cfg->u32Tx_EmiosChannel, EMIOS_GPIO_MODE );
			/* arrange for the first interrupt to come one bit time from now */
			//eMIOS_TIMER_COUNT(1) = u32CntrVal + EMSCI_BIT_TIME; pemsci_cfg->u32Tx_EmiosChannel
			eMIOS_TIMER_COUNT( pemsci_cfg->u32Tx_EmiosChannel ) = u32CntrVal + EMSCI_BIT_TIME;
			//eMIOS_SET_MODE(1,EMIOS_SAOC_MODE);pemsci_cfg->u32Tx_EmiosChannel
			eMIOS_SET_MODE( pemsci_cfg->u32Tx_EmiosChannel, EMIOS_SAOC_MODE );
 /* interrupt the CPU (the Tx buffer is empty) */
#if (  EMSCI_USE_INTERRUPTS  )
			pemsci_cfg->TxEndNotification( );
#endif
		}
	}															/*	SECTION 4: this section handles transmission  */
																/*		out of the Tx pin						  */
	if( pemsci->tx_shift != 0U )
	{
		/* the shift register is not empty */
		/* set-up transmission of the LSB bit on next output compare action */
		if( ( pemsci->tx_shift & 0x0001U ) == 0x0001U )
		{
			/* the LSB of the shift register is 1, make sure the timer sets the pin on next compare */
			eMIOS_SET_EDGE_POLARITY( pemsci_cfg->u32Tx_EmiosChannel, 1U );
		}
		else
		{
			/* the LSB of the shift register is 0, make sure the timer clears the pin on next compare */
			eMIOS_SET_EDGE_POLARITY( pemsci_cfg->u32Tx_EmiosChannel, 0U );
		}														/*	  shift the register down, LSB is not needed  */
																/*		  anymore								  */
		pemsci->tx_shift = pemsci->tx_shift>>1U;
		/* there a at least one bit to be transmitted, so clear the interrupt flag */
		eMIOS_STATUS_FLAG( pemsci_cfg->u32Tx_EmiosChannel ) = 0x01U;
	}															/*	SECTION 5: this section tries to reload the   */
																/*		shift register							  */
	if( pemsci->tx_shift == 0U )
	{
		/* the character has completely transmitted, try reload it from the buffer */
		if( pemsci->tx_buffer != EMSCI_TX_EMPTY )
		{
			/* Tx buffer is not empty */
			/* copy the character across, create a start bit in LSB and get rid of the MSB */
			pemsci->tx_shift = pemsci->tx_buffer<<1U;

			if( ( pemsci->tx_buffer & 0x8000U ) == 0x0000U )
			{
				/* the character is was a plain char to be transmitted - need to supply the stop bit */
				pemsci->tx_shift |= ( 0x0002U<<EMSCI_BIT_COUNT );
				/* mask-out any unused bits */
				pemsci->tx_shift
				= pemsci->tx_shift & ( 0xffffU>>( 16U - EMSCI_BIT_COUNT - 2U ) );
			}													/*		invalidate the buffer contents			  */
			pemsci->tx_buffer = EMSCI_TX_EMPTY;
 /* interrupt the CPU (the Tx buffer is empty) */
#if (  EMSCI_USE_INTERRUPTS  )
			pemsci_cfg->TxEndNotification( );
#endif
		}
		else
		{
			/* the shift register is empty & there is nothing to transmit */
			/* mark the transmitter as idle */
			pemsci->tx_in_progress = 0U;
			/* check if reception is in progrees at this time */
			if( pemsci->rx_in_progress == 0U )
			{
				/* receiver is idle */
				/* lock the semaphore first */
				//while((_ssem(EMSCI_SEMAPHORE),_ssem(EMSCI_SEMAPHORE))==0);
				/* disable the output compare interrupt */
				eMIOS_DISABLE_CHANNEL_INTERRUPT( pemsci_cfg->u32Tx_EmiosChannel );
			/* release the semaphore */
			//_csem(EMSCI_SEMAPHORE);
			}
			else
			{
				/* receiver is active -> the interrupt is left enabled */
				/* clear the interrupt flag instead */
				eMIOS_STATUS_FLAG( pemsci_cfg->u32Tx_EmiosChannel ) = 0x01U;
			}
		}
	}

	u32Match_1
	= ( uint32_t )
		( ( uint32_t )EMIOS.CH[ pemsci_cfg->u32Tx_EmiosChannel ].CADR.R
		   + ( uint32_t )EMSCI_BIT_TIME );
	/* Reload Output Compare Counter */
	eMIOS_SET_COUNT_A( pemsci_cfg->u32Tx_EmiosChannel, u32Match_1 );
	/* Clear interrupt flag */
	eMIOS_MATCH_EVENT_FLAG( pemsci_cfg->u32Tx_EmiosChannel ) = 1U;
	/* Clear counter overflow flag */
	eMIOS_CLEAR_OVERFLOW_FLAG( pemsci_cfg->u32Tx_EmiosChannel );
}

#endif



