/** ########################################################################################################################
**     Filename  : bootloader.C
**     Project   : bootloader
**     Processor : MC56F8006_48_LQFP
**     Version   : Driver 01.10
**     Compiler  : Metrowerks DSP C Compiler
**     Date/Time : 4/13/2006, 2:06 PM
**     Abstract  : Static Serial bootloader for 56F800x/56F802x/3x. 
**     Author:	   William Jiang,Petr Stancik
**     Description:
				This bootloader configures SCI rx interrupt as fast interrupt, so the last vector in the vector table 
				should not contain any JSR/BSR to the fast interrupt service routine, instead, use JMP to
				the fast interrupt service routine as below:
				  JMP  fisr_sci_rx_full                // Interrupt no. 50 (Unused) - ivINT_USER6  
				
				The CPU interrupt must be disabled shadow registers initialization.
				
 	   		    It also requires modification of user's application code; 
 				on _EntryPoint function (after vector table) the folowing sequence must be
 				placed:
				extern unsigned int _vba;
				INTC_VBA = (int)&_vba >>7;  // restore VBA register to point to user defined vector table
 				vba is defined in the linker command file as below:
			        .interrupt_vectors :
			        {
			          # interrupt vectors
			          F_vba = .;
			          * (interrupt_vectors.text)
			        } > .p_Interrupts
			    where interrupt_vectors.text is defined as below
				#pragma define_section interrupt_vectors "interrupt_vectors.text"  RX
				#pragma section interrupt_vectors begin
				static asm void _vect(void) {
				  JMP  0x1B00          # Hardware reset vector must jump to 0x1B00          
				  JMP  0x1B00          # COP reset vector must jump to 0x1B00           
				  JSR  Cpu_Interrupt                   
				  JSR  Cpu_Interrupt                   
				  JSR  Cpu_Interrupt                  
				  JSR  Cpu_Interrupt  
				  ...
					}
				 #pragma section interrupt_vectors end
				                  
			    NOTE: 
			    The above first two reset vectors must replaces those defined by ProcessorExpert or CodeWarrior.   	
    			For 56F802x, the program flash starts from 0x4000, so the .p_Interrupts segment is defined as below:
    		    .p_Interrupts  (RWX) : ORIGIN = 0x00004000, LENGTH = 0x00000080

    			For 56F803x, the program flash starts from 0, so the .p_Interrupts segment is defined as below:
    		    .p_Interrupts  (RWX) : ORIGIN = 0x0000000, LENGTH = 0x00000080
				Users can change the starting address of the vector table to fit their applications.
  				
 				Because the bootloader reads boot delay value
 				from address 0x1BFF,users must define the boot delay value in 0x1BFF by adding a memory segment .xBootCfg 
 				and a memory section .ApplicationConfiguration  in
 				the linker command file  
 				as below:
     			#Serial bootloader configuration section
    			.xBootCfg (RWX) : ORIGIN = 0x00001AFE, LENGTH = 0x2
 				
 
 
 				#Serial bootloader configuration section
		        .ApplicationConfiguration :
		        {
		        	  # Application Entry Point must be written here
		              WRITEH(F_EntryPoint);
		              
		              # Bootloader start delay in seconds
		              WRITEH(10);      # 10 is the boot delay value in seconds before bootloader
		              				   # starts up the application code
		              
		        } > .xBootCfg
 		
**		Change History:	
**			
**			Jan. 24,2008	removed some characters and outputs to reduce the code size in flash
**							so as to fix the issue of overwriting flash configuration area.
**			Nov. 28,2008	Added support for 56F800x
**
** #########################################################################################################################
*/

/* MODULE bootloader */

/* Including used modules for compiling procedure */
#include "Cpu.h"
#include "Events.h"
#include "SCI1.h"
#include "PE_Types.h"
#include "PE_Error.h"
#include "PE_Const.h"
/* Include shared modules, which are used for whole project */
#include "PE_Types.h"
#include "PE_Error.h"
#include "PE_Const.h"
#include "IO_Map.h"
/* Include custom modules */
#include "bootloader_lib.h"


UWord32  StartAddress;
UWord16  StartDelay;
char address[4];

char StrCopyright[]     = "\r\nMC56F800x/1x/2x/3x Serial Bootloader v2.0\r\n";
char StrWaitingSrec[]   = "Waiting for application S-Record.\r\n";
char StrLoaded[]        = "Download Complete\r\n";
char StrStarted[]       = "Application started!\r\n\0"; // NOTE: the last ending byte 0 is added 
														// to workaround the compiler issue with
														// even # of chars in the string that 
														// will not be padded with 0 by the compiler
char StrLock[]          = "Lock Flash?";
char StrNewLine[]       = "\r\n\0";
char StrErr[]           = "\r\nError: ";

void bootTimerCheck(void);
void bootExit(void);
void bootTimerInit(void);

// Define page buffer to store the last page containing flash configuration words
#pragma define_section flash_page_section "flash_page.page"  RW
#pragma section flash_page_section begin
word page_buf[FLASH_PAGE_SIZE];
#pragma section flash_page_section end

/*****************************************************************************
* void main(void)
*
* Main application.
*
* In:  n/a
*
* Out: n/a
*****************************************************************************/
void main(void)
{
  UInt16 state;
  char *temp_ptr;
  char ch;
  Int16 i; 
  extern word _vba;
  
  
  INTC_VBA = ((word)&_vba) >> (21-INTC_VBA_BITS);			   /* Set Vector Base Address */

  
  /*** Processor Expert internal initialization. DON'T REMOVE THIS CODE!!! ***/
  PE_low_level_init();
  /*** End of Processor Expert internal initialization.                    ***/
    
  /****************************************************************************
  / read start delay if no delay jump to user application
  /****************************************************************************/
    StartDelay = pmem_read((unsigned int *)BOOT_START_DELAY_PLACE) & 0x00FF;

    if(!StartDelay)  bootExit();
    else bootTimerInit(); 
   
  /* Write your code here */

   // Initialize R0,R1,M01, and N for circular buffer (including shadows)
   asm {
        swap shadows                    ;// Switch to shadow registers
        moveu.w #(RX_DATA_SIZE-1),M01   ;// Set buffer size
        moveu.w #rx_data,R0             ;// Set starting address
        moveu.w #QSCI_STAT,R1           ;// Use R1 for SCI status register
        moveu.w #0,N                    ;// N is unused
        swap shadows                    ;// Switch back to normal registers
   }

   // Initialize globals
   mem_init(rx_data, NULL, RX_DATA_SIZE);
   data_out = data_in = rx_data; 
   
   state = INITIAL_STATE;
   temp_ptr = rx_data;

   // Disable protection
   setReg(FM_PROT,BOOT_PROT_VALUE);
   
   // Output banner
   sci_tx(&StrCopyright[0]);
   
   // Now it is safe to enable interrupts
   Cpu_EnableInt();

   do  {
      // State: Initial State
      if (state == INITIAL_STATE) {
         status = 0;
         sci_tx(&StrWaitingSrec[0]);
         sci_tx_char(XON);
         state = WAIT_FOR_S;
      }

      if (data_in != temp_ptr) {
         Timer_Disable();

         // State: Wait for Header "S"
         if (state == WAIT_FOR_S) {
            temp_ptr = data_out;
            if (get_char(&temp_ptr) == 'S') { state = WAIT_FOR_0; }
            else { get_char(&data_out); }
         }
         // State: Wait for Header "0"
         else if (state == WAIT_FOR_0) {
            if (get_char(&temp_ptr) == '0') { state = WAIT_FOR_EOL; }
            else {
               get_char(&data_out);
               state = WAIT_FOR_S;
            }
         }
         // State: Wait for EOL
         else if (state == WAIT_FOR_EOL) {
            if (get_char(&temp_ptr) == '\r') {
               if (!(status & TX_XOFF)) {
                  sci_tx_char(XOFF);
                  status |= TX_XOFF;
               }
               srec_decode();
               temp_ptr = data_out;
            }
         }
      }
      else {
         if (status & TX_XOFF) {
            sci_tx_char(XON);
            status &= ~TX_XOFF;
         }
      }
      // State: Error
      if (status & DOWNLOAD_ERROR) {
         sci_tx(StrErr);
         sci_tx(int_to_string(status));
         state = INITIAL_STATE;
         sci_tx(StrNewLine);
      }
      
      bootTimerCheck();
      
   } while (status != DOWNLOAD_OK);
   
   sci_tx(StrNewLine); 
   sci_tx(&StrLoaded[0]);
  
   // Prompt user to lock flash
   sci_tx(&StrLock[0]);
   ch = 0;
   do
   { 
   	  if (data_in != temp_ptr)
   	  {
   	  	ch = get_char(&temp_ptr);      	   	  	
   	  }
   	  
   }while ((ch != 'y') && (ch != 'n'));
   //
   sci_tx(StrNewLine); 

   if(ch == 'y')
   {
   		// Save last page
   		for(i = 0; i < FLASH_PAGE_SIZE;i++)
   		{
   			page_buf[i] = pmem_read((word*)FLASH_LAST_PAGE+i);   			
   		}
   		// Erase last page if necessary
   		if(page_buf[FLASH_PAGE_SIZE-9] != 0xFFFFu)
   		{
   			fperase(FLASH_LAST_PAGE);   			
   		}
   		// Enable security
   	#if 0
   		page_buf[FLASH_PAGE_SIZE-9]= FLASH_SECURITY_L;
   		page_buf[FLASH_PAGE_SIZE-8]= FLASH_SECURITY_H; 
   	#else
   		page_buf[i-9] = FLASH_SECURITY_L;
   		page_buf[i-8] = FLASH_SECURITY_H;
   	#endif 
   		
   		// Now write back to the flash
   		fprogram(FLASH_LAST_PAGE,page_buf,FLASH_PAGE_SIZE); 		
   }
 
   bootExit();
}

/***************************************************************************
* Functions:   bootTimerCheck()
*
* Description: Check TIMER specified in Timer bean
*
* Returns:     1 - timer is reached defined timevalue
*              0 - count in progress
*
* Notes:
*
**************************************************************************/
void bootTimerCheck(void)
{
#ifdef	MC56F800x
    if(TMR1_SCTRL & TMR1_SCTRL_TCF_MASK)
#else
    if(TMRA1_SCTRL & TMRA1_SCTRL_TCF_MASK)
#endif    
    {
#ifdef	MC56F800x
        TMR1_SCTRL&=~TMR1_SCTRL_TCF_MASK;
#else	
        TMRA1_SCTRL&=~TMRA1_SCTRL_TCF_MASK;
#endif        
        if (--StartDelay == 0) 
        {
        	Timer_Disable();
            bootExit();	
        }
    }
}

/***************************************************************************
* Functions:   bootExit()
*
* Description: read the new appliaction start address
*              and start user's application

* Returns:     None
*
* Notes:
*
**************************************************************************/

void bootExit(void)
{
    int i=0;
    
   sci_tx(&StrStarted[0]);

#ifdef	MC56F800x
//   while(!getRegBit(SCI_STAT,TIDLE)){};
#else    
   while(!getRegBit(QSCI0_STAT,TIDLE)){};
#endif    

	// Disable all interrupts
	Cpu_DisableInt();
    
    // Disable timer
    Timer_Disable();
    
    // Restore Vector Base Address (VBA)
    INTC_VBA = 0;
    
    
  /****************************************************************************
  / read user start address
  /****************************************************************************/
   StartAddress = pmem_read((unsigned int *)BOOT_START_ADDRESS_PLACE);

    asm(move.l StartAddress, N);
    asm(jmp (N));
}


void bootTimerInit(void)
{   
#ifdef	MC56F800x
	/* Enable timer clock */
	SIM_PCE |= (SIM_PCE_TA1_MASK | SIM_PCE_TA0_MASK) ;
	
	/* Use 32-bit cascaded timer counter */
	
	TMR0_LOAD = 0;
	TMR0_CNTR = 0;
	TMR0_COMP1 = 31250;		// generate 125ms
	                        
	
	TMR1_LOAD = 0;   
	TMR1_CNTR = 0;                          
	TMR1_COMP1 = 8;
	
	/* Set up the cascaded timer 1 & 0 */
		
	TMR1_CTRL = 0xE820; 	// enabled  timer 1 as cascaded counter    
	
	
	TMR0_CTRL = 0x3E20;     // enabled  timer 0      
                            // [12:09]  = 1111 Count Source IPbus clock divide by 128    
							// [8:7]    = 00                                             
							// [6]      = 0 ONCE Count repeatedly                       
							// [5]      = 1 LENGTH Roll over                            
							// [4]      = 0 DIR Count up                                 
							// [3]      = 0                                              
							// [2:0]    = 000 OFLAG asserted                             
#else
	/* Enable timer clock */
	SIM_PCE1 |= (SIM_PCE1_TA1_MASK | SIM_PCE1_TA0_MASK) ;
	
	/* Use 32-bit cascaded timer counter */
	
	TMRA0_LOAD = 0;
	TMRA0_CNTR = 0;
	TMRA0_COMP1 = 31250;		// generate 125ms
	                        
	
	TMRA1_LOAD = 0;   
	TMRA1_CNTR = 0;                          
	TMRA1_COMP1 = 8;
	
	/* Set up the cascaded timer 1 & 0 */
		
	TMRA1_CTRL = 0xE820; 	// enabled  timer 1 as cascaded counter    
	
	
	TMRA0_CTRL = 0x3E20;     // enabled  timer 0      
                            // [12:09]  = 1111 Count Source IPbus clock divide by 128    
							// [8:7]    = 00                                             
							// [6]      = 0 ONCE Count repeatedly                       
							// [5]      = 1 LENGTH Roll over                            
							// [4]      = 0 DIR Count up                                 
							// [3]      = 0                                              
							// [2:0]    = 000 OFLAG asserted                             
#endif
}


/* END bootloader */
/*
** ###################################################################
**
**     This file was created by UNIS Processor Expert 2.98 [03.78]
**     for the Freescale 56800 series of microcontrollers.
**
** ###################################################################
*/
