/************************************************************************************
*
* (c) Copyright 2009, Freescale, Inc.  All rights reserved.
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale.
*
************************************************************************************/
#include "stdafx.h"
#include <stdio.h>
#include "Engine.h"
#include "UART.h"

/************************************************************************************
*************************************************************************************
* Private prototypes
*************************************************************************************
************************************************************************************/
static ENGCmdStatus_t WaitForConfirm(HANDLE hComm, uint32_t timeout);

/************************************************************************************
*************************************************************************************
* Private memory declarations
*************************************************************************************
************************************************************************************/
ENGCommand_t command; //One command, used for both TX and RX

/************************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
************************************************************************************/

/***********************************************************************************/
ENGCmdStatus_t ENG_Erase(HANDLE hComm, uint32_t address, uint32_t confirmTimeout)
{
  //Erase the FLASH
  command.commandId = engEraseReq_c;
  command.fields.eraseReq.address = address;
  
  //Try to send the command
  if(UART_SendCmd(hComm, (char*)&command, sizeof(ENGEraseReq_t) + 1) != gUARTSuccess_c)
  {
    return gEngCommError_c;
  }
  
  //Wait for confirm. Return to the caller the respective status.
  return WaitForConfirm(hComm, confirmTimeout);
}

/***********************************************************************************/
ENGCmdStatus_t ENG_Write(HANDLE hComm, uint32_t address, uint8_t* pBuffer, uint16_t length, uint32_t confirmTimeout)
{
  //Ensure we have a valid length and source data
  if((length > ENG_BUFFER_SIZE)||(pBuffer == NULL))
  {
    return gEngInvalidReq_c;
  }

  //Format the commad
  command.commandId = engWriteReq_c;
  command.fields.writeReq.startAddress = address;
  command.fields.writeReq.length = length;
  //Copy the data to be written into the command buffer
  memcpy(command.fields.writeReq.buffer, pBuffer, length);

  //Try to send the command
  if(UART_SendCmd(hComm, (char*)&command, sizeof(ENGWriteReq_t) + 1 - ENG_BUFFER_SIZE + length) != gUARTSuccess_c)
  {
    return gEngCommError_c;
  }
  
  //Wait for confirm. Return to the caller the respective status.
  return WaitForConfirm(hComm, confirmTimeout);
}

/***********************************************************************************/
ENGCmdStatus_t ENG_Read(HANDLE hComm, uint32_t address, uint8_t* pBuffer, uint16_t length, uint32_t readTimeout)
{
  ENGCmdStatus_t confirmStatus;

  //Ensure we have a valid length and receive pointer
  if((length > ENG_BUFFER_SIZE)||(pBuffer == NULL))
  {
    return gEngInvalidReq_c;
  }
  
  //Format the command
  command.commandId = engReadReq_c;
  command.fields.readReq.startAddress = address;
  command.fields.readReq.length = length;

  //Try to send the command
  if(UART_SendCmd(hComm, (char*)&command, sizeof(ENGReadReq_t) + 1) != gUARTSuccess_c)
  {
    return gEngCommError_c;
  }

  //Wait for the confirm - it should actually be a readResp
  confirmStatus = WaitForConfirm(hComm, readTimeout);

  //We should have receive gEngNoConfirm_c a confirm with an error code
  if(confirmStatus != gEngNoConfirm_c)
  {
    if(confirmStatus != gEngSuccessOp_c)
    {
      return confirmStatus;
    }
    else
    {
      //If we ended here, we received a confirm with status success instead of a 
      //  readResp. This is an error, so return an error code.
      return gEngExecError_c;
    }
  }
  else
  {
    //We have not receive a confirm - verify it is a read response. Even if no message
    //  was received, it should still not be a read response and thus we should know
    //  that the operations has failed.
    if(command.commandId == engReadResp_c)
    {
      //Verify the status of the read response
      if(command.fields.readResp.cmdStatus != gEngSuccessOp_c)
      {
        return (ENGCmdStatus_t)command.fields.readResp.cmdStatus;
      }
      else
      {
        //We received a read response with status sucess. Copy the data to the caller and return.
        memcpy(pBuffer, command.fields.readResp.buffer, length);
        return gEngSuccessOp_c;
      }
    }
    else //We have not received the expected read response, so return an error code.
    {
      return gEngExecError_c;
    }
  }
}

/***********************************************************************************/
ENGCmdStatus_t ENG_Commit(HANDLE hComm, uint32_t length, ENGSecureOption_t secureOption, uint32_t commitTimeout)
{
  //Format the commad
  command.commandId = engCommitReq_c;
  command.fields.commitReq.binLength = length;
  command.fields.commitReq.secured = secureOption;

  //Try to send the command
  if(UART_SendCmd(hComm, (char*)&command, sizeof(ENGCommitReq_t) + 1) != gUARTSuccess_c)
  {
    return gEngCommError_c;
  }
  
  //Wait for confirm. Return to the caller the respective status.
  return WaitForConfirm(hComm, commitTimeout);
}

/************************************************************************************
*************************************************************************************
* Private functions
*************************************************************************************
************************************************************************************/

/************************************************************************************
*  Wait for a message from the SSL running on the MC1322x, and verify if the message
* is a confirm.
*
*  Input parameters:
*  - hComm: comm port opened for the communication with the MC1322x.
*  - timeout: time, in miliseconds, to wait for a message from the SSL running 
*        on the MC1322x.
*  Return:
*  - ENGCmdStatus_t: status of the commit operation.
************************************************************************************/
static ENGCmdStatus_t WaitForConfirm(HANDLE hComm, uint32_t timeout)
{
  UARTStatus_t uartResult;
  unsigned short bytesReceived;

  //Wait for the confirm
  uartResult = UART_ReceiveCmd(hComm, timeout, (char*)&command, &bytesReceived);
  if( uartResult != gUARTSuccess_c)
  {
    //Convert the UART error into a engine error
    if(uartResult == gUARTTimeOut_c) return gEngNoConfirm_c;
    if(uartResult == gUARTCrcError_c) return gEngCRCError_c;
    return gEngCommError_c;
  }
  else
  {
    //Verify if we received a confirm
    if(command.commandId != engCmdCnf_c)
    {
      return gEngNoConfirm_c;
    }
    else
    {
      return (ENGCmdStatus_t)command.fields.cmdCnf.cmdStatus;
    }
  }
}