/************************************************************************************
*
* (c) Copyright 2009, Freescale, Inc.  All rights reserved.
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale.
*
************************************************************************************/
#include "EmbeddedTypes.h"
#include "Platform.h"
#include "Engine.h"
#include "UART.h"
#include "Main.h"

/***********************************************************************************/

/************************************************************************************
*************************************************************************************
* Private macros
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Private prototypes
*************************************************************************************
************************************************************************************/
static void GpioUart1Init(void);
static void putch(uint8_t data);
static uint8_t getch(void);

/************************************************************************************
*************************************************************************************
* Private memory declarations
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Public memory declarations
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
************************************************************************************/

/***********************************************************************************/
void UART_Init(void)
{
  GpioUart1Init();

#if (SYSTEM_CLOCK==24000)
  // fclk 24MHz
  #define MOD 10000
  #define INC 767 //for 115200 baudrate
#else
  #error SYSTEM_CLOCK not defined
#endif /** SYSTEM_CLOCK==24 */

  UART1.Ucon = 0;
  UART1.Ubr = (INC << 16) | MOD;
  //              mRx        mTx          FCE        FCpol       RxEna      TxEna
  UART1.Ucon = (1 << 14) | (1 << 13) | (0 << 12) | (0 << 11)  | (1 << 1) | (1 << 0);
}

/***********************************************************************************/
bool_t UART_ReceiveCmd(uint8_t* pBuffer, uint16_t* pBytesReceived)
{
  bool_t crcOK = TRUE;
  uint8_t computedCRC = 0;
  uint8_t receivedCRC = 0;
  uint16_t length = 0;
  uint8_t byte;
  //Look for the start of the packet
  do
  {
    byte = getch();
  }while(byte != UART_SOF);
  
  //Receive the length of the packet
  *((uint8_t*)&length) = getch(); 
  *((uint8_t*)&length + 1) = getch();

  //Copy the length to the caller
  *pBytesReceived = length;

  //Start receiving the length and computing the CRC
  while(length--)
  {
    byte = getch();
    computedCRC += byte;
    *pBuffer++ = byte;
  }
  //Receive the CRC for the packet
  receivedCRC = getch();

  //Test the computed CRC against the received CRC
  crcOK = (computedCRC == receivedCRC);

  //Return the status to the caller
  return crcOK;
}

/***********************************************************************************/
void UART_SendCmd(uint8_t* pBuffer, uint16_t bytesToSend)
{
  uint8_t computedCRC = 0;
  uint8_t data;
  
  //Send start of frame
  putch(UART_SOF);
  
  //Send the length of the packet
  putch(*((uint8_t*)&bytesToSend));
  putch(*((uint8_t*)&bytesToSend + 1));
  
  //Send the buffer and compute the CRC
  while(bytesToSend--)
  {
    data = *(pBuffer++);
    putch(data);
    computedCRC+= data;
  }
  
  //Send the CRC
  putch(computedCRC);
}

/***********************************************************************************/
void UART_SendRawData(uint8_t* pBuffer, uint16_t bytesToSend)
{
  while(bytesToSend--)
  {
    putch(*(pBuffer++));
  }
}

/************************************************************************************
*************************************************************************************
* Private functions
*************************************************************************************
************************************************************************************/

/************************************************************************************
*  Configures the GPIO for UART1 usage.
*
*  Input parameters:
*  - None.
*  Return:
*  - Nothing.
************************************************************************************/
static void GpioUart1Init(void)
{
  register unsigned long tmpReg;

  tmpReg = GPIO.FuncSel0 & ~((FN_MASK << GPIO_UART1_RX_fnpos) | (FN_MASK << GPIO_UART1_TX_fnpos));
  GPIO.FuncSel0 = tmpReg | ((FN_ALT << GPIO_UART1_RX_fnpos) | (FN_ALT << GPIO_UART1_TX_fnpos));
  tmpReg = GPIO.FuncSel1 & ~((FN_MASK << GPIO_UART1_CTS_fnpos) | (FN_MASK << GPIO_UART1_RTS_fnpos));
  GPIO.FuncSel1 = tmpReg | ((FN_ALT << GPIO_UART1_CTS_fnpos) | (FN_ALT << GPIO_UART1_RTS_fnpos));
}

/************************************************************************************
*  Send one byte over the UART.
*
*  Input parameters:
*  - data: the byte to send.
*  Return:
*  - Nothing.
************************************************************************************/
static void putch(uint8_t data)
{
  while(UART1.Utxcon == 0);
  UART1.Udata = data;
}

/************************************************************************************
*  Receive one byte from the UART.
*
*  Input parameters:
*  - None.
*  Return:
*  - uint8_t: received byte.
************************************************************************************/
static uint8_t getch(void)
{
  while(UART1.Urxcon == 0);
  return UART1.Udata;
}
