/*
 * File:		ezport.c
 * Purpose:		EzPORT transfer functions and interrupt handler
 *
 * Notes:		
 *  
 */

#include "common.h"
#include "ezport.h"
#include "qspi.h"

/*
 * EzPORTinit: starts EzPORT interface
 *
 * Parameters: none
 *
 * Return : error flag.
 */
uint8 EzPORTinit()
{
	uint8 temp;
	vuint32 time_out = TIME_OUT_VALUE;

	/* force EzPort mode on the other board */
	if(EzPORTmode())
	{
		return 1;		/* error */
	}
	
	/* start QSPI module */
	QSPIinit();
	
	if(EzPORT_RDSR(&temp) || temp != EP_SR_NFS)
	{
		return 1;		/* error */
	}
	
	/* send write enable command */
	EzPORT_WREN();
	
	if(EzPORT_RDSR(&temp) || temp != EP_SR_WEN)
	{
		return 1;		/* error */
	}
	
	/* set flash clock frequency in the other 5213 */
	EzPORT_WRCR(CLOCK_5213);
	
	temp = EP_SR_WIP;
	
	while(temp & EP_SR_WIP)
	{
		EzPORT_RDSR(&temp);
		
		if(!(time_out--))
			break;		
	}
	
	EzPORT_RDSR(&temp);
	
	if(temp != EP_SR_CRL)
	{
		return 1;
	}
	
	/* no error */
	return 0;
}

/*
 * EzPORTerase: send erase command
 *
 * Parameters: none
 *
 * Return : error flag.
 */
uint8 EzPORTerase()
{
	uint8 temp;
	vuint32 time_out = TIME_OUT_VALUE;

	/* one write start */
	if(EzPORT_WREN())
	{
		return 1;
	}
	
	/* erase entire flash */
	EzPORT_BE();
	
	temp = EP_SR_WIP;
	
	while(temp & EP_SR_WIP)
	{
		EzPORT_RDSR(&temp);
		
		if(!(time_out--))
			break;
	}	
	
	return 0;
}

/*
 * EzPORTerase: send erase command
 *
 * Parameters: address:address where it starts
 *             buffer: array containing bytes to write
 *             length: number of bytes to write
 *             verify: if a verify is needed
 *
 * Return : error flag.
 */
uint8 EzPORTs19(uint32 address, uint8 *buffer, uint32 length, uint8 verify)
{
	uint8 temp,error_flag;
	uint8 remainders, within_a_page;
	uint32 i;
	uint8 read_buffer[256];
	uint32 buffer_end;
	
	vuint32 time_out = TIME_OUT_VALUE;
	
	/* if length % 4 != 0 */
	/* this happens if last opcode is 16-bit length instead of 32-bit */
	if(length%4)
	{
		/* correct the length */
		buffer[length]   = buffer[length-2];
		buffer[length+1] = buffer[length-1];
		length += 2;
	}
	
	error_flag = 0;
	buffer_end = address+length-1;

	/* it fits in a single flash page */
	if((address>>8) == (buffer_end>>8))
	{
		/* one write start */
		if(EzPORT_WREN())
		{
			return 1;
		}
		
		if(EzPORT_PP(address,length,(void *)buffer))
		{
			return 1;
		}
		
		temp = EP_SR_WIP;
	
		while(temp & EP_SR_WIP)
		{
			EzPORT_RDSR(&temp);
			
			if(!(time_out--))
				break;
		}
		
	}
	/* it doesn't fit in a single flash page */
	else
	{
		remainders = (address + length)%256;
		within_a_page = length - remainders;
		
		/* one write starts */
		if(EzPORT_WREN())
		{
			return 1;
		}
		
		if(EzPORT_PP(address,within_a_page,&buffer[0]))
		{
			return 1;
		}
		
		temp = EP_SR_WIP;
	
		while(temp & EP_SR_WIP)
		{
			EzPORT_RDSR(&temp);
			
			if(!(time_out--))
				break;
		}
		
		/* one write ends */
		
		/* second write starts */
		if(EzPORT_WREN())
		{
			return 1;
		}
		
		if(EzPORT_PP(address+within_a_page,remainders,&buffer[within_a_page]))
		{
			return 1;
		}
		
		temp = EP_SR_WIP;
	
		while(temp & EP_SR_WIP)
		{
			EzPORT_RDSR(&temp);
			
			if(!(time_out--))
				break;
		}
		/* second write ends */
		
	}
	
	/* verify arrays */
	if(verify)
	{
		if(EzPORT_FAST_READ(address, length, &read_buffer[0]))
		{
			return 1;
		}
		
		for(i=0;i<length;i++)
		{
			if(read_buffer[i] != buffer[i])
			{
				printf("Programming error");
				printf("Address = %x Read data = %x, Expected = %x\n",
				       address+i,
				       read_buffer[i],
				       buffer[i]);
			}
		}
		
	}
	
	return 0;
}

/*
 * EzPORTmode: set the other board into EzPORT mode
 *
 * Parameters: none.
 *
 * Return : error flag.
 */
uint8 EzPORTmode()
{
	vuint16 timeout = TIME_OUT_VALUE;

	/* setting PORT_TC[0] and PORT_TC[1] as GPIO */
	EZPORT_GPIO_CONFIG;

	/* Configure as GPIO pins to assert RCON*/
	MCF_GPIO_PQSPAR = 0x0; 						/* GPIO function*/
	
	/* CS0 pin as output */
	MCF_GPIO_DDRQS = MCF_GPIO_DDRQS_DDRQS3;
	
	/* asserting CS0 pin */
	MCF_GPIO_PORTQS = MCF_GPIO_PORTQS_PORTQS3;
	
	/* set CS0 for inactive high-level */
	MCF_QSPI_QWR = MCF_QSPI_QWR_CSIV;
	
	/* select OUT values */
	EZPORT_RSTIN_OUT  = 1;
	
	/* setting pins direction */
	EZPORT_RSTOUT_DDR = 0;
	EZPORT_RSTIN_DDR  = 1;
		
	/* resetting other board */
	EZPORT_RSTIN_OUT = 0;
	
	while(EZPORT_RSTOUT_IN)
	{
		if(!(timeout--))
			return 1;
	}
	/* waiting for /RSTO assertion */
	
	/* CS0/RCON pin = 0 */
	MCF_GPIO_PORTQS &= ~MCF_GPIO_PORTQS_PORTQS3;
	
	/* other board starts leaving reset phase */
	EZPORT_RSTIN_OUT = 1;	
	
	while(!EZPORT_RSTOUT_IN)
	{
		if(!(timeout--))
			return 1;
	}
	/* waiting for /RSTO des-assertion */
	
	/* CS0/RCON pin = 0 */
	MCF_GPIO_PORTQS |= MCF_GPIO_PORTQS_PORTQS3;
	
	/* leaving RSTO and RSTI as input */
	EZPORT_RSTIN_DDR  = 0;
	EZPORT_RSTOUT_DDR = 0;
	
	return 0;
}

/******************************INTERNAL FUNCTIONS******************************/


uint8 EzPORT_WREN()
{
	qspi_state_t temp;
	
	/* commands to send */
	uint16 array = EZPORT_WREN;
	
	/* filling structure to send to qspi driver */
	temp.bits_per_data = 8;		/* byte */
	temp.receive_flag = 0;		/* tx */
	temp.size = 1;				/* words size */
	temp.data = &array;			/* tx buffer */

	/* sending and receiving through qspi */
	if(QSPIsendReceive(&temp))
	{
		return 1;		/* error */
	}
	
	/* everything's fine */
	return 0;
}

uint8 EzPORT_WRDI()
{
	qspi_state_t temp;
	
	/* commands to send */
	uint16 array = EZPORT_WRDI;

	/* filling structure to send to qspi driver */
	temp.bits_per_data = 8;		/* byte */
	temp.receive_flag = 0;		/* tx */
	temp.size = 1;				/* words size */
	temp.data = &array;			/* tx buffer */

	/* sending and receiving through qspi */
	if(QSPIsendReceive(&temp))
	{
		return 1;		/* error */
	}
	
	/* everything's fine */
	return 0;
}

uint8 EzPORT_BE()
{
	qspi_state_t temp;
	
	/* commands to send */
	uint16 array = EZPORT_BE;

	/* filling structure to send to qspi driver */
	temp.bits_per_data = 8;		/* byte */
	temp.receive_flag = 0;		/* tx */
	temp.size = 1;				/* words size */
	temp.data = &array;			/* tx buffer */

	/* sending and receiving through qspi */
	if(QSPIsendReceive(&temp))
	{
		return 1;		/* error */
	}
	
	/* everything's fine */
	return 0;
}

uint8 EzPORT_RESET()
{
	qspi_state_t temp;
	
	/* commands to send */
	uint16 array = EZPORT_RESET;

	/* filling structure to send to qspi driver */
	temp.bits_per_data = 8;		/* byte */
	temp.receive_flag = 0;		/* tx */
	temp.size = 1;				/* words size */
	temp.data = &array;			/* tx buffer */

	/* sending and receiving through qspi */
	if(QSPIsendReceive(&temp))
	{
		return 1;		/* error */
	}
	
	/* everything's fine */
	return 0;	
}

uint8 EzPORT_WRCR(uint8 data)
{
	qspi_state_t temp;
	
	/* commands to send */
	uint16 array = ((EZPORT_WRCR<<8) | data);

	/* filling structure to send to qspi driver */
	temp.bits_per_data = 16;	/* word */
	temp.receive_flag = 0;		/* tx */
	temp.size = 1;				/* words size */
	temp.data = &array;			/* tx buffer */

	/* sending and receiving through qspi */
	if(QSPIsendReceive(&temp))
	{
		return 1;		/* error */
	}
	
	/* everything's fine */
	return 0;	
}

uint8 EzPORT_RDSR(uint8 *data)
{
	qspi_state_t temp;
	
	/* commands to send */
	uint16 array = ((EZPORT_RDSR<<8) | DONT_CARE);

	/* filling structure to send to qspi driver */
	temp.bits_per_data = 16;	/* word */
	temp.receive_flag = 1;		/* rx */
	temp.size = 1;				/* words size */
	temp.data = &array;			/* rx buffer */

	/* sending and receiving through qspi */
	if(QSPIsendReceive(&temp))
	{
		return 1;		/* error */
	}	
	
	/* getting status register */
	*data = (temp.data[0])&0x00FF;
	
	/* everything's fine */
	return 0;
}

uint8 EzPORT_PP(uint32 address, uint32 size, void *data)
{
	qspi_state_t temp;
	uint32 i;
	uint16 *data_pointer;
	uint16 array[128/* max number */ + 2 /* command + address */];

	data_pointer = (uint16 *)data;
	
	/* filling structure to send to qspi driver */
	temp.bits_per_data = 16;	/* word */
	temp.receive_flag = 0;		/* tx */
	temp.size = size/2 + 2;		/* words size */
	temp.data = &array[0];		/* tx buffer */
	
	/* commands + address to send */
	array[0] = (EZPORT_PP<<8) | (address>>16);
	array[1] = address & 0x0000FFFF;
	
	/* setting data to be sent */
	for(i=0;i<(temp.size-2);i++)
	{
		array[i+2] = data_pointer[i];
	}

	/* sending and receiving through qspi */
	if(QSPIsendReceive(&temp))
	{
		return 1;		/* error */
	}	
	
	/* everything's fine */
	return 0;
}

uint8 EzPORT_SE(uint32 address)
{
	qspi_state_t temp;
	uint16 array[2];
	
	/* filling structure to send to qspi driver */
	temp.bits_per_data = 16;	/* word size */
	temp.receive_flag = 0;		/* tx */
	temp.size = 2;				/* words size */
	temp.data = &array[0];		/* tx buffer */
	
	/* commands + address to send */
	array[0] = (EZPORT_SE<<8) | (address>>16);
	array[1] = (address & 0xFFFF);
	
	/* sending and receiving through qspi */
	if(QSPIsendReceive(&temp))
	{
		return 1;		/* error */
	}
	
	/* everything's fine */
	return 0;
}

uint8 EzPORT_FAST_READ(uint32 address, uint32 size, void *data)
{
	uint32 i;
	qspi_state_t temp;
	uint8 *data_pointer;
	uint8 array[256 /*tentative max number*/+ 5/*cmd+add+dummy*/];
	
	data_pointer = (uint8 *)data;

	/* filling structure to send to qspi driver */
	temp.bits_per_data = 16;	/* word size */
	temp.receive_flag = 1;		/* rx */
	temp.size = (size/2) + 3;	/* words size */
	temp.data = (uint16 *)array;/* rx buffer */
	
	/* commands + address + dummy byte to send */
	array[0] = EZPORT_FAST_READ;
	array[1] = address>>16;
	array[2] = (address>>8)&0xFF;
	array[3] = (address)&0xFF;
	/* array[2] = DONT_CARE; */

	/* sending and receiving through qspi */
	if(QSPIsendReceive(&temp))
	{
		return 1;		/* error */
	}
	
	/* getting data just received */
	for(i=0;i<size;i++)
	{
		data_pointer[i] = array[i+5];
	}
	
	/* everything's fine */
	return 0;
}

uint8 EzPORT_READ(uint32 address, uint32 size, void *data)
{
	uint32 i;
	qspi_state_t temp;
	uint8 *data_pointer;
	uint8 array[256 /*tentative max number*/+ 4/*cmd+add*/];
	
	data_pointer = (uint8 *)data;

	/* filling structure to send to qspi driver */
	temp.bits_per_data = 16;	/* word size */
	temp.receive_flag = 1;		/* rx */
	temp.size = (size/2) + 3;	/* words size */
	temp.data = (uint16 *)array;/* rx buffer */
	
	/* commands + address + dummy byte to send */
	array[0] = EZPORT_READ;
	array[1] = address>>16;
	array[2] = (address>>8)&0xFF;
	array[3] = (address)&0xFF;

	/* sending and receiving through qspi */
	if(QSPIsendReceive(&temp))
	{
		return 1;		/* error */
	}
	
	/* getting data just received */
	for(i=0;i<size;i++)
	{
		data_pointer[i] = array[i+4];
	}
	
	/* everything's fine */
	return 0;
}