/*
 * File:        serial_s19.c
 * Purpose:     Routines to download Freescale S-Records via serial port.
 *
 * Notes:       Implements Xon/Xoff software handshaking
 */

#include "common.h"
#include "serial_s19.h"
#include "stdlib.h"
#include "ezport.h"

/**********************************FUNCTIONS*****************************/

uint8 overflow[60];
int ovchars;
int ovindex = 0;
uint32 timeout = 0;

/*
 * dl_getchar: returns serial character from UART module
 *
 * Parameters: none.
 *
 * Return : none.
 */
static uint8 dl_getchar(void)
{   
    uint8 rval;
	
	/* in case overflow occurs */
    if (ovchars)
    {
        rval = overflow[ovindex++];
        /* is overflow array empty? */
        if (!--ovchars)
        {
            /* Send Xon character */
            out_char (CTRL_XON);
            ovindex = 0;
        }
        return rval;
    }
    else
    {
        return in_char();
    }
}

/*
 * dl_gethexvalue: change hex number from ascii to hex value
 *
 * Parameters:
 *          character: ascii number
 *
 * Return :
 *          hex number
 */
static uint8 dl_gethexvalue(uint8 character)
{
    switch (character)
    {
        case '0':
        case '1':
        case '2':
        case '3':
        case '4':
        case '5':
        case '6':
        case '7':
        case '8':
        case '9':
            return (character - '0');
        case 'A':
        case 'a':
            return 10;
        case 'B':
        case 'b':
            return 11;
        case 'C':
        case 'c':
            return 12;
        case 'D':
        case 'd':
            return 13;
        case 'E':
        case 'e':
            return 14;
        case 'F':
        case 'f':
            return 15;
        default:
            return 0;
    }
}

/*
 * dl_getS: This routine skips all characters until an `S' is received
 *
 * Parameters: none.
 *
 * Return :
 *          type of S-record
 */
static uint8 dl_getS(void)
{
    /*
     * This routine skips all characters until an `S' is received.
     * It then returns the next character.
     * This routine is here in order to not echo characters back
     * out as they are received during a download.
     *
     * This code ignores NL,CR-LF end of line issues by waiting
     * for 'S'.
     */
    uint8 c;

    while (TRUE)
    {
    	/* read character */
        c = dl_getchar();
        if (c == 'S')
            break;
    }

    /* Get type of S-record */
    return dl_getchar();
}

/*
 * dl_getSpair: return a pair of characters received from serial terminal
 *
 * Parameters: none.
 *
 * Return :
 *          a pair of characters
 */
static uint8 dl_getSpair(void)
{
    uint8 ch;
    uint8 upper;

    ch = dl_getchar();
    upper = (uint8)(dl_gethexvalue(ch) << 4);
    ch = dl_getchar();
    	return (upper | dl_gethexvalue(ch));
}

/*
 * flash_dlio_vda: check if an address is within FLASH addressable space
 *
 * Parameters:
 *			addr: address in FLASH memory
 *
 * Return :
 *          valid address flag
 */
static uint8 flash_dlio_vda(ADDRESS addr)
{

    if ((FLASH_BASE_SIZE) && 
        (FLASH_BASE_ADDRESS <= addr) &&
        (addr <= (ADDRESS)(FLASH_BASE_ADDRESS + (FLASH_BASE_SIZE - 1))))
    {
        return 1;
    }
    /*
     * Address was not found within a registered Flash device
     */
    return 0;
}

/*
 * dl_download_srec: receive a s19 file from a serial terminal and send it
 *					 through QSPI to EzPort
 *
 * Parameters:
 *			verify: comparison flag
 *
 * Return :
 *          error flag
 */
uint8 dl_download_srec(uint8 verify)
{
    uint8 checksum, i, index, done, type, data, length;
    ADDRESS address;
    uint8 buffer[150];
    
    done = FALSE;
    ovchars = 0;

    /* Send Xon character */
    out_char(CTRL_XON);

    while (!done)
    {
        /* Get start of S-record */
        type = dl_getS();
        
        /* Get record length */
        length = dl_getSpair();
        checksum = length;

        /* Take appropriate action */
        switch (type)
        {
            case '1':
            case '2':
            case '3':
                timeout = 0;
                address = NULL;
                type -= '0';
                for (i = 0; i <= type ; i++)
                {
                    /* Formulate address */
                    data = dl_getSpair();
                    address = (address << 8) | data;
                    
                    /* Maintain 8-bit checksum */
                    checksum = (data + checksum) & 0x00FF;
                    length--;                    
                }              
                
                if ((index = flash_dlio_vda((ADDRESS)address)) != 0)
                {
                    /* Get data and put into buffer */
                    for (i = 0; i < (length - 1); i++)
                    {
                        buffer[i] = dl_getSpair(); 
                    }
                    
                    /* Get checksum byte */
                    data = dl_getSpair();
                    
                    /* Send Xoff character */
                    out_char(CTRL_XOFF);

                    /* Catch the overflow until XOFF char takes effect */
                    while (timeout++ < CHAR_WAIT_TIMEOUT)
                    {
                        if (char_present())
                        {
                            overflow[ovchars++] = in_char();
                            timeout = 0;  
                        }
                    }
                    
                    /* Write buffered data to Flash using EzPort */
                    if(EzPORTs19(address, &buffer[0], i, verify))
                    {
                    	printf("Error during spi tx\n");
                    	return FALSE;
                    }
                                
                    /* Calculate checksum */
                    for (i = 0; i < (length - 1); i++)
                    {
                        checksum = (buffer[i] + checksum) & 0x00FF;
                    }
					
					/* Compare checksum */
                    if (((data - ~checksum) & 0x00FF) != 0)
                    {
                        printf("Error:  S-record checksum error!\n");
                        printf("Expected:  %2X,  Received: %2X\n",
                            data,checksum);
                        printf("Address: %8X\n ",address);
                    }

                    /* Send Xon if no overflow occured */
                    if (!ovchars)
                    	out_char ( CTRL_XON);
                }
                else
                {
                    printf("Invalid download address: %#08X\n",address);
                    return FALSE;
                }

                break;
            case '7':
            case '8':
            case '9':
            	/* check for final S-record */
                address = NULL;
                type = type - '0';
                type = 10 - type;
                for(i = 0; i < 3; i++)
                {
                    data = dl_getSpair();
                    checksum = (data + checksum) & 0x00FF;
                    address = (address << 8) | data;
                    length--;
                }
                while (length-- > 1)
                {
                    data = dl_getSpair();
                    checksum = (data + checksum) & 0x00FF;
                }

                data = dl_getSpair();

                if (((data - ~checksum) & 0x00FF) != 0)
                {
                    printf("Error: S-record checksum error!\n");
                    return FALSE;
                }
                else
                {
                	/* reset the other board to show the code */                
                	EzPORT_RESET();
                
                    printf("\nS-record download successful!\n");
                }
                /* exit from the while */
                done = TRUE;
                
                /* Clear out the \n in the receive buffer */
                in_char();
                break;
            case '0':
            case '4':
            case '5':
            case '6':
            default:
                break;
        }
    }
    return TRUE;
}