/******************************************************************************
*
* Freescale Inc.
* (c) Copyright 2004-2009 Freescale Inc.
* ALL RIGHTS RESERVED.
*
****************************************************************************//*!
*
* @file     rf_comm.c
* @version  0.0.16.0
* @date     Oct-16-2009
* @author   b20253
* 
* @brief    Main file for RF_comm driver
*           
* 
*******************************************************************************/

#include <hidef.h> /* for EnableInterrupts macro */
#include "derivative.h"


#include "pub_def.h"
#include "simple_mac.h"
#include "mcu_hw_config.h"
#include "MC13192_hw_config.h"
#include "drivers.h"

#include "rf_comm_cfg.h"
#include "rf_comm.h"
#include "rf_comm_pr.h"


extern void RfComm_TestCB(byte lbl,word val);
extern void Delay_Xms(byte loop);


#pragma CONST_SEG  EEPROM
extern volatile const word nvnetnum;

void ConnDataCB(byte *data, byte sensor);            

#ifdef __HCS08__
 #pragma DATA_SEG SHORT _DATA_ZEROPAGE
#endif


/***********************************
*  local variables 
***********************************/
static  byte send_buffer_cnt;
static  byte rx_buff_pending;
static  byte tx_unique_number_ID = 0;
static  byte rx_unique_number_ID = 128;
static  byte alive_cnt;
static  byte data_buffer_length;
static  byte rx_pending_cnt;


byte i,j,idx, event = 0;

LINK_STATE link_state = NOT_CONNECTED;
tTxPacket tx_packet;
tRxPacket rx_packet;



/***********************************
*  MACRO SECTION
***********************************/
#define ALIVE_CNT_MAX  5
#define TIMEOUT_MSG    4
#define TIMEOUT_LONG   500    
#define TIMEOUT_SHORT  1
#define MAX_SEND_BUFFER_CNT 20
#define OPEN_RX_TIMEOUT     15000

static volatile byte timeout_msg   = TIMEOUT_MSG;
static volatile word timeout_alive = TIMEOUT_LONG;

#pragma DATA_SEG DEFAULT
 
 byte tx_data_buffer[DATA_BUFFER_LEN];
 byte rx_data_buffer[DATA_BUFFER_LEN];

#pragma CONST_SEG DEFAULT

/***********************************
* All functions for state machine 
***********************************/
const PFC action_function[EVENT_COUNT] =
{
 RF_Comm_TX_done,
 RF_Comm_Init,
 RF_Comm_SendAck,
 RF_Comm_SendBuffer,
 RF_Comm_SendNack,
 RF_Comm_Check_device
 };

/*=======================================================================*/
/* Function: RFinit                                                      */
/* Parameters: none                                                      */
/* Return: none                                                          */
/*                                                                       */
/* Function function initialize all neccessary rf variables              */
/*=======================================================================*/
void RF_Comm_Init(void)
{
    DisableInterrupts;
    MCUInit();                                   // MCU initialization
    RadioInit();                                 // RF initialization
    
    tx_packet.u8DataLength = 0;	                 // Initialize the tx_packet global
    tx_packet.pu8Data = &tx_data_buffer[0];      // Set the pointer to point to the tx_buffer

    rx_packet.u8DataLength = 0;
    rx_packet.pu8Data = &rx_data_buffer[0];
    rx_packet.u8MaxDataLength = DATA_BUFFER_LEN;
    rx_packet.u8Status = 0;				               // initialize status to 0.
	  
	  (void)MLMEMC13192PAOutputAdjust(MAX_POWER);  // Set MAX power setting
	  (void)MLMESetMC13192TmrPrescale(3);          // 250 KHz for internal ZigBee clock
     MLMESetChannelRequest(SMAC_WORK_CHANNEL);
    
    RF_Comm_ChangeLinkState(NOT_CONNECTED);
 
    data_buffer_length = 0;    
    EnableInterrupts;
    
    alive_cnt = 0;    
    event = EVENT_CHECK;                          // event search device
    Timer_init();
}

/*=======================================================================*/
/* Function: RF_Comm_Check_device                                        */
/* Parameters: none                                                      */
/* Return: none                                                          */
/*                                                                       */
/* Function Manage new pylon and checks all sensor if its still active   */
/*=======================================================================*/
void RF_Comm_Check_device(void)
{
  if(alive_cnt++ >= ALIVE_CNT_MAX)               // if alive counter is full, link_state -> NOT_CONNECTED
    {
     alive_cnt = 0;     
     timeout_alive = TIMEOUT_LONG;              // set the long time
     RF_Comm_ChangeLinkState(NOT_CONNECTED);
    }
   else
    {
      timeout_alive = TIMEOUT_MSG;               // set short time and send Handshake
      if(data_buffer_length) 
       {
         RF_Comm_TxBuffFlush();    
       }
       else  
         {
           RF_Comm_TxGenerator(RF_COMM_CMD_HANDSHAKE); 
           timeout_alive += (RFC_TIMER_CNT & 0x0003);
         }
      }
 }
 
/*=======================================================================*/
/* Function: RF_Comm_SendAck                                             */
/* Parameters: none                                                      */
/* Return: none                                                          */
/*                                                                       */
/*=======================================================================*/
void RF_Comm_SendAck(void)
{
  RF_Comm_TxGenerator(RF_COMM_CMD_ACK);
} 
/*=======================================================================*/
/* Function: RF_Comm_SendNack                                             */
/* Parameters: none                                                      */
/* Return: none                                                          */
/*                                                                       */
/*=======================================================================*/
void RF_Comm_SendNack(void)
{
  RF_Comm_TxGenerator(RF_COMM_CMD_NACK);
} 
/*=======================================================================*/
/* Function: RF_Comm_SendBuffer                                          */
/* Parameters: none                                                      */
/* Return: none                                                          */
/*                                                                       */
/*=======================================================================*/
void RF_Comm_SendBuffer(void)
{
  #if DATA_ACK_RECEIVE == SUPPORTED
   
    if(send_buffer_cnt > 0) 
     {
       send_buffer_cnt--;
       timeout_msg = TIMEOUT_MSG;
       RF_Comm_TxGenerator(RF_COMM_CMD_DATA);
    
     } else
        {
          RF_Comm_ChangeLinkState(NOT_CONNECTED);
        }
        
   #else
       RF_Comm_TxGenerator(RF_COMM_CMD_DATA);
       data_buffer_length = 0;
   #endif    
}
/*=======================================================================*/
/* Function: RF_Comm_TX_done                                             */
/* Parameters: none                                                      */
/* Return: none                                                          */
/*                                                                       */
/*=======================================================================*/
void RF_Comm_TX_done(void)
{
  if(rx_buff_pending)              // this will not occur if tx data were sent (this happens if ACK is sent & data were not handled after RX) - 2nd and more try
  {
    timeout_msg = TIMEOUT_SHORT;
  }
  else
  {
  
   MLMERXDisableRequest();
   
   if(MLMERXEnableRequest(&rx_packet,OPEN_RX_TIMEOUT) != SUCCESS)
      MLMERXEnableRequest(&rx_packet,OPEN_RX_TIMEOUT);
  }
  
  #if DATA_ACK_RECEIVE == UNSUPPORTED
    RF_Comm_ChangeLinkState(READY);
  #endif
}
/*=======================================================================*/
/* Function: Timer_init                                                  */
/* Parameters: none                                                      */
/* Return: none                                                          */
/*                                                                       */
/*=======================================================================*/
void Timer_init(void)
{
  RFC_TIMER_INIT  
}

/*=======================================================================*/
/* Function: Timer_Interrupt                                             */
/* Parameters: none                                                      */
/* Return: none                                                          */
/*                                                                       */
/*=======================================================================*/
#if defined(__HCS08__) || defined(__HC08__)
  void interrupt Timer_Interrupt(void)           
#else
  __interrupt void Timer_Interrupt(void)
#endif
{
  if(timeout_msg)
  {
   timeout_msg--;
    if(timeout_msg == 0) 
    {
      if(rx_buff_pending)
        {
        
        #ifdef DATA_RECEIVE_CB
        
         if(DATA_RECEIVE_CB(RF_C(&rx_packet)->data, rx_buff_pending))
          {
            if(((rx_pending_cnt++) % 4) == 0)
             {
              event |= EVENT_SEND_NACK;
              timeout_msg = TIMEOUT_SHORT;
             }
          }
          else
          { 
            rx_buff_pending = 0;
            event |= EVENT_SEND_ACK;
          }
          
         #endif 
          
        }
         #if DATA_ACK_RECEIVE == SUPPORTED
            else if(link_state == BUSY)                
            {
              event |= EVENT_SEND_BUFF;
            }
         #endif   
     }
    }     
  if(timeout_alive)
  {
   timeout_alive--;
    if(timeout_alive == 0) 
    {
        event |= EVENT_CHECK;
    }
  }
 RFC_CLEAR_TOF
}
/*********************************************************************/
void MLMEMC13192ResetIndication (void)
{
}

/*=======================================================================*/
/* Function: MCPSDataIndication                                          */
/* Parameters: *rx_packet - pointer on received packet structure         */
/* Return: none                                                          */
/*                                                                       */
/*=======================================================================*/
void MCPSDataIndication(tRxPacket *rx_packet)
{
 if (rx_packet->u8Status == SUCCESS)
	{
    if(RF_C(rx_packet)->netnum == RF_COMM_NETNUM) 
    {
      alive_cnt = 0;
      timeout_alive = TIMEOUT_LONG;
      
      if(link_state == NOT_CONNECTED)
        RF_Comm_ChangeLinkState(READY);    
      
      switch(RF_C(rx_packet)->command) 
      {
        case RF_COMM_CMD_DATA:              /* RECEIVED DATA PACKET */
         
         LED3 = LEDON;
         
         #if DATA_ACK_SEND == SUPPORTED
         
          if(RF_C(rx_packet)->packetId != rx_unique_number_ID) // received different data packet, than last?? 
           {
              rx_unique_number_ID = RF_C(rx_packet)->packetId; // insert the unique ID to structure of packet 
          
            #ifdef DATA_RECEIVE_CB                             // CALLBACK TO USER APPLICATION 
          
              rx_buff_pending = (byte)(rx_packet->u8DataLength - (DATA_BUFFER_LEN - RF_COMM_DATA_BUFFER_LEN));
          
              if(DATA_RECEIVE_CB(RF_C(rx_packet)->data, rx_buff_pending))
               {
                 event|= EVENT_SEND_NACK;                
               }
              else
               { 
                 rx_buff_pending = 0;
                 event|= EVENT_SEND_ACK;    
               }                                                               
            #endif   
          }
          
          else
           {
             event |= EVENT_SEND_ACK;
           }
         
         #else 
         
            LED1 = LEDON;
           
           #ifdef DATA_RECEIVE_CB 
           
            rx_buff_pending = (byte)(rx_packet->u8DataLength - (DATA_BUFFER_LEN - RF_COMM_DATA_BUFFER_LEN));
            DATA_RECEIVE_CB(RF_C(rx_packet)->data, rx_buff_pending);
            rx_buff_pending = 0;
          
           #endif
            event |= EVENT_TXDONE;
         
         #endif
         
          break;
          
         case RF_COMM_CMD_HANDSHAKE:  /* RECEIVED HANDSHAKE PACKET */
        
          event|= EVENT_SEND_ACK;
         
          break;

         case RF_COMM_CMD_ACK:
          
           if(link_state == BUSY)
            {
              if(RF_C(rx_packet)->packetId == tx_unique_number_ID)
              {
                data_buffer_length = 0;
              }
            }
                
          RF_Comm_ChangeLinkState(READY);
          event |= EVENT_TXDONE;            /* open receive window by event tx_done */
         break;

        case RF_COMM_CMD_NACK:
           event |= EVENT_TXDONE;           /* open receive window by event tx_done */
         break;
         
         default:
           event |= EVENT_TXDONE;
         break;  
       }
     }
  	}
 
  else
   {
    event |= EVENT_TXDONE;
   }
   
 LED3 = LEDOFF;
 LED1 = LEDOFF;
}
/*=======================================================================*/
/* Function: RF_Comm_ChangeLinkState                                     */
/*  - This function changes actual link state to request state           */
/* Parameters: new_state - new state of internal LINK STATE              */
/* Return: none                                                          */
/*                                                                       */
/*=======================================================================*/
#ifdef RFCOMM_CHANGE_STATE_CB
  void RFCOMM_CHANGE_STATE_CB(LINK_STATE new_state);          
#endif

static void RF_Comm_ChangeLinkState(LINK_STATE new_state) 
{
  if(link_state != new_state) 
  {
    link_state = new_state;
          
    #ifdef RFCOMM_CHANGE_STATE_CB 
       RFCOMM_CHANGE_STATE_CB(new_state);  
    #endif
  
    if(link_state == NOT_CONNECTED) 
    {
      data_buffer_length = 0;
    }
  }
}
/*=======================================================================*/
/* Function: RF_Comm_TxGenerator                                         */
/* Parameters: tx_command - possible of transmitted command (ACK, DATA,  */
/*                          NACK,HANDSHAKE)                              */
/* Return: none                                                          */
/*                                                                       */
/*=======================================================================*/
void RF_Comm_TxGenerator(byte tx_command)

 {
  RF_C(&tx_packet)->netnum   = RF_COMM_NETNUM; 
  RF_C(&tx_packet)->command  = tx_command;
  RF_C(&tx_packet)->packetId = rx_unique_number_ID;
  RF_C(&tx_packet)->deviceId = RFC_DEBUG_ID;
  
  tx_packet.u8DataLength = RF_COMM_HEADER_LEN;
  
  if(tx_command == RF_COMM_CMD_DATA)
   {
     LED1 =  LEDON;
     RF_C(&tx_packet)->packetId = tx_unique_number_ID;
     tx_packet.u8DataLength    += data_buffer_length; 
   } 

  MLMERXDisableRequest();
  DisableInterrupts;
  MLMEDelayTransceiver(MLMEGetActualTime() + 100, 0, &tx_packet,&rx_packet,TX_MODE);
  EnableInterrupts;

  LED1 = LEDOFF;
}
/*=======================================================================*/
/* Function: RF_Comm_Poll - INTERNAL STATE MACHINE - (Poll Function)     */
/* Parameters: none                                                      */
/* Return: none                                                          */
/*                                                                       */
/*=======================================================================*/
void RF_Comm_Poll(void)
{
  if(event)                         // rf event decoder
  {
    unsigned char i_mask;
    unsigned char idx;
   
    for(i_mask=1, idx=0; idx < EVENT_COUNT; i_mask<<= 1, idx++)
    {
      if(event & i_mask)            // this event occur??
      {
        action_function[idx]();     // manage all events by array of functions
        event &= ~i_mask;           // clear event flag
      }  
    }
  }
}
/*=======================================================================*/
/* Function: RF_Comm_Init1 - Init event                                  */
/* Parameters:                                                           */
/* Return: none                                                          */
/*                                                                       */
/*=======================================================================*/
void RF_Comm_Init1(void)
{
  event = EVENT_INIT;	
}
/*=======================================================================*/
/* Function: RF_Comm_TxBuff                                              */
/* Parameters: *ptr_data - pointer on data, which can be sent            */
/*             len - total length of the data                            */
/* Return: none                                                          */
/*                                                                       */
/*=======================================================================*/
byte RF_Comm_TxBuff(byte *ptr_data,byte len) 
{
  byte *ptr_data_RF;
  
  if((link_state != READY)||(RF_COMM_DATA_BUFFER_LEN - data_buffer_length) == 0)
   return 0;

  ptr_data_RF = &RF_C(&tx_packet)->data[data_buffer_length];
  
  while((len--) && (data_buffer_length++ < RF_COMM_DATA_BUFFER_LEN))
  {
    *(ptr_data_RF++) = *(ptr_data++);
  }

  if((RF_COMM_DATA_BUFFER_LEN - data_buffer_length) == 0)
    RF_Comm_TxBuffFlush(); 

  return len;
}
/*=======================================================================*/
/* Function: RF_Comm_TxBuffFlush                                         */
/* Parameters: none                                                      */
/* Return: none                                                          */
/*                                                                       */
/*=======================================================================*/
byte RF_Comm_TxBuffFlush(void)
{
  if(!link_state == READY)
    return 0;
  
   RF_Comm_ChangeLinkState(BUSY);
   tx_unique_number_ID++;             
 
   send_buffer_cnt = MAX_SEND_BUFFER_CNT;
   event |= EVENT_SEND_BUFF;
   
   return 1;
}
/*=======================================================================*/
/* Function: RF_Comm_TxBuffPending                                       */
/* Parameters: none                                                      */
/* Return: none                                                          */
/*                                                                       */
/*=======================================================================*/
byte RF_Comm_TxBuffPending(void) 
{
   // return free size of buffer
   return(byte)(RF_COMM_DATA_BUFFER_LEN - data_buffer_length);
}
/*=======================================================================*/
/* Function: RF_Comm_GetLinkState                                        */
/* Parameters: none                                                      */
/* Return: none                                                          */
/*                                                                       */
/*=======================================================================*/
LINK_STATE RF_Comm_GetLinkState(void) 
{
  return link_state;
}
