/**************************************************************************/
/* FILE NAME: main.c                       COPYRIGHT (c) Freescale 2009   */
/*                                                All Rights Reserved     */
/* DESCRIPTION:                                                           */
/* Implelments micro stepping using the SMC module.                           */
/* Will move the stepper motor to a chosen position  */
/* Will perform acceleration and decelleration as part of the movement				*/
/*                                                                        */
/* SETUP:                                                                 */
/* Software tested on Spectrum EVB using Lauterbach and GHS 5.0.5 compiler  */
/*                                                                        */
/*========================================================================*/
/* REV      AUTHOR        DATE        DESCRIPTION OF CHANGE               */
/* ---   -----------    ----------    ---------------------               */
/* 0.1    D. Mckenna       19/Mar/09    Initial Version                     */
/* 0.2    D. Mckenna       26/Mar/09    Added position and acceleration support                     */
/**************************************************************************/
#include "MPC5606S_2.h"
#include "Ports.h"


/*========================================================================*/
/*                          DEFINES                                       */
/*========================================================================*/
// Select SMC Channels to use
#define CHANNELA 0
#define CHANNELB 1


/*========================================================================*/
/*                      GLOBAL VARIABLES                                  */
/*========================================================================*/
int8_t direction; // can be -1 or 1 - bck or fwd. Should only be changed at end of full step (e.g. microstep 0,4,8 or 12)
uint16_t new_pos; /* posotion motor should move to */
uint16_t current_pos; /* current motor position */
uint16_t accel_value, decel_value; /* position at which accel/decel should occur */
uint8_t move_complete; /* flag so that main cde knows when motor has reached desired position */


/*========================================================================*/
/*                          PROTOTYPES                                    */
/*========================================================================*/
void MC_MODE_INIT();
void DISABLE_WATCHDOG();
void PIT0_INIT();
void PIT_CH0_ISR();
uint16_t SetupPLL(uint16_t);
void STEP_INIT(void);
void STEP_TO(uint16_t);
/*========================================================================*/
/*                          FUNCTIONS                                     */
/*========================================================================*/

int main(void)
{
	uint32_t i;
	
	DISABLE_WATCHDOG();

	MC_MODE_INIT();   // select 16MHz IRC

	SetupPLL(32);

	//Assign I/O on PD0 - PD3
	for (i=46;i<50;i++)
	{
		SIU.PCR[i].R = 0x600; //select SMC function and enable Output
	}
	

	SMC.CTL0.B.MCPRE = 2;	//Divide incoming clock by 4
	SMC.CTL1.B.RECIRC = 1;	//Recirculation on low side (static channels low)
	SMC.PER.R = 0x400;	//Period is 0x400 clocks


	//Configure SMC
	//Configure SMC channels

	SMC.CC[CHANNELA].B.MCOM = 3;	//Dual-Full H-Bridge mode
	SMC.CC[CHANNELA].B.MCAM = 1;	//Left aligned
	
	
	SMC.CC[CHANNELB].B.MCOM = 3;	//Dual-Full H-Bridge mode
	SMC.CC[CHANNELB].B.MCAM = 1;	//Left aligned
	
	
	
	
	/* initialise all variables */
	direction = -1; /* start with motor turning fwds */
	move_complete = 0;
	accel_value = decel_value = 0xffff;
	current_pos = 0xfffe;

	
	// Configure the interrupts and interrupt controller
	xcptn_xmpl();     

	// Enable Interrupts at Core
	enableIrq();      

	// CONFIG INTC
	INTC.PSR[59].R  = 0x02;  /* PIT 0 */
	
	STEP_INIT();	/* ensures motor is at zero upon startup */
	direction = 1;
	while(1)
	{
		STEP_TO(1500);
		while(move_complete==0);
		STEP_TO(400);
		while(move_complete==0);
	}
} /* main */

void PIT_CH0_ISR(void)
{
	//local variables
	static uint8_t microstep = 0; //current number of microsteps per 90o rotation
	static int32_t channelA_inc = 0; 
	static int32_t channelB_inc = 0;
	static int8_t actual_direction = 1;//local variable giving the direction of rotation. Used to ensure that direction is only changed on step 0,4,8 and 12)
	
	
	PIT.CH[0].TFLG.B.TIF    = 1;  /* clear interrupt flag */
	microstep = (microstep + actual_direction)%16;
	if(microstep==255)
		microstep = 15;
	current_pos = current_pos + actual_direction;
	SMC.DC[CHANNELA].R += channelA_inc;	//alter duty cycle based on direction
	SMC.DC[CHANNELB].R += channelB_inc;	//alter duty cycle based on direction
	switch(microstep)
	{
		case 0:	//fullstep 0
			actual_direction = direction;
			if(direction==1)
			{
				SMC.DC[CHANNELA].R = 0x0400;
				SMC.DC[CHANNELB].R = 0x0000;
			}
			else
			{	
				SMC.DC[CHANNELA].R = 0x0400;
				SMC.DC[CHANNELB].R = 0x8000;
			}
			channelA_inc = -0x100;
			channelB_inc = 0x100;				
		break;
		case 4: //fullstep 1
			actual_direction = direction;
			if(direction==1)
			{			
				SMC.DC[CHANNELA].R = 0x8000;	
				SMC.DC[CHANNELB].R = 0x0400;
			}
			else
			{	
				SMC.DC[CHANNELA].R = 0x0000;
				SMC.DC[CHANNELB].R = 0x0400;				
			}
			channelA_inc = 0x100;
			channelB_inc = -0x100;
		break;
		case 8: //fullstep 2
			actual_direction = direction;
			if(direction==1)
			{	
				SMC.DC[CHANNELA].R = 0x8400;
				SMC.DC[CHANNELB].R = 0x8000;
			}
			else
			{	
				SMC.DC[CHANNELA].R = 0x8400;
				SMC.DC[CHANNELB].R = 0x0000;	
			}
			channelA_inc = -0x100;
			channelB_inc = 0x100;
		break;
		case 12: //fullstep 3
			actual_direction = direction;
			if(direction==1)
			{
				SMC.DC[CHANNELA].R = 0x0000;	
				SMC.DC[CHANNELB].R = 0x8400;
			}
			else
			{	
				SMC.DC[CHANNELA].R = 0x8000;
				SMC.DC[CHANNELB].R = 0x8400;
			}
			channelA_inc = 0x100;
			channelB_inc = -0x100;
		break;
	}
	if(current_pos == accel_value)
		PIT.CH[0].LDVAL.R = 0x6fff;
	if(current_pos == decel_value)
		PIT.CH[0].LDVAL.R = 0xdfff;
	if(current_pos == new_pos)
	{
		PIT.CH[0].TCTRL.B.TEN = 0;// disable stepper motor
		move_complete = 1;
	}
}

void MC_MODE_INIT()
{
	/* Enbable all peripheral clocks */
	CGM.SCDC.R = 0x80808080;
	/* Setting RUN Configuration Register ME_RUN_PC[0] */
	ME.RUNPC[0].R=0x000000FE; /* Peripheral ON in every mode */
	CGM.OCDSSC.B.SELCTL=1; 	   	   //Select Fast IRC (16 MHz)

	/* Re-enter in DRUN mode to update */
	ME.MCTL.R = 0x30005AF0;     	  /* Mode & Key */
	ME.MCTL.R = 0x3000A50F;     	  /* Mode & Key */  
}                                                                                                                                                                                                                         

void DISABLE_WATCHDOG()
{
	SWT.SR.R = 0x0000c520; /* key */
	SWT.SR.R = 0x0000d928; /* key */
	SWT.CR.R = 0x8000010A; /* disable WEN */
}                                                                                                                                                                                                                                                                   

//Initial PLL to freqMHz and select as system bus - assume 8MHz crystal
//freq must be >= 16 and <= 64
//Returns value actually set

void PIT0_INIT(vint32_t LDVAL)
{
	PIT.CH[0].LDVAL.R         = LDVAL;  /* load PIT counter */
	PIT.CH[0].TCTRL.B.TIE     = 1;      /* enable interrupt */
	PIT.CH[0].TCTRL.B.TEN     = 1;      /* enable channel */
	PIT.PITMCR.B.MDIS         = 0;      /* enable PIT module */
}

uint16_t SetupPLL(uint16_t freq)
{
    if (freq > 64) freq = 64;
    if (freq < 16) freq = 16;
    ME.DRUN.B.XOSC0ON=1;			       /* Switch on external osc */
    ME.DRUN.B.SYSCLK=2;			       /* Select external osc */
/* RE enter the drun mode, to update the configuration */
    ME.MCTL.R = 0x30005AF0;        /* Mode & Key */
    ME.MCTL.R = 0x3000A50F;        /* Mode & Key inverted */
    while(ME.GS.B.S_OSC==0);			   /* Wait for external osc to stabilize */
    while(ME.GS.B.S_MTRANS==1);      /* Wait for mode entry to complete */
    while(ME.GS.B.S_CURRENTMODE!=3); /* Check DRUN mode has been entered */
	
/* Max bus 64Mhz, VCO 256-512Mhz */
/* Fpll = XTAL * NDIV / IDF / ODF = 64MHz */
/* Fvco = XTAL * NDIV / IDF = 512MHz */
    if (freq >= 32)
    {
	CGM.FMPLL[0].CR.B.IDF=0x0000; 	/* Divide by 1 */
	CGM.FMPLL[0].CR.B.ODF=0x2;	/* Divide by 8 */
	CGM.FMPLL[0].CR.B.NDIV=freq;	/* Loop divide by freq */
    }
    else if (freq >= 16)
    {
	CGM.FMPLL[0].CR.B.IDF=0x0000;	/* Divide by 1 */
	CGM.FMPLL[0].CR.B.ODF=0x3;	/* Divide by 16 */
	CGM.FMPLL[0].CR.B.NDIV=freq*2;	/* Loop divide by freq*2 */
    }

//    CGM.FMPLL[1].CR.B.EN_PLL_SW	= 1;	//Enable progressive PLL switching
	
    ME.DRUN.B.PLL1ON=1;     /* enable pll */      
      
/* RE enter the drun mode, to update the configuration */
    ME.MCTL.R = 0x30005AF0;     	  /* Mode & Key */
    ME.MCTL.R = 0x3000A50F;         /* Mode & Key inverted */
    while(ME.GS.B.S_MTRANS==1);		    /* Wait for mode entry to complete */
    while(ME.GS.B.S_CURRENTMODE!=0x3);/* Check DRUN mode has been entered */
		
    while(CGM.FMPLL[0].CR.B.S_LOCK==0);
	
    ME.DRUN.B.SYSCLK=0x4;  /* system clock is PLL */
    ME.MCTL.R = 0x30005AF0;     	  /* Mode & Key */
    ME.MCTL.R = 0x3000A50F;         /* Mode & Key inverted */
    while(ME.GS.B.S_MTRANS==1);		    /* Wait for mode entry to complete */
    while(ME.GS.B.S_CURRENTMODE!=0x3);/* Check DRUN mode has been entered */

/* ME_GS Poll Global status register to get current system clock
   fail if system clock is not pll
   0000 16MHz internal RC oscillator
   0001 divided 16MHz internal RC oscillator
   0010 4MHz crystal oscillator
   0011 divided 4MHz crystal oscillator
   0100 system PLL
   1111 system clock is disabled */

    while(ME.GS.B.S_SYSCLK != 4){}; /* fail is stuck here	 */
    return freq;
}

void STEP_INIT(void)
{
	/* ensures motor is a zero upon startup */
	uint32_t i;
	direction = -1;
	PIT0_INIT(0xdfff);		//Initialise PIT interrupt - value sets speed
	for(i=0;i<0x1500000;i++);//wait until stepper motor is at position 0
	PIT.CH[0].TCTRL.B.TEN = 0;// disable stepper motor
	current_pos = 0;
}

void STEP_TO(uint16_t dest)
{
	/* function to move motor to a position designated by dest */
	/* dest must be an integer between 0 and 8000*/
	new_pos = dest;
	if(new_pos>current_pos)
	{
		direction = 1;
		move_complete = 0;
		if((new_pos - current_pos)>500)
		{
			/* find acceleration and decelleration values */
			accel_value = current_pos + (0.1 * (new_pos - current_pos));
			decel_value = current_pos + (0.9 * (new_pos - current_pos));
		}else
			accel_value = decel_value = 0xffff;
		PIT.CH[0].TCTRL.B.TEN = 1;// enable stepper motor
	}
	else if (new_pos<current_pos)
	{
		direction = -1;
		move_complete = 0;
		if((current_pos - new_pos)>500)
		{
			/* find acceleration and decelleration values */
			decel_value = current_pos - (0.9 * (current_pos - new_pos));
			accel_value = current_pos - (0.1 * (current_pos - new_pos));
		}else
			accel_value = decel_value = 0xffff;
		PIT.CH[0].TCTRL.B.TEN = 1;// enable stepper motor
	}
	else
		move_complete = 1;	//current pos == new pos, do nothing
}
