/******** IrDA.c ********/

#include "XTWR-MCF51JE.h"
#include "IrDA.h"
#include "SDbyIrDA.h"
#include "sci.h"

UINT8 u8irda_tx_counter;
UINT8 u8irda_rx_counter;
UINT8 u8irda_tx_status;
UINT8 u8irda_rx_status;
UINT8 u8irda_tx_data;
UINT8 u8irda_rx_data;

/***************************************************************************************/
/******************** INITIALIZATION AND TRANSMISSION FUNCTIONS ************************/
/***************************************************************************************/

void vfnCMT_init(void)
{  
   //TWO BITS FOR DIVISOR SELECTOR: 0-> CLOCK/1, 1-> CLOCK/2, 2-> CLOCK/4, 3-> CLOCK/8
   #if (DIVISOR == 1)
     CMTMSC_CMTDIV = 1;
   #endif  
   
   #if (DIVISOR == 2)
     CMTMSC_CMTDIV = 2;
   #endif  
   
   #if (DIVISOR == 4)
     CMTMSC_CMTDIV = 3;
   #endif  
   
   CMTMSC_EXSPC = 0;              //Extended Space Enable-1 / Disable-0
   CMTMSC_BASE = 0;               //Baseband Enable-1 / Disable-0   
   CMTMSC_FSK = 0;                //FSK mode select: 0-time or baseband / 1-FSK mode
   CMTMSC_EOCIE = 0;              //End of cycle interrupt Enabled-1 / Disabled-0
   
   CMTOC_CMTPOL = 1;              //IRO polarity: 0: active low / 1-active high
   CMTOC_IROPEN = 1;              //IRO pin enabled-1 / disabled-0
   SOPT3_CMT_PAD = 1;             //Double Drive Strength on IRO pin Enable-1/ Disable-0  
      
   CMTCGH1 = (UINT8)HIGH_COUNT;   //Carrier generator values         
   CMTCGL1 = (UINT8)LOW_COUNT;  
   
   CMTCMD34 = 0;                  //Modulator value (space)
   CMTMSC_MCGEN = 0;              //Modulator and Carrier Generator Enabled-1 / Disabled-0   
}

/***************************************************************************************/

UINT8 u8ACMP_rx_init (UINT8 acmp_level)
{ 
   if (0<acmp_level>31) return 1;
   
   /***PRACMP init***/     
   PRACMPCS = 0;                 /* ACMP disabled */
   /*PRACMPCS_ACIEN = 1;*/       /* interrupt */
   PRACMPCS_ACOPE = 0;           /* output on ACMPO */
   
   PRACMPC0_ACPSEL = 7;          /* internal programmable reference*/ 
   PRACMPC0_ACNSEL = 1;          /* external reference */ /* CMPP1 for XTWR */
   
   PRACMPC1_PRGINS = 1;          /* using external Vdd reference (probably 3.3V) */
   PRACMPC2 = 0x02;              /* external analog channel 1 is allowed!!! */
   PRACMPC1_PRGEN = 1;           /* start programmable */
   PRACMPC1_PRGOS = acmp_level;  /* comparator reference level */
   PRACMPCS_ACEN = 1;            /* ACMP enabled */
   /***PRACMP end***/
   
   return 0;
}

/***************************************************************************************/

void vfnTPM1_rx_init(void)
{
  SOPT2_ACIC = 1;           // ACMP output to TPM1CH0 input: 0-disabled, 1-enabled 
  TPM1SC_TOIE = 0;          // Timer overflow interrupt (0- disabled, 1- enabled)
  TPM1MOD = TPM_PERIOD;
  
  //TMP ch 0 mode
  TPM1SC_CPWMS = 0;         // Center-aligned PWM: 0-disabled, 1-enabled
  
  TPM1C0SC_MS0B = 0;        // Input Capture mode
  TPM1C0SC_MS0A = 0;
  
  TPM1C0SC_ELS0B = 0;       // Rising edge input capture
  TPM1C0SC_ELS0A = 1;
  
  TPM1C0SC_CH0IE = 1;       // Interrupt enable on channel 0 event

  //THREE BITS FOR DIVISOR SELECTOR: 
  //0-> 1, 1-> 2, 2-> 4, 3-> 8, 4-> 16, 5-> 32, 6-> 64, 7-> 128   
  #if (DIVISOR == 1)
    TPM1SC_PS = 1;
  #endif  
   
  #if (DIVISOR == 2)
    TPM1SC_PS = 2;
  #endif  
   
  #if (DIVISOR == 4)
    TPM1SC_PS = 3;
  #endif  
    
  TPM1SC_CLKSB = 0;         // Clock source selection (B:A) (0 disables the module)
  TPM1SC_CLKSA = 1;         // 00-> NONE, 01-> BUS, 10-> FIXED FREQUENCY, 11-> EXTERNAL                          
}

/***************************************************************************************/

void vfnIrDA_transmission(UINT8 u8data)
{
   UINT16 u16stop = (UINT16)(STOP_BIT);
   u8irda_tx_data = u8data;
 
   while ((u8irda_tx_status == IRDA_SENDING)   ||
          (u8irda_rx_status == IRDA_RECEIVING)){};
   
   u8irda_tx_counter = 0;   
   u8irda_tx_status = IRDA_SENDING;
   
   CMTOC_IROPEN = 1;                      //Modulator gate enabled
   CMTCMD12 = (UINT16)(CMT_PERIOD / 2);   //Modulator value (mark)        
   CMTMSC_MCGEN = 1;                      //Modulator and Carrier Generator enabled
   (void)CMTMSC;
   CMTCMD12 = (UINT16)(CMT_PERIOD);
   CMTMSC_EOCIE = 1;                      //End of cycle interrupt enabled
   
   while (u8irda_tx_status == IRDA_SENDING){};   
   while (u16stop) u16stop--; 
}

/***************************************************************************************/

void IrDA_Message(UINT8 *string)
{
    while(*string) vfnIrDA_transmission(*string++);        
}



/***************************************************************************************/
/**************************** INTERRUPT SERVICES ***************************************/
/***************************************************************************************/

void interrupt VectorNumber_Vcmt CMT_eoc(void) 
{           
    //Clear interrupt flag
    (void)CMTMSC;
    (void)CMTCMD2; 

    if (u8irda_tx_counter < 8)
    {
      if ((u8irda_tx_data>>u8irda_tx_counter)&0x01) CMTOC_IROPEN = 0;
      else CMTOC_IROPEN = 1;

      u8irda_tx_counter++;               //Count bit number
    } 
    
    else if (u8irda_tx_counter == 8)
    {
      CMTOC_IROPEN = 0;                  //Modulator gate open
      CMTMSC_EOCIE = 0;                  //End of cycle interrupt disabled
      CMTMSC_MCGEN = 0;                  //Modulator and Carrier Generator disabled
      u8irda_tx_status = IRDA_TX_IDLE;
    }
}

/***************************************************************************************/

void interrupt VectorNumber_Vtpm1ovf TPM1_rx_space(void) 
{
    //Clear interrupt flag
    (void)TPM1SC;
    TPM1SC_TOF = 0;
    
    if(u8irda_rx_counter==9)
    {
      u8irda_rx_status = IRDA_RX_IDLE;
      u8irda_rx_counter = 0;
      TPM1SC_TOIE = 0;      
      
      /* Add here the code for interruption on data received */          
      
      if(u8irda_status == IrDA_SD_TO_SCI)
      { 
        vfnSCI_SendChr(USED_PORT, u8irda_rx_data);
      }
      
      /*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*/
      
    }       
    else  
    {
       u8irda_rx_data |= (1<<(u8irda_rx_counter-1));   
    }  
    
    if(u8irda_rx_counter) 
    {
       u8irda_rx_counter++;
    }
}

/***************************************************************************************/
 
void interrupt VectorNumber_Vtpm1ch0 TPM1_rx_pulse(void) 
{
    //Clear interrupt flag
    (void)TPM1C0SC;
    TPM1C0SC_CH0F = 0;
    
    //Disable timer count and interrupt
    TPM1SC_TOIE = 0;
    TPM1SC_CLKSB = 0;         
    TPM1SC_CLKSA = 0;         
    
    if(!u8irda_rx_counter) 
    {
      u8irda_rx_status = IRDA_RECEIVING;
      u8irda_rx_data = 0;
    }
    u8irda_rx_counter++;
    
    TPM1CNT = 0;
    (void)TPM1SC;
    TPM1SC_TOF = 0;
    
    TPM1SC_CLKSB = 0;         
    TPM1SC_CLKSA = 1;         
    
    TPM1SC_TOIE = 1;
}

/***************************************************************************************/