/*******************************************************************************
*
*	Freescale Semiconductor Inc.
*	(c) Copyright 2010 Freescale Semiconductor Inc.
*	ALL RIGHTS RESERVED.
*
****************************************************************************//*!
*
*	@file sci_app_01.c
*	
*	@version 0.0.4.0
*	
*	@lastmodusr B16958
*	
*	@lastmoddate Jul-1-2010
*	
********************************************************************************
*
*	@brief eDMA emulating single SCI transmit
*
*	This example demonstrates how the eDMA could be used to emulate SCI.
*	It uses LINFlex0 to receive one byte from serial line, and sends the same
*	byte using single DMA channel.
*	Configuration:
*		- LINFlex0 - Rx only on PCR_19 (PB3)
*		- DMA channel 0 writes to GPIO PCR_18 (PB2)
*		- DMA triggered using PIT_0 with always enabled source (periodic triggers only)
*		- SW task: On reception of one byte using UART transmit the same byte using DMA
*		- Connection: Standard wiring on EVB: PB2 and PB3 -> SCI transciever
*	
*	@project AN4147
*	
*	@author B16958
*	
*	@ingroup sci_tx
*
******************************************************************************/

/******************************************************************************
* Includes
******************************************************************************/
#include "sci_app_01.h"
#include "MPC560xB.h"

/******************************************************************************
* Internal Function Declarations
******************************************************************************/


static void SciInit01(void);

static void SciRun01(void);


/******************************************************************************
* External Function Definitions
******************************************************************************/
/***************************************************************************//*!
*
* @brief Function interface to execute example 1 of SCI communication emulation.
*
******************************************************************************/
extern void SciApp01(void)
{
	SciInit01();
	
	SciRun01();
}

/******************************************************************************
* Internal Function Definitions
******************************************************************************/

/***************************************************************************//*!
*
* @brief Initializes the first SCI application to transmit one byte at at time
*			using one eDMA channel triggered periodically by PIT
*
*	Initialization of the following peripherals:
*		- SIU: PCR registers for Rx and Tx signals
*		- LINFLEX_0: As UART reciever on 9.600kBauds, 8bit data bits
*		- PIT_0: To periodically trigger DMA at 9600Hz (one bit per timer overflow)
*		- DMAMUX: Routes PIT_0 flag to activate channel 0 DMA
*		- eDMA0: Setup one byte data movements
*			- from GPIO's PG15-PH9 (10 iterations of minor loop)
*			- to GPIO PB2 (SCI Tx)
*			- disable on completion (after one byte transmited, turns the eDMA off)
*
*	@ingroup sci_tx
*
******************************************************************************/
static void SciInit01(void)
{
	/* Initialize PCR registers */
	 SIU.PCR[19].B.PA = 0;          /* Pin asigned to LINFlex Rx */
	 SIU.PCR[19].B.IBE = 1;         /* Input buffer enable */
	 SIU.PCR[18].B.PA = 0;          /* Pin asigned as GPIO for DMA to perform SCI Tx */
	 SIU.PCR[18].B.OBE = 1;         /* Output buffer enable */
	
	/* Initialize LINFlex as SCI receiver (UART, 9.600kb, 8data, 1stop, no parity) */
	LINFLEX_0.LINCR1.B.SLEEP = 0; /* Resets sleep mode */
	LINFLEX_0.LINCR1.B.INIT = 1;   /* Sets initialization mode */
	LINFLEX_0.UARTCR.B.UART = 1; /* Sets UART mode */
	LINFLEX_0.UARTCR.B.WL = 1;   /* 8 data bits */
	LINFLEX_0.UARTCR.B.PCE = 0;    /* Parity controll disabled */
	LINFLEX_0.UARTCR.B.RXEN = 1; /* Enable Receiver */
	LINFLEX_0.UARTCR.B.TXEN = 0; /* Disable Transmitter */
	LINFLEX_0.UARTCR.B.TDFL = 0; /* Sets 1Byte data lenght for Transmitter */
	LINFLEX_0.UARTCR.B.RDFL = 0; /* Sets 1Byte data lenght for Receiver */
	LINFLEX_0.LINIBRR.B.DIV_M = 0x34; /* Clock divisor's mantisa */
	LINFLEX_0.LINFBRR.B.DIV_F = 0; /* Clock divisor's fraction */	
	LINFLEX_0.LINCR1.B.INIT = 0;   /* Resets initialization mode */

	/* Initialize PIT 0 timer for periodic triggers of DMA channel 0 */
 	PIT.PITMCR.B.MDIS = 0;	/* enable the module */
  	PIT.CH[0].LDVAL.R=833;	/* setup 9600@8MHZ overflow rate */
  	PIT.CH[0].TFLG.B.TIF=1; /* clear the flag */


	/* Initialize the DMA channel 0 */
	DMAMUX.CHCONFIG[0].R =0x00; /* disable the channel activation source */
	/* Setup the TCD */
	EDMA.TCD[0].SADDR = (vuint32_t)( &(SIU.GPDO[120].R)); /* Load data from byte-wise register PG15(=start bit), PH0-PH8(data), PH9(=stopBit) */
	EDMA.TCD[0].SSIZE = 0; /* Read 2**0 = 1 byte per transfer */
	EDMA.TCD[0].SOFF = -1; /* After transfer, decrement 1 */
	EDMA.TCD[0].SLAST = 10; /* After major loop, back to the beginning (10 iterations) */
	EDMA.TCD[0].SMOD = 0; /* Source modulo feature not used */
	EDMA.TCD[0].DADDR = (vuint32_t) &(SIU.GPDO[18].R); /* writing to PCR 18 (GPIO-Tx) */
	EDMA.TCD[0].DSIZE =  0; /* Write 2**0 = 1 byte per transfer */
	EDMA.TCD[0].DOFF = 0; /* Do not increment destination addr */
	EDMA.TCD[0].DLAST_SGA = 0; /* After major loop no change to destination addr */
	EDMA.TCD[0].DMOD = 0; /* Destination modulo feature not used */
	EDMA.TCD[0].NBYTES = 1; /* Transfer 1 byte per minor loop */
	EDMA.TCD[0].BITER = 10; /* 10 iterations of major loop */
	EDMA.TCD[0].CITER = 10; /* Initialize current iteraction count */
	EDMA.TCD[0].D_REQ = 1; /* Disable channel when major loop is done (sending one byte only) */
	EDMA.TCD[0].INT_HALF = 0; /* no interrupt in half */
	EDMA.TCD[0].INT_MAJ = 0; /* no interrupt in major loop completion */
	EDMA.TCD[0].CITERE_LINK = 0; /* Linking is not used */
	EDMA.TCD[0].BITERE_LINK = 0;
	EDMA.TCD[0].MAJORE_LINK = 0;
	EDMA.TCD[0].BITERLINKCH = 0;	
	EDMA.TCD[0].CITERLINKCH = 0;	
	EDMA.TCD[0].MAJORLINKCH = 0;	/* no linkage */
	EDMA.TCD[0].E_SG = 0;
	EDMA.TCD[0].BWC = 0; /* Default bandwidth control- no stalls */
	EDMA.TCD[0].START = 0; /* Initialize status flags */
	EDMA.TCD[0].DONE = 0;
	EDMA.TCD[0].ACTIVE = 0;
	/* route DMA source to always enabled channel and enable periodic triggers */
	DMAMUX.CHCONFIG[0].R =0xC0 | 0x1F; /* only PIT0 -- always enabled channel */

}

/***************************************************************************//*!
*
* @brief	Polls the LINFlex reciever and if anything recieved, send it back
*				using eDMA
*
*	Data to be transmitted are located at PG15-PH9, where
*		- PG15 is a start bit
*		- PH0-PH8 are data bits
*			- accessed using GPDO[112-119] using eDMA (serial)
*			- accessed using PGPDO[3] using application (parallel)
*		- PH9 is a stop bit
*
*	Data transmission is started enabling the DMA channel, also we need to clear 
*		the flag and restart the timer, as we are sending one byte at a time. In case
*		of multi-byte transmission this wouldn't be necessary.
*
*	@ingroup sci_tx
*
******************************************************************************/
static void SciRun01(void)
{
  unsigned volatile char *pTxData=(void *)0;
  volatile int i=0;
  unsigned char recievedData;
  
	/* SETUP the Data to be sent */
  	pTxData=((unsigned char *)&SIU.PGPDO[3].R)+2; /* 8 data bits: PH0 - PH8 */
	SIU.GPDO[111].R = 1; /* STOP bit:  PH9 */
	SIU.GPDO[120].R = 0; /* START bit: PG15 */
 	
 	/* setup the line to logical high -- to get the first start bit right */
 	SIU.GPDO[18].R = 1;

  for (;;) 
  {
  	while ((LINFLEX_0.UARTSR.B.DRF)!=1) {}; /* Wait until a new data is recieved */ 
    LINFLEX_0.UARTSR.B.DRF = 1; /* Reset DTF flag */
	recievedData = (unsigned char)LINFLEX_0.BDRM.B.DATA4;
    
    /* set the received word to the word to be transmitted */
    *pTxData = recievedData;
    
	/* now let's start the DMA transmission */
  	PIT.CH[0].TCTRL.B.TEN=0; /* stop the timer */
  	PIT.CH[0].TFLG.B.TIF=1;	 /* clear the flag */
	EDMA.SERQR.R = 0; 		/* enable DMA 0 */
 	PIT.CH[0].TCTRL.B.TEN=1; /* and now start the timer */
  }
    

}
