/*******************************************************************************
*
*	Freescale Semiconductor Inc.
*	(c) Copyright 2010 Freescale Semiconductor Inc.
*	ALL RIGHTS RESERVED.
*
****************************************************************************//*!
*
*	@file sci_app_02.c
*	
*	@version 0.0.4.0
*	
*	@lastmodusr B16958
*	
*	@lastmoddate Jul-1-2010
*	
********************************************************************************
*
*	@brief eDMA emulating multiple SCI transmits
*
*	This example demonstates how the eDMA could be used to emulate SCI.
*	Sending a string of bytes using chaining two eDMA channels
*	Configuration:
*		- DMA channel 0 writes to GPIO PCR_18 (PB2)
*		- DMA channel 0 activated using periodic triggers
*		- DMA channel 1 takes one byte from buffers and link to the channel 0
*		- DMA channel 1 activated explicitely by SW to transmit burst of bytes
*		- Connection: Standard wiring on EVB: PB2 and PB3 -> SCI transciever
*	
*	@project AN4147
*	
*	@author B16958
*	
*	@ingroup sci_tx
*
******************************************************************************/

/******************************************************************************
* Includes
******************************************************************************/
#include "sci_app_02.h"
#include "MPC560xB.h"


/******************************************************************************
* Internal variable Definitions
******************************************************************************/
static volatile unsigned char *pTxData=((unsigned char *)&SIU.PGPDO[3].R)+2; /*!< Pointer 
* to the data prepared for transmission, i.e. serialization. This is a reference
* to GPIO parallel access registers */
static  unsigned char stringBuffer[12]="Hello World!"; /*!< String which is periodiacally
* transmitted over serial line. Note that the transmission has fixed number of data,
* equals to the size of this byte array. */

/******************************************************************************
* Internal Function Declarations
******************************************************************************/


static void SciInit02(void);

static void SciRun02(void);


/******************************************************************************
* External Function Definitions
******************************************************************************/
extern void SciApp02(void)
{
	SciInit02();
	
	SciRun02();
}

/******************************************************************************
* Internal Function Definitions
******************************************************************************/

/***************************************************************************//*!
*
* @brief Initializes the SCI application to transmit one byte continuouslly,
*			whilst this byte is modified by other channel.
*
*	Initialization of the following peripherals:
*		- SIU: PCR registers for UART Tx signals
*		- PIT_0: To periodically trigger DMA at 9600Hz (one bit per timer overflow)
*		- DMAMUX: Routes PIT_0 flag to activate channel 0 DMA
*		- eDMA0: Setup one byte data movements
*			- from GPIO's PG15-PH9 (10 iterations of minor loop)
*			- to GPIO PB2 (SCI Tx)
*			- link to the channel 1 after major loop.
*		- eDMA1: Setup one byte transfer from data buffer to PH0-PH7 (data to be sent)
*
*	@ingroup sci_tx
*
******************************************************************************/
static void SciInit02(void)
{
	/* Initialize PCR registers */
	 SIU.PCR[18].B.PA = 0;          /* Pin asigned as GPIO for DMA to perform SCI Tx */
	 SIU.PCR[18].B.OBE = 1;         /* Output buffer enable */
 	/* setup the line to logical high -- to get the first start bit right */
 	SIU.GPDO[18].R = 1;
	
	/* Initialize PIT 0 timer for periodic triggers of DMA channel 0 */
 	PIT.PITMCR.B.MDIS = 0;	/* enable the module */
  	PIT.CH[0].LDVAL.R=833;	/* setup 9600@8MHZ overflow rate */
  	PIT.CH[0].TFLG.B.TIF=1; /* clear the flag */

	/* Initialize the DMA channel 0 */
	DMAMUX.CHCONFIG[0].R =0x00; /* disable the channel activation source */
	/* Setup the TCD for channel 0 --> single SCI Tx */
	EDMA.TCD[0].SADDR = (vuint32_t)( &(SIU.GPDO[120].R)); /* Load data from byte-wise register PG15(=start bit), PH0-PH8(data), PH9(=stopBit) */
	EDMA.TCD[0].SSIZE = 0; /* Read 2**0 = 1 byte per transfer */
	EDMA.TCD[0].SOFF = -1; /* After transfer, decrement 1 */
	EDMA.TCD[0].SLAST = 10; /* After major loop, back to the beginning (10 iterations) */
	EDMA.TCD[0].SMOD = 0; /* Source modulo feature not used */
	EDMA.TCD[0].DADDR = (vuint32_t) &(SIU.GPDO[18].R); /* writing to PCR 18 (GPIO-Tx) */
	EDMA.TCD[0].DSIZE =  0; /* Write 2**0 = 1 byte per transfer */
	EDMA.TCD[0].DOFF = 0; /* Do not increment destination addr */
	EDMA.TCD[0].DLAST_SGA = 0; /* After major loop no change to destination addr */
	EDMA.TCD[0].DMOD = 0; /* Destination modulo feature not used */
	EDMA.TCD[0].NBYTES = 1; /* Transfer 1 byte per minor loop */
	EDMA.TCD[0].BITER = 10; /* 10 iterations of major loop */
	EDMA.TCD[0].CITER = 10; /* Initialize current iteraction count */
	EDMA.TCD[0].D_REQ = 0; /* Disable channel when major loop is done (sending one byte only) */
	EDMA.TCD[0].INT_HALF = 0; /* no interrupt in half */
	EDMA.TCD[0].INT_MAJ = 0; /* no interrupt in major loop completion */
	EDMA.TCD[0].CITERE_LINK = 0; /* no link after minor loop */
	EDMA.TCD[0].BITERE_LINK = 0;
	EDMA.TCD[0].MAJORE_LINK = 1; /* Linking the next channel to process buffering after major loop */
	EDMA.TCD[0].BITERLINKCH = 0;	
	EDMA.TCD[0].CITERLINKCH = 0;	
	EDMA.TCD[0].MAJORLINKCH = 1;	/* Link to channel 1 */
	EDMA.TCD[0].E_SG = 0;
	EDMA.TCD[0].BWC = 0; /* Default bandwidth control- no stalls */
	EDMA.TCD[0].START = 0; /* Initialize status flags */
	EDMA.TCD[0].DONE = 0;
	EDMA.TCD[0].ACTIVE = 0;

	/* Setup the TCD for channel 1 --> buffer for SCI Tx */
	DMAMUX.CHCONFIG[1].R =0x00; /* disable the channel activation source */
	EDMA.TCD[1].SADDR = (vuint32_t)(stringBuffer+1); /* Load data from the buffer (start with 2nd element) */
	EDMA.TCD[1].SSIZE = 0; /* Read 2**0 = 1 byte per transfer */
	EDMA.TCD[1].SOFF = 1; /* After transfer, decrement 1 */
	EDMA.TCD[1].SLAST = -12; /* After major loop, back to the beginning (-12 iterations) */
	EDMA.TCD[1].SMOD = 0; /* Source modulo feature not used */
	EDMA.TCD[1].DADDR = (vuint32_t)pTxData; /* writing to PCR 18 (GPIO-Tx) */
	EDMA.TCD[1].DSIZE =  0; /* Write 2**0 = 1 byte per transfer */
	EDMA.TCD[1].DOFF = 0; /* Do not increment destination addr */
	EDMA.TCD[1].DLAST_SGA = 0; /* After major loop no change to destination addr */
	EDMA.TCD[1].DMOD = 0; /* Destination modulo feature not used */
	EDMA.TCD[1].NBYTES = 1; /* Transfer 1 byte per minor loop */
	EDMA.TCD[1].BITER = 12; /* 12 iterations of major loop */
	EDMA.TCD[1].CITER = 11; /* Initialize current iteraction count, (start with 2nd data!) */
	EDMA.TCD[1].D_REQ = 0; /* Disable channel when major loop is done (sending the buffer once) */
	EDMA.TCD[1].INT_HALF = 0; /* no interrupt in half */
	EDMA.TCD[1].INT_MAJ = 0; /* no interrupt in major loop completion */
	EDMA.TCD[1].CITERE_LINK = 0; /* Linking disabled */
	EDMA.TCD[1].BITERE_LINK = 0;
	EDMA.TCD[1].BITERLINKCH = 0;
	EDMA.TCD[1].CITERLINKCH = 0;
	EDMA.TCD[1].MAJORE_LINK = 0;
	EDMA.TCD[1].E_SG = 0;
	EDMA.TCD[1].BWC = 0; /* Default bandwidth control- no stalls */
	EDMA.TCD[1].START = 0; /* Initialize status flags */
	EDMA.TCD[1].DONE = 0;
	EDMA.TCD[1].ACTIVE = 0;


	/* route DMA source to always enabled channel and enable periodic triggers */
	DMAMUX.CHCONFIG[0].R =0xC0 | 0x1F; /* only PIT0 -- always enabled channel */

}

/***************************************************************************//*!
*
* @brief	Transmits the message "Hello World!" continously on serial line.
*
*	Data to be transmitted are located at PG15-PH9, where
*		- PG15 is a start bit
*		- PH0-PH8 are data bits
*			- accessed using GPDO[112-119] using eDMA (serial)
*			- accessed using PGPDO[3] using application (parallel)
*		- PH9 is a stop bit
*
*	Data transmission is started enabling the DMA channel 0, which transmits one byte.
*			After the transmission, links to the channel 1, which pops other byte from
*			transmitting buffer. Channel 0 is not dissabled, so transmission continues
*			with all the messages.
*			
*	These transmissions are performed only by eDMA, so that there is no SW task after 
*			activation of the two channels and the timer. This function is never exitted
*			and waits in the empty neverending loop.
*
*	@ingroup sci_tx
*
******************************************************************************/
static void SciRun02(void)
{
  
	/* SETUP the first data to be sent */
	*pTxData = stringBuffer[0];	/* 8 data bits: PH0 - PH8 */
	SIU.GPDO[111].R = 1; /* STOP bit:  PH9 */
	SIU.GPDO[120].R = 0; /* START bit: PG15 */
 	
 	/* setup the line to logical high -- to get the first start bit right */
 	SIU.GPDO[18].R = 1;

  	PIT.CH[0].TFLG.B.TIF=1;	 /* clear the timer flag */
 	PIT.CH[0].TCTRL.B.TEN=1; /* and now start the timer */

	EDMA.SERQR.R = 0; 		/* start DMA channel 0 */

  for (;;) 
  {
  }
   

}
