/*******************************************************************************
*
*	Freescale Semiconductor Inc.
*	(c) Copyright 2010 Freescale Semiconductor Inc.
*	ALL RIGHTS RESERVED.
*
****************************************************************************//*!
*
*	@file sci_app_04.c
*	
*	@version 0.0.4.0
*	
*	@lastmodusr B16958
*	
*	@lastmoddate Jul-1-2010
*	
********************************************************************************
*
*	@brief eDMA emulating multiple SCI transmits with condition
*
*	This example demonstates how the eDMA could be used to transmit SCI data 
*	of variable length
*	Configuration:
*		- DMA channel 0 writes to GPIO PCR_18 (PB2) and is acitvated by PIT
*		- DMA channel 1 takes one byte from buffers and link from the channel 0
*		- DMA channel 2 sends one byte from buffers also to the comparator (is linked from the channel 1)
*		- DMA channel 3 is activated either by SW (to start the timer) or by eMIOS-comparator (on match)
*			to stop transmitting the data.
*		- Connection: Standard wiring on EVB: PB2 and PB3 -> SCI transciever
*	
*	@project AN4147
*	
*	@author B16958
*	
*	@ingroup sci_tx
*
******************************************************************************/

/******************************************************************************
* Includes
******************************************************************************/
#include "sci_app_04.h"
#include "MPC560xB.h"


/******************************************************************************
* Internal variable Definitions
******************************************************************************/
static volatile unsigned char *pTxData=((unsigned char *)&SIU.PGPDO[3].R)+2; /*!< Pointer 
* to the data prepared for transmission, i.e. serialization. This is a reference
* to GPIO parallel access registers */
static  unsigned char stringBuffer[125]="Hello World! -please use period at the end of a sentence.";
/*!< String which is transmitted over serial line. The end of this buffer is indicated by the last character
* '.'. The data are compared during the transmission with the end character and the transmission
* is stopped on match.  */
static  unsigned long int toggleTimer[2]=
{
	0, /* disable the timer at first channel iteration */
	0  /* disable the timer in next iteration (leftover- not used) */
};
/*!< This array is used to toggle (start/stop) periodic interrupt timer, meaning
* to enable/disable the transmission. Enabling is done by the application, whereas
* disabling is automatic, when a stop character is indicated. */

/******************************************************************************
* Internal Function Declarations
******************************************************************************/


static void SciInit04(void);

static void SciRun04(void);


/******************************************************************************
* External Function Definitions
******************************************************************************/
extern void SciApp04(void)
{
	SciInit04();
	
	SciRun04();
}

/******************************************************************************
* Internal Function Definitions
******************************************************************************/

/***************************************************************************//*!
*
* @brief Initializes the SCI application to transmit one byte continuouslly,
*			whilst this byte is modified by the channel 1. Channel 2 transmitt
*			this byte to the comparator, which triggers channel 3 when match occurs.
*			Channel 3 then disables the PIT, i.e. the data transmittion.
*
*	Initialization of the following peripherals:
*		- SIU: PCR registers for UART Tx signals
*		- PIT_0: To periodically trigger DMA at 9600Hz (one bit per timer overflow)
*		- DMAMUX
*			- Routes PIT_0 flag to activate channel 0 DMA
*			- Routes eMIOS flag (from comparator) to trigger channel 3
*		- eDMA0: Setup one byte data movements (to transmit one SCI byte)
*		- eDMA1: Setup one byte transfer from data buffer to PH0-PH7 (data to be sent)
*		- eDMA2: Setup one byte transfer from data buffer to eMIOS comparator
*		- eDMA3: Setup to enable/disable the PIT. This channel is triggered from eMIOS comparator. 
*
*	@ingroup sci_tx
*
******************************************************************************/
static void SciInit04(void)
{


	/* Initialize PCR registers */
	SIU.PCR[18].B.PA = 0;          /* Pin asigned as GPIO for DMA to perform SCI Tx */
	SIU.PCR[18].B.OBE = 1;         /* Output buffer enable */

 	/* setup the line to logical high -- to get the first start bit right */
 	SIU.GPDO[18].R = 1;
	
	/* Initialize PIT 0 timer for periodic triggers of DMA channel 0 */
 	PIT.PITMCR.B.MDIS = 0;	/* enable the module */
  	PIT.CH[0].LDVAL.R=833;	/* setup 9600@8MHZ overflow rate */
  	PIT.CH[0].TFLG.B.TIF=1;	 /* clear the timer flag */
 	PIT.CH[0].TCTRL.B.TEN=0; /* and disable the timer */

	/* Initialize the eMIOS0_ch0 for single action output compare */
	EMIOS_0.MCR.B.MDIS=0;
	EMIOS_0.MCR.B.GTBE=1; /* global time base out enable*/
	EMIOS_0.MCR.B.GPREN=1; /* enable prescaler */
	EMIOS_0.MCR.B.GPRE=0; /* presclaer 1:1 */
	EMIOS_0.OUDIS.R=0; /* enable all outputs */
	EMIOS_0.UCDIS.R=0; /* enable all channels */
	EMIOS_0.CH[0].A.R=0;
	EMIOS_0.CH[0].CCR.B.UCPRE=0; /* 1:1 */
	EMIOS_0.CH[0].CCR.B.UCPEN=1; /* enable prescaler */
	EMIOS_0.CH[0].CCR.B.DMA=1;	 /* DMA, not interrupt */
	EMIOS_0.CH[0].CCR.B.FCK=1; /* use main clock */
	EMIOS_0.CH[0].CCR.B.FEN=1; /* enable Flag -> DMA */
	EMIOS_0.CH[0].CCR.B.BSL=3; /* all channels internal counter bus */
	EMIOS_0.CH[0].CCR.B.EDSEL=1; /* saoc: output flip-flop toggled */
	EMIOS_0.CH[0].CCR.B.EDPOL=0; /* falling edge */
	EMIOS_0.CH[0].CCR.B.MODE=3; /* Single Action Output Compare */	


	/* Initialize the DMA channel 0 */
	DMAMUX.CHCONFIG[0].R =0x00; /* disable the channel activation source */
	DMAMUX.CHCONFIG[1].R =0x00; /* disable the channel activation source */
	DMAMUX.CHCONFIG[2].R =0x00; /* disable the channel activation source */
	DMAMUX.CHCONFIG[3].R =0x00; /* disable the channel activation source */
	/* Setup the TCD for channel 0 --> single SCI Tx */
	EDMA.TCD[0].SADDR = (vuint32_t)( &(SIU.GPDO[120].R)); /* Load data from byte-wise register PG15(=start bit), PH0-PH8(data), PH9(=stopBit) */
	EDMA.TCD[0].SSIZE = 0; /* Read 2**0 = 1 byte per transfer */
	EDMA.TCD[0].SOFF = -1; /* After transfer, decrement 1 */
	EDMA.TCD[0].SLAST = 10; /* After major loop, back to the beginning (10 iterations) */
	EDMA.TCD[0].SMOD = 0; /* Source modulo feature not used */
	EDMA.TCD[0].DADDR = (vuint32_t) &(SIU.GPDO[18].R); /* writing to PCR 18 (GPIO-Tx) */
	EDMA.TCD[0].DSIZE =  0; /* Write 2**0 = 1 byte per transfer */
	EDMA.TCD[0].DOFF = 0; /* Do not increment destination addr */
	EDMA.TCD[0].DLAST_SGA = 0; /* After major loop no change to destination addr */
	EDMA.TCD[0].DMOD = 0; /* Destination modulo feature not used */
	EDMA.TCD[0].NBYTES = 1; /* Transfer 1 byte per minor loop */
	EDMA.TCD[0].BITER = 10; /* 10 iterations of major loop */
	EDMA.TCD[0].CITER = 10; /* Initialize current iteraction count */
	EDMA.TCD[0].D_REQ = 0; /* Disable channel when major loop is done (sending one byte only) */
	EDMA.TCD[0].INT_HALF = 0; /* no interrupt in half */
	EDMA.TCD[0].INT_MAJ = 0; /* no interrupt in major loop completion */
	EDMA.TCD[0].CITERE_LINK = 0; /* no link after minor loop */
	EDMA.TCD[0].BITERE_LINK = 0;
	EDMA.TCD[0].MAJORE_LINK = 1; /* Linking the next channel to process buffering after major loop */
	EDMA.TCD[0].BITERLINKCH = 0;	
	EDMA.TCD[0].CITERLINKCH = 0;	
	EDMA.TCD[0].MAJORLINKCH = 1;	/* Link to channel 1 */
	EDMA.TCD[0].E_SG = 0;
	EDMA.TCD[0].BWC = 0; /* Default bandwidth control- no stalls */
	EDMA.TCD[0].START = 0; /* Initialize status flags */
	EDMA.TCD[0].DONE = 0;
	EDMA.TCD[0].ACTIVE = 0;
	/* Setup the TCD for channel 1 --> buffer for SCI Tx */
	EDMA.TCD[1].SADDR = (vuint32_t)(stringBuffer+1); /* Load data from the buffer (start with 2nd element) */
	EDMA.TCD[1].SSIZE = 0; /* Read 2**0 = 1 byte per transfer */
	EDMA.TCD[1].SOFF = 1; /* After transfer, increment 1 */
	EDMA.TCD[1].SLAST = 0; /* After major loop no change */
	EDMA.TCD[1].SMOD = 0; /* Source modulo feature not used */
	EDMA.TCD[1].DADDR = (vuint32_t)pTxData; /* writing to PCR 18 (GPIO-Tx) */
	EDMA.TCD[1].DSIZE =  0; /* Write 2**0 = 1 byte per transfer */
	EDMA.TCD[1].DOFF = 0; /* Do not increment destination addr */
	EDMA.TCD[1].DLAST_SGA = 0; /* After major loop no change to destination addr */
	EDMA.TCD[1].DMOD = 0; /* Destination modulo feature not used */
	EDMA.TCD[1].NBYTES = 1; /* Transfer 1 byte per minor loop */
	EDMA.TCD[1].BITER = 1; /* 1 iterations of major loop */
	EDMA.TCD[1].CITER = 1; /* Initialize current iteraction count */
	EDMA.TCD[1].D_REQ = 0; /* Disable channel when major loop is done (sending the buffer once) */
	EDMA.TCD[1].INT_HALF = 0; /* no interrupt in half */
	EDMA.TCD[1].INT_MAJ = 0; /* no interrupt in major loop completion */
	EDMA.TCD[1].CITERE_LINK = 0; /* Linking disabled */
	EDMA.TCD[1].BITERE_LINK = 0;
	EDMA.TCD[1].BITERLINKCH = 0;
	EDMA.TCD[1].CITERLINKCH = 0;
	EDMA.TCD[1].MAJORE_LINK = 1;
	EDMA.TCD[1].MAJORLINKCH = 2;	/* Link to channel 2 */
	EDMA.TCD[1].E_SG = 0;
	EDMA.TCD[1].BWC = 0; /* Default bandwidth control- no stalls */
	EDMA.TCD[1].START = 0; /* Initialize status flags */
	EDMA.TCD[1].DONE = 0;
	EDMA.TCD[1].ACTIVE = 0;
	/* Setup the TCD for channel 2 --> send the word to be sent also to the comparator */
	EDMA.TCD[2].SADDR = (vuint32_t)pTxData; /* Load data prepared for Tx */
	EDMA.TCD[2].SSIZE = 0; /* Read 2**0 = 1 byte per transfer */
	EDMA.TCD[2].SOFF = 0; /* After transfer no change */
	EDMA.TCD[2].SLAST = 0; /* After major loop no change */
	EDMA.TCD[2].SMOD = 0; /* Source modulo feature not used */
	EDMA.TCD[2].DADDR = (vuint32_t)(&(EMIOS_0.CH[0].A.R))+3; /* writing to compare register (bytewise only.. +3) */
	EDMA.TCD[2].DSIZE =  0; /* Write 2**0 = 1 byte per transfer */
	EDMA.TCD[2].DOFF = 0; /* Do not increment destination addr */
	EDMA.TCD[2].DLAST_SGA = 0; /* After major loop no change to destination addr */
	EDMA.TCD[2].DMOD = 0; /* Destination modulo feature not used */
	EDMA.TCD[2].NBYTES = 1; /* Transfer 1 byte per minor loop */
	EDMA.TCD[2].BITER = 1; /* 12 iterations of major loop */
	EDMA.TCD[2].CITER = 1; /* Initialize current iteraction count, (start with 2nd data!) */
	EDMA.TCD[2].D_REQ = 0; /* Disable channel when major loop is done (sending the buffer once) */
	EDMA.TCD[2].INT_HALF = 0; /* no interrupt in half */
	EDMA.TCD[2].INT_MAJ = 0; /* no interrupt in major loop completion */
	EDMA.TCD[2].CITERE_LINK = 0; /* Linking disabled */
	EDMA.TCD[2].BITERE_LINK = 0;
	EDMA.TCD[2].BITERLINKCH = 0;
	EDMA.TCD[2].CITERLINKCH = 0;
	EDMA.TCD[2].MAJORE_LINK = 0;
	EDMA.TCD[2].MAJORLINKCH = 0;
	EDMA.TCD[2].E_SG = 0;
	EDMA.TCD[2].BWC = 0; /* Default bandwidth control- no stalls */
	EDMA.TCD[2].START = 0; /* Initialize status flags */
	EDMA.TCD[2].DONE = 0;
	EDMA.TCD[2].ACTIVE = 0;
	/* Setup the TCD for channel 3 --> disabling PIT0 to stop transmission on stop character ('.') */	
	EDMA.TCD[3].SADDR = (vuint32_t)(&toggleTimer); /* memory structure to enable/disable PIT0 */
	EDMA.TCD[3].SSIZE = 2; /* Read 2**2 = 4 byte per transfer */
	EDMA.TCD[3].SOFF = 0; /* After transfer no increment */
	EDMA.TCD[3].SLAST = 0; /* After major loop no change */
	EDMA.TCD[3].SMOD = 0; /* Source modulo feature not used */
	EDMA.TCD[3].DADDR = (vuint32_t)(&(PIT.CH[0].TCTRL.R)); /* PIT_0 - enable */
	EDMA.TCD[3].DSIZE =  2; /* Write 2**2 = 4 byte per transfer */
	EDMA.TCD[3].DOFF = 0; /* Do not increment destination addr */
	EDMA.TCD[3].DLAST_SGA = 0; /* After major no change to destination addr */
	EDMA.TCD[3].DMOD = 0; /* Destination modulo feature not used */
	EDMA.TCD[3].NBYTES = 4; /* Transfer 4 byte per minor loop */
	EDMA.TCD[3].BITER = 1; /* 1 iterations of major loop */
	EDMA.TCD[3].CITER = 1; /* Initialize current iteraction count, (start with 2nd data!) */
	EDMA.TCD[3].D_REQ = 0; /* Disable channel when major loop is done (sending the buffer once) */
	EDMA.TCD[3].INT_HALF = 0; /* no interrupt in half */
	EDMA.TCD[3].INT_MAJ = 0; /* no interrupt in major loop completion */
	EDMA.TCD[3].CITERE_LINK = 0; /* Linking disabled */
	EDMA.TCD[3].BITERE_LINK = 0;
	EDMA.TCD[3].BITERLINKCH = 0;
	EDMA.TCD[3].CITERLINKCH = 0;
	EDMA.TCD[3].MAJORE_LINK = 0; 
	EDMA.TCD[3].MAJORLINKCH = 0; 
	EDMA.TCD[3].E_SG = 0;
	EDMA.TCD[3].BWC = 0; /* Default bandwidth control- no stalls */
	EDMA.TCD[3].START = 0; /* Initialize status flags */
	EDMA.TCD[3].DONE = 0;
	EDMA.TCD[3].ACTIVE = 0;

	/* proceed with eMIOS magic */
	EMIOS_0.CH[0].CCR.B.MODE=0; /* go to the GPIO mode to be able to write to CNT */	
	EMIOS_0.MCR.B.GTBE=0; /* global time base out disable */
	EMIOS_0.MCR.B.GPREN=1; /* disable prescaler */
	EMIOS_0.CH[0].CSR.B.FLAG=1; /* reset the flag */
	EMIOS_0.CH[0].C.R=(unsigned long)'.'; /* period is the sign for end of transmission */
	EMIOS_0.CH[0].CCR.B.MODE=3; /* go back to the SAOC mode to use the comparator */	


	/* route DMA source to always enabled channel and enable periodic triggers */
	DMAMUX.CHCONFIG[0].R =0xC0 | 0x1F; /* only PIT0 -- always enabled channel */
	DMAMUX.CHCONFIG[3].R =0x80 | 0x0D; /* eMIOS0, channel 0 */

}

/***************************************************************************//*!
*
* @brief	Transmits the message "Hello World   etc" from the designated buffer on serial line.
* 			This message could have variable lenght, the end of the message is indicated by the end
* 			character '.'.
*
*	Data transmission is started enabling the DMA channel 0, which transmits one byte.
*			After the transmission, links to the channel 1, which pops other byte from
*			transmitting buffer. Triggering for channel 0 is dissabled by channel 3,
*			when a stop character found '.'
*
*	@ingroup sci_tx
*
******************************************************************************/
static void SciRun04(void)
{
	/* SETUP the first data to be sent */
	*pTxData = stringBuffer[0];	/* 8 data bits: PH0 - PH8 */
	SIU.GPDO[111].R = 1; /* STOP bit:  PH9 */
	SIU.GPDO[120].R = 0; /* START bit: PG15 */

	EDMA.SERQR.R = 3; 		/* enable DMA channel 3 -- comparator check */
 	
 	/* setup the line to logical high -- to get the first start bit right */
 	SIU.GPDO[18].R = 1;

  	PIT.CH[0].TFLG.B.TIF=1;	 /* clear the timer flag */
 	PIT.CH[0].TCTRL.B.TEN=1; /* and now start the timer */

	EDMA.SERQR.R = 0; 		/* start DMA channel 0 */

  for (;;) 
  {
  }
   

}
