/*******************************************************************************
*
*	Freescale Semiconductor Inc.
*	(c) Copyright 2010 Freescale Semiconductor Inc.
*	ALL RIGHTS RESERVED.
*
****************************************************************************//*!
*
*	@file spi_app_02.c
*	
*	@version 0.0.2.0
*	
*	@lastmodusr B16958
*	
*	@lastmoddate May-31-2010
*	
********************************************************************************
*
*	@brief eDMA emulating SPI slave
*
*	This example demonstates how the eDMA could be used to emulate SPI slave,
*	sending and recieving data
*	Configuration:
*		- DMA channel 0 SCK activated by SPI master (eMIOS0-CH0-SAIC)
*		- DMA channel 1 processes data in
*		- DMA channel 2 processes data out
*		- CS is solved by SW, but could be as an additional DMA channel
*		- Connection: Connect DSPI to GPIO used for eDMA spi emulation
*			- PC4: DSPI1_SIN  -> PB0 (DMA SOUT)
*			- PC5: DSPI1_SOUT -> PB1 (DMA SIN)
*			- PC2: DSPI1_SCK  -> PA0 (DMA SCK)
*			- PC3: DSPI1_CS0  -> PB3 (DMA CS0)
*	
*	@project AN4147
*	
*	@author B16958
*	
*	@ingroup spi_slave
*
******************************************************************************/

/******************************************************************************
* Includes
******************************************************************************/
#include "spi_app_02.h"
#include "MPC560xB.h"


/******************************************************************************
* Internal variable Definitions
******************************************************************************/
static  unsigned char *pInData=((unsigned char *)&SIU.PGPDO[3].R)+1; /*!< Pointer 
* to the input data accessing parallel GPIO registers. */
static  unsigned char *pOutData=((unsigned char *)&SIU.PGPDO[3].R)+2;/*!< Pointer 
* to the output data accessing parallel GPIO registers. */
static  unsigned char startDmaChannel[2]=
{
	2, /* CPHA=1, first write = DMA channel 2 */
	1  /* then read = DMA channel 1*/
};
/*!< This array is used to explicitly start DMA channels for data reading and data writing. 
Note, that the order depends on the selected SPI properties CPHA and CPOL, which in our example 
equal to 1 and 0 respectively. (upon the first edge sending, i.e. starting channel 2) */

/******************************************************************************
* Internal Function Declarations
******************************************************************************/


static void SpiInit02(void);

static void SpiRun02(void);


/******************************************************************************
* External Function Definitions
******************************************************************************/
extern void SpiApp02(void)
{
	SpiInit02();
	
	SpiRun02();
}

/******************************************************************************
* Internal Function Definitions
******************************************************************************/

/***************************************************************************//*!
*
* @brief Initializes the SPI slave using eDMA and also the DSPI peripheral
*			as SPI master to check the data
*
*	Initialization of the following peripherals:
*		- SIU: PCR registers for DMA SPI master and DSPI SPI slave
*		- DSPI: spi master, 8 bit data.
*		- eMIOS: channel 0 as input capture unit (need DMA trigger, 5607B doesn't mux SIU to DMA)
*		- DMAMUX: Routes eMIOS0_CH0 to activate channel 0 DMA
*		- eDMA0: Setup to alternately trigger:
*			-channel_1 to recieve data (after iteration)
*			-channel_2 to transmit data (after completion)
*		- eDMA1: Setup one byte transfer from GPIO to PH0-PH7 (reading)
*		- eDMA2: Setup one byte transfer from PH8-PH15 to GPIO (writing)
*
*	@ingroup spi_slave
*
******************************************************************************/
static void SpiInit02(void)
{
	/* Initialize PCR registers */
	/* DMA SPI */
	 SIU.PCR[16].B.PA = 0;          /* PB0: DMA SPI SOUT */
	 SIU.PCR[16].B.OBE = 1;         /* Output buffer enable */
	 SIU.PCR[17].B.PA = 0;          /* PB1: DMA SPI SIN */
	 SIU.PCR[17].B.IBE = 1;         /* Input buffer enable */
	 SIU.PCR[0].B.PA = 1;          /* PA0: DMA SPI SCK */
	 SIU.PCR[0].B.IBE = 1;         /* Input buffer enable */
	 SIU.PCR[19].B.PA = 0;          /* PB3: DMA SPI CS */
	 SIU.PCR[19].B.IBE = 1;         /* Input buffer enable */
	 
	 /*  DSPI */
	 SIU.PCR[34].B.PA = 1;          /* PC[2]: DSPI1 SCK */
	 SIU.PCR[34].B.OBE = 1;         /* Output buffer enable */
	 SIU.PCR[35].B.PA = 1;          /* PC[3]: DSPI1 CS0_1 */
	 SIU.PCR[35].B.OBE = 1;         /* Output buffer enable */
	 SIU.PCR[36].B.PA = 0;          /* PC[4]: DSPI1 SIN_1 */
	 SIU.PCR[36].B.IBE = 1;         /* Input buffer enable */
	 SIU.PCR[37].B.PA = 1;          /* PC[5]: DSPI1 SOUT_1 */
	 SIU.PCR[37].B.OBE = 1;         /* Output buffer enable */
	
	/* Initialize the eMIOS0_ch0 for single action input capture */
	EMIOS_0.MCR.B.MDIS=0;
	EMIOS_0.MCR.B.GTBE=1; /* global time base out enable*/
	EMIOS_0.MCR.B.GPREN=1; /* enable prescaler */
	EMIOS_0.MCR.B.GPRE=0; /* presclaer 1:1 */
	EMIOS_0.OUDIS.R=0; /* enable all outputs */
	EMIOS_0.UCDIS.R=0; /* enable all channels */
	EMIOS_0.CH[0].A.R=833;
	EMIOS_0.CH[0].CCR.B.UCPRE=0; /* 1:1 */
	EMIOS_0.CH[0].CCR.B.UCPEN=1; /* enable prescaler */
	EMIOS_0.CH[0].CCR.B.DMA=1;	 /* DMA, not interrupt */
	EMIOS_0.CH[0].CCR.B.FCK=1; /* use main clock */
	EMIOS_0.CH[0].CCR.B.FEN=1; /* enable Flag -> DMA */
	EMIOS_0.CH[0].CCR.B.BSL=3; /* all channels internal counter bus */
	EMIOS_0.CH[0].CCR.B.EDSEL=1; /* saoc: output flip-flop toggled */
	EMIOS_0.CH[0].CCR.B.EDPOL=0; /* falling edge */
	EMIOS_0.CH[0].CCR.B.MODE=2; /* Single Action Input Capture */	

	/* DSPI settings as slave device */
	DSPI_1.MCR.R = 0x81013C00;		/* SPI, Master, PCS0 active low */
	DSPI_1.MCR.R |= 0x00003000;		/* Disable both FIFOs */
	DSPI_1.CTAR[0].R = 0x3A000008;	/* 8 bit, CPHA,CPOL= 1,0 */

	/* Initialize the DMA channel 0 -- SPI clock */
	DMAMUX.CHCONFIG[0].R =0x00; /* disable the channel activation source */
	/* Setup the TCD for channel 0 --> single SCI Tx */
	EDMA.TCD[0].SADDR = (vuint32_t)( &(startDmaChannel)); /* Generate Clock structure, 0&1 */
	EDMA.TCD[0].SSIZE = 0; /* Read 2**0 = 1 byte per transfer */
	EDMA.TCD[0].SOFF = 1; /* After transfer, decrement 1 */
	EDMA.TCD[0].SLAST = -2; /* After major loop, back to the beginning (2 iterations) */
	EDMA.TCD[0].SMOD = 0; /* Source modulo feature not used */
	EDMA.TCD[0].DADDR = (vuint32_t) &(EDMA.SSBR.R); /* starting DMA channel */
	EDMA.TCD[0].DSIZE =  0; /* Write 2**0 = 1 byte per transfer */
	EDMA.TCD[0].DOFF = 0; /* Do not increment destination addr */
	EDMA.TCD[0].DLAST_SGA = 0; /* After major loop no change to destination addr */
	EDMA.TCD[0].DMOD = 0; /* Destination modulo feature not used */
	EDMA.TCD[0].NBYTES = 1; /* Transfer 1 byte per minor loop */
	EDMA.TCD[0].BITER = 2; /* 2 iterations of major loop */
	EDMA.TCD[0].CITER = 2; /* Initialize current iteraction count */
	EDMA.TCD[0].D_REQ = 0; /* Disable channel when major loop is done (sending one byte only) */
	EDMA.TCD[0].INT_HALF = 0; /* no interrupt in half */
	EDMA.TCD[0].INT_MAJ = 0; /* no interrupt in major loop completion */
	EDMA.TCD[0].CITERE_LINK = 0; /* no linking */
	EDMA.TCD[0].BITERE_LINK = 0;
	EDMA.TCD[0].MAJORE_LINK = 0; 
	EDMA.TCD[0].BITERLINKCH = 0;
	EDMA.TCD[0].CITERLINKCH = 0;	
	EDMA.TCD[0].MAJORLINKCH = 0;
	EDMA.TCD[0].E_SG = 0;
	EDMA.TCD[0].BWC = 0; /* Default bandwidth control- no stalls */
	EDMA.TCD[0].START = 0; /* Initialize status flags */
	EDMA.TCD[0].DONE = 0;
	EDMA.TCD[0].ACTIVE = 0;
	/* Setup the TCD for channel 1 -- SPI data IN */
	EDMA.TCD[1].SADDR = (vuint32_t)&(SIU.GPDI[17].R); /* Load data from SIN pin PB1 */
	EDMA.TCD[1].SSIZE = 0; /* Read 2**0 = 1 byte per transfer */
	EDMA.TCD[1].SOFF = 0; /* After transfer no change */
	EDMA.TCD[1].SLAST = 0; /* After major loop no change */
	EDMA.TCD[1].SMOD = 0; /* Source modulo feature not used */
	EDMA.TCD[1].DADDR = (vuint32_t)( &(SIU.GPDO[104].R)); /* writing to internal (GPDO bit-buffer) */
	EDMA.TCD[1].DSIZE =  0; /* Write 2**0 = 1 byte per transfer */
	EDMA.TCD[1].DOFF = 1; /* Do increment destination addr */
	EDMA.TCD[1].DLAST_SGA = -8; /* After major loop go back 8 bytes to destination addr */
	EDMA.TCD[1].DMOD = 0; /* Destination modulo feature not used */
	EDMA.TCD[1].NBYTES = 1; /* Transfer 1 byte per minor loop */
	EDMA.TCD[1].BITER = 8; /* 12 iterations of major loop */
	EDMA.TCD[1].CITER = 8; /* Initialize current iteraction count, (start with 2nd data!) */
	EDMA.TCD[1].D_REQ = 0; /* Disable channel when major loop is done (sending the buffer once) */
	EDMA.TCD[1].INT_HALF = 0; /* no interrupt in half */
	EDMA.TCD[1].INT_MAJ = 0; /* no interrupt in major loop completion */
	EDMA.TCD[1].CITERE_LINK = 0; /* Linking disabled */
	EDMA.TCD[1].BITERE_LINK = 0;
	EDMA.TCD[1].BITERLINKCH = 0;
	EDMA.TCD[1].CITERLINKCH = 0;
	EDMA.TCD[1].MAJORE_LINK = 0;/* Link after major loop to disable further transmission */
	EDMA.TCD[1].MAJORLINKCH = 0; /* after completion: link to channel 3 */	
	EDMA.TCD[1].E_SG = 0;
	EDMA.TCD[1].BWC = 0; /* Default bandwidth control- no stalls */
	EDMA.TCD[1].START = 0; /* Initialize status flags */
	EDMA.TCD[1].DONE = 0;
	EDMA.TCD[1].ACTIVE = 0;
	/* Setup the TCD for channel 2 -- SPI data OUT */
	EDMA.TCD[2].SADDR = (vuint32_t)( &(SIU.GPDO[112].R)); /* read from internal (GPDO) bit buffer */
	EDMA.TCD[2].SSIZE = 0; /* Read 2**0 = 1 byte per transfer */
	EDMA.TCD[2].SOFF = 1; /* After transfer no change */
	EDMA.TCD[2].SLAST = -8; /* After major loop no change */
	EDMA.TCD[2].SMOD = 0; /* Source modulo feature not used */
	EDMA.TCD[2].DADDR = (vuint32_t)&(SIU.GPDO[16].R); /* writing to PCR 17 (SOUT=PB0) */
	EDMA.TCD[2].DSIZE =  0; /* Write 2**0 = 1 byte per transfer */
	EDMA.TCD[2].DOFF = 0; /* Do increment destination addr */
	EDMA.TCD[2].DLAST_SGA = 0; /* After major loop go back 8 bytes to destination addr */
	EDMA.TCD[2].DMOD = 0; /* Destination modulo feature not used */
	EDMA.TCD[2].NBYTES = 1; /* Transfer 1 byte per minor loop */
	EDMA.TCD[2].BITER = 8; /* 12 iterations of major loop */
	EDMA.TCD[2].CITER = 8; /* Initialize current iteraction count, (start with 2nd data!) */
	EDMA.TCD[2].D_REQ = 0; /* Disable channel when major loop is done (sending the buffer once) */
	EDMA.TCD[2].INT_HALF = 0; /* no interrupt in half */
	EDMA.TCD[2].INT_MAJ = 0; /* no interrupt in major loop completion */
	EDMA.TCD[2].CITERE_LINK = 0; /* Linking after minor loop disabled */
	EDMA.TCD[2].BITERE_LINK = 0;
	EDMA.TCD[2].BITERLINKCH = 0;
	EDMA.TCD[2].CITERLINKCH = 0;
	EDMA.TCD[2].MAJORE_LINK = 0; 
	EDMA.TCD[2].MAJORLINKCH = 0;
	EDMA.TCD[2].E_SG = 0;
	EDMA.TCD[2].BWC = 0; /* Default bandwidth control- no stalls */
	EDMA.TCD[2].START = 0; /* Initialize status flags */
	EDMA.TCD[2].DONE = 0;
	EDMA.TCD[2].ACTIVE = 0;

	/* route DMA source eMIOS to get started by external edge */
	DMAMUX.CHCONFIG[0].R =0x80 | 0x0D; /* eMIOS0, channel 0 */

}

/***************************************************************************//*!
*
* @brief	Transmits the message "Hello World!" and other message from
*			SPI master (DSPI_1) to SPI slave (emulated) and back.
*
*	SPI slave is triggered using eMIOS channels (instead of external interrupt)-channel0.
*		This channel launches other two channels (2 and 1), which perform 
*		data transmission and reception.
*
*	@ingroup spi_slave
*
******************************************************************************/
static void SpiRun02(void)
{
volatile unsigned char i;
unsigned char putData[12]="Hello World!";
unsigned char getData[12]="sir,yes sir!";

volatile unsigned char slaveGot[12];
volatile unsigned char masterGot[12];

	/* enable DMA channels 0,1,2,3 */
	EDMA.SERQR.R = 0;
	EDMA.SERQR.R = 1;
	EDMA.SERQR.R = 2;
	EDMA.SERQR.R = 3;


	/* prepare some data */
	for (i=0; i<12; i++)
	{
		/* preparing the data to be sent on slave */
		*pOutData=putData[i];
		
		/* now initiate transmission from the master */
		DSPI_1.PUSHR.R = 0x0C010000 + (unsigned long)getData[i];

		/* waiting till transmission finishes */
		while(DSPI_1.SR.B.TCF == 0) {}; 
		DSPI_1.SR.B.TCF = 1;
		
		/* retrieving the data */
		masterGot[i]=(unsigned char)DSPI_1.POPR.R; /* what was actually received on DSPI master */
		slaveGot[i]=*pInData;					  /* what was sent to the DMA slave */
		
	}
	

  for (;;) 
  {
  	/* check what master and slave received */
  }
   

}
