#include <hidef.h>      /* for EnableInterrupts macro */
#include "derivative.h" /* include peripheral declarations */

#include "em_main.h"
#include "sci_flowtron.h"
#include "rtc_driver_ext.h"
#include "lcd.h"
#include "task_mgr.h"
#include "math.h"
#include "mtim8_timer.h"   

#define AVERAGE_CYCLE 25

#define OFFSET 32767
#define FACT_ACTP 77200
#define FACT_VOLT 10
#define FACT_CURR 77


extern void printf(unsigned char*,...);
extern int sprintf( char * str, const char *format,...);
extern void lcd_PrintString(unsigned char *str);

void fcd_enable_on_lcd_pins(void);
void mcu_init(void);
void pracmp_init(void);
void adc1_init(void);
void adc2_init(void);
void pdb_init(void);
void rtc_init(void);
void gpio_init(void);

uint32_t RMS_calc (int16_t *);
uint32_t Sqrt_r(uint32_t);
void Power_Calc(int16_t *, int16_t *, power_vec_t  *);

void (*ptr_next_task)(void);   // task pointer
unsigned long timer_task;      // timer runs 1 msec
unsigned int  time_to_sec=500;
uint16_t set_timer;

// tasks
void tsk_welcome(void);
void tsk_show_measurements(void);
void tsk_show_date(void);
void tsk_show_time(void);
void tsk_show_tamper(void);
void tsk_show_irms (void);
void tsk_show_vrms (void);
void tsk_show_freq (void);
void tsk_show_kva (void);
void tsk_show_energy (void);
void tsk_lcd_demo (void);
void tsk_lcd_blink (void);
void tsk_rtc_demo (void);
void tsk_show_kw (void); 
void tsk_show_pf (void); 
void measurement (void);


#define UART_PORT 3
#define BUS_CLK  16776195
#define SAMPLE 32
#define BUFFER 2

uint32_t mtim_count;
uint8_t  freq_sample_done;
uint8_t  cycle;
uint16_t result;

uint8_t sample;
uint8_t buffer;
uint8_t buffer_ready;
uint8_t pre_buffer;


emeter_t em;

uint8_t avg_index;
uint16_t voltage_disp_temp;
uint16_t voltage_disp;
uint16_t current_disp_temp;
uint16_t current_disp;
int32_t active_power_temp;
int32_t active_power_disp;
int32_t apparent_power_temp;
int32_t apparent_power_disp;
int32_t power_factor_disp;

int16_t resultA[BUFFER][SAMPLE];
int16_t resultB[BUFFER][SAMPLE];


uint8_t tamper_count;


//------------------------------------------------------------------------
//push button services 
//------------------------------------------------------------------------


unsigned char SW3_CLICK;   //for switch 3
unsigned char SW5_CLICK;   //for switch 5
unsigned char Set_Time;    //The user can set the time when its set to 1
unsigned char Set_Date;    //The user can set the date when its set to 1

unsigned char Count_1sec;  //one sec counter
unsigned char set_mode;    //The time/ date can be set in this mode
unsigned char set_part;    //if time is getting set it decides whether sec/hour/min is getting set
                           //if date is getting set it decides whether day/date/year/month is getting set

unsigned char curr_hour, curr_min, curr_sec; //stores the current values of hour , min , sec respectively

uint16_t curr_year; //stores the current value of year  
uint8_t  curr_date; //stores the current value of date
uint8_t  curr_month; //stores the current value of month
enum weekday_names curr_wday; //stores the current week day
uint8_t disp_string[20]="";   //Variable to store the string to be displayed on LCD

void setLCDDate(void);  //Displays the value of date on LCD
void setLCDClock(void); //Displays the value of clock on LCD
void clearLCDDTx(void); //function to clear the DOTs on LCD
void MTIM_Callback(unsigned char);//callback function in case of MTIM interrupt
void KBI_Init(void); //Initializes the KBI

/* Function to display clock value on LCD*/
void setLCDClock() 
{
  sprintf(disp_string, "   %02d%02d%02d", curr_hour, curr_min, curr_sec);
  lcd_PrintString(disp_string);
}

/*Function to display date value on LCD*/
void setLCDDate() 
{
 switch(curr_wday)
      {
        case 0 :
           sprintf(disp_string,"SUN%02ld%02ld%02ld",(uint32_t)curr_date,(uint32_t)curr_month,(uint32_t)curr_year);
           break;
        case 1 :
           sprintf(disp_string,"MON%02ld%02ld%02ld",(uint32_t)curr_date,(uint32_t)curr_month,(uint32_t)curr_year);
           break;
        case 2 :
           sprintf(disp_string,"TUE%02ld%02ld%02ld",(uint32_t)curr_date,(uint32_t)curr_month,(uint32_t)curr_year);
           break;
        case 3 :
           sprintf(disp_string,"WED%02ld%02ld%02ld",(uint32_t)curr_date,(uint32_t)curr_month,(uint32_t)curr_year);
           break;
        case 4 :
           sprintf(disp_string,"THU%02ld%02ld%02ld",(uint32_t)curr_date,(uint32_t)curr_month,(uint32_t)curr_year);
           break;
        case 5 :
           sprintf(disp_string,"FRI%02ld%02ld%02ld",(uint32_t)curr_date,(uint32_t)curr_month,(uint32_t)curr_year);
           break;
        case 6 :
           sprintf(disp_string,"SAT%02ld%02ld%02ld",(uint32_t)curr_date,(uint32_t)curr_month,(uint32_t)curr_year);
           break;
      }

      LCDClear();
      LCDWriteSegment(FREESCALE,0);
      LCDWriteSegment(COLx,5);
      LCDWriteSegment(COLx,7);
      LCDWriteSegment(DATE,0);
      lcd_PrintString(disp_string);

}

/*function to clear the DOTs on LCD*/
void clearLCDDTx()
{
  unsigned char i;
  
  for(i=0;i<10;i++)
  {
    LCDClearSegment(DTx, i);
  }
}

/*Callback function when MTIM interrupt occurs*/
void MTIM_Callback(unsigned char param)
{
 
   Disable_MTIM1();
  if((!PTDD_PTDD5) & (Set_Time | Set_Date))  //if SW5 is pressed and either of the Set_Time or Set_Date is  set
  {
    Count_1sec++;
    if (Count_1sec == 20)      //if SW5 is pressed for certain time
    {
        Count_1sec = 0;
        
        if(set_mode == 1)        
        { 
          if(Set_Time)          //if Set_Time mode is set
          {  
          setLCDClock();
          LCDWriteSegment(COLx, 5);
          LCDWriteSegment(COLx, 7);
          set_mode = 2;         //clock set mode
          set_part = 0;
          } 
          else if(Set_Date)
          {
            set_mode = 2;       //date set mode
            set_part =0; 
          }
        }
        else if(set_mode == 2)    //while leaving the set mode the values set are stored in RTC
        {
          curr_year += 2000;
          if(rtc_setDateAndTime(curr_year , curr_month, curr_date, curr_wday, curr_hour, curr_min ,curr_sec)) 
          {
           while(1);
          }
          set_mode = 0;
          clearLCDDTx();
        } 
    }
    else                         
    {
      Set_TimeOut_MTIM1(50);     
      Enable_MTIM1();
    }
  }
  else
  {
    if(set_mode == 1)
    {  
      set_mode = 0;
      SW5_CLICK = 1;            
    }
  }
  return;
}

/*Initializes the KBI required to scroll between menus*/
void KBI_Init()
{
  SCGC1_KBI = 1;
  SCGC3_PTD = 1;
  SCGC4_MUXCTRL = 1;
  
  PTDPF3 = 0x11;
  PTDPE_PTDPE5 = 1;
  PTDPE_PTDPE4 = 1;
  
  KBIPE = 0x30;
  KBIES = 0x00;
  
  KBISC = 0x04;
  KBISC_KBIE = 1;
}


//------------------------------------------------------------------------
// RTC ISR callback
//------------------------------------------------------------------------ 
void isr_callback(unsigned int interrupt_mask, unsigned char tamper_mask)
{

  if(tamper_mask & RTC_ISR_TMPR1) 
  {
    tamper_count = tamper_count+ 1;
  }
  return;
}

void gpio_init() 
{
 SCGC3_PTB = 1;
  PTBPF2 = 0x00;
  PTBDD_PTBDD3 = 1;
  PTBDD_PTBDD2 = 1;
  PTBPE_PTBPE3 = 1;
  PTBD_PTBD2 = 1;
  PTBD_PTBD3 = 1;
}

void delay(void)
 {
  unsigned int m;
  for(m=0;m<=65534;m++);
  for(m=0;m<=65534;m++);
  for(m=0;m<=65534;m++);
  for(m=0;m<=65534;m++);
  for(m=0;m<=65534;m++);
  for(m=0;m<=65534;m++);
  for(m=0;m<=65534;m++);
  for(m=0;m<=65534;m++);
 }
 
 void irq_init(void) 
 {
   SCGC2_IRQ = 1;
   PTAPF3_A5 = 4;
   IRQSC_IRQPE = 1;
   IRQSC_IRQIE = 1;
   
 }
   
//------------------------------------------------------------------------
// Main Funciton
//------------------------------------------------------------------------
void main(void) {
 
 
 //unsigned char result;
  EnableInterrupts; /* enable interrupts */

  sample = 0;
  buffer = 0;
  buffer_ready =0;
  freq_sample_done = 0;

  em.powervec.kwh = 0;
 
  mcu_init(); //initializes the MCU by configuring the clock
  
  SCGC3_PTA = 1;
     
  MTIM1_Init(&MTIM_Callback); //initializes the MTIM1 passing the address of MTIM_Callback 
                              //for the function to be called back in case of interrupt 
  
  LCDInit();  //initializes the LCD
  KBI_Init(); //initializes KBI
  gpio_init();
  rtc_init(); //initializes RTC
  tamper_count = StandbyRAM[1];
  
  SCI_Init(SCI_Index0,0);   //initializes SCI 1
 
  
  adc1_init();
  adc2_init();
  pdb_init();
  
  printf("welcome");
  next_task_t(tsk_welcome,3000); // the pointer of the task is pointed towards tsk_welcome
  for(;;) {
  
  ptr_next_task();
  
  measurement();
  switch(set_mode)
  {
    case 1:
      
      Set_TimeOut_MTIM1(50);
      Enable_MTIM1();
      while(set_mode == 1);      
      break;
  } 
 }
                    
}

//------------------------------------------------------------------------
// Measurement routine:
// Calculate the power vectors and update the structure
// Need to calculate the time ??
//------------------------------------------------------------------------
void measurement()
{
   uint8_t active_power_negate;

   if(buffer_ready) {
    
   buffer_ready = 0;  
   Power_Calc(resultB[pre_buffer],resultA[pre_buffer],&em.powervec);
   voltage_disp_temp += em.powervec.v;
   current_disp_temp += em.powervec.a;
   active_power_temp += em.powervec.kw;
   apparent_power_temp +=em.powervec.kva;

   avg_index++;

   if(avg_index == AVERAGE_CYCLE)
     {
       active_power_negate = 0;       
       voltage_disp = voltage_disp_temp/AVERAGE_CYCLE; 
       current_disp = current_disp_temp/AVERAGE_CYCLE;
       
       if(active_power_temp < 0) { 
         active_power_temp = (-1 * active_power_temp);
         active_power_negate = 1;
       }
       
       active_power_disp = (active_power_temp/AVERAGE_CYCLE);
       
       if(active_power_negate) {        
       active_power_disp = (-1 * active_power_disp);
       }
       
       apparent_power_disp = apparent_power_temp/AVERAGE_CYCLE;
       
       if(apparent_power_disp == active_power_disp)
       power_factor_disp = 1;
       else
       power_factor_disp = (int32_t)(active_power_disp*100)/apparent_power_disp;
       
       avg_index = 0;
       voltage_disp_temp = 0;
       current_disp_temp = 0;
       active_power_temp = 0;
       apparent_power_temp = 0;       
     }
   }
}
//------------------------------------------------------------------------
// Task Welcome defination
//------------------------------------------------------------------------
void tsk_welcome(void) 
 {
   uint16_t i;
   for(i=0;i<60000;i++);
    _asm(nop);
   LCDClear();
   LCDWriteSegment(FREESCALE,0);
   lcd_PrintString("GW64 Demo");
   if (SW3_CLICK) 
   {
    SW3_CLICK = 0;
   next_task_t(tsk_show_measurements,10000);
   }
   if (SW5_CLICK) 
   {
    SW5_CLICK = 0;
    next_task_t(tsk_show_measurements,10000);
   }
     
  } 
//------------------------------------------------------------------------
// Task Measurement defination
//------------------------------------------------------------------------
void tsk_show_measurements () {
   uint16_t i;
  
   for(i=0;i<60000;i++);
    _asm(nop);
   LCDClear();
   LCDWriteSegment(FREESCALE,0);
   lcd_PrintString(" E METER ");
   em.powervec.kwh =0;
   if(SW3_CLICK) 
   {
    SW3_CLICK = 0;
    next_task_t(tsk_rtc_demo,10000);
   }
   if(SW5_CLICK) {
    SW5_CLICK = 0 ;
    next_task_t(tsk_show_irms,10000);
   }
}
//------------------------------------------------------------------------
// Task show IRMS Measurement defination
//------------------------------------------------------------------------
void tsk_show_irms () {

uint8_t disp_string[10];
  
if(current_disp < 15)        //any current value below 1.5 is 0
   current_disp = 0;

if(current_disp > 300)
  current_disp = 300;        //any current above 30 is 30 
 
  sprintf(disp_string,"IRMS  %3ld",(uint32_t)current_disp);
  
  LCDClear();
  LCDWriteSegment(FREESCALE,0);
  LCDWriteSegment(AMPS,0);
  LCDWriteSegment(DTx,8);
  lcd_PrintString(disp_string);

  if(SW3_CLICK) 
  { SW3_CLICK = 0;
    next_task_t(tsk_welcome,10000);
  }
  if(SW5_CLICK) 
  {
   SW5_CLICK = 0; 
    next_task_t(tsk_show_vrms,10000);
  }
}
//------------------------------------------------------------------------
// Task show VRMS Measurement defination
//------------------------------------------------------------------------
void tsk_show_vrms () {

 uint8_t disp_string[10];
 
  if(voltage_disp <55 )
  voltage_disp =0;
  
  if(voltage_disp > 2200)
  voltage_disp = 2300;
  
  sprintf(disp_string,"VRMS %4ld",(uint32_t)voltage_disp);
  LCDClear();
  LCDWriteSegment(FREESCALE,0);
  LCDWriteSegment(VOLTS,0);
  lcd_PrintString(disp_string);
  LCDWriteSegment(DTx,8);

  if(SW3_CLICK) 
  { 
    SW3_CLICK = 0;
    next_task_t(tsk_rtc_demo,10000);
  }
  if(SW5_CLICK)
  { 
    SW5_CLICK =0;
    next_task_t(tsk_show_kw,10000);
  }
}
//------------------------------------------------------------------------
// Task show power Measurement defination
//------------------------------------------------------------------------
void tsk_show_kva () {

  uint8_t disp_string[10];
  
  if((voltage_disp < 55) || (current_disp < 15))
  apparent_power_disp =0;
  
  sprintf(disp_string,"APP%5ldW",apparent_power_disp);
  
  LCDClear();
  LCDWriteSegment(FREESCALE,0);
  lcd_PrintString(disp_string);
 
  if(SW3_CLICK)
  {
   SW3_CLICK =0; 
   next_task_t(tsk_rtc_demo,10000);
  }
  if(SW5_CLICK)
  {
   SW5_CLICK =0;
   next_task_t(tsk_show_pf,10000);
  }
}
//------------------------------------------------------------------------
// Task show power Measurement defination
//------------------------------------------------------------------------
void tsk_show_kw () 
{

  uint8_t disp_string[10];
  if((voltage_disp < 55) || (current_disp < 15))
  active_power_disp =0;
  
  sprintf(disp_string,"ACT%5ldW",active_power_disp);
  
  LCDClear();
  LCDWriteSegment(FREESCALE,0);
  lcd_PrintString(disp_string);
 
  if(SW3_CLICK) 
  {
    SW3_CLICK =0;
    next_task_t(tsk_rtc_demo,10000);
  }
  if(SW5_CLICK) 
  {
    SW5_CLICK =0;
    next_task_t(tsk_show_kva,10000);
  }
}
//------------------------------------------------------------------------
// Task show power Measurement defination
//------------------------------------------------------------------------
void tsk_show_pf () 
  
{
  uint8_t disp_string[10];
  if((voltage_disp < 55) || (current_disp < 15))
  power_factor_disp =0;
  if(power_factor_disp > 100)
  power_factor_disp = 1;
  
  if(power_factor_disp < -100)
  power_factor_disp = -1;
  
  sprintf(disp_string,"PF   %04ld",power_factor_disp);
  
  LCDClear();
  LCDWriteSegment(FREESCALE,0);
  
  lcd_PrintString(disp_string);
  LCDWriteSegment(DTx,7);
  if(SW3_CLICK) 
  {
    SW3_CLICK =0;
    next_task_t(tsk_rtc_demo,10000);
  }
  if(SW5_CLICK) 
  {
    SW5_CLICK =0;
    next_task_t(tsk_show_energy,10000);
  }
}
//------------------------------------------------------------------------
// Task show energy Measurement defination
//------------------------------------------------------------------------
void tsk_show_energy () {


  uint8_t disp_string[10];
//  int32_t energy_disp;

 // energy_disp = (em.powervec.kwh/3600);   // display energy upto 3 decimal stored in wh

  sprintf(disp_string,"E %07ld",(em.powervec.kwh/3600));
  LCDClear();
  lcd_PrintString(disp_string);
  LCDWriteSegment(FREESCALE,0);
  LCDWriteSegment(KWH,0);
  LCDWriteSegment(DTx,6);

  
  if(SW3_CLICK) 
  {
    SW3_CLICK =0;
     next_task_t(tsk_rtc_demo,20000);
  }
  
  if(SW5_CLICK) 
  {
    SW5_CLICK =0;
    next_task_t(tsk_show_irms,20000);
  }
}
//------------------------------------------------------------------------
// Task show Freq Measurement defination
//------------------------------------------------------------------------
void tsk_show_freq() 
{
  uint32_t freq;
  uint8_t disp_string[10];
  if(freq_sample_done == 1) {      
    freq_sample_done = 0;
    
    // Calculate and display freq 
    freq = (uint32_t)((BUS_CLK*10)/(32*mtim_count));
  }

  if(freq > 300 & freq <700)       
    {
        sprintf(disp_string,"FR %3ld Hz",freq);
  
        LCDClear();
        LCDWriteSegment(FREESCALE,0);
        lcd_PrintString(disp_string);
   }
  
  LCDWriteSegment(DTx,6);
  lcd_PrintString("FR  507Hz");

  if(SW3_CLICK) 
  {
    SW3_CLICK =0;
    next_task_t(tsk_rtc_demo,10000);
  }
  
  if(SW5_CLICK) 
  {
    SW3_CLICK =0;
    next_task_t(tsk_show_energy,10000);
  }
}
//------------------------------------------------------------------------
// Task show date
//------------------------------------------------------------------------
void tsk_rtc_demo () 
{
  uint16_t i;
 
  for(i=0;i<60000;i++);
   _asm(nop);
  LCDClear();
  LCDWriteSegment(FREESCALE,0);
  lcd_PrintString("RTC DEMO");
  if(SW3_CLICK) 
  {
    SW3_CLICK =0;
    next_task_t(tsk_lcd_demo,10000);
  }
  if(SW5_CLICK) 
  {
    SW5_CLICK =0;
    next_task_t(tsk_show_date,10000);
  }
}


//------------------------------------------------------------------------
// Task show date
//------------------------------------------------------------------------
void tsk_show_date ()
{ 
  Set_Date =1;
  if(set_mode != 2) 
  {
    
      curr_year = (rtc_getYear()-2000);
      
      curr_month = rtc_getMonth();
      rtc_getDayAndDate(&curr_wday, &curr_date);
      switch(curr_wday)
      {
        case 0 :
           sprintf(disp_string,"SUN%02ld%02ld%02ld",(uint32_t)curr_date,(uint32_t)curr_month,(uint32_t)curr_year);
           break;
        case 1 :
           sprintf(disp_string,"MON%02ld%02ld%02ld",(uint32_t)curr_date,(uint32_t)curr_month,(uint32_t)curr_year);
           break;
        case 2 :
           sprintf(disp_string,"TUE%02ld%02ld%02ld",(uint32_t)curr_date,(uint32_t)curr_month,(uint32_t)curr_year);
           break;
        case 3 :
           sprintf(disp_string,"WED%02ld%02ld%02ld",(uint32_t)curr_date,(uint32_t)curr_month,(uint32_t)curr_year);
           break;
        case 4 :
           sprintf(disp_string,"THU%02ld%02ld%02ld",(uint32_t)curr_date,(uint32_t)curr_month,(uint32_t)curr_year);
           break;
        case 5 :
           sprintf(disp_string,"FRI%02ld%02ld%02ld",(uint32_t)curr_date,(uint32_t)curr_month,(uint32_t)curr_year);
           break;
        case 6 :
           sprintf(disp_string,"SAT%02ld%02ld%02ld",(uint32_t)curr_date,(uint32_t)curr_month,(uint32_t)curr_year);
           break;
      }

      LCDClear();
      LCDWriteSegment(FREESCALE,0);
      LCDWriteSegment(COLx,5);
      LCDWriteSegment(COLx,7);
      LCDWriteSegment(DATE,0);
      lcd_PrintString(disp_string);
  }
  else 
  {
     unsigned char digit = 3 + (2*set_part);

      
          LCDWriteSegment(DTx, digit-2);
          LCDWriteSegment(DTx, digit);
  }


  if(SW3_CLICK) 
  {
    SW3_CLICK =0;
    Set_Date =0;
    next_task_t(tsk_lcd_demo,10000);
  }  
  if(SW5_CLICK) 
  {
    SW5_CLICK = 0;
    Set_Date =0;
    next_task_t(tsk_show_time,10000);
  }
}
//------------------------------------------------------------------------
// Task show time 
//------------------------------------------------------------------------
void tsk_show_time() {

Set_Time =1;
 if(set_mode != 2)
    {
      curr_hour = rtc_getHour();
      curr_min = rtc_getMin();
      curr_sec = rtc_getSeconds();
      sprintf(disp_string, "   %02d%02d%02d", curr_hour, curr_min, curr_sec);

      lcd_PrintString(disp_string);
      if(curr_sec%2)
      {
        LCDWriteSegment(COLx, 5);
        LCDWriteSegment(COLx, 7);
      }
      else
      {
        LCDClearSegment(COLx, 5);
        LCDClearSegment(COLx, 7);
      }
      (void)rtc_clearInterrupt(RTC_ISR_FRE1HZ);

    } 
    
 else 
   {
    
     unsigned char digit = 4 + (2*set_part), sec = rtc_getSeconds();

        if(sec%2)
        {
          LCDWriteSegment(DTx, digit-1);
          LCDWriteSegment(DTx, digit);
          LCDWriteSegment(DTx, digit+1);
        }
        else
        {
          LCDClearSegment(DTx, digit-1);
          LCDClearSegment(DTx, digit);
          LCDClearSegment(DTx, digit+1);
        } 
   }
  
 
      
   if(SW3_CLICK)    
   {
    SW3_CLICK=0;
    Set_Time=0;    
    next_task_t(tsk_lcd_demo,10000); 
   }
   
   if(SW5_CLICK)
   {
     SW5_CLICK = 0;
     Set_Time=0;
     next_task_t(tsk_show_tamper,10000);
   }
}

//------------------------------------------------------------------------
// Task show tamper 
//------------------------------------------------------------------------
void tsk_show_tamper() {

  uint8_t disp_string[10];

  uint16_t i;
  for(i=0;i<10000;i++);
   _asm(nop);

  LCDClear(); 
  LCDWriteSegment(FREESCALE,0);
  sprintf(disp_string,"TAMPER%3ld",(uint32_t)tamper_count);

  LCDWriteSegment(SNOWFLAKE,0);
  lcd_PrintString(disp_string);

  if(tamper_count != StandbyRAM[1])
   {
    irtc_unprotectRegisters();    
    StandbyRAM[1] = (uint8_t)(tamper_count);
   }   
   
   if(SW3_CLICK) 
   {
    SW3_CLICK =0;
    next_task_t(tsk_lcd_demo,10000);
   }
   if(SW5_CLICK) 
   {
     SW5_CLICK =0;
    next_task_t(tsk_show_date,10000);
   }
}

//------------------------------------------------------------------------
// Task show LCD 
//------------------------------------------------------------------------
void tsk_lcd_demo() 
{
  uint16_t i;
  for(i=0;i<60000;i++);
   _asm(nop);
  LCDClear();
  LCDWriteSegment(FREESCALE,0);
  lcd_PrintString("LCD Demo");
  if(SW3_CLICK) 
  {
    SW3_CLICK =0;
    next_task_t(tsk_welcome,10000);
  }
  if(SW5_CLICK)
  { 
    SW5_CLICK = 0; 
    LCDBlinkState();
    next_task_t(tsk_lcd_blink,10000);
  }
}
//------------------------------------------------------------------------
// task blink LED
//------------------------------------------------------------------------
void tsk_lcd_blink ()
{ 
  lcd_PrintString("LCD BLINK");       
  if(SW3_CLICK){
    SW3_CLICK =0;
    LCDOnOffState();
    next_task_t(tsk_welcome,10000);
  }
  if(SW5_CLICK) 
  {
    SW5_CLICK=0;
    next_task_t(tsk_lcd_blink,10000);
  }
}

//------------------------------------------------------------------------
// MCU Init
//------------------------------------------------------------------------
void mcu_init() {
  SOPT1_COPT = 0;

  ICSC1      = 0x06;  
  ICSC2      = 0x00; // 0x80 for 4 Mhz 0x00 for 16 Mhz
  ICSTRM     = *(unsigned char *)0xFFAF; 
  ICSSC      = 0x00;  // 16 Mhz bus clock for 20 Mhz put it 0x20        
  fcd_enable_on_lcd_pins(); 

  SCGC4_MUXCTRL=1;
  
  // Enable all port control clock
  SCGC3      = 0xff;
}
//------------------------------------------------------------------------
// RTC Init
//------------------------------------------------------------------------
void rtc_init() 
{
 (void)rtc_setConfig(RTC_CLKOUT_1HZ, RTC_DAYLIGHTSAVINGS_DISABLE, 0, &isr_callback);
 rtc_setTamper1Config(1,0,63,1);
 
 if(StandbyRAM[0] != 0xFF)
  { 
  if(!rtc_setDateAndTime(2010, February, 22, Monday , 14, 46, 00))
    {
     //success
    }
    irtc_unprotectRegisters();    
    StandbyRAM[0] = 0xFF;
    StandbyRAM[1] = 0x00;
  }
  
}

//------------------------------------------------------------------------
// ADC init 
// System clock is 16 Mhz, use ADCK as bus clock and use prescale /4.
//
// So ADC clock freq = 16/4 = 4Mhz
//
// ADC configuration
//    Long sample time, default long sample time 24 ADCK clock
//    16 Bit single ended mode.
//
// Conversion time = SFC ADDER + AVGNUM(BCT+LSTAdder+HSCAdder)
//
// SFC ADDER = 3 ADCK + 5 bus clock
// AVG NUM   = 1
// BCT       = 25ADCK
// LSTAdder  = 6 ADCK
// HSCAdder  = 0 ADCK
//
// Conversion time  = 34 ADCK + 5 Bus clock
//              
//                  = 8.8Usec     // Worst case can be improved
//
// PDB must generate the trigger afer 50 us.                 
//------------------------------------------------------------------------
void adc1_init() { 
  uint16_t temp; 
  SCGC1_ADC0   = 1;     // Enable clock for ADC
  ADC0CFG1     = 0x5C;  // Bus clock/4 ADLSMP Mode =16 bit input clock
  ADC0CFG2     = 0x03;  // Long sample time
  ADC0SC2      = 0x40;  // Hardware trigger select
  ADC0SC3      = 0x07;   // Hardware average enable with 32 samples
 
  ADC0SC1A     = 0x01;  // Conversion on channel 1 current
  ADC0SC1B     = 0x02;  // Conversion on channel 1 current
  
}

void adc2_init() {
  SCGC1_ADC1   = 1;     // Enable clock for ADC
  ADC1CFG1     = 0x5C;  // Bus clock/4 ADLSMP Mode =12 bit input clock
  ADC1CFG2     = 0x03;  // Long sample time
  ADC1SC2      = 0x40;  // Hardware trigger select
  ADC1SC3      = 0x07;   // Hardware average enable with 32 samples
 
  ADC1SC1A     = 0x00;  // Conversion on channel 0 voltage
  //ADC1SC1B     = 0x02;  // Conversion on channel 1 current

}
//------------------------------------------------------------------------
// PDB init
// System clock 16Mhz. PDB Prescale 128 so PDB clock freq 156.25Khz
//
// For 128 Samples per cycle i.e 20 msec.Trigger should be at 156.25 Usec
// so modulus value should be 1000 i.e 0x3E8
//
// Consider ADC running at 1Mhz. Conversion time will be x
//
// Time Preiod of PDB clock is 6.4Usec.
//
// So DLYA = 0;          = 0x0000; 
//    DLYB = (50+10);    = 0x0009;
//    IDLY = (2*50 +10); = 0x0012;
//------------------------------------------------------------------------
void pdb_init() {
  SCGC4_PDB     = 1;   // Enable the clock of the PDB block
  
  PDBMOD        = 5000;
  // Configuration for channel 1
  PDBCH1CR      = 0x0012;
  PDBCH1DLYA    = 0;
  PDBCH1DLYB    = 253;
  // Configuration for channel 2
  PDBCH2CR      = 0x0012;
  PDBCH2DLYA    = 0;
  PDBCH2DLYB    = 253;

  PDBIDLY       = 1989;
  PDBSC         = 0x205F;  // Prescale 2
  PDBSC         |= 0x0020;

}
//------------------------------------------------------------------------
// Enable FCD on lCD pins
//------------------------------------------------------------------------
void fcd_enable_on_lcd_pins(void) {  
  //Configuring FCD on LCD Pads
  SCGC2_LCD  = 1;
  LCDSUPPLY |= 0x03;
  LCDC1     |= 0x04;
  SCGC2     |= 0x40; //Enabling clock for Prescalar block  
}

//------------------------------------------------------------------------
// Config pracmp for zcd 
//------------------------------------------------------------------------
void pracmp_init ()
{
  // Enable clock for the pracmp 
  SCGC5_PRACMP1 = 1;

  // Config PTA2 for input PRACMP CMPP0
  //PTAPF2_A2 = 4;

  // Config PTH0 for input PRACMP CMPP5
  PTHPF1_H0 = 0x1;

  // Config PTG4 for  cmpout
  //PTGPF3_G4 = 1;
  
  PRACMP1CS = 0x91; // Enable comp, active int on rising and falling edge
  PRACMP1C1 = 0xCF; // Enable PRG and select value 3.3/2 = 1.65
  PRACMP1C0 = 0x57; // Take + input from CMPP0 and - input from PRG
  PRACMP1C2 = 0xFF; // Enble the external input 1
}
//------------------------------------------------------------------------
// ISR for PRACMP1 interrupt 
//------------------------------------------------------------------------
interrupt VectorNumber_Vpracmp1 void ACMP1_ISR()
{ 
  if(cycle == 0) {
    mtim_count        = (uint32_t)MTIM3CNT;    // Read the mtim16 count register  
  }
  else if (cycle == 1) {
    mtim_count        = mtim_count + (uint32_t)MTIM3CNT;
  }

  MTIM3SC_TRST        = 1;            // Reset the MTIM16 counter
  PRACMP1CS_ACMPF     = 0;            // Clear the ACMP interrupt flag

  PTCD_PTCD4         = !PTCD_PTCD4;  // Toggel Red LED

  if(cycle == 0) {
     cycle = 1;
  }
  else if(cycle == 1) {
    cycle  = 0;
    freq_sample_done   = 1; 
  }
}

//------------------------------------------------------------------------
// ISR for PDB to read the ADC A and B regoster
//------------------------------------------------------------------------
interrupt VectorNumber_Vpdb void PDB_ISR ()
{
  // Current sample done
  if(ADC0SC1A_COCOA) 
   {
    
    resultA[buffer][sample]   = ADC0RA - OFFSET;
  
   }
  
  if (ADC1SC1A_COCOA) 
   {  
    resultB[buffer][sample]   = ADC1RA - OFFSET;
   }

   sample = sample + 1;

   if(sample == SAMPLE){

     pre_buffer    = buffer;
     sample        = 0;
     buffer_ready  = 1;

     if (buffer)
       buffer = 0;
     else 
       buffer = 1;
      
      //disable_pdb
      //PDBSC_EN = 0;
   }

  // Clear the interrupt flag
  PDBSC |= 0x0080;
}
//------------------------------------------------------------------------
// ISR for PDB to read the ADC A and B regoster
//------------------------------------------------------------------------
interrupt VectorNumber_Vpdb_err void PDBERR_ISR ()
{
  while(1);
}

 void interrupt VectorNumber_Vkbi KBI_ISR()
{
  unsigned char ptd = PTDD;
  KBISC_KBACK = 1;
  
  ptd &= (PTDD_PTDD4_MASK|PTDD_PTDD5_MASK);

  if(!(ptd & PTDD_PTDD5_MASK))
  {
    if(set_mode == 0)
    {
      set_mode = 1;
    }
    else if(set_mode == 1)
    {
      set_mode = 0;
    }
    else if(set_mode == 2)
    {
      set_part++;
      if(Set_Time)
      set_part %= 3;
      
      if(Set_Date)
      set_part %= 4;
      
      clearLCDDTx();

      Set_TimeOut_MTIM1(50);
      Enable_MTIM1();
    }
  }
   
  if(!(ptd & PTDD_PTDD4_MASK))
  { 
   if((set_mode == 2)) 
    {
       if(Set_Time) 
       {
        
           switch(set_part)
            {
              case 0:
                curr_hour++;
                curr_hour %= 24;
                break;
              case 1:
                curr_min++;
                curr_min %= 60;
                break;
              case 2:
                curr_sec++;
                curr_sec %= 60;
                break;
            }
           setLCDClock();
       }
      else if(Set_Date)
      {
        switch(set_part) 
        {
          case 0:
             curr_wday++;
             curr_wday %= 7;
             break;
          case 1:
             curr_date++;
             curr_date %= 32;
             if(!curr_date)
             curr_date++;
             break;
          case 2:
             curr_month++;
             curr_month %= 13;
             if(!curr_month)
             curr_month++;
             break;
          case 3:
             curr_year++;
             curr_year %=21;
             break;
        }
        setLCDDate();
      }
       
    } 
    else
    {
      SW3_CLICK = 1;
    }
  
  }
    
  
}


//------------------------------------------------------------------------
//   RMS
//
//   Input parameter: pointer to 16-bit usigned.
//   Output parameter: unsigned 16-bits.
//------------------------------------------------------------------------
uint32_t  RMS_calc(int16_t *input)
{
  uint16_t i;
  int32_t x;

  uint32_t y; 
  double acc  = 0; 
                                 
  for(i=0; i<SAMPLE; i=i+1){               // execute interaction      
    x = ((int32_t)(input[i])) * ((int32_t)(input[i]));
    acc = acc + x;           
  }      
  y = (uint32_t)(acc/SAMPLE);
  return(Sqrt_r(y));                  // output square root of Sum
} 


//------------------------------------------------------------------------   
//   Power and Energy Calculations
//
//   Input parameter: -Voltage pointer to 16-bit signed.
//                    -Current pointer to 16-bit signed.
//                    -Output Vector pointer.
//                    -Output Time vector pointer.
//   Output parameter: unsigned 16-bits.
//------------------------------------------------------------------------                
   
void Power_Calc(int16_t *V, int16_t *I, power_vec_t *Out)
{
  uint16_t i;
  int32_t x;
  int16_t *tempV = V, *tempI = I;             // local pointers

  double acc;
  double kwh_temp;

 
  
  for(i = 0;i < SAMPLE; i++){                // interaction for the 256 samples
      x = ((int32_t)(tempV[i])) * ((int32_t)(tempI[i]));
      acc = acc + x;
  }
  
                                              
  Out->kw = (int32_t)(acc/(SAMPLE*FACT_ACTP));                         // output result
 
  kwh_temp = (acc/(SAMPLE*FACT_ACTP));                                 // Energy for 20ms 
  kwh_temp = ((kwh_temp * 2) /100);

 
  //if(((Out->kwh + (int32_t)kwh_temp)> 2147483647) || ((Out->kwh + (int32_t)kwh_temp) < -268435456)) 
  //Out->kwh =0;
  
  Out->kwh =  Out->kwh  + (int32_t)kwh_temp;                           // Store energy in watthour
  
                                           
  /* Voltage RMS Calculation */
  Out->v = (uint16_t)RMS_calc(V);                                                   
  Out->v = (uint16_t)((Out->v)/FACT_VOLT);
  

  /* Current RMS Calculation */
  Out->a = (uint16_t)RMS_calc(I);                    
  Out->a = (uint16_t)(Out->a/FACT_CURR);
                                         
  /* Aparent Power Calculation */
  Out->kva = (int32_t)(((int32_t)((Out->v)/10) * (int32_t)(Out->a))/10);
  
  /* If Aparent power is smaller then active then correct the output */
  if((Out->kva) < (Out->kw)) Out->kva = Out->kw;                            

 }               
//------------------------------------------------------------------------
/**
 * \brief  Calculates the square root of the value
 * \author  Luis Puebla
 * \param   value
 * \return  (unsigend long)square root
 * \todo    
 * \warning
 */
//------------------------------------------------------------------------   
uint32_t Sqrt_r(uint32_t value)
{
  uint8_t num_iter     = 0;
  uint32_t result      = 1000;
  uint32_t last_result = 1000;
  
  do
	{
      last_result = result;
      result = (result + value/result)/2;
	}
  while (num_iter++ < 7 && last_result!= result);
  return result;
}
//------------------------------------------------------------------------   
