/*
 *                         Copyright 2004
 *                    Green Hills Software, Inc.
 *
 *    This program is the property of Green Hills Software, Inc,
 *    its contents are proprietary information and no part of it
 *    is to be disclosed to anyone except employees of Green Hills
 *    Software, Inc., or as agreed in writing signed by the President
 *    of Green Hills Software, Inc.
 *
 */

#ifndef _THREAD_H_
#define _THREAD_H_

#include <signal.h>
#include <stdio.h>
#include <time.h>
#include <setjmp.h>

#if !defined(_SIGMAX)
#  if defined(SOLARIS20)
#    define _SIGMAX MAXSIG
#  elif defined(__linux) || defined(MSW) || defined(__LYNX)
#    define _SIGMAX (NSIG-1)
#  endif
#endif

#if defined(_SIGMAX)
typedef void (*SignalHandler)(int);

enum __ghs_ThreadLocalStorage_specifier {
    __ghs_TLS_asctime_buff,
    __ghs_TLS_tmpnam_space,
    __ghs_TLS_strtok_saved_pos,
    __ghs_TLS_Errno,
    __ghs_TLS_gmtime_temp,
    __ghs_TLS___eh_globals,
    __ghs_TLS_SignalHandlers
};

/* For compatibility with older libraries without
 * __ghs_GetThreadLocalStorageItem(), the symbol is declared weak.
 * In a future release, the symbol will be strong.
 */
#pragma weak __ghs_GetThreadLocalStorageItem
extern void *__ghs_GetThreadLocalStorageItem(int);

/* Define __ghs_SafeGetThreadLocalStorageItem() for referencing fields in
 * ThreadLocalStorage from outside libsys.  We check for
 * __ghs_GetThreadLocalStorageItem() which is the preferred entry point, but
 * for compatibility with older releases, we will also accept
 * GetThreadLocalStorage().
 *
 * For systems that are meant to work in a MULTI 5.0 environment or later,
 * one need only implement __ghs_GetThreadLocalStorageItem().  If compatibility
 * with earlier releases is also required, one must implement
 * GetThreadLocalStorage(), too.
 */
#ifdef __ghs_pic
#pragma weak __ghs_undefined_func
extern void *__ghs_undefined_func(int);
#define __ghs_SafeGetThreadLocalStorageItem(item) \
    ((&__ghs_GetThreadLocalStorageItem == &__ghs_undefined_func) ? \
	    (&(GetThreadLocalStorage()->item)) : \
	    (__ghs_GetThreadLocalStorageItem(__ghs_TLS_ ## item)))	
#else
#define __ghs_SafeGetThreadLocalStorageItem(item) \
    (!(&__ghs_GetThreadLocalStorageItem) ? \
	    (&(GetThreadLocalStorage()->item)) : \
	    (__ghs_GetThreadLocalStorageItem(__ghs_TLS_ ## item)))
#endif

/* Use of ThreadLocalStorage and GetThreadLocalStorage() from outside
 * of libsys is deprecated.
 */
typedef struct
{
	int			Errno;
	SignalHandler 		SignalHandlers[_SIGMAX];
	char			tmpnam_space[L_tmpnam];
	char			asctime_buff[30];
	char			*strtok_saved_pos;
	struct tm		gmtime_temp;
	void 			*__eh_globals;
} ThreadLocalStorage;

/*
 * Some of the functions in this file are to be implemented by the user for
 * use by the Green Hills Libraries.  See the Building book for more
 * information.
 *
 */

/* Return pointer to thread local storage */
#pragma weak GetThreadLocalStorage
ThreadLocalStorage *GetThreadLocalStorage(void);

/* Acquire a lock which can be obtained from within an interrupt handler. */
#pragma weak __ghs_InterruptLock
void __ghs_InterruptLock(void);

/* Release the lock acquired via __ghs_InterruptLock. */
#pragma weak __ghs_InterruptUnlock
void __ghs_InterruptUnlock(void);

#ifdef __ghs_pic
#define __ghs_weak_sym_check(_fn) ((void*)_fn != (void*)__ghs_undefined_func)
#else
#define __ghs_weak_sym_check(_fn) (_fn)
#endif

/* A "static" buffer to be shared between ind_gmtm.c and ind_tmzn.c */
extern struct tm __ghs_static_gmtime_temp;

/* Define wrappers for profiling hooks to call interrupt lock if available,
 * otherwise fall back on _ghsLock.  The interrupt lock is necessary if
 * profiling may be used for code executing at interrupt level, or in some
 * other context (e.g. kernel-internal) where the ghsLock cannot be used. */
#define __ghs_ProfileLock() { 						\
    if (__ghs_weak_sym_check(__ghs_InterruptLock))			\
	__ghs_InterruptLock();						\
    else								\
	__ghsLock();							\
}
#define __ghs_ProfileUnlock() { 					\
    if (__ghs_weak_sym_check(__ghs_InterruptUnlock))			\
	__ghs_InterruptUnlock();					\
    else								\
	__ghsUnlock();							\
}

/* Acquire general lock.  Blocks until the lock becomes available. */
/* Implemented by the user for Green Hills libraries.  */
void __ghsLock(void);

/* Release general lock */
/* Implemented by the user for Green Hills libraries.  */
void __ghsUnlock(void);

/* Save arbitrary state across a setjmp() */
/* Implemented by the user for Green Hills libraries.  */
int  __ghs_SaveSignalContext(jmp_buf);

/* Restore arbitrary state across a longjmp() */
/* Implemented by the user for Green Hills libraries.  */
void __ghs_RestoreSignalContext(jmp_buf);

#if (!defined (EMBEDDED) && !defined (__OSE) && !defined (MSW)) || defined(MINIMAL_STARTUP)
__inline void __ghsLock(void) { }
__inline void __ghsUnlock(void) { }
#endif /* !EMBEDDED && !__OSE && !MSW */


#define GHSLOCK		__ghsLock();
#define GHSUNLOCK	__ghsUnlock();

/* macros used in stdio library source */
#ifndef __disable_thread_safe_extensions
# define LOCKFILE(f)	flockfile(f);
# define TRYLOCKFILE(f)	ftrylockfile(f);
# define UNLOCKFILE(f)	funlockfile(f);
# define LOCKCREATE(f)	flockcreate(f);
# define LOCKCLEANUP(f)	flockdestroy(f);
# define LOCKINUSE(f)   __ghs_flock_in_use(f)
/* prototypes for FILE lock routines (not in POSIX API) */

void ** __ghs_flock_ptr(void *);

/* Acquire lock for FILE *addr */
/* Implemented by the user for Green Hills libraries.  */
void __ghs_flock_file(void *);

/* Release lock for FILE *addr */
/* Implemented by the user for Green Hills libraries.  */
void __ghs_funlock_file(void *);

/* Non blocking acquire lock for FILE *addr.  May return -1 if */ 
/* this cannot be implemented. Returns 0 on success and nonzero. */
/* Implemented by the user for Green Hills libraries.  */
int __ghs_ftrylock_file(void *);

int __ghs_flock_in_use(void *);

/* Callbacks to initialize local lock data structures before they */ 
/* are used. */
/* Implemented by the user for Green Hills libraries.  */
void __ghs_flock_create(void **);
void __ghs_flock_destroy(void *);
#else
# define LOCKFILE(f)
# define TRYLOCKFILE(f)	-1;	/* no lock obtained */
# define UNLOCKFILE(f)
# define LOCKCREATE(f)	
# define LOCKCLEANUP(f)
# define LOCKINUSE(f) 0
#endif
/* prototypes for FILE lock routines (not in POSIX API) */
void flockcreate(FILE *stream);
void flockdestroy(FILE *stream);
/* End New */

#ifdef __Declare_Initialization_Routines__
#pragma weak __gh_signal_init
void __gh_signal_init(SignalHandler *);
#pragma weak __gh_iob_init
extern void __gh_iob_init(void);
#pragma weak __gh_error_init
extern void __gh_error_init(void);
#pragma weak __gh_lock_init

/* A callback to initialize the lock data structure before it is used. */
/* This function is provided by the user for use by the Green Hills libraries.
 */
extern void __gh_lock_init(void);
#if defined(__OSE)
#else
#pragma weak __cpp_exception_init
#endif /* __OSE */
extern void __cpp_exception_init(void **);
#if defined(__OSE)
#else
#pragma weak __cpp_exception_cleanup
#endif /* __OSE */
extern void __cpp_exception_cleanup(void **);
#endif /* __Declare_Initialization_Routines__ */
#endif /* defined(_SIGMAX) */
#endif /* _THREAD_H_ */

