/******************************************************************************
								    Copyright (c) Freescale 2010
File Name    :  main.c

Current Revision :	Revision: 2.0 

PURPOSE: main program entry.                       
                                                                          
DataSheet : MC9S12XHY256RMV1.11.pdf
                                                                       
DESCRIPTION:  function main() providing initial program entry.
                                                         
                                                                          
UPDATE HISTORY                                                            
REV  AUTHOR    DATE        DESCRIPTION OF CHANGE                          
---  ------    --------    ---------------------                          
1.0  r28318    17/09/09    - initial coding
2.0  b30269    13/10/10    - S12XHY  migration, removed magic numbers and created ENUM
                             Current_Test_Mode. Created defines for SCI module configu
                             ration.
                             Cases to awake from Stop and Pseudo stop modes added. 

 *******************************************************************
 * File created by: Freescale East Kilbride MSG Applications Group *
 *******************************************************************


                                                                          
******************************************************************************/
/*===========================================================================*/
/* Freescale reserves the right to make changes without further notice to any*/
/* product herein to improve reliability, function, or design. Freescale does*/
/* not assume any  liability arising  out  of the  application or use of any */
/* product,  circuit, or software described herein;  neither  does it convey */
/* any license under its patent rights  nor the  rights of others.  Freescale*/
/* products are not designed, intended,  or authorized for use as components */
/* in  systems  intended  for  surgical  implant  into  the  body, or  other */
/* applications intended to support life, or  for any  other application  in */
/* which the failure of the Freescale product  could create a situation where*/
/* personal injury or death may occur. Should Buyer purchase or use Freescale*/
/* products for any such intended  or unauthorized  application, Buyer shall */
/* indemnify and  hold  Freescale  and its officers, employees, subsidiaries,*/
/* affiliates,  and distributors harmless against all claims costs, damages, */
/* and expenses, and reasonable  attorney  fees arising  out of, directly or */
/* indirectly,  any claim of personal injury  or death  associated with such */
/* unintended or unauthorized use, even if such claim alleges that  Freescale*/
/* was negligent regarding the  design  or manufacture of the part. Freescale*/
/* and the Freescale logo* are registered trademarks of Freescale Ltd.       */
/*****************************************************************************/

/************************* Include Files *************************************/
/*general includes */
#include <hidef.h>  /* also includes boolean definitions in stdtypes.h     */
#include <MC9S12XHY256.h>     /* derivative information */
#include "target.h" 
#include "S12HY_peripherals.h" /* includes peripherals definitions and FSL data types */

/************************* typedefs ******************************************/
typedef enum
{
   RUN_M = 0,
   STOP_M,
   PSEUDO_STOP_M
	 
} CLK_TEST_MODES;

/************************* #defines ******************************************/
#define SW1_RUN_MODE            0xE0
#define SW2_PSEUDO_STOP_MODE    0xD0
#define SW3_STOP_MODE           0xB0
#define SW4_SWITCH_MODE         0x70


#define HIGHER_NIBBLE_MASK      0xF0
#define S_BIT_CLR_MASK          0x7F
#define TRNS_OFF                0x04

/* SCI baud rate = SCI bus clock / (16 * SCInBR[12:0])*/
#define SCI_40MHZ_9600_BR       0x104
#define SCI_16MHZ_9600_BR       0x068
#define SCI_08MHZ_9600_BR       0x034
#define SCI_04MHZ_9600_BR       0x01A


/************************* Constants *****************************************/
#pragma CONST_SEG DEFAULT

/************************* Global Variables **********************************/
#pragma DATA_SEG DEFAULT

CLK_TEST_MODES Current_Test_Mode = SW1_RUN_MODE;

/************************* function prototypes *******************************/
#pragma CODE_SEG DEFAULT

/************************* Functions *****************************************/
#pragma CODE_SEG DEFAULT
/******************************************************************************
Function Name  : Service_WD
Engineer       : b30269	
Date           : 14/10/10
Parameters     : NONE
Returns        : NONE
Notes          : COP time-out period is restarted. using sequence stated in: 
                 MC9S12XHY256RMV1.11.pdf, 7.4.1.5 Computer Operating Properly 
                 Watchdog (COP). 
                  
******************************************************************************/
void Service_WD(void)
{
   ARMCOP = 0x55; 
   ARMCOP = 0xAA;
}

/******************************************************************************
Function Name  : main
Engineer       : b30269	
Date           : 14/10/10
Parameters     : NONE
Returns        : NONE
Notes          : main routine called by Startup.c.
                 This function will configure CRG module in the S12XHY256 MC to
                 demonstrate the following modes:
                 
                 RUN_MODE
                 PSEUDO_STOP_MODE
                 STOP_MODE
                 SELF_CLK_MODE
                 
                 Previous modes are selected via Reset and Switches.
                  
******************************************************************************/

void main(void) 
{
   unsigned int   outer_counter;		
   unsigned int   inner_counter;   

   PTR      = PTR_PTR0_MASK;   /* Initialize PORT R 0 High; turn off LED                     */
   DDRR     = PTR_PTR0_MASK;   /* PORTR 0 = output                                           */

   PER1AD   = (HIGHER_NIBBLE_MASK | TRNS_OFF);            /* Enable pull ups on Switches 1-4, and Transceiver Force Off */
   ATDDIENL = HIGHER_NIBBLE_MASK;/* Enable digital input buffer on ANx pin                   */
     
   /* Wait for pull ups to come into effect */
   for (outer_counter=0; outer_counter<1000; outer_counter++)
   {	
      for (inner_counter=0; inner_counter<100; inner_counter++)
      {
         ;                     /* Intentionally left empty                                   */ 
      } 
   }

   COPCTL   = COPCTL_CR_MASK;  /* set WD period to max allowed ,                             */
																																									  
   Service_WD();			         /* Refresh WD for first time                                  */
   
   asm ANDCC #S_BIT_CLR_MASK   /* Enable STOP mode, clear S bit                              */

   EnableInterrupts;
   PIE1AD_PIE1AD7 = 1;         /* Enable Interrupt for  SW4                                  */
   
   switch (PT01AD & HIGHER_NIBBLE_MASK)
   {
      case SW1_RUN_MODE: 

         /* 
            Initialise the system clock 8 MHz Crystal, 
            40 MHz Bus CLK (80 MHz VCO, 80 MHz PLL)  
         */
                
         SYNR    = 0x44;     /* SYNDIV = 3, VCO frequency 48 - 80 MHz  */       

         REFDV   = 0x80;     /* REFDIV = 10, REFCLK frequency 6-12 MHz */       

         POSTDIV = 0x00;     /* POSTDIV = 0 FPLL = FVCO                */

         while(!CRGFLG_LOCK)
         {
            ;                /* wait for VCO to stabilize              */
         }			 
         
         Service_WD();
         
         CLKSEL_PLLSEL = 1;  /* Switch clk to use PLL                  */

         Current_Test_Mode = RUN_M;		   
         /* Configure the SCI to run at 9600 baud and Tx Clock Mode and Clock speeds */

         SCI_Config(SCI_40MHZ_9600_BR);
         DisplayString("DEMOS12XHY256 Board Running Normal Mode");
         LF;
         DisplayString("80 MHz VCO Clock, 80 MHz PLL Clock, 40 MHz Bus Clock, from 8 MHz Crystal");
         LF;
         LF;
    
         break;
      
      case SW2_PSEUDO_STOP_MODE:
      
         /* 
            Initialise the system clock for Pseudo Stop Mode 
            - 32MHz VCO, 16MHz PLL, 8MHz Bus CLK, from 8MHz Crystal 
         */
				 
         SYNR    = 0x01;     /* SYNDIV = 1, VCO frequency 32 - 48 MHz         */

         REFDV   = 0x80;     /* REFDIV = 10 REFCLK frequency 6-12 MHz         */     

         POSTDIV = 0x01;     /* POSTDIV = 1                                   */

         while(!CRGFLG_LOCK)
         {
            ;                /* wait for VCO to stabilize                     */
         }			 
         
         Service_WD();
         
         CLKSEL_PLLSEL = 1;  /* Switch clk to use PLL                         */
      	 
      	 Current_Test_Mode = PSEUDO_STOP_M;
      	 
      	 CLKSEL_PSTP = 1;    /* Enable pseudo stop mode                       */
      	 PLLCTL_PCE  = 1;    /* COP continues running during pseudo stop mode */

      	 
      	 
      	 /* 
      	    Configure the SCI to run at 9600 baud and Tx Clock Mode and Clock speeds 
      	 */

         SCI_Config(SCI_08MHZ_9600_BR);
         DisplayString("DEMOS12XHY256 Board Running PSEUDO STOP Mode");
         LF;
         DisplayString("32 MHz VCO Clock, 16 MHz PLL Clock, 8 MHz Bus Clock, from 8 MHz Crystal");
         LF;
         LF;

         break;
      
      case SW3_STOP_MODE:
                    
         /* 
            Initialise the system clock to Stop mode - 64 MHz VCO, 
            32 MHz PLL Clock, 16 MHz bus CLK, from 8MHz External Clock 
         */
				 
         SYNR    = 0x43;      /* SYNDIV = 3, VCO frequency 48 - 64 MHz  */

         REFDV   = 0x80;      /* REFDIV = 10 REFCLK frequency 6-12MHz   */ 

         POSTDIV = 0x01;      /* POSTDIV = 1                            */

         while(!CRGFLG_LOCK)
         {
            ;                 /* wait for VCO to stabilize              */
         }			 
         
         CLKSEL_PLLSEL = 1;   /* Switch clk to use PLL                  */
         CLKSEL_XCLKS  = 1;    /* external clk to use PLL                */
         
         Service_WD();
         
         Current_Test_Mode = STOP_M;
      	 
      	 CLKSEL_PSTP = 0;    /* Disable pseudo stop mode                             */
      	 PLLCTL_PCE  = 0;    /* COP stops running during pseudo stop mode            */
       
         
         /* Configure the SCI to run at 9600 baud and Tx Clock Mode and Clock speeds */

         SCI_Config(SCI_16MHZ_9600_BR);
         DisplayString("DEMOS12XHY256 Board Running User Configured STOP Mode");
         LF;
         
         DisplayString("64 MHz VCO Clock, 32 MHz PLL Clock, 16 MHz Bus Clock, from 8 MHz External Clock");
         LF;
         LF;
      
         break;
      
      default: /* Run Mode */
      {
          
         /* 
            Initialise the system clock to Stop mode - 32 MHz VCO, 
            4 MHz PLL Clock,8 MHz bus CLK, from 8MHz External Clock 
         */
				 
         SYNR    = 0x01;      /* SYNDIV = 3, VCO frequency 32 - 48 MHz  */

         REFDV   = 0x80;      /* REFDIV = 10 REFCLK frequency 6-12MHz   */ 

         POSTDIV = 0x02;      /* POSTDIV = 2                            */

         while(!CRGFLG_LOCK)
         {
            ;                 /* wait for VCO to stabilize              */
         }			 
         
         CLKSEL_PLLSEL = 1;   /* Switch clk to use PLL                  */
         CLKSEL_XCLKS  = 1;   /* external clk to use PLL                */

         Service_WD();
         
         Current_Test_Mode = RUN_M;
         /* Configure the SCI to run at 9600 baud and Tx Clock Mode and Clock speeds */
				 SCI_Config(SCI_04MHZ_9600_BR);
         
         DisplayString("DEMO9S12XHY256 Board Running DEFAULT Mode");
         LF;
         
         DisplayString("System clocks are derived from OSCCLK (fBUS = fOSC / 2)");
         LF;
         DisplayString("Loop controlled Pierce Oscillator is selected");
         LF;
         DisplayString("8MHz System Clock, 4MHz Bus Clock");
         LF;
         LF;
    
         break;
       
      }/* default */
   }/* switch */
      
   ECLKCTL = 0x00;         /* Enable ECLK = Bus Clock, on PH2                        */                   

   for(;;)
   {
      PTR ^= PTR_PTR0_MASK;
      /* Delay to see LED1 toggle */
      for (outer_counter = 0; outer_counter < 2000; outer_counter++)
  	  {	
   		   for (inner_counter = 0; inner_counter < 500; inner_counter++)
  		   {
  		      ;              /* Intentionally left empty */
  		   } 
	    }
     
      Service_WD();
      
   }/* for */ 		
}/* main */

#pragma CODE_SEG NON_BANKED
#pragma TRAP_PROC
__interrupt VectorNumber_Vportad void PORTAD_ISR(void) 
{
  
  PIF1AD_PIF1AD7 = 1;
  
  if((PSEUDO_STOP_M == Current_Test_Mode) ||
     (STOP_M        == Current_Test_Mode)) 
  {
     PTR_PTR0 = 1;  /* Turn of LED1 */
     asm(STOP);
           
            /*
               Note: After pressing SW4 the MC will enter:
               
               Pseudo stop mode:
                  Leaving the WD active and after WD expires MC will leave STOP
                  Mode then reset and enter to default Self clk Mode: 4MHz BUS at PH2 pin
              
               Stop Mode:  
                  MC will stop all internal clocks and stay sleep, No frequency 
                  signal is shown on the scope at PH2 pin. 
                  To take out the MC off from this mode press reset.
  
  */} /* inner if */
}
#pragma CODE_SEG DEFAULT
