/********************************************************************** 

Filename : lcddrv.c

Description:  Performs user friendly initialization 
              of LCD Control Register 0 and 1. Source file
              initializes LCD duty cycle and bias.  By
              providing the program with your desired frame
              frequency and given oscillator frequency, this
              program calculates the correct clock divider.
              User is also capable of enabling special modes
              like Pseudostop or Wait modes.  Front plane start
              and stop points are also entered. Finally, the 
              program enables the LCD to the on state. 
              
              Additional routines provide the user with the 
              ability to turn on and off all the LCD segments.
                
************************************************************************/

#include <mc9s12xhy256.h>     /* derivative information */
#include "lcddrv.h"


extern void time_delay(int);

extern char * LCD_pos[NUMRAMREGS_FORGLASS];
extern char  LCD_position;


// Type declarations for LCD data display types

// --- Temperature Display Group ---

extern enum  t_temp_type  {inside_temp,outside_temp,engine_temp} temp_type;   // Temperature type being displayed
extern enum  t_temp_scale {C,F} temp_scale;                                   // Temperature scale being displayed

// --- Compass Display Group ---

extern enum t_compass_direction {N,NE,E,SE,S,SW,W,NW } direction;  // Compass direction displayed


// --- Odometer Display Group ---

// --- Clock/Mileage Display Group ---

// --- Transmission Display Group ---

extern enum t_transmission_gear {Park,Reverse,Neutral,Drive,Third,Second,First} gear;  // Gear selection


// --- Cruise/TCS/TPMS Display Group ---
 
 
 
 
 //*************************************
 //Internal Funtions
 //*************************************
//=======================================================


//=======================================================
void CONFIG_CLKSOURCE() 
  {                        //Configure LCD clock
      LCDCR0_LCLK = 5;		//Divide clk by 32768= 61Hz

  }


//=======================================================
//=======================================================
//void SET_CONFIG_VSUPPLY(/*char lcd_power_mode, char lcd_glass_voltage*/ )
  //{ 
  //	char vsupplu_temp;

  /*
  #if LCDPWR_VLCD
    LCDSUPPLY_VSUPPLY = 2;
      #if DOUBLERMODE	  //valid mode is using the buffer					
            LCDSUPPLY_LCDCPMS = 0;//DOUBLER MODE  // LCD3VGLASS
  					LCDSUPPLY_BBYPASS = 0;          

      #elif	 TRIPLERMODE
            LCDSUPPLY_LCDCPMS = 1;//TRILPER MODE //LCD5VGLASS
            #if LCDBUFFERSTATE
                  LCDSUPPLY_BBYPASS = 0;
            #else
                  LCDSUPPLY_BBYPASS = 1;          
            #endif    
      #endif
  #elif LCDPWR_VLLLN
     LCDSUPPLY_VSUPPLY = 3;
  #elif  LCDPWR_VDD //* 0 for pwr to vll2 * / //or 1 / *for pwr to vll3* /
      #if  VDDTOVLL2_2V       //for 3V LCD glass
          LCDSUPPLY_VSUPPLY = 0;
          //#if LCD3VGLASS					
            LCDSUPPLY_LCDCPMS = 0;	//DOUBLER MODE  // LCD3VGLASS
          //#endif       
      #elif VDDTOVLL2_3D3V     //for 5V LCD glass
         LCDSUPPLY_VSUPPLY = 0 ;
          //#if LCD5VGLASS
            LCDSUPPLY_LCDCPMS = 1;  //TRILPER MODE //LCD5VGLASS
          //#endif       
         
      #elif VDDTOVLL3_3V       //for 3V LCD glass
         LCDSUPPLY_VSUPPLY = 1 ;
          //#if LCD3VGLASS						
            LCDSUPPLY_LCDCPMS = 0;//DOUBLER MODE		// LCD3VGLASS
          //#endif              
     #endif
  #endif
   
    LCDSUPPLY_LCDCPEN = 1;
  */  
    
 // } 
 
//=======================================================
//=======================================================
void SET_LCDCR1_REG()
  {
  #if LCDINWAIT 
      LCDCR1_LCDSWAI = 0;
  #else
      LCDCR1_LCDSWAI = 1;
  #endif
  }

//=======================================================
//=======================================================
void SET_BIAS()
  {
  //Configure LCD duty & bias
  #ifdef D1B1
      LCDCR0_BIAS = 0;	//Bias = 1/1
  #endif    
  #ifdef D2B2
      LCDCR0_BIAS = 0;	//Bias = 1/2
  #endif    
  #ifdef D2B3
      LCDCR0_BIAS = 1;	//Bias = 1/3
  #endif    
  #ifdef D3B3
      LCDCR0_BIAS = 1;	//Bias = 1/3
  #endif    
  #ifdef D4B3
      LCDCR0_BIAS = 1;	//Bias = 1/3
  #endif    
  }					
//=======================================================
//=======================================================
void SET_DUTY() 
  {
  //Configure LCD duty & bias
  #ifdef D1B1
      LCDCR0_DUTY = 1;	//Duty = 1/1
  #endif    
  #ifdef D2B2
      LCDCR0_DUTY = 2;	//Duty = 1/2
  #endif    
  #ifdef D2B3
      LCDCR0_DUTY = 3;	//Duty = 1/2
  #endif    
  #ifdef D3B3
      LCDCR0_DUTY = 3;	//Duty = 1/3
  #endif    
  #ifdef D4B3
      LCDCR0_DUTY = 0;	//Duty = 1/4
  #endif    


  /*
  			// 1/2 Duty Cycle 
  #if DUTY1BY2			
          LCDCR0_DUTY1 = 0; 
          LCDCR0_DUTY0 = 1;
  #elif	 DUTY1BY3
        // 1/3 Duty Cycle 
          LCDCR0_DUTY1 = 1; 
          LCDCR0_DUTY0 = 0;
  #elif	 DUTY1BY4
        // 1/4 Duty Cycle 
          LCDCR0_DUTY1 = 1; 
          LCDCR0_DUTY0 = 1;
  #endif
  */

  }

//=======================================================
//=======================================================
void  SET_LCD_FRAME_FREQU() 
{
  
		//user must deterime values for duty and LCLK[2:0] based on the application
		SET_DUTY();
		
#if DUTY1BY2			
      /* 1/2 Duty Cycle */
      #if D2FF128D08         
          LCDCR0_LCLK = 3;
      #elif D2FF64D04          
          LCDCR0_LCLK = 4;
      #elif D2FF32D02          
          LCDCR0_LCLK = 5;
      #elif D2FF16D01          
          LCDCR0_LCLK = 6;
      #endif
#elif	 DUTY1BY3
      /* 1/3 Duty Cycle */
      #if D3FF170D77          
          LCDCR0_LCLK = 2;
      #elif D3FF85D38          
          LCDCR0_LCLK = 3;
      #elif D3FF42D69          
          LCDCR0_LCLK = 4;
      #elif D3FF29D34          
          LCDCR0_LCLK = 5;
      #endif
#elif	 DUTY1BY4
      /* 1/4 Duty Cycle */
      #if D4FF128D08         
          LCDCR0_LCLK = 2;
      #elif D4FF64D04          
          LCDCR0_LCLK = 3;
      #elif D4FF32D02          
          LCDCR0_LCLK = 4;
      #elif D4FF16D01          
          LCDCR0_LCLK = 5;
      #endif
#endif


}
//=======================================================

//=======================================================
void ENABLE_FP()
  {               //Enable all segments
  FPENR0 = 0xFF;	//Segments 0..7
  FPENR1 = 0xFF;	//Segments 8..15
  FPENR2 = 0xFF;	//Segments 16..23
  FPENR3 = 0xFF;	//Segments 24..31
  FPENR4 = 0xFF;	//Segments 32..39
  }


 //*************************************
 //External Funtions
 //*************************************
//=======================================================
//=======================================================
//=======================================================

//=======================================================
//void ENABLE_BLINK(byte cmd_enable){ LCDBCTL_BLINK = cmd_enable;}

//=======================================================
void ENABLE_LCD(byte cmd_enable)  { LCDCR0_LCDEN = cmd_enable;}

//=======================================================
void LCD_init() 
  {
  CONFIG_CLKSOURCE();	          //Configure clock source
  SET_LCDCR1_REG();             //Configure operation in stop/wait
  SET_LCD_FRAME_FREQU();        //Configure  frame frequncy
  SET_DUTY();
  SET_BIAS();
	ENABLE_FP();	                //Enable Frontplanes
	ENABLE_LCD(ON);	              //Enable LCD
  }

//=======================================================
//=======================================================
void LCD_RAM_SET_ALL()  
  {
  byte i;
  for (i=0; i < NUMRAMREGS_FORGLASS; i++) {*LCD_pos[i] =  0xF;}
  }

//=======================================================
//=======================================================
void LCD_RAM_CLEAR_ALL()  
  {
  byte i;
  for (i=0; i < NUMRAMREGS_FORGLASS; i++) { *LCD_pos[i] =  0x0;}
  }



//--------------------------------------------------------------------
 //=======================================
void INIT_LCD_NUMS(void) 
  {
  tU08 DIGITAL_POS_1,DIGITAL_POS_2,DIGITAL_POS_3,DIGITAL_POS_4,DIGITAL_POS_5,DIGITAL_POS_6,DIGITAL_POS_7,DIGITAL_POS_8, DIGITAL_POS_9, DIGITAL_POS_10, DIGITAL_POS_11, DIGITAL_POS_12, DIGITAL_POS_13;
    
    DIGITAL_POS_1  = display_pos_1_value_; 
    DIGITAL_POS_2  = display_pos_2_value_; 
    DIGITAL_POS_3  = display_pos_3_value_; 
    DIGITAL_POS_4  = display_pos_4_value_; 
    DIGITAL_POS_5  = display_pos_5_value_; 
    DIGITAL_POS_6  = display_pos_6_value_; 
    DIGITAL_POS_7  = display_pos_7_value_; 
    DIGITAL_POS_8  = display_pos_8_letter_S; 
    DIGITAL_POS_9  = display_pos_9_value_1; 
    DIGITAL_POS_10 = display_pos_10_value_2;
    DIGITAL_POS_11 = display_pos_11_letter_X ; 
    DIGITAL_POS_12 = display_pos_12_letter_h ;
    DIGITAL_POS_13 = display_pos_13_letter_y; 
    
    
    MDelay(6000);
    
  }

//=======================================
void SET_LOGO(byte FSL_logo, byte HY_64_Part_Num, byte Jellyfish_Name, byte Jellyfish_Body, byte Jellyfish_Tentacles) 
  {
  SEG_P21 = FSL_logo;                 //Set LCDRAM for Freescale Logo
  SEG_P1  = HY_64_Part_Num;           //Set LCDRAM for Part Number
  SEG_P2  = Jellyfish_Name;           //Set LCDRAM for Part Name
  SEG_S1  = Jellyfish_Body;           //Set LCDRAM for Part Graphic 1
  SEG_P3  = Jellyfish_Tentacles;      //Set LCDRAM for Part Graphic 2
  }

//=======================================
void SET_ODO_TITLE(byte odo, byte miles, byte tripA, byte tripB, byte km) 
  {
    SEG_P40 = odo;   //set LCDRAM for ODO Seg
    SEG_P34 = miles;   // set LCDRAM for miles seg
    SEG_P41 = tripA;  // set LCDRAM for TRIPA seg
    SEG_P42 = tripB;  // set LCDRAM for TRIPB seg 
    SEG_P35 = km;   // set LCDRAM for miles seg 
  }

//=======================================

void OFF_TITLE(void) 
  {
    tU08 DIGITAL_POS_4,DIGITAL_POS_5,DIGITAL_POS_6,DIGITAL_POS_7,DIGITAL_POS_8, DIGITAL_POS_9, DIGITAL_POS_10, DIGITAL_POS_11, DIGITAL_POS_12, DIGITAL_POS_13;
    
    DIGITAL_POS_4  = display_pos_4_value_OFF;
    DIGITAL_POS_5  = display_pos_5_value_OFF;
    DIGITAL_POS_6  = display_pos_6_value_OFF;
    DIGITAL_POS_7  = display_pos_7_value_OFF;
    DIGITAL_POS_8  = display_pos_8_value_OFF;
    DIGITAL_POS_9  = display_pos_9_value_OFF;
    DIGITAL_POS_10 = display_pos_10_value_OFF;
    DIGITAL_POS_11 = display_pos_11_value_OFF;
    DIGITAL_POS_12 = display_pos_12_value_OFF;
    DIGITAL_POS_13 = display_pos_13_value_OFF;
  }

//=======================================


void HAZARD_TITLE(void) 
  {
    tU08 DIGITAL_POS_8, DIGITAL_POS_9, DIGITAL_POS_10, DIGITAL_POS_11, DIGITAL_POS_12, DIGITAL_POS_13;
    
    DIGITAL_POS_8  = display_pos_8_letter_H;
    DIGITAL_POS_9  = display_pos_9_letter_A;
    DIGITAL_POS_10 = display_pos_10_letter_Z;
    DIGITAL_POS_11 = display_pos_11_letter_A;
    DIGITAL_POS_12 = display_pos_12_letter_R;
    DIGITAL_POS_13 = display_pos_13_letter_D; 
    
/*    DIGITAL_POS_8  = display_pos_8_letter_H_FLASH;
    DIGITAL_POS_9  = display_pos_9_letter_A_FLASH;
    DIGITAL_POS_10 = display_pos_10_letter_Z_FLASH;
    DIGITAL_POS_11 = display_pos_11_letter_A_FLASH;
    DIGITAL_POS_12 = display_pos_12_letter_R_FLASH;
    DIGITAL_POS_13 = display_pos_13_letter_D_FLASH;   */
  }

//=======================================
void  SET_VOLT_BAR(byte local_st, byte local_bars) {
  
  char i;
  byte  temp_array[6];
  
  if (local_st == ON)   //Set LCDRAM
    {
    SEG_P14 = 1;  //Turn on Voltage label
		                                                      //fill icon bars 1-6 as requested 
    for (i=0; i < 6; i++)          {temp_array[i] = OFF;} // Clear buffer
    for (i=0; i < local_bars; i++) {temp_array[i] = ON;}  // Fill up to requested value
    
	  SEG_P20 = temp_array[0];	 //icon bars 1-6
    SEG_P19 = temp_array[1];
    SEG_P18 = temp_array[2];
    SEG_P17 = temp_array[3];
    SEG_P16 = temp_array[4];      
    SEG_P15 = temp_array[5];      
    }
  else		
    {
    SEG_P14 = OFF;  //Turn off Voltage label
    SEG_P20 = OFF;	//icon bars 1-6
    SEG_P19 = OFF;
    SEG_P18 = OFF;
    SEG_P17 = OFF;
    SEG_P16 = OFF;      
    SEG_P15 = OFF;      
    }
} 

//=======================================
void  SET_TEMP_BAR(byte local_st, byte local_bars) {
  
  char i;
  byte  temp_array[6];
      
  if (local_st == ON)       //Set LCDRAM
    {
    SEG_P6 = ON;            //Turn on Temperature label
		                                                      //fill icon bars 1-6 as requested 
    for (i=0; i < 6; i++)          {temp_array[i] = OFF;} // Clear buffer
    for (i=0; i < local_bars; i++) {temp_array[i] = ON;}  // Fill up to requested value
    
	  SEG_P7  = temp_array[0];	 //icon bars 1-6
    SEG_P8  = temp_array[1];
    SEG_P9  = temp_array[2];
    SEG_P10 = temp_array[3];
    SEG_P11 = temp_array[4];      
    SEG_P12 = temp_array[5];      
    }
  else		
    {
    SEG_P6  = OFF;   //Turn off Temperature label
    SEG_P7  = OFF;  //icon bars 1-6
    SEG_P8  = OFF;	
    SEG_P9  = OFF;
    SEG_P10 = OFF;
    SEG_P11 = OFF;
    SEG_P12 = OFF;      
    }
} 


//=======================================
void  SET_TEMPSEGS(enum t_temp_type temp_types, enum t_temp_scale temp_scales, char degree_st) 
  {
  switch (temp_types) 
    {
    case inside_temp:   SEG_P4 = ON;  break;  // "INSIDE"  label
    case outside_temp:  SEG_S2 = ON;  break;  // "OUTSIDE" label
    case engine_temp:   SEG_P5 = ON;  break;  // "ENGINE"  label
    }
   
  SEG_P6 = ON;     // "TEMP" label
  
  if(temp_scales == C)  SEG_C=ON; else SEG_F=ON;    // "C" or "F"
  }


//=======================================
void  SET_TEMP_DIGITS(char t, tU08 temp_value) 
  {
  
  tU08  DIGIT_POS_1,DIGIT_POS_2,DIGIT_POS_3;    // Temporary values for each of the 3 positions
  
  // ----- Set temp value display here.....
  // Dealing with digit positions 1,2, & 3, given 8 bit value
 
  DIGIT_POS_1 = temp_value/100;   // Digit 1 = temp_value/100

  switch(DIGIT_POS_1) 
    {
    case 0: display_pos_1_value_0   break;
    case 1: display_pos_1_value_1   break;
    case 2: display_pos_1_value_2   break;
    case 3: display_pos_1_value_3   break;
    case 4: display_pos_1_value_4   break;
    case 5: display_pos_1_value_5   break;
    case 6: display_pos_1_value_6   break;
    case 7: display_pos_1_value_7   break;
    case 8: display_pos_1_value_8   break;
    case 9: display_pos_1_value_9   break;
    }
  
  DIGIT_POS_2 = (temp_value - (100*DIGIT_POS_1))/10;   // Digit 2 = (temp_value - 100*DIGIT1)/10

  switch(DIGIT_POS_2) 
    {
    case 0: display_pos_2_value_0   break;
    case 1: display_pos_2_value_1   break;
    case 2: display_pos_2_value_2   break;
    case 3: display_pos_2_value_3   break;
    case 4: display_pos_2_value_4   break;
    case 5: display_pos_2_value_5   break;
    case 6: display_pos_2_value_6   break;
    case 7: display_pos_2_value_7   break;
    case 8: display_pos_2_value_8   break;
    case 9: display_pos_2_value_9   break;
    }

  DIGIT_POS_3 = (temp_value - ((100*DIGIT_POS_1) + (10*DIGIT_POS_2)));   // Digit 3 = (temp_value - (100*DIGIT1) - (10*DIGIT2))

  switch(DIGIT_POS_3) 
    {
    case 0: display_pos_3_value_0   break;
    case 1: display_pos_3_value_1   break;
    case 2: display_pos_3_value_2   break;
    case 3: display_pos_3_value_3   break;
    case 4: display_pos_3_value_4   break;
    case 5: display_pos_3_value_5   break;
    case 6: display_pos_3_value_6   break;
    case 7: display_pos_3_value_7   break;
    case 8: display_pos_3_value_8   break;
    case 9: display_pos_3_value_9   break;
    }

  }



//=======================================
void  SET_COMPASS(byte local_st, enum t_compass_direction directions) 
  {
  // -------------- Clear all compass segments
  SEG_T1 = OFF;   // Surround ring for compass
  SEG_T2 = OFF;	  // N
  SEG_T9 = OFF;   // NE
  SEG_T5 = OFF;   // E
  SEG_T8 = OFF;   // SE
  SEG_T3 = OFF;   // S
  SEG_T7 = OFF;   // SW
  SEG_T4 = OFF;   // W
  SEG_T6 = OFF;   // NW

  if (local_st == ON)
    {
    SEG_T1  = ON;                   // Turn on surround ring for compass
    switch(directions) 
      {
      case N : SEG_T2 = ON; break;  // Turn on relevant segment based on direction
      case NE: SEG_T9 = ON; break;
      case E : SEG_T5 = ON; break;
      case SE: SEG_T8 = ON; break;
      case S : SEG_T3 = ON; break;
      case SW: SEG_T7 = ON; break;
      case W : SEG_T4 = ON; break;
      case NW: SEG_T6 = ON; break;
      }
    }
  else		
    {
    SEG_T1  = 0;  // Surround ring for compass
    }
} 

//=======================================
void  SET_ODO_DIGITS(char t, UINT32 odo_value) 
  {

  UINT32  DIGIT_POS_8,DIGIT_POS_9,DIGIT_POS_10,DIGIT_POS_11,DIGIT_POS_12,DIGIT_POS_13;    // Temporary values for each of the 6 positions
  
  DIGIT_POS_8 = (odo_value)/100000;             // 100,000s digit
  
  switch(DIGIT_POS_8) 
    {
    case 0: display_pos_8_value_0   break;
    case 1: display_pos_8_value_1   break;
    case 2: display_pos_8_value_2   break;
    case 3: display_pos_8_value_3   break;
    case 4: display_pos_8_value_4   break;
    case 5: display_pos_8_value_5   break;
    case 6: display_pos_8_value_6   break;
    case 7: display_pos_8_value_7   break;
    case 8: display_pos_8_value_8   break;
    case 9: display_pos_8_value_9   break;
    }
  
  DIGIT_POS_9 = (odo_value-(100000*DIGIT_POS_8))/10000;   // 10,000s digit

  switch(DIGIT_POS_9) 
    {
    case 0: display_pos_9_value_0   break;
    case 1: display_pos_9_value_1   break;
    case 2: display_pos_9_value_2   break;
    case 3: display_pos_9_value_3   break;
    case 4: display_pos_9_value_4   break;
    case 5: display_pos_9_value_5   break;
    case 6: display_pos_9_value_6   break;
    case 7: display_pos_9_value_7   break;
    case 8: display_pos_9_value_8   break;
    case 9: display_pos_9_value_9   break;
    }

  DIGIT_POS_10 = (odo_value - (100000*DIGIT_POS_8 + 10000*DIGIT_POS_9))/1000;   // 1,000s digit

  switch(DIGIT_POS_10) 
    {
    case 0: display_pos_10_value_0   break;
    case 1: display_pos_10_value_1   break;
    case 2: display_pos_10_value_2   break;
    case 3: display_pos_10_value_3   break;
    case 4: display_pos_10_value_4   break;
    case 5: display_pos_10_value_5   break;
    case 6: display_pos_10_value_6   break;
    case 7: display_pos_10_value_7   break;
    case 8: display_pos_10_value_8   break;
    case 9: display_pos_10_value_9   break;
    }

  DIGIT_POS_11 = (odo_value-((100000*DIGIT_POS_8)+(10000*DIGIT_POS_9)+(1000*DIGIT_POS_10)))/100;   // 100s digit
  
  switch(DIGIT_POS_11) 
    {
    case 0: display_pos_11_value_0   break;
    case 1: display_pos_11_value_1   break;
    case 2: display_pos_11_value_2   break;
    case 3: display_pos_11_value_3   break;
    case 4: display_pos_11_value_4   break;
    case 5: display_pos_11_value_5   break;
    case 6: display_pos_11_value_6   break;
    case 7: display_pos_11_value_7   break;
    case 8: display_pos_11_value_8   break;
    case 9: display_pos_11_value_9   break;
    }
  
  DIGIT_POS_12 = (odo_value-((100000*DIGIT_POS_8)+(10000*DIGIT_POS_9)+(1000*DIGIT_POS_10)+(100*DIGIT_POS_11)))/10;   // 10s digit

  switch(DIGIT_POS_12) 
    {
    case 0: display_pos_12_value_0   break;
    case 1: display_pos_12_value_1   break;
    case 2: display_pos_12_value_2   break;
    case 3: display_pos_12_value_3   break;
    case 4: display_pos_12_value_4   break;
    case 5: display_pos_12_value_5   break;
    case 6: display_pos_12_value_6   break;
    case 7: display_pos_12_value_7   break;
    case 8: display_pos_12_value_8   break;
    case 9: display_pos_12_value_9   break;
    }

  DIGIT_POS_13 = (odo_value-((100000*DIGIT_POS_8)+(10000*DIGIT_POS_9)+(1000*DIGIT_POS_10)+(100*DIGIT_POS_11)+(10*DIGIT_POS_12)));   // 1s digit

  switch(DIGIT_POS_13) 
    {
    case 0: display_pos_13_value_0   break;
    case 1: display_pos_13_value_1   break;
    case 2: display_pos_13_value_2   break;
    case 3: display_pos_13_value_3   break;
    case 4: display_pos_13_value_4   break;
    case 5: display_pos_13_value_5   break;
    case 6: display_pos_13_value_6   break;
    case 7: display_pos_13_value_7   break;
    case 8: display_pos_13_value_8   break;
    case 9: display_pos_13_value_9   break;
    }

  }


 //=======================================
void  SET_TRIPA_DIGITS(char t, tU32 tripA_value) 
  {

  tU32  DIGIT_POS_8,DIGIT_POS_9,DIGIT_POS_10,DIGIT_POS_11,DIGIT_POS_12,DIGIT_POS_13;    // Temporary values for each of the 6 positions
  
  DIGIT_POS_8 = (tripA_value)/100000;             // 100,000s digit
  
  switch(DIGIT_POS_8) 
    {
    case 0: display_pos_8_value_0   break;
    case 1: display_pos_8_value_1   break;
    case 2: display_pos_8_value_2   break;
    case 3: display_pos_8_value_3   break;
    case 4: display_pos_8_value_4   break;
    case 5: display_pos_8_value_5   break;
    case 6: display_pos_8_value_6   break;
    case 7: display_pos_8_value_7   break;
    case 8: display_pos_8_value_8   break;
    case 9: display_pos_8_value_9   break;
    }
  
  DIGIT_POS_9 = (tripA_value-(100000*DIGIT_POS_8))/10000;   // 10,000s digit

  switch(DIGIT_POS_9) 
    {
    case 0: display_pos_9_value_0   break;
    case 1: display_pos_9_value_1   break;
    case 2: display_pos_9_value_2   break;
    case 3: display_pos_9_value_3   break;
    case 4: display_pos_9_value_4   break;
    case 5: display_pos_9_value_5   break;
    case 6: display_pos_9_value_6   break;
    case 7: display_pos_9_value_7   break;
    case 8: display_pos_9_value_8   break;
    case 9: display_pos_9_value_9   break;
    }

  DIGIT_POS_10 = (tripA_value - (100000*DIGIT_POS_8 + 10000*DIGIT_POS_9))/1000;   // 1,000s digit

  switch(DIGIT_POS_10) 
    {
    case 0: display_pos_10_value_0   break;
    case 1: display_pos_10_value_1   break;
    case 2: display_pos_10_value_2   break;
    case 3: display_pos_10_value_3   break;
    case 4: display_pos_10_value_4   break;
    case 5: display_pos_10_value_5   break;
    case 6: display_pos_10_value_6   break;
    case 7: display_pos_10_value_7   break;
    case 8: display_pos_10_value_8   break;
    case 9: display_pos_10_value_9   break;
    }

  DIGIT_POS_11 = (tripA_value-((100000*DIGIT_POS_8)+(10000*DIGIT_POS_9)+(1000*DIGIT_POS_10)))/100;   // 100s digit
  
  switch(DIGIT_POS_11) 
    {
    case 0: display_pos_11_value_0   break;
    case 1: display_pos_11_value_1   break;
    case 2: display_pos_11_value_2   break;
    case 3: display_pos_11_value_3   break;
    case 4: display_pos_11_value_4   break;
    case 5: display_pos_11_value_5   break;
    case 6: display_pos_11_value_6   break;
    case 7: display_pos_11_value_7   break;
    case 8: display_pos_11_value_8   break;
    case 9: display_pos_11_value_9   break;
    }
  
  DIGIT_POS_12 = (tripA_value-((100000*DIGIT_POS_8)+(10000*DIGIT_POS_9)+(1000*DIGIT_POS_10)+(100*DIGIT_POS_11)))/10;   // 10s digit

  switch(DIGIT_POS_12) 
    {
    case 0: display_pos_12_value_0   break;
    case 1: display_pos_12_value_1   break;
    case 2: display_pos_12_value_2   break;
    case 3: display_pos_12_value_3   break;
    case 4: display_pos_12_value_4   break;
    case 5: display_pos_12_value_5   break;
    case 6: display_pos_12_value_6   break;
    case 7: display_pos_12_value_7   break;
    case 8: display_pos_12_value_8   break;
    case 9: display_pos_12_value_9   break;
    }

  DIGIT_POS_13 = (tripA_value-((100000*DIGIT_POS_8)+(10000*DIGIT_POS_9)+(1000*DIGIT_POS_10)+(100*DIGIT_POS_11)+(10*DIGIT_POS_12)));   // 1s digit

  switch(DIGIT_POS_13) 
    {
    case 0: display_pos_13_value_0   break;
    case 1: display_pos_13_value_1   break;
    case 2: display_pos_13_value_2   break;
    case 3: display_pos_13_value_3   break;
    case 4: display_pos_13_value_4   break;
    case 5: display_pos_13_value_5   break;
    case 6: display_pos_13_value_6   break;
    case 7: display_pos_13_value_7   break;
    case 8: display_pos_13_value_8   break;
    case 9: display_pos_13_value_9   break;
    }

  }
  
  //=======================================
void  SET_TRIPB_DIGITS(char t, tU32 tripB_value) 
  {
  
  tU32  DIGIT_POS_8,DIGIT_POS_9,DIGIT_POS_10,DIGIT_POS_11,DIGIT_POS_12,DIGIT_POS_13;    // Temporary values for each of the 6 positions
  
  DIGIT_POS_8 = (tripB_value)/100000;             // 100,000s digit
  
  switch(DIGIT_POS_8) 
    {
    case 0: display_pos_8_value_0   break;
    case 1: display_pos_8_value_1   break;
    case 2: display_pos_8_value_2   break;
    case 3: display_pos_8_value_3   break;
    case 4: display_pos_8_value_4   break;
    case 5: display_pos_8_value_5   break;
    case 6: display_pos_8_value_6   break;
    case 7: display_pos_8_value_7   break;
    case 8: display_pos_8_value_8   break;
    case 9: display_pos_8_value_9   break;
    }
  
  DIGIT_POS_9 = (tripB_value-(100000*DIGIT_POS_8))/10000;   // 10,000s digit

  switch(DIGIT_POS_9) 
    {
    case 0: display_pos_9_value_0   break;
    case 1: display_pos_9_value_1   break;
    case 2: display_pos_9_value_2   break;
    case 3: display_pos_9_value_3   break;
    case 4: display_pos_9_value_4   break;
    case 5: display_pos_9_value_5   break;
    case 6: display_pos_9_value_6   break;
    case 7: display_pos_9_value_7   break;
    case 8: display_pos_9_value_8   break;
    case 9: display_pos_9_value_9   break;
    }

  DIGIT_POS_10 = (tripB_value - (100000*DIGIT_POS_8 + 10000*DIGIT_POS_9))/1000;   // 1,000s digit

  switch(DIGIT_POS_10) 
    {
    case 0: display_pos_10_value_0   break;
    case 1: display_pos_10_value_1   break;
    case 2: display_pos_10_value_2   break;
    case 3: display_pos_10_value_3   break;
    case 4: display_pos_10_value_4   break;
    case 5: display_pos_10_value_5   break;
    case 6: display_pos_10_value_6   break;
    case 7: display_pos_10_value_7   break;
    case 8: display_pos_10_value_8   break;
    case 9: display_pos_10_value_9   break;
    }

  DIGIT_POS_11 = (tripB_value-((100000*DIGIT_POS_8)+(10000*DIGIT_POS_9)+(1000*DIGIT_POS_10)))/100;   // 100s digit
  
  switch(DIGIT_POS_11) 
    {
    case 0: display_pos_11_value_0   break;
    case 1: display_pos_11_value_1   break;
    case 2: display_pos_11_value_2   break;
    case 3: display_pos_11_value_3   break;
    case 4: display_pos_11_value_4   break;
    case 5: display_pos_11_value_5   break;
    case 6: display_pos_11_value_6   break;
    case 7: display_pos_11_value_7   break;
    case 8: display_pos_11_value_8   break;
    case 9: display_pos_11_value_9   break;
    }
  
  DIGIT_POS_12 = (tripB_value-((100000*DIGIT_POS_8)+(10000*DIGIT_POS_9)+(1000*DIGIT_POS_10)+(100*DIGIT_POS_11)))/10;   // 10s digit

  switch(DIGIT_POS_12) 
    {
    case 0: display_pos_12_value_0   break;
    case 1: display_pos_12_value_1   break;
    case 2: display_pos_12_value_2   break;
    case 3: display_pos_12_value_3   break;
    case 4: display_pos_12_value_4   break;
    case 5: display_pos_12_value_5   break;
    case 6: display_pos_12_value_6   break;
    case 7: display_pos_12_value_7   break;
    case 8: display_pos_12_value_8   break;
    case 9: display_pos_12_value_9   break;
    }

  DIGIT_POS_13 = (tripB_value-((100000*DIGIT_POS_8)+(10000*DIGIT_POS_9)+(1000*DIGIT_POS_10)+(100*DIGIT_POS_11)+(10*DIGIT_POS_12)));   // 1s digit

  switch(DIGIT_POS_13) 
    {
    case 0: display_pos_13_value_0   break;
    case 1: display_pos_13_value_1   break;
    case 2: display_pos_13_value_2   break;
    case 3: display_pos_13_value_3   break;
    case 4: display_pos_13_value_4   break;
    case 5: display_pos_13_value_5   break;
    case 6: display_pos_13_value_6   break;
    case 7: display_pos_13_value_7   break;
    case 8: display_pos_13_value_8   break;
    case 9: display_pos_13_value_9   break;
    }

  }
  
//=======================================
void  SET_GEAR (char local_st, enum t_transmission_gear gears) 
  {
  if(SEG_S5==OFF) SEG_S5  = ON;          // Turn on PRND321 segment if not already on
  
  // -------------- Clear all transmission segments
  SEG_P = OFF; SEG_R = OFF; SEG_N = OFF; SEG_D = OFF; SEG_3 = OFF; SEG_2 = OFF; SEG_1 = OFF; 

  if (local_st == ON)
    {
    switch(gears) 
      {
      case Park     : SEG_P = ON; break;  // Turn on relevant segment based on selected gear
      case Reverse  : SEG_R = ON; break;
      case Neutral  : SEG_N = ON; break;
      case Drive    : SEG_D = ON; break;
      case Third    : SEG_3 = ON; break;
      case Second   : SEG_2 = ON; break;
      case First    : SEG_1 = ON; break;
      }
    }
  else		
    {
//    SEG_T1  = 0;  // Surround ring for compass
    }
  
}  

//=======================================
void SET_DIST_TITLE(byte decimal, byte distance, byte remain) 
  {
    SEG_R3 = decimal;   //set LCDRAM for ODO Seg
    SEG_P25 = distance;   // set LCDRAM for miles seg
    SEG_P27 = remain;  // set LCDRAM for TRIPA seg

  }

 //=======================================
void DIST_REMAIN (char t, tU16 dist_value) 
  {
   
  tU16  DIGIT_POS_4, DIGIT_POS_5, DIGIT_POS_6, DIGIT_POS_7;    // Temporary values for each of the 6 positions
  
  DIGIT_POS_4 = (dist_value)/1000;             // 1000s digit
  
  switch(DIGIT_POS_4) 
    {
    case 0: display_pos_4_value_0   break;
    case 1: display_pos_4_value_1   break;
    case 2: display_pos_4_value_2   break;
    case 3: display_pos_4_value_3   break;
    case 4: display_pos_4_value_4   break;
    case 5: display_pos_4_value_5   break;
    case 6: display_pos_4_value_6   break;
    case 7: display_pos_4_value_7   break;
    case 8: display_pos_4_value_8   break;
    case 9: display_pos_4_value_9   break;
    }
  
  DIGIT_POS_5 = (dist_value-(1000*DIGIT_POS_4))/100;   // 10,000s digit

  switch(DIGIT_POS_5) 
    {
    case 0: display_pos_5_value_0   break;
    case 1: display_pos_5_value_1   break;
    case 2: display_pos_5_value_2   break;
    case 3: display_pos_5_value_3   break;
    case 4: display_pos_5_value_4   break;
    case 5: display_pos_5_value_5   break;
    case 6: display_pos_5_value_6   break;
    case 7: display_pos_5_value_7   break;
    case 8: display_pos_5_value_8   break;
    case 9: display_pos_5_value_9   break;
    }

  DIGIT_POS_6 = (dist_value - (1000*DIGIT_POS_4 + 100*DIGIT_POS_5))/10;   // 1,000s digit

  switch(DIGIT_POS_6) 
    {
    case 0: display_pos_6_value_0   break;
    case 1: display_pos_6_value_1   break;
    case 2: display_pos_6_value_2   break;
    case 3: display_pos_6_value_3   break;
    case 4: display_pos_6_value_4   break;
    case 5: display_pos_6_value_5   break;
    case 6: display_pos_6_value_6   break;
    case 7: display_pos_6_value_7   break;
    case 8: display_pos_6_value_8   break;
    case 9: display_pos_6_value_9   break;
    }

  DIGIT_POS_7 = (dist_value-((1000*DIGIT_POS_4)+(100*DIGIT_POS_5)+(10*DIGIT_POS_6)))/1;   // 100s digit
  
  switch(DIGIT_POS_7) 
    {
    case 0: display_pos_7_value_0   break;
    case 1: display_pos_7_value_1   break;
    case 2: display_pos_7_value_2   break;
    case 3: display_pos_7_value_3   break;
    case 4: display_pos_7_value_4   break;
    case 5: display_pos_7_value_5   break;
    case 6: display_pos_7_value_6   break;
    case 7: display_pos_7_value_7   break;
    case 8: display_pos_7_value_8   break;
    case 9: display_pos_7_value_9   break;
    }

  }


 //=======================================
void TYRE_CONFIG (byte cruise, byte traction, byte pressure, byte off) 
  {
    SEG_P36 = cruise;      //set LCDRAM for cruise Seg
    SEG_P37 = traction;    // set LCDRAM for traction seg
    SEG_P39 = pressure;   // set LCDRAM for tyre pressure seg
   // SEG_S4  = engaged;   // set LCDRAM for traction seg
    SEG_S3  = off;  // set LCDRAM for tyre pressure seg
  }


/*
//=======================================
void  SET_COLONS(byte local_lcdrammode, byte col1_st, byte col2_st, byte col3_st, byte minus_st) {
  
		    //configure LCDRAM as SEGONOFF or SEGBLINKEN
  LCDCMD_LCDDRMS = local_lcdrammode;
    
		  LCDRAM3_FP7BP2 = col1_st;	 //col1_st
      LCDRAM9_FP19BP2 = col2_st;	 //col2_st
      LCDRAM13_FP27BP2 = col3_st;	 //col3_st
      LCDRAM1_FP3BP3 = minus_st;	 //minus_st
      
  //default configure LCDRAM as SEGONOFF 
  LCDCMD_LCDDRMS = SEGONOFF;
} 

//=======================================
void  SET_DECIMALS(byte local_lcdrammode, byte deci1_st,byte deci2_st,byte deci3_st,char deci4_st,char deci5_st,char deci6_st,char deci7_st,char deci8_st, char deci9_st) {
     
  //configure LCDRAM as SEGONOFF or SEGBLINKEN
  LCDCMD_LCDDRMS = local_lcdrammode;
  //Set LCDRAM
  LCDRAM1_FP3BP2 = deci1_st;	 //deci1_st
  LCDRAM3_FP7BP3 = deci2_st;	 //decN_st
  LCDRAM5_FP11BP2 = deci3_st;	 //decN_st
  LCDRAM7_FP15BP2 = deci4_st;	 //decN_st
  LCDRAM9_FP19BP3 = deci5_st;	 //decN_st
  LCDRAM11_FP23BP2 = deci6_st;	 //decN_st
  LCDRAM13_FP27BP3 = deci7_st;	 //decN_st
  LCDRAM15_FP31BP2 = deci8_st;	 //decN_st
  LCDRAM17_FP35BP2 = deci9_st;	 //decN_st
  //default configure LCDRAM as SEGONOFF 
  LCDCMD_LCDDRMS = SEGONOFF;
} 



*/


/*---------------------------------------------------------------------------*/
                              /* Revision Log */
/*---------------------------------------------------------------------------*/
/*
* + Aug 30th '10 B30269 Jose Cisneros
*  - Jelly fish part number and Jelly fish simbol not displayed for Seawasp DEMO
*  - S12XhY now is displayed during init time of DEMO in place of S12HY.  
*  - ENABLE_LCD(), SET_ODO_TITLE(), SET_VOLT_BAR(), SET_TEMP_BAR(), SET_ODO_DIGITS(),
*    SET_DIST_TITLE() argument types changed to be the same as definition, warnings
*    cleanup. 
*/