/*****************************************************************************
 * FILE NAME: main.c                            COPYRIGHT (c) FREESCALE 2011 *
 * DESCRIPTION:                                        All Rights Reserved   *
 ****************************************************************************/
 
/* Notes:  */
/*  1. MMU not initialized; must be done by debug scripts or BAM                                    */
/*  2. SRAM not initialized; must be done by debug scripts or in a crt0 type file                   */

/* Use one of the next two pairs of lines for header files                                          */

#include "mpc563m.h"                    /* Use proper include file such as mpc5510.h or mpc5554.h   */
#include "mpc563m_vars.h"                           /* MPC563m specific variables                   */
#include "etpu_util.h"                              /* useful utility routines                      */
#include "etpu_set1.h"                              /* eTPU standard function set 1                 */
#include "etpu_pwm.h"                               /* eTPU PWM API                                 */
#include "stm.h"

/* User written include files                                                                       */
#include "main.h"                                   /* include application specific defines.        */

uint32_t *fs_free_param;                            /* pointer to the first free parameter          */

void initSysclk (void) {
                                                        
 
  FMPLL.ESYNCR2.R = 0x00000002;
  FMPLL.ESYNCR1.R = 0xF0000000 + 40;                /* Set to 40MHz                                 */
  
  
  while (FMPLL.SYNSR.B.LOCK != 1) {};               /* Wait for FMPLL to LOCK                       */
   
  FMPLL.ESYNCR1.R = 0xF0000000 + 64;                /* Set to 64MHz                                 */
 
}


static uint32_t Result     = 0;                     /* ADC conversion result                        */
static uint32_t ResultInMv = 0;                     /* ADC conversion result in millivolts          */

void initADC0(void) {
  EQADC.CFPR[0].R = 0x80800F01;                     /* Send CFIFO 0 a ADC0 configuration command    */
                                                    /* enable ADC0 & sets prescaler= divide by 32   */ 
  EQADC.CFCR[0].R = 0x0410;                         /* Trigger CFIFO 0 using Single Scan SW mode    */
  while (EQADC.FISR[0].B.EOQF !=1) {}               /* Wait for End Of Queue flag                   */
  EQADC.FISR[0].B.EOQF = 1;                         /* Clear End Of Queue flag                      */
}

void SendConvCmd (void) {
  EQADC.CFPR[0].R = 0x80001100;                     /* Conversion command: convert channel 17       */
                                                  /* with ADC0, set EOQ, and send result to RFIFO 0 */ 
  EQADC.CFCR[0].R = 0x0410;                         /* Trigger CFIFO 0 using Single Scan SW mode    */
}

void ReadResult(void) {
  while (EQADC.FISR[0].B.RFDF != 1){}               /* Wait for RFIFO 0's Drain Flag to set         */
  Result = EQADC.RFPR[0].R;                         /* ADC result                                   */        
  ResultInMv = (uint32_t)((5000*Result)/0x3FFC);    /* ADC result in millivolts                     */        
  EQADC.FISR[0].B.RFDF = 1;                         /* Clear RFIFO 0's Drain Flag                   */
  EQADC.FISR[0].B.EOQF = 1;                         /* Clear CFIFO's End of Queue flag              */
  }



void main ()
{
  int32_t  error_code;                              /* Returned value from etpu API functions       */
  uint16_t DutyCycle;
  int16_t  State;

  initSysclk();                                     /* Initialize PLL to 64 MHz                     */
  
  initADC0();                  	                    /* Enable ADC0 only on eQADC                    */
  
                                                    /* Initialize eTPU hardware                     */
  fs_etpu_init (             my_etpu_config, 
                (uint32_t *) etpu_code, 
                             sizeof (etpu_code),
                (uint32_t *) etpu_globals, 
                             sizeof (etpu_globals));
  
  
  
  /* Initialize eTPU channel ETPU_A[5]            */
  error_code = fs_etpu_pwm_init  (5,                /*  Channel ETPU_A[5]                           */
                                  FS_ETPU_PRIORITY_MIDDLE,
                                  1000,             /* Frequency = 1000 Hz                          */
                                  2500,             /* Duty cycle = 2500/100 = 25%                  */
                                  FS_ETPU_PWM_ACTIVEHIGH,
                                  FS_ETPU_TCR1,
                                  1000000);         /* Timebase (TCR1) freq is 1 MHz                */

  SIU.PCR[119].R = 0x0E00;                          /* Configure pad for signal ETPU_A[5] output    */
  
  
  fs_timer_start ();                                /* Enable all timebases                         */
  
  
 
  error_code = fs_etpu_pwm_update (5,               /* Channel ETPU_A[5]                            */
                                   2000,            /* New frequency = 2KHz	                        */
                                   6000,            /* New duty cycle = 6000/100= 60%               */
                                   1000000);        /* Timebase (TCR1) freq = 1 MHz                 */


  DutyCycle = 1;                                    /* Limit the duty cycle from 10% to 90%         */
  State     = 0;                                    /* Initialized the state machine                */
  while(1)                                          /* Wait forever                                 */
  {
    
    SendConvCmd();                                  /* Send one conversion command */
    
    ReadResult();                                   /* Read result */



    error_code = fs_etpu_pwm_update (5,             /* Channel ETPU_A[5]                            */
                                   2000,            /* New frequency = 2KHz	                        */
                                   (100 - DutyCycle)*100,   /* New duty cycle = 6000/100= 60%               */
                                   1000000);        /* Timebase (TCR1) freq = 1 MHz                 */


    State = (uint16_t)(16000 - (Result - 312));
    
    if (State <= 1)
    {
        DutyCycle = 1;        
    }
    else
    {
        DutyCycle = (uint16_t)(State/160);
        if (DutyCycle > 99)
        {
            DutyCycle = 99;
        }
    }
    
    
  }
}











