/****************************************************************************
 * FILE NAME: etpuc_ppa.c                    COPYRIGHT (c) FREESCALE 2004   *
 * DESCRIPTION:                                     All Rights Reserved     *
 *==========================================================================*
 * ORIGINAL AUTHOR: Richard Soja                                            *
 * $Log: etpuc_ppa.c,v $
 * Revision 1.2  2004/11/30 15:58:48  r47354
 * Add #defines for FM bits in API.
 *
 * Revision 1.1  2004/11/02 13:48:07  r12110
 * -First version in CVS.
 * -Added support for building function stand alone or in a set.
 *   -This includes using the global error function if avaialable.
 * -Updated comments in auto generated file to include version number.
 *
 *..........................................................................*
 * 0.00  R. Soya         Feb/04    Initial version of file.                 *
 * 0.1   G. Emerson   7/Sep/04     Changes for integration into set1        *
 *                                 Added flags for testing visibility       *
 * 0.2   G. Emerson   21/Oct/04    Post review changes                      *
 *                                 Removed testing instrumentation etc      *
 *                                 Added comments                           *
 ****************************************************************************/
#ifndef __ETPUC_H
#include <etpuc.h>          /*Defines eTPU hardware*/
#endif

#ifndef __ETPUC_COMMON_H
#include <eTPUc_common.h>   /*Standard way to use eTPU*/
#endif

#ifdef PPA_FUNCTION_NUMBER /* function is being compiled as part a set? */
#pragma ETPU_function PPA, alternate @ PPA_FUNCTION_NUMBER;
#else
#pragma ETPU_function PPA, alternate;  /* stand alone */
#endif

/* Some macro functions */
#define ScheduleTimeoutA(AbsTime) erta = AbsTime; EnableMatchA(); ClearMatchALatch()
#define Assign(a,b)              (b == 1) ? a = 1: a = 0;

/* Host Service Requests */
#define INIT_TCR1 7
#define INIT_TCR2 5

/* Flag definitions */

/*Measurement inactive/active flag*/
#define Measurement flag1
/* Only for setting flag1 (can't use for testing flag) */
#define Active TRUE
#define Inactive FALSE

/* Last pin level */
#define LastPin flag0
#define Low 0
#define High 1

/* Function Mode bits definition */
/* Type of measurement */
#define PulseMeasurement fm0 == 0
#define LowPulse fm1 == 0
#define HighPulse fm1 == 1

#define PeriodMeasurement fm0 == 1
#define FallingEdge fm1 == 0
#define RisingEdge fm1 == 1

#define PulseLow 0 /*Used by the API only*/
#define PulseHigh 2 /*Used by the API only*/

#define PeriodFall 1 /*Used by the API only*/
#define PeriodRise 3 /*Used by the API only*/

/* Some error values */
#define LowPulseError 1
#define HighPulseError 2
#define Overflow 128

/* Hardware definitions */
#define EventTime1 erta
#define EventTime2 ertb
#define Transition_A tdlA == 1

/****************************************************************************
* Channel Flag usage
* Flag0 is used to store the level of the pin sampled in the prvious thread.
* Flag1 is used to indictae wether measurement is active or not
****************************************************************************/

/*--------------------------------------------------------------------------+
| FUNCTION PARAMETERS:                                                      |
| LastTime : This 24 bit value is used by eTPU only. It contains the time of|
|            the last transition during active measurement of a single pulse|
|            or period.                                                     |
|    Accum : This 24 bit value should be initialised to 0 by CPU and        |
|            updated only by the eTPU. The parameter contains the current   |
|            uncompleted period or pulse measurement while measurement is in|
|            progress                                                       |
|  StatAcc : This 8 bit value contains a single bit flag to indicate that   |
|            the currently active accumulation has overflowed a 24 bit      |
|            value. The parameter is initialized to 0 by the CPU and        |
|            updated by the eTPU. The eTPU copies this value to the Status  |
|            parameter when the measurement completes, then resets StatAcc  |
|            to 0. An overflow is indicated when StatAcc is 0x80.           |
|SampleTime: This 24 bit value defines the rate at which the measurement is |
|            updated in the absence of an edge. This allows the host to make|
|            an interim measurement without waiting for the current pulse or|
|            period to complete. The CPU should initialize this value to the|
|            number of timebase counts desired between each sample; the eTPU|
|            schedules the first sample event when a measurement is         |
|            activated by a valid edge. If the sample rate is greater than  |
|            the maximum width between active pulse or period edges, then no|
|            sampling will occur. Note that excessively high sample rates   |
|            (very small values of SampleTime) may result in performance    |
|            degradation of the eTPU.                                       |
|PeriodCount:This 24 bit value is initialized to 0 by the CPU and updated   |
|            only by the eTPU. It contains the current number of accumulated|
|            pulses or periods since initialization or since the last       |
|            measurement was returned in parameter Result                   |
| MaxCount : This 24 bit value specifies the number of pulses or periods    |
|            that should be measured. Initialized by the CPU and tested by  |
|            the eTPU.                                                      |
|   Result : This 24 bit value is updated only by the eTPU. It contains the |
|            final total accumulation when the measurement completes. The   |
|            parameter Status indicates whether the result has overflowed   |
|   Status : This 8 bit value indicates whether the completed measurement,  |
|            available in parameter Result, has overflowed. At the end of   |
|            the measurement, the eTPU copies StatAcc to Status after it has|
|            copied Accum to Result, and then the eTPU sets the channel     |
|            interrupt status flag and DMA request flag. This ensures the   |
|            measurement result and its status are coherent.                |
+--------------------------------------------------------------------------*/

struct ChannelParameters
{int24 Accum;
 int8  StatAcc;
 int24 SampleTime,
       LastTime,
       PeriodCount,
       MaxCount,
       Result;
 int8  Status;
} register_chan_base *this;


void CheckPin()
{
/*---------------------------------------------------------------------------------------+
| This re-writes the chan value and at the same time clears the transition latch (tdl).  |
| This causes the pin state to be latched.                                               |
| If the pin does not transition after it is latched, tdl will remain clear.             |
| If the pin changes state after it is latched, tdl will be set again, and processed     |
| after this thread quits.| The design of the hardware ensures the pin state and value   |
| of tdl are coherent.                                                                   |
| tdl does not have to be checked in the current thread.                                 |
| It is the current (latched in this sub routine) and last pin levels that determine     |
| the operation of the state machine                                                     |
| The current pin level is obtained from channel.PSS                                     |
+----------------------------------------------------------------------------------------*/

    ClearTransLatch();
    chan = chan;
}

void AccumTime()
{
    this->Accum = this->Accum + (EventTime1 - this->LastTime);
    if (CC.C == 1)
    {
       this->StatAcc = this->StatAcc | Overflow;
       SetChannelInterrupt();    /* Generate Interrupt and DMA requests */
       SetDataTransferInterrupt();
    }
}

void Read1Retrig()
{
this->LastTime = EventTime1;
ScheduleTimeoutA(EventTime1 + this->SampleTime);
}

void Read2Retrig()
{
this->LastTime = EventTime2;
ScheduleTimeoutA(EventTime2 + this->SampleTime);
}

void CheckCount()
{
   this->PeriodCount++;
   if (this->PeriodCount >= this->MaxCount)
   {

      this->Result = this->Accum;
      this->Status = this->StatAcc;
      this->Accum = 0;
      this->StatAcc = 0;
      this->PeriodCount = 0;

      SetChannelInterrupt();
      SetDataTransferInterrupt(); /* Generate Interrupt and DMA requests */
    }
}

void MeasGlitch()
{
this->Accum = this->Accum + EventTime2 - EventTime1;
}

void PPA(struct ChannelParameters this)
{
/*--------------------------------------------------------------------------+
| THREAD NAME: THREAD1                                                      |
| DESCRIPTION:  Initialise channel for PPA using TCR2 as timebase.          |
|Disable matches in thread in case there are any previous pending matches.  |
|This could happen if the channel was running another function before this  |
|one.                                                                       |
|Configure channel to use TCR2.                                             |
|Proceed as per thread THREAD2 (from label init)                            |
|                                                                           |
+--------------------------------------------------------------------------*/
	if ( hsr == INIT_TCR2 ) /*  Required for API*/
	{
			DisableMatchesInThread();
			/* set up time base for TCR1*/
			ActionUnitA( MatchTCR2, CaptureTCR2, GreaterEqual);
	    goto init;

	}
/*--------------------------------------------------------------------------+
| THREAD NAME: THREAD2                                                      |
| DESCRIPTION:  Initialise channel for PPA using TCR1 as timebase.          |
|Disable matches in thread in case there are any previous pending matches.  |
|This could happen if the channel was running another function before this  |
|one.                                                                       |
|Configure channel to use TCR1.                                             |
|Disable the output buffer as this is an input function.                    |
|Prevent the pin from toggling as a result of matches on ActionUnitA/B.     |
|Clear latches. Set Measurement = Inactive.                                 |
|Configure the channel mode.                                                |
|Enable event handling.                                                     |
|Set IPACA/B dependant upon mode                                            |
|Set LastPin dependant upon mode                                            |
|                                                                           |
+--------------------------------------------------------------------------*/
	else if ( hsr == INIT_TCR1 ) /*  Required for API*/
	{
			DisableMatchesInThread();

			/* set up time base for TCR1*/
			ActionUnitA( MatchTCR1, CaptureTCR1, GreaterEqual);

init:

      DisableOutputBuffer(); /* required for Puma */

	    OnMatchA(NoChange);  /* Needed so ouptut pin does not get toggled */
	    OnMatchB(NoChange);  /* Needed so ouptut pin does not get toggled */

      ClearAllLatches();
	    Measurement = Inactive;
	    SingleMatchSingleTransition(); /* Enable service request when first edge occurs */

	    EnableEventHandling();
	    if (PulseMeasurement)
	    {
			/*--------------------------------------------------------------------------+
			| STATE NAME: INIT1                                                         |
			+--------------------------------------------------------------------------*/

					OnTransA(any_trans);   /* Detect on any edge on both input capture registers */
					OnTransB(any_trans);
	        CheckPin();
	        Assign(LastPin,SampledInputPin); /* channel struct member can't be Assigned directly with var. */
	    }
	    else                      /* initialize period measurement */
	    {
			/*--------------------------------------------------------------------------+
			| STATE NAME: INIT2                                                         |
			+--------------------------------------------------------------------------*/

	        LastPin = 0;          /* Force period measurement to only use states with LastPin=0 */
					OnTransB(no_detect);
					OnTransA(high_low);
	        if (RisingEdge) OnTransA(low_high);
	    }
	}
/*--------------------------------------------------------------------------+
| THREAD NAME: THREAD3                                                      |
| DESCRIPTION:                                                              |
| Measurement inactive service request, LastPin high                        |
| If entered due to a transition; then:                                     |
| 	Latch the pin state and clear transition detect latch                   |
|   Set LastPin to the sample pin state                                     |
|   Execute state TRANS1 or TRANS2 or TRANS3 or TRANS4 dependant upon       |
|     PulseMeasurement/PinSampledState                                      |
| Else must have been entered due to a match; then                          |
|   Execute state MATCH1                                                    |
| NB Can't be period measurement because LastPin must be 0 for Period       |
|    Measurement                                                            |
|                                                                           |
+--------------------------------------------------------------------------*/
	else if (((Measurement==Inactive) && (LastPin==High) && m1) ||
				   ((Measurement==Inactive) && (LastPin==High) && m2))
	{
	    if (Transition_A)
	    {
	       CheckPin();	      /* Also clears transition flag */
	       Assign(LastPin,SampledInputPin);	/* update Last_Pin state */
	       if (LowPulse)
	       {
	          if (SampledInputPin == 0)       /* Start normal low measurement */
	          {
						/*--------------------------------------------------------------------------+
						| STATE NAME: TRANS1                                                        |
						| Conditions under which this state is executed:-                           |
						|   Measurement = Inactive                                                  |
						|   LastPin = High                                                          |
						|   Transition occurred                                                     |
						|   Mode is Low pulse measurement                                           |
						|   Channel.PSS is 0                                                        |
						|Start normal low measurement                                               |
						|Record erta in LastTime                                                    |
      		 	|Clear match flagA                                                          |
						|Schedule a match SampleTime in the future                                  |
						|Set Measurement to Active                                                  |
						|                                                                           |
						+--------------------------------------------------------------------------*/

	             Read1Retrig();
	             Measurement = Active;
	          }
	          else                /* a short negative pulse occurred */
	          {
						/*--------------------------------------------------------------------------+
						| STATE NAME: TRANS2                                                        |
						| Conditions under which this state is executed:-                           |
						|   Measurement = Inactive                                                  |
						|   LastPin = High                                                          |
						|   Transition occurred                                                     |
						|   Mode is Low pulse measurement                                           |
						|   Channel.PSS is 1                                                        |
						|i.e. a short negative pulse occurred                                       |
						|Add short pulse width to Accum                                             |
      		 	|Clear match flagA                                                          |
						|Schedule a match SampleTime in the future                                  |
						|If sufficient LowPulses have been accumulated update Result with Accum     |
						|  and Status takes the value of StatAcc; StatAcc get set to 0              |
						|Measurement remains Inactive                                               |
						|                                                                           |
						+--------------------------------------------------------------------------*/

	             MeasGlitch();   /* read both capture registers */
	             CheckCount();	/* Retain inactive state */
	          }
	       }
	       else	/* High pulse measurement */
	       {
	          if (SampledInputPin == 1)       /* Glitched start of high measurement */
	          {
						/*--------------------------------------------------------------------------+
						| STATE NAME: TRANS3                                                        |
						| Conditions under which this state is executed:-                           |
						|   Measurement = Inactive                                                  |
						|   LastPin = High                                                          |
						|   Transition occurred                                                     |
						|   Mode is High pulse measurement                                          |
						|   Channel.PSS is 1                                                        |
						|i.e. a glitch occurred at the start of a high pulse measurement            |
						|Record ertb in LastTime                                                    |
      		 	|Clear match flagA                                                          |
						|Schedule a match SampleTime in the future                                  |
						|Set Measurement to Active                                                  |
						|                                                                           |
						+--------------------------------------------------------------------------*/

	             Read2Retrig();	/* Start high measurement on second capture edge */
	             Measurement = Active;
	          }
	          else	/* First edge goes low, so wait for next edge */
						/*--------------------------------------------------------------------------+
						| STATE NAME: TRANS4                                                        |
						| Conditions under which this state is executed:-                           |
						|   Measurement = Inactive                                                  |
						|   LastPin = High                                                          |
						|   Transition occurred                                                     |
						|   Mode is High pulse measurement                                          |
						|   Channel.PSS is 0                                                        |
						|i.e. The first edge was falling                                            |
						|Do nothing; wait for next edge                                             |
						+--------------------------------------------------------------------------*/

	          {}      /* Do_Nothing */
	       }
	    }    /* end Transition_A */
 			else /* It's an invalid match */
 			{
 			/*--------------------------------------------------------------------------+
			| STATE NAME: MATCH1                                                        |
			| Conditions under which this state is executed:-                           |
			|   Measurement = Inactive                                                  |
			|   LastPin = High                                                          |
			|   Match occurred                                                          |
			|This is an invalid match condition                                         |
			|Clear match flagA                                                          |
			|                                                                           |
			+--------------------------------------------------------------------------*/

     	ClearMatchALatch();
 			}
	}        /* end Measurement inactive service request, LastPin high */
/*--------------------------------------------------------------------------+
| THREAD NAME: THREAD4                                                      |
| DESCRIPTION:                                                              |
| Measurement inactive service request, LastPin low                         |
| If entered due to a transition; then:                                     |
| 	If doing pulse measurement                                              |
| 		Latch the pin state and clear transition detect latch                 |
|   	Set LastPin to the sample pin state                                   |
|   	Execute state TRANS5 or TRANS6 or TRANS7 or TRANS8 dependant upon     |
|   	  PulseMeasurement/PinSampledState                                    |
|		Else doing period measurement                                           |
|   	Execute state TRANS9                                                  |
| Else must have been entered due to a match; then                          |
|   Execute state MATCH2                                                    |
|                                                                           |
+--------------------------------------------------------------------------*/
	else if (((Measurement==Inactive) && (LastPin==Low) && m1) ||
					 ((Measurement==Inactive) && (LastPin==Low) && m2))
	{
	    if (Transition_A)
	    {
	       if (PulseMeasurement)
	       {
	          CheckPin();	      /* Also clears transition flag */
	          Assign(LastPin,SampledInputPin);	/* update Last_Pin state */
	          if (HighPulse)
	          {
	             if (SampledInputPin == 1)    /* Start normal high measurement */
	             {
			  			 /*--------------------------------------------------------------------------+
							 | STATE NAME: TRANS5                                                        |
						 	 | Conditions under which this state is executed:-                           |
						 	 |   Measurement = Inactive                                                  |
						 	 |   LastPin = Low                                                           |
						 	 |   Transition occurred                                                     |
							 |   Mode is High  pulse measurement                                         |
							 |   Channel.PSS is 1                                                        |
							 |Start normal high measurement                                              |
							 |Record erta in LastTime                                                    |
      		 	   |Clear match flagA                                                          |
							 |Schedule a match SampleTime in the future                                  |
							 |Set Measurement to Active                                                  |
							 |                                                                           |
							 +--------------------------------------------------------------------------*/

	                Read1Retrig();
	                Measurement = Active;
	             }
	             else	/* a glitch occurred */
	             {
			  			 /*--------------------------------------------------------------------------+
							 | STATE NAME: TRANS6                                                        |
						   | Conditions under which this state is executed:-                           |
					     |   Measurement = Inactive                                                  |
						   |   LastPin = Low                                                           |
						   |   Transition occurred                                                     |
						   |   Mode is High pulse measurement                                          |
						   |   Channel.PSS is 0                                                        |
						   |i.e. a short positive pulse occurred                                       |
						   |Add short pulse width to Accum                                             |
      		 	   |Clear match flagA                                                          |
						   |Schedule a match SampleTime in the future                                  |
						   |If sufficient HighPulses have been accumulated update Result with Accum    |
						   |  and Status takes the value of StatAcc; StatAcc get set to 0              |
						   |Measurement remains Inactive                                               |
						   |                                                                           |
							 +--------------------------------------------------------------------------*/

	                MeasGlitch();	/* read both capture registers */
	                CheckCount();	/* Retain inactive state */
	             }
	          }
	          else	/* Low pulse measurement */
	          {
	             if (SampledInputPin == 0)	/* Glitched start of low measurement */
	             {
			  			 /*--------------------------------------------------------------------------+
							 | STATE NAME: TRANS7                                                        |
						   | Conditions under which this state is executed:-                           |
						   |   Measurement = Inactive                                                  |
						   |   LastPin = Low                                                           |
						   |   Transition occurred                                                     |
						   |   Mode is Low pulse measurement                                           |
						   |   Channel.PSS is 0                                                        |
						   |i.e. a glitch occurred at the start of a low pulse measurement             |
						   |Record ertb in LastTime                                                    |
      		 	   |Clear match flagA                                                          |
  					   |Schedule a match SampleTime in the future                                  |
						   |Set Measurement to Active                                                  |
						   |                                                                           |
						   +--------------------------------------------------------------------------*/
	                Read2Retrig(); /* Start low measurement on second capture edge */
	                Measurement = Active;
	             }
			  			 /*--------------------------------------------------------------------------+
							 | STATE NAME: TRANS8                                                        |
						   | Conditions under which this state is executed:-                           |
							 |   Measurement = Inactive                                                  |
							 |   LastPin = Low                                                           |
							 |   Transition occurred                                                     |
							 |   Mode is Low pulse measurement                                           |
							 |   Channel.PSS is 1                                                        |
							 |i.e. The first edge was rising                                             |
							 |Do nothing; wait for next edge                                             |
							 +--------------------------------------------------------------------------*/
	             else             /* First edge goes high, so wait for next edge */
	             {}               /* Do_Nothing */
	          }
	       }
	       else                   /* It's a period measurement */
	       {
			   /*--------------------------------------------------------------------------+
				 | STATE NAME: TRANS9                                                        |
				 | Conditions under which this state is executed:-                           |
				 |   Measurement = Inactive                                                  |
				 |   LastPin = Low                                                           |
				 |   Transition occurred                                                     |
				 |   Mode is Period measurement                                              |
				 |i.e. Start of period measurment                                            |
				 |Clear transition detect latch                                              |
				 |Record erta in LastTime                                                    |
         |Clear match flagA                                                          |
				 |Schedule a match SampleTime in the future                                  |
				 |Set Measurement to Active                                                  |
				 |                                                                           |
				 +--------------------------------------------------------------------------*/

            ClearTransLatch();
	          Read1Retrig();
	          Measurement = Active;
	       } /* end period measurement */
	    }     /* end Transition_A */
	    else /* It's an invalid match */
	    {
			/*--------------------------------------------------------------------------+
			| STATE NAME: MATCH2                                                        |
			| Conditions under which this state is executed:-                           |
			|   Measurement = Inactive                                                  |
			|   LastPin = Low                                                           |
			|   Match occurred                                                          |
			|This is an invalid match condition                                         |
			|Clear match flag A                                                     |
			|                                                                           |
			+--------------------------------------------------------------------------*/

	        ClearMatchALatch();
	    }
    }    /* end Measurement inactive service request, LastPin low */
/*--------------------------------------------------------------------------+
| THREAD NAME: THREAD5                                                      |
| DESCRIPTION:                                                              |
| Measurement active service request, LastPin low                           |
| If entered due to a transition; then:                                     |
| 	If doing pulse measurement                                              |
| 		Latch the pin state and clear transition detect latch                 |
|   	Set LastPin to the sample pin state                                   |
|   	Execute state TRANS10 or TRANS11 or TRANS12 dependant upon            |
|   	  PulseMeasurement/PinSampledState                                    |
|		Else doing period measurement                                           |
|   	Execute state TRANS13                                                 |
| Else must have been entered due to a match; then                          |
|   Execute state MATCH3                                                    |
|                                                                           |
+--------------------------------------------------------------------------*/
  else if (((Measurement==Active) && (LastPin==Low) && m1) ||
   			   ((Measurement==Active) && (LastPin==Low) && m2))
      {
          if (Transition_A)
          {
              if (PulseMeasurement)
              {
                 CheckPin();
                 Assign(LastPin,SampledInputPin);    /* update Last_Pin state */
                 if (LowPulse)
                 {
                    if (SampledInputPin == 1)        /* Stop normal low measurement */
                    {
								    /*--------------------------------------------------------------------------+
									  | STATE NAME: TRANS10                                                       |
						   			| Conditions under which this state is executed:-                           |
							 			|   Measurement = active                                                    |
							 			|   LastPin = Low                                                           |
							 			|   Transition occurred                                                     |
							 			|   Mode is Low pulse measurement                                           |
							 			|   Channel.PSS is 1                                                        |
							 			|Stop Normal Low Pulse Measurement                                          |
										|Add TCRs elapsed between this event and the previous event to Accum        |
										|If Accum overflows send a Channel Interrupt and DMA Request                |
						   			|If sufficient Periods have been accumulated update Result with Accum       |
						   			|  and Status takes the value of StatAcc; StatAcc get set to 0              |
						   			|Measurement set to Inactive                                                |
										|                                                                           |
									  +--------------------------------------------------------------------------*/

                       AccumTime();
                       CheckCount();     /* Update measurement if all done */
                       Measurement = Inactive;
                    }
                    else	           /* A narrow positive pulse occurred */
                    {
								    /*--------------------------------------------------------------------------+
									  | STATE NAME: TRANS11                                                       |
						   			| Conditions under which this state is executed:-                           |
							 			|   Measurement = active                                                    |
							 			|   LastPin = Low                                                           |
							 			|   Transition occurred                                                     |
							 			|   Mode is Low pulse measurement                                           |
							 			|   Channel.PSS is 0                                                        |
							 			|A narrow positive pulse occurred                                           |
										|Add TCRs elapsed between this event and the previous event to Accum        |
										|If Accum overflows send a Channel Interrupt and DMA Request                |
						   			|Record ertb in LastTime                                                    |
            		 	  |Clear match flagA                                                          |
						   			|Schedule a match SampleTime in the future                                  |
						   			|If sufficient Periods have been accumulated update Result with Accum       |
						   			|  and Status takes the value of StatAcc; StatAcc get set to 0              |
						   			|Measurement remains Active                                                 |
										|                                                                           |
										+--------------------------------------------------------------------------*/

                       AccumTime();      /* Accumulate time to rising edge, */
                       Read2Retrig();    /* start next measurement on second edge (measurement stays active) */
                       CheckCount();     /* and update measurement if all done */
                    }
                 }
                 else	                 /* Illegal state for High pulse measurement - should never happen */
                 {
								    /*--------------------------------------------------------------------------+
									  | STATE NAME: TRANS12                                                       |
						   			| Conditions under which this state is executed:-                           |
							 			|   Measurement = active                                                    |
							 			|   LastPin = Low                                                           |
							 			|   Transition occurred                                                     |
							 			|   Mode is High pulse measurement                                          |
							 			|Illegal state for High pulse measurement                                   |
										|Set HighPulseError flag in StatAcc                                         |
										|Set Measurement to Inactive                                                |
										|                                                                           |
									  +--------------------------------------------------------------------------*/

                    this.StatAcc = this.StatAcc | HighPulseError;
                    Measurement = Inactive;    /* Disable measurement */
                 }
              }
              else                       /* it's a period measurement */
              {
							/*--------------------------------------------------------------------------+
							| STATE NAME: TRANS13                                                       |
				 			| Conditions under which this state is executed:-                           |
				 			|   Measurement =Active                                                     |
				 			|   LastPin = Low                                                           |
				 			|   Transition occurred                                                     |
				 			|   Mode is Period measurement                                              |
              |Falling edge of rising edge period measurement OR                          |
              |rising edge of falling edge period measurement				 			                |
              |Clear transition detect latch                                              |
							|Add TCRs elapsed between this event and the previous event to Accum        |
							|If Accum overflows send a Channel Interrupt and DMA Request                |
				 			|Record erta in LastTime                                                    |
				 			|Schedule a match SampleTime in the future                                  |
						  |If sufficient Periods have been accumulated update Result with Accum       |
						  |  and Status takes the value of StatAcc; StatAcc get set to 0              |
				 			| Measurement remains Active                                                |
				 			+--------------------------------------------------------------------------*/

                 ClearTransLatch();
                 AccumTime();
                 Read1Retrig();
                 CheckCount();           /* measurement always stays active in period mode, until re-initialized */
              }
          }
          else    /* it is a match */
          {
					/*--------------------------------------------------------------------------+
					| STATE NAME: MATCH3                                                        |
					| Conditions under which this state is executed:-                           |
					|   Measurement = Active                                                    |
					|   LastPin = Low                                                           |
					|   Match occurred                                                          |
					|Add TCRs elapsed between this event and the previous event to Accum        |
					|If Accum overflows send a Channel Interrupt and DMA Request                |
				 	|Record erta in LastTime                                                    |
				 	|Clear match flagA                                                          |
				 	|Schedule a match SampleTime in the future                                  |
					|                                                                           |
					+--------------------------------------------------------------------------*/

              AccumTime();
              Read1Retrig();
          }    /* end Transition_A */
      } /* end Measurement active service request, LastPin low */
/*--------------------------------------------------------------------------+
| THREAD NAME: THREAD6                                                      |
| DESCRIPTION:                                                              |
| Measurement active service request, LastPin high                          |
| If entered due to a transition; then:                                     |
| 	Latch the pin state and clear transition detect latch                   |
|   Set LastPin to the sample pin state                                     |
|   Execute state TRANS14 or TRANS15 or TRANS16 dependant upon              |
|     PulseMeasurement/PinSampledState                                      |
| Else must have been entered due to a match; then                          |
|   Execute state MATCH4                                                    |
| NB Can't be period measurement because LastPin must be 0 for Period       |
|    Measurement                                                            |
|                                                                           |
+--------------------------------------------------------------------------*/
else if (((Measurement==Active) && (LastPin==High) && m1) ||
 			   ((Measurement==Active) && (LastPin==High) && m2))
      {
          if (Transition_A)
          {
              CheckPin();
              Assign(LastPin,SampledInputPin);	/* update Last_Pin state */
              if (HighPulse)
              {
                  if (SampledInputPin == 0)        /* Stop normal high measurement */
                  {
								  /*--------------------------------------------------------------------------+
									| STATE NAME: TRANS14                                                       |
						   		| Conditions under which this state is executed:-                           |
							 		|   Measurement = active                                                    |
							 		|   LastPin = HIgh                                                          |
							 		|   Transition occurred                                                     |
							 		|   Mode is High pulse measurement                                          |
							 		|   Channel.PSS is 0                                                        |
							 		|Stop High Pulse Measurement                                                |
									|Add TCRs elapsed between this event and the previous event to Accum        |
									|If Accum overflows send a Channel Interrupt and DMA Request                |
						   		|If sufficient Periods have been accumulated update Result with Accum       |
						   		|  and Status takes the value of StatAcc; StatAcc get set to 0              |
						   		|Measurement set to Inactive                                                |
									|                                                                           |
								  +--------------------------------------------------------------------------*/

                      AccumTime();
                      CheckCount();    /* Update measurement if all done */
                      Measurement = Inactive;
                  }
                  else	               /* A narrow negative pulse occurred */
                  {
							    /*--------------------------------------------------------------------------+
								  | STATE NAME: TRANS15                                                       |
						  		| Conditions under which this state is executed:-                           |
									|   Measurement = active                                                    |
									|   LastPin = High                                                          |
									|   Transition occurred                                                     |
									|   Mode is High pulse measurement                                          |
							 		|   Channel.PSS is 1                                                        |
							 		|A short negative pulse occurred                                            |
									|Add TCRs elapsed between this event and the previous event to Accum        |
									|If Accum overflows send a Channel Interrupt and DMA Request                |
						   		|Record ertb in LastTime                                                    |
            		  |Clear match flagA                                                          |
						   		|Schedule a match SampleTime in the future                                  |
						   		|If sufficient Periods have been accumulated update Result with Accum       |
						   		|  and Status takes the value of StatAcc; StatAcc get set to 0              |
						   		|Measurement remains Active                                                 |
									|                                                                           |
									+--------------------------------------------------------------------------*/

                      AccumTime();     /* Accumulate time to falling edge */
                      Read2Retrig();   /* start next measurement on second edge (measurement stays active) */
                      CheckCount();    /* and update measurement if all done */

                  }
              }
              else	               /* Illegal state for Low pulse measurement - should never happen */
              {
							/*--------------------------------------------------------------------------+
							| STATE NAME: TRANS16                                                       |
						  | Conditions under which this state is executed:-                           |
							|   Measurement = active                                                    |
							|   LastPin = High                                                          |
							|   Transition occurred                                                     |
							|   Mode is Low pulse measurement                                           |
							|Illegal state for Low pulse measurement                                    |
							|Set LowPulseError flag in StatAcc                                          |
							|Set Measurement to Inactive                                                |
							|                                                                           |
							+--------------------------------------------------------------------------*/

                  this.StatAcc = this.StatAcc | LowPulseError;
                  Measurement = Inactive;    /* Disable measurement */
              }
          }
          else    /* it is a match */
          {
 					/*--------------------------------------------------------------------------+
					| STATE NAME: MATCH4                                                        |
					| Conditions under which this state is executed:-                           |
					|   Measurement = active                                                    |
					|   LastPin = High                                                          |
					|   Match occurred                                                          |
					|Add TCRs elapsed between this event and the previous event to Accum        |
					|If Accum overflows send a Channel Interrupt and DMA Request                |
				 	|Record erta in LastTime                                                    |
				 	|Clear match flagA                                                          |
				 	|Schedule a match SampleTime in the future                                  |
					|                                                                           |
					+--------------------------------------------------------------------------*/

              AccumTime();
              Read1Retrig();
          }    /* end Transition_A */
      } /* end Measurement active service request, LastPin High */

	  else
		/*--------------------------------------------------------------------------+
		| THREAD NAME: THREAD7                                                      |
		| DESCRIPTION: Something has gone wrong. :-(                                |
		|                Call the Global error routine if available and end.        |
		+--------------------------------------------------------------------------*/
	  {
#ifdef GLOBAL_ERROR_FUNC
        Global_Error_Func();
#else
		ClearAllLatches();
#endif
	  }
}

/* Information exported to Host CPU program */
#pragma write h, (::ETPUfilename (cpu/etpu_ppa_auto.h));
#pragma write h, ( );
#pragma write h, (/****************************************************************);
#pragma write h, ( * WARNING this file is automatically generated DO NOT EDIT IT! *);
#pragma write h, ( *                                                              *);
#pragma write h, ( * FILE NAME: etpu_ppa_auto.c      COPYRIGHT (c) Freescale 2004 *);
#pragma write h, ( *                                      All Rights Reserved     *);
#pragma write h, ( * This file generated by:                                      *);
#pragma write h, ( * $RCSfile: etpuc_ppa.c,v $ $Revision: 1.2 $);
#pragma write h, ( *                                                              *);
#pragma write h, ( * This file provides an interface between eTPU code and CPU    *);
#pragma write h, ( * code. All references to the PPA function should be made with *);
#pragma write h, ( * information in this file. This allows only symbolic          *);
#pragma write h, ( * information to be referenced which allows the eTPU code to be*);
#pragma write h, ( * optimized without effecting the CPU code.                    *);
#pragma write h, ( ****************************************************************/);
#pragma write h, (#ifndef _ETPU_PPA_AUTO_H_ );
#pragma write h, (#define _ETPU_PPA_AUTO_H_ );
#pragma write h, ( );
#pragma write h, (/* Function Configuration Information */);
#pragma write h, (::ETPUliteral(#define FS_ETPU_PPA_FUNCTION_NUMBER) PPA_FUNCTION_NUMBER );
#pragma write h, (::ETPUliteral(#define FS_ETPU_PPA_TABLE_SELECT) ::ETPUentrytype(PPA) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_PPA_NUM_PARMS) ::ETPUram(PPA) );
#pragma write h, ( );
#pragma write h, (/* Host Service Request Definitions -timer options*/);
#pragma write h, (::ETPUliteral(#define FS_ETPU_PPA_INIT_TCR1) INIT_TCR1  );
#pragma write h, (::ETPUliteral(#define FS_ETPU_PPA_INIT_TCR2) INIT_TCR2  );
#pragma write h, ( );
#pragma write h, (/* Function Mode Bit Definitions - pulse/period measurement options - mode options*/);
#pragma write h, (::ETPUliteral(#define FS_ETPU_PPA_LOW_PULSE) PulseLow  );
#pragma write h, (::ETPUliteral(#define FS_ETPU_PPA_HIGH_PULSE) PulseHigh  );
#pragma write h, (::ETPUliteral(#define FS_ETPU_PPA_FALLING_EDGE) PeriodFall  );
#pragma write h, (::ETPUliteral(#define FS_ETPU_PPA_RISING_EDGE) PeriodRise  );
#pragma write h, ( );
#pragma write h, (/* Parameter Definitions */);
#pragma write h, (::ETPUliteral(#define FS_ETPU_PPA_ACCUM_OFFSET) ::ETPUlocation (PPA, this.Accum) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_PPA_SAMPLE_TIME_OFFSET) ::ETPUlocation (PPA, this.SampleTime) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_PPA_PERIOD_COUNT_OFFSET) ::ETPUlocation (PPA, this.PeriodCount) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_PPA_MAX_COUNT_OFFSET) ::ETPUlocation (PPA, this.MaxCount) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_PPA_RESULT_OFFSET) ::ETPUlocation (PPA, this.Result) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_PPA_STAT_ACC_OFFSET) ::ETPUlocation (PPA, this.StatAcc) );
#pragma write h, (::ETPUliteral(#define FS_ETPU_PPA_STATUS_OFFSET) ::ETPUlocation (PPA, this.Status) );
#pragma write h, ( );
#pragma write h, (#endif /* _ETPU_PPA_AUTO_H_ */);

