
/******************************************************************************
  FILE        : ADC_Cfg.c
  PURPOSE     : Configure ADC module and provide some ADC services like 
                adc signal filtering.
  MACHINE     : Freescale 68HC12 (Target)
  LANGUAGE    : ANSI-C
  HISTORY     : b30269 First version created
******************************************************************************/


/*===========================================================================*\
    standard header files
\*===========================================================================*/
/*===========================================================================*\
    Additional header files
\*===========================================================================*/
#include "adc_cfg.h"
#include "pwm_cfg.h"
/*===========================================================================*\
*      #define constants 
\*===========================================================================*/

/* Notice that the followint filter contants are tuned to have a cut off 
   frequency of 50 Hz 
*/
#define K0               (uchar) 12
#define K1               (uchar)244

#define OPEN_LOAD_THR    (int) 0x0F00
/* Delay to start diagnostics */
#define ON_DELAY_INIT    0x000FFFFF

/*===========================================================================*\
*      Local type declarations (enum, struct, union, typedef)
\*===========================================================================*/

/*===========================================================================*\
*      Local object definitions must defined as static!
\*===========================================================================*/
static bool OpenLoad        = FALSE;
/*===========================================================================*\
*      Exported object definitions
\*===========================================================================*/
U_Word16_T ADC_Ch1_Flt;
uchar      ADC_CH1_Raw_8bit = 0;
unsigned long On_Delay      = ON_DELAY_INIT;
/*===========================================================================*\
*     Local function prototypes
\*===========================================================================*/
void  ADC_LowPassFilter(uint8_t *input, U_Word16_T *output);
/*===========================================================================*\
*      Function definitions
\*===========================================================================*/

/*===========================================================================*\
*
*  Function Name: ADC_An1Init
*
*  Description:   ADC module configuration for:
*                 No WRAP
*                 External trigger (PP1 (PWM CH1)).
*                 Discharge cap before sampling.
*                 External trigger type to high level trigger.
*                 Enable external trigger.
*                 Right justified conversion result.
*                 Single convertion per trigger.
*                 Sample time for 4 CLKs.
*                 No Prescaler used.
*
*
*  Parameters:    None.    
*
*  Return Value:  None  
*
\*===========================================================================*/
void ADC_An1Init(void)
{
   /* Not all ATD registers are 0ed at reset */
   ATD_Reg_Clr(ATDCTL0);
   ATD_Reg_Clr(ATDCTL1);
   ATD_Reg_Clr(ATDCTL3);
   ATD_Reg_Clr(ATDCTL4);
   /* Select wrap function */
   ATD_Wrap(NO_GRAP_MSK);
   /* Determine if external trigger is required */  
   ATD_Ext_Trig_Sel(ATDCTL1_ETRIGSEL_MASK|ATDCTL1_ETRIGCH0_MASK);
   /* Determine if AD capacitor required to be discharged after conversions */   
   ATD_Cap_Disch_En(ATDCTL1_SMP_DIS_MASK);
   /* ETRIG1 PP1(PWM channel 1) */
   ATD_Ext_Trig_Type(ATDCTL2_ETRIGLE_MASK | ATDCTL2_ETRIGP_MASK);
   //ATD_EndOFConvEn(ATDCTL2_ASCIE_MASK);
   ATD_Ext_Trig_En(ATDCTL2_ETRIGE_MASK);  
   /* ADC result Justification selection */
   ATD_Result_Just(ATDCTL3_DJM_MASK);
   ATD_Number_Conv(ATDCTL3_S1C_MASK);
   /* Select amount of ADC clk cycles to finish a conversion */
   ATD_Set_CLK_Conv(SAMPLE_TIME_4CLKS_MSK);
   /* Select ADC converion prescaler */
   ATD_Set_Prescaler(NO_PRESCALER_MSK);
   /* Enable CHx to convert */
   ATD_Ch_Conv(ATD_CH1); 
}

/*===========================================================================*\
*
*  Function Name: ADC_LowPassFilter
*
*  Description: This function filteres input value and wtores the result into 
*               output in a 16 bit format, the filter characteristics are
*               determined by K0 and K1 constants as shown inthe following 
*               equation ADC_Sn = K1 * ADC_S1 + K0 * ADC_S0.
*
*  Parameters:     pointer to filter input and pointer to filter output
*
*  Return Value:   None
*
\*===========================================================================*/
void ADC_LowPassFilter(uint8_t *input, U_Word16_T *output)
{
   uint16_t   in_aux;
   uint16_t   out_aux;

   in_aux  = (uint16_t)(*input * K0);
   out_aux = ((uint16_t)(output->Field.LoByte * K1) >> 8);
   out_aux += (uint16_t)(output->Field.HiByte * K1);
   /* Intentionaly not verifying overflow */
   output->Word     = in_aux + out_aux;
}

/*===========================================================================*\
*
*  Function Name: OpenLoadMonitor
*
*  Description: This function is used form an extern procedure to monitor
*               the Open load diagnostic, this function stops PWM signal and 
*               initializes a delay for the next load activation.
*
*  Parameters:  None   
*
*  Return Value:   None
*
\*===========================================================================*/

void  OpenLoadMonitor(void)
{
   if((OPEN_LOAD_THR > ADC_Ch1_Flt.Word) &&
       !PWM_Off)
   {
      PWM_Off  = TRUE;
      OpenLoad = TRUE;
      On_Delay = ON_DELAY_INIT;
   }
   else
   {
      OpenLoad = FALSE;
   }
}

#pragma CODE_SEG NON_BANKED
#pragma TRAP_PROC
__interrupt VectorNumber_Vrti void RTI_AdcFilter(void) 
{
  CPMUFLG_RTIF = 1; /* clear interrupt flag */
  
  /**********************************/
  ADC_CH1_Raw_8bit = ATDDR0L;
  ADC_LowPassFilter(&ADC_CH1_Raw_8bit, &ADC_Ch1_Flt);

}
#pragma CODE_SEG DEFAULT
