/*****************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2004-2007 Freescale Semiconductor, Inc.
* (c) Copyright 2001-2004 Motorola, Inc.
* ALL RIGHTS RESERVED.
*
******************************************************************************
*
* File Name: ucan.h
*
* Description: uCAN public header file
*
* $Version: 2.0.17.0$
*
*****************************************************************************/

#ifndef __UCAN_H
#define __UCAN_H

#ifndef __TYPES_H
#error uCAN: types.h must be included before ucan.h
#endif

// assure that user's configuration is applied
#include "ucan_cfg.h"

/***********************************
*  public data types
***********************************/

// return codes
typedef enum
{
	UCAN_OK = 0,
	UCAN_ERR_MODE,		// bad mode
	UCAN_ERR_MBCFG,		// MB not properly configured
	UCAN_ERR_MB,		// wrong MB number
	UCAN_ERR_LEN,		// bad length
	
} UCANRESULT;

// buffer modes
typedef enum
{
	UCAN_BM_CLOSED,		// inactive
	UCAN_BM_RXDF,		// receive mode
	UCAN_BM_TXDF,		// transmit mode

	UCAN_BM_COUNT
} UCANBMODE;

// buffer Load and Read modes (data modes)
typedef enum
{
	UCAN_DM_COPY,		// simple copy per 16bit
	UCAN_DM_SWAB16,		// simple copy per 16bit, bytes are swapped in each word
	UCAN_DM_HALF8,		// build from (split to) 8 words each storing single byte

	// Little or Big endian modes depends on selected byte order
	#if	UCAN_DATA_LSBFIRST
	UCAN_DM_COPY16_LE = UCAN_DM_COPY,
	UCAN_DM_COPY16_BE = UCAN_DM_SWAB16,
	#else
	UCAN_DM_COPY16_LE = UCAN_DM_SWAB16,
	UCAN_DM_COPY16_BE = UCAN_DM_COPY,
	#endif

} UCANDMODE;


// transmit codes
typedef enum
{
	UCAN_TX_ONCE,		// transmit data once
	UCAN_TX_AUTORESP,	// auto transmit responses to RTR
	UCAN_TX_RTR,		// transmit RTR request, then become RX, prevent callback after TX
	UCAN_TX_RTR_TXCB,	// transmit RTR request, then become RX, do callback also after TX
	
	UCAN_TXMODE_COUNT
} UCANTXMODE;

// MB status
typedef enum
{
	// flags
	UCAN_BS_RX_FLAG = 0x80,   // unread message pending (either OK or overrun)
	
	// status values
	UCAN_BS_VOID   = 0x00,                    // no action pending
	UCAN_BS_ERR    = 0x01,                    // error getting status
	UCAN_BS_BUSY   = 0x02,                    // MB is under driver's control - don't touch unless calling AbortMB
	UCAN_BS_RXFULL = 0x03 | UCAN_BS_RX_FLAG,  // MB holds received data
	UCAN_BS_RXFULL_OLD = 0x03,				  // returned when nothing new happened
	UCAN_BS_RXOVR  = 0x04 | UCAN_BS_RX_FLAG,  // MB holds received data, some data were lost
	UCAN_BS_RXOVR_OLD  = 0x04,				  // returned when nothing new happened
	
} UCANBSTATUS;

// Complete MB information for ReadAllMB
typedef struct
{
	UWord16  length;		// received length
	UWord16  timeStamp;		// timestamp of the frame
	UWord16* pDataBuff;		// pointer to data buffer which will receive frame data
	UWord32  realID;		// real ID received in the frame
	
} UCANRXINFO;

// calback function type
typedef void (*UCAN_MBCALLBACK_FUNC)(UWord16);

// flag to specify Extended ID (same as in FlexCAN and msCAN low-level drivers)
#define UCAN_ID_IDE 0x80000000L

/***********************************
*  public functions
***********************************/

// driver functions
UCANRESULT UCAN_Init(void);
UCANRESULT UCAN_Sleep(void);
UCANRESULT UCAN_Wakeup(void);
UWord16    UCAN_IsSynchronized(void);

// MB operations
UCANRESULT UCAN_ConfigMB(UWord16 bNum, UCANBMODE bMode, UWord32 id);
UCANRESULT UCAN_SetRxMaskGlobal(UWord32 mask);
UCANRESULT UCAN_SetRxMaskMB(UWord16 bNum, UWord32 mask);

// interrupt-safe MB operations
UCANRESULT UCAN_SetCallbackMB(UWord16 bNum, UCAN_MBCALLBACK_FUNC pCallbackFunc);
UCANBSTATUS UCAN_CheckStatusMB(UWord16 bNum);
UCANRESULT UCAN_AbortMB(UWord16 bNum);
UCANRESULT UCAN_TransmitOnceMB(UWord16 bNum);
UCANRESULT UCAN_TransmitMB(UWord16 bNum, UCANTXMODE txMode);

// interrupt-safe data movement functions
UCANRESULT UCAN_LoadMB(UWord16 bNum, UWord16* pSrcBuff, UCANDMODE copyMode, UWord16 len);
UWord16    UCAN_ReadMB(UWord16 bNum, UWord16* pDestBuff, UCANDMODE copyMode);
UCANRESULT UCAN_ReceiveMB(UWord16 bNum, UCANRXINFO* pDestInfo, UCANDMODE copyMode);

/************************************************************
*  include low-level specific header (inline functions etc.)
************************************************************/

#if UCAN_USE_MSCAN
#include "ucan_ms.h"
#else
#include "ucan_flex.h"
#endif

#endif /* __UCAN_H */
