/* main.c - Software vector mode program using C isr */
/* Jan 03 2012 S.Mihalik- Initial version based on AN2865 INTC example */
/* Copyright Freescale Semiconductor, Inc. 2012. All rights reserved. */
 
#include "MPC5606S_M07N.h" /* Use proper include file */

extern const vuint32_t IntcIsrVectorTable[];

void initModesAndClock(void) {
  ME.MER.R = 0x0000001D;          /* Enable DRUN, RUN0, SAFE, RESET modes */
                                  /* Initialize PLL before turning it on: */
  CGM.FMPLL[0].CR.R = 0x02400100; /* 8 MHz xtal: Set PLL0 to 64 MHz */   
  ME.RUN[0].R = 0x001F0074;       /* RUN0 cfg: 16MHzIRCON,OSC0ON,PLL0ON,syclk=PLL */
  ME.RUNPC[1].R = 0x00000010; 	  /* Peri. Cfg. 1 settings: only run in RUN0 mode */
  ME.PCTL[62].R = 0x01;           /* SGL   : select ME.RUNPC[1] */  
  ME.PCTL[68].R = 0x01;           /* SIUL  : select ME.RUNPC[1] */  
  ME.PCTL[73].R = 0x01;           /* EMIOS1: select ME.RUNPC[1] */  
                                  /* Mode Transition to enter RUN0 mode: */
  ME.MCTL.R = 0x40005AF0;         /* Enter RUN0 Mode & Key */
  ME.MCTL.R = 0x4000A50F;         /* Enter RUN0 Mode & Inverted Key */  
  while (ME.GS.B.S_MTRANS) {}     /* Wait for mode transition to complete */    
                                  /* Note: could wait here using timer and/or I_TC IRQ */
  while(ME.GS.B.S_CURRENTMODE != 4) {} /* Verify RUN0 is the current mode */
}

void initPeriClkGen(void) {
  CGM.SC_DC[0].R = 0x80;        /* Enable peri set 1 (SGM) with sysclk/1 */
  CGM.AC2_SC.R   = 0x03000000;  /* Aux Clk2 (EMIOS_1): select PLL for clock source */
  CGM.AC2_DC.R   = 0x80000000;  /* Aux Clk2 (EMIOS_1): enable input clk= Aux Clk2 / 1 */
}

void disableWatchdog(void) {
  SWT.SR.R = 0x0000c520;     /* Write keys to clear soft lock bit */
  SWT.SR.R = 0x0000d928; 
  SWT.CR.R = 0x8000010A;     /* Clear watchdog enable (WEN) */
}        

void initINTC(void) {	
  INTC.MCR.B.HVEN = 0;       /* Initialize for SW vector mode */
  INTC.MCR.B.VTES = 0;       /* Use default vector table 4B offsets */
  INTC.IACKR.R = (uint32_t) &IntcIsrVectorTable[0];    /* INTC ISR table base */
}

void initEMIOS_1(void) {  

  EMIOS_1.CH[17].CADR.R = 500;      /* Count n eMIOS clocks for OWPFMB duty cycle */
  EMIOS_1.CH[17].CBDR.R = 1000;     /* Count n eMIOS clocks for OWPFMB period */
  EMIOS_1.CH[17].CCR.R= 0x02000058; /* UCPRE==0, UCPREN=1, MODE=OPWFM */  
  /* Next line done for test purposes to observe PWM signal */
  SIU.PCR[117].R = 0x0A00;          /* Port PJ12: eMIOS (opt.2), output */ 

  EMIOS_1.CH[18].CADR.R = 5;       /* Count n eMIOS clocks for OWPFMB duty cycle */
  EMIOS_1.CH[18].CBDR.R = 10;      /* Count n eMIOS clocks for OWPFMB period */
  EMIOS_1.CH[18].CCR.R= 0x02000058; /* UCPRE==0, UCPREN=1, MODE=OPWFM */  
  /* Next line done for test purposes to observe PWM signal */
  SIU.PCR[51].R = 0x0E00;          /* Port PD5: eMIOS (opt.e), output */ 
                                   /* Start eMIOS_1 */
  EMIOS_1.MCR.B.GPRE = 63;         /* eMIOS clk = 64 MHz / (n+1) = 1 MHz */
  EMIOS_1.MCR.B.GPREN = 1;         /* enable prescaler and eMIOS clock */
}

void initSGL_mono_dur_irq(void) {
  SGL.MODE_SEL.B.M_P = 1;          /* Select monotonic sound */
  SGL.MODE_SEL.B.PRE = 0;          /* Prescale 64 MHz clock by 1 */
  SGL.MODE_SEL.B.CH1_SEL = 0x9;    /* Select EMIOS_1 ch 17: 1KHz, 50% duty */
  SGL.MODE_SEL.B.CH2_SEL = 0xA;    /* Select EMIOS_1 ch 18: 2KHz, 50% duty */
  SGL.MODE_SEL.B.SOUND_CTRL = 0;   /* Sound control: No sound (reset default) */
  SIU.PCR[40].R = 0x0A00;          /* Port PC10: SGL (opt. 2) output */
  /* The following two lines apply if using Freescale MPC5606S_DEMO_V2 EVB: */
  SIU.PCR[43].R = 0x0200;          /* Port PC13: GPIO output for BATT switch */
  SIU.GPDO[43].R = 1;              /* Enable BATT switch to speaker circuit */
  
  SGL.SOUND_DURATION.R = 64000000; /* Duration: # of SGLclks (1 sec) */     
  SGL.MODE_SEL.B.SDCIE = 1;        /* Enable sound duration complete interrupt */
  INTC.PSR[183].R = 3;             /* Set interrupt priority */
  SGL.MODE_SEL.B.SOUND_CTRL = 5;   /* Sound ctrl: dummy write */    
  SGL.MODE_SEL.B.SOUND_CTRL = 1;   /* Sound ctrl: continuous sound with duration*/
}

void enableIrq(void) {
  INTC.CPR.B.PRI = 0;          /* Lower INTC's current priority */
  asm(" wrteei 1");	    	   /* Enable external interrupts */
}

void main (void) {	
  vuint32_t i = 0;			    /* Dummy idle counter */

  initModesAndClock();      /* MPC56xxP/B/S: Initialize mode entries, set sysclk = 64 MHz*/
  initPeriClkGen();         /* Generated peripheral clock to SGL, other set 1 peris */
  disableWatchdog();        /* Disable watchdog */
  initIrqVectors();	        /* Initialize exceptions: only need to load IVPR */
  initINTC();               /* Initialize INTC for software vector mode */
  initEMIOS_1();            /* Initialize eMIOS 1 channels 17 & 18 */
  initSGL_mono_dur_irq();   /* Init SGL: Mono, no sound, prescale by 1, emios1 ch 17, 18 */
                            /*           with duration interrupt enabled */ 
  enableIrq();		   	    /* Ensure INTC current prority=0 & enable IRQ */
 
  while (1) { 
  	i++;
  }	
}

void SGL_ISR(void) {                /* Decrease duty cycle until 0 */
  if (EMIOS_1.CH[18].CADR.R > 0) {  /* If duty cycle >0 */
  	EMIOS_1.CH[18].CADR.R = EMIOS_1.CH[18].CADR.R - 1;  /* decrease duty cycle*/
  }
  else 
  {
    SGL.MODE_SEL.B.SDCIE = 0;        /* Disable further interrupts  */
  }
  SGL.MODE_SEL.B.SOUND_CTRL = 0;     /* Sound ctrl: first write */    
  SGL.SGL_STATUS.R = 0x00000001;     /* Clear flag done after writing SOUND_CTRL=0 */
  SGL.MODE_SEL.B.SOUND_CTRL = 1;     /* Sound ctrl: re-arm cont. sound with dur. */
}

