 /******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2008 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
***************************************************************************//*!
*
* @file     dspi.c
*
* @author   b15651
*
* @version  1.0.1.0
*
* @date     Apr-28-2010
*
* @brief    DSPI - DSPI module on MPC5643L
*
*******************************************************************************
*
* This file includes initial setting functions of MPC5604P DSPI module.
*
******************************************************************************/
#include "MPC5554.h"
#include "dspi.h"
#include "MPC5554_appconfig.h"

/******************************************************************************
* DSPI functions
*****************************************************************************/

/***************************************************************************//*!
@brief          DSPi init function

@param[in,out]  void

@return         void

@details        This function initializes basic start-up setting of the DSPI
                modules 0 to 2.

@note

@warning
******************************************************************************/
void DSPI_init(void)
{
    pDSPI0  = &DSPI_A;
    pDSPI1  = &DSPI_B;
    pDSPI2  = &DSPI_C;

    /*-------------------------------------------------------------------------
     * DSPI0 Reset
    --------------------------------------------------------------------------*/
    pDSPI0->MCR.R               = 0x1;
    pDSPI0->CTAR[0].R           = 0x38000000;
    pDSPI0->CTAR[1].R           = 0x38000000;
    pDSPI0->CTAR[2].R           = 0x38000000;
    pDSPI0->CTAR[3].R           = 0x38000000;
    pDSPI0->RSER.R              = 0x0;
    /*-------------------------------------------------------------------------
     * DSPI1 Reset
    --------------------------------------------------------------------------*/
    pDSPI1->MCR.R               = 0x1;
    pDSPI1->CTAR[0].R           = 0x38000000;
    pDSPI1->CTAR[1].R           = 0x38000000;
    pDSPI1->CTAR[2].R           = 0x38000000;
    pDSPI1->CTAR[3].R           = 0x38000000;
    pDSPI1->RSER.R              = 0x0;
    /*-------------------------------------------------------------------------
     * DSPI2 Reset
    --------------------------------------------------------------------------*/
    pDSPI2->MCR.R               = 0x1;
    pDSPI2->CTAR[0].R           = 0x38000000;
    pDSPI2->CTAR[1].R           = 0x38000000;
    pDSPI2->CTAR[2].R           = 0x38000000;
    pDSPI2->CTAR[3].R           = 0x38000000;
    pDSPI2->RSER.R              = 0x0;
    /*-------------------------------------------------------------------------
     * DSPI0 Control registers
    --------------------------------------------------------------------------*/
    #ifdef DSPI0_MCR                  /* MCR register */
        pDSPI0->MCR.R           = DSPI0_MCR;
    #endif

    #ifdef DSPI0_TCR                   /* TCR register */
        pDSPI0->TCR.R           = DSPI0_TCR;
    #endif

    #ifdef DSPI0_CTAR0                   /* CTAR[0] register */
        pDSPI0->CTAR[0].R       = DSPI0_CTAR0;
    #endif

    #ifdef DSPI0_CTAR1                   /* CTAR[1] register */
        pDSPI0->CTAR[1].R       = DSPI0_CTAR1;
    #endif

    #ifdef DSPI0_CTAR2                   /* CTAR[2] register */
        pDSPI0->CTAR[2].R       = DSPI0_CTAR2;
    #endif

    #ifdef DSPI0_CTAR3                   /* CTAR[3] register */
        pDSPI0->CTAR[3].R       = DSPI0_CTAR3;
    #endif

    #ifdef DSPI0_RSER                   /* RSER register */
        pDSPI0->RSER.R          = DSPI0_RSER;
    #endif

    /*-------------------------------------------------------------------------
     * DSPI1 Control registers
    --------------------------------------------------------------------------*/
    #ifdef DSPI1_MCR                  /* MCR register */
        pDSPI1->MCR.R           = DSPI1_MCR;
    #endif

    #ifdef DSPI1_TCR                   /* TCR register */
        pDSPI1->TCR.R           = DSPI1_TCR;
    #endif

    #ifdef DSPI1_CTAR0                   /* CTAR[0] register */
        pDSPI1->CTAR[0].R       = DSPI1_CTAR0;
    #endif

    #ifdef DSPI1_CTAR1                   /* CTAR[1] register */
        pDSPI1->CTAR[1].R       = DSPI1_CTAR1;
    #endif

    #ifdef DSPI1_CTAR2                   /* CTAR[2] register */
        pDSPI1->CTAR[2].R       = DSPI1_CTAR2;
    #endif

    #ifdef DSPI1_CTAR3                   /* CTAR[3] register */
        pDSPI1->CTAR[3].R       = DSPI1_CTAR3;
    #endif

    #ifdef DSPI1_RSER                   /* RSER register */
        pDSPI1->RSER.R          = DSPI1_RSER;
    #endif

    /*-------------------------------------------------------------------------
     * DSPI2 Control registers
    --------------------------------------------------------------------------*/
    #ifdef DSPI2_MCR                  /* MCR register */
        pDSPI2->MCR.R           = DSPI2_MCR;
    #endif

    #ifdef DSPI2_TCR                   /* TCR register */
        pDSPI2->TCR.R           = DSPI2_TCR;
    #endif

    #ifdef DSPI2_CTAR0                   /* CTAR[0] register */
        pDSPI2->CTAR[0].R       = DSPI2_CTAR0;
    #endif

    #ifdef DSPI2_CTAR1                   /* CTAR[1] register */
        pDSPI2->CTAR[1].R       = DSPI2_CTAR1;
    #endif

    #ifdef DSPI2_CTAR2                   /* CTAR[2] register */
        pDSPI2->CTAR[2].R       = DSPI2_CTAR2;
    #endif

    #ifdef DSPI2_CTAR3                   /* CTAR[3] register */
        pDSPI2->CTAR[3].R       = DSPI2_CTAR3;
    #endif

    #ifdef DSPI2_RSER                   /* RSER register */
        pDSPI2->RSER.R          = DSPI2_RSER;
    #endif

}

/**************************************************************************//*!
@brief          DSPI0 byte transmision

@param[in,out]  *SetPtr, *ptDataIn
@param[in]      u8DataOut

@return         void

@details        The function transmit 8bits from SPI module

@note           This function transmit 8bits from DSPI0 unit

@warning
******************************************************************************/
void DSPI0_transmit_SPI_8(uint8_t u8DataOut, MC33937_SET_T *SetPtr, uint8_t *ptDataIn)
{
    static  uint32_t data_out;

    // data preparation - assert CSx signal, CTAS register for transfer
    data_out            = (uint32_t)u8DataOut;
    data_out           |= ((1<<(16+SetPtr->DspiCSsignal)) | ASSERT_CTAR0);
    // Clear TCF bit transmit complete flag
    pDSPI0->SR.B.TCF    = 1;
    // Write to receive register
    pDSPI0->PUSHR.R     = data_out;
    // While Tx reg. not empty
    while (pDSPI0->SR.B.TCF != 1) {}
    *ptDataIn           = (uint8_t)pDSPI0->POPR.R;
}

/**************************************************************************//*!
@brief          DSPI1 byte transmision

@param[in,out]  *SetPtr, *ptDataIn
@param[in]      u8DataOut

@return         void

@details        The function transmit 8bits from SPI module

@note           This function transmit 8bits from DSPI1 unit

@warning
******************************************************************************/
void DSPI1_transmit_SPI_8(uint8_t u8DataOut, MC33937_SET_T *SetPtr, uint8_t *ptDataIn)
{
    static  uint32_t data_out;

    // data preparation - assert CSx signal, CTAS register for transfer
    data_out            = (uint32_t)u8DataOut;
    data_out           |= ((1<<(16+SetPtr->DspiCSsignal)) | ASSERT_CTAR0);
    // Clear TCF bit transmit complete flag
    pDSPI1->SR.B.TCF    = 1;
    // Write to receive register
    pDSPI1->PUSHR.R     = data_out;
    // While Tx reg. not empty
    while (pDSPI1->SR.B.TCF != 1) {}
    *ptDataIn           = (uint8_t)pDSPI1->POPR.R;
}

/**************************************************************************//*!
@brief          DSPI2 byte transmision

@param[in,out]  *SetPtr, *ptDataIn
@param[in]      u8DataOut

@return         void

@details        The function transmit 8bits from SPI module

@note           This function transmit 8bits from DSPI2 unit

@warning
******************************************************************************/
void DSPI2_transmit_SPI_8(uint8_t u8DataOut, MC33937_SET_T *SetPtr, uint8_t *ptDataIn)
{
    static  uint32_t data_out;

    // data preparation - assert CSx signal, CTAS register for transfer
    data_out            = (uint32_t)u8DataOut;
    data_out           |= ((1<<(16+SetPtr->DspiCSsignal)) | ASSERT_CTAR0);
    // Clear TCF bit transmit complete flag
    pDSPI2->SR.B.TCF    = 1;
    // Write to receive register
    pDSPI2->PUSHR.R     = data_out;
    // While Tx reg. not empty
    while (pDSPI2->SR.B.TCF != 1) {}
    *ptDataIn           = (uint8_t)pDSPI2->POPR.R;
}

/**************************************************************************//*!
@brief          DSPI0 word transmision

@param[in,out]  *ptDataIn
@param[in]      u16DataOut

@return         void

@details        The function transmit 16bits from SPI module

@note           This function transmit 16bits from DSPI0 unit with use CTAR1

@warning
******************************************************************************/
void DSPI0_transmit_SPI_16(uint16_t u16DataOut, uint16_t *ptDataIn)
{
    static  uint32_t data_out;

    // data preparation - assert CSx signal, CTAS register for transfer
    data_out            = (uint32_t)u16DataOut;
    data_out           |= (ASSERT_CS0 | ASSERT_CTAR1);
    // Clear TCF bit transmit complete flag
    pDSPI0->SR.B.TCF    = 1;
    // Write to receive register
    pDSPI0->PUSHR.R     = data_out;
    // While Tx reg. not empty
    while (pDSPI0->SR.B.TCF != 1) {}
    *ptDataIn           = (uint16_t)pDSPI0->POPR.R;
}

/**************************************************************************//*!
@brief          DSPI1 word transmision

@param[in,out]  *ptDataIn
@param[in]      u16DataOut

@return         void

@details        The function transmit 16bits from SPI module

@note           This function transmit 16bits from DSPI1 unit with use CTAR1

@warning
******************************************************************************/
void DSPI1_transmit_SPI_16(uint16_t u16DataOut, uint16_t *ptDataIn)
{
    static  uint32_t data_out;

    // data preparation - assert CSx signal, CTAS register for transfer
    data_out            = (uint32_t)u16DataOut;
    data_out           |= (ASSERT_CS0 | ASSERT_CTAR1);
    // Clear TCF bit transmit complete flag
    pDSPI1->SR.B.TCF    = 1;
    // Write to receive register
    pDSPI1->PUSHR.R     = data_out;
    // While Tx reg. not empty
    while (pDSPI1->SR.B.TCF != 1) {}
    *ptDataIn           = (uint16_t)pDSPI1->POPR.R;
}

/**************************************************************************//*!
@brief          DSPI2 word transmision

@param[in,out]  *ptDataIn
@param[in]      u16DataOut

@return         void

@details        The function transmit 16bits from SPI module

@note           This function transmit 16bits from DSPI2 unit with use CTAR1

@warning
******************************************************************************/
void DSPI2_transmit_SPI_16(uint16_t u16DataOut, uint16_t *ptDataIn)
{
    static  uint32_t data_out;

    // data preparation - assert CSx signal, CTAS register for transfer
    data_out            = (uint32_t)u16DataOut;
    data_out           |= (ASSERT_CS0 | ASSERT_CTAR1);
    // Clear TCF bit transmit complete flag
    pDSPI2->SR.B.TCF    = 1;
    // Write to receive register
    pDSPI2->PUSHR.R     = data_out;
    // While Tx reg. not empty
    while (pDSPI2->SR.B.TCF != 1) {}
    *ptDataIn           = (uint16_t)pDSPI2->POPR.R;
}

