/******************************************************************************
*                                                       
*       Copyright (C) 2008 Freescale Semiconductor, Inc.
*       All Rights Reserved								              
*														                            
* Filename:     msCANdrv.c                
*														                            
* Revision:      										                    
*														                            
* Functions:    msCAN driver routines for single msCAN module 
*												                            
* Description:  
*
* Notes:        
*
******************************************************************************/

/******************************************************************************
 * Include 
 ******************************************************************************/
#include <hidef.h>                  /* for EnableInterrupts macro */
#include "msCANvar.c"               /* msCAN driver variables */
#include "msCANstd.h"               /* Standard defines */
#include "msCANdrv.h"               /* General defines, enums, prototypes */
#include "msCANinc.h"               /* Internal typedefs, prototypes */
#include "msCANreg.h"               /* msCAN register defines */

/****************************************************************************/

/******************************************************************************
VERSION
******************************************************************************/

#define     VERSION MSCANMCF51_01

/******************************************************************************
BENCHMARK
*******************************************************************************
If BENCHMARK defined, the following port pins are toggled:
PORTB
Bit 0:      Set at start of ReceiveISR, cleared at end.
Bit 1:      Set at start of TransmitISR, cleared at end.
******************************************************************************/
#ifdef  BENCHMARK
#define PORTB (*((volatile UINT8*)0x0001))
#endif

/******************************************************************************
Compiler dependancies
******************************************************************************/
unsigned char u8gvbuffer;//ALL
unsigned char u8CANTxOK;
/***************************************************************************
 * Function :   CAN_Init
 *
 * Description: Initiliase msCAN driver .
 * 
 * Returns:     ERR_OK if no error, otherwise error code.                   
 *
 * Notes:       If CMPTX argument is used, the fn could take the duration of
 *              a single message transmission to return 
 *
 **************************************************************************/
UINT8 CAN_Init(UINT8 rmode)
{
      UINT8 errorStatus,                               /* error code from Reset_CAN */
            i;                                         /* loop counter */


#if defined(EEPROM_SECURITY_CHECK)
												/* test for correct value in EEPROM	*/
	if (CAN_dataCheck != EEPROM_SECURITY)
	{
												/* if test fails do not initialise	*/
		return(ERR_OK);
	}
#endif /* EEPROM_SECURITY_CHECK	*/

      errorStatus = CAN_Reset(rmode);                      /* reset msCAN */

      if (errorStatus != ERR_OK)
      {
            return (errorStatus);
      }

      CANCTL1 = CANCTL1_Def;                       /* initialise hardware registers */
      CANBTR0 = CANBTR0_Def;
      CANBTR1 = CANBTR1_Def;
      CANIDAC = CANIDAC_Def;

      for (i = 0; i < NO_OF_FILTER_REG; i++)          
      {
            *(&CANIDAR0 + i) = CANIDFilter_Def[i];              /* initialise hardware acceptance filter  */
      }

      CANCTL0 &= ~INITRQ_MASK;                                     /* clear soft reset */
      CANCTL0 = CANCTL0_Def;                                 /* must write separate from clearing SFTRST */

      for (i = 0 ; i < NoOfMB ; i++)                         /* configure all MB's to CLOSED, NODATA */
      {
            BufferMode_CAN[i] = CLOSED;
            BufferStatus_CAN[i] = NODATA;
      }

      MB_MSCAN[0] = 0xff;                                    /* more efficient than for loop */
      MB_MSCAN[1] = 0xff;
      MB_MSCAN[2] = 0xff;
      
      MBP_MSCAN[0] = 0xff;
      MBP_MSCAN[1] = 0xff;
      MBP_MSCAN[2] = 0xff;
      
#if (NO_OF_MSCAN_TXBUFFER != 3)
#error MB_MSCAN not initialised
#endif
      
      if (CANCTL1 & INITAK_MASK) return ERR_INIT;
      
      EnableInterrupts; /* enable interrupts */;                                                   /* clear interrupt mask */
      CANRIER = 0x01;                                        /* enable Receive Interrupts */

      if ((CANRFLG & BUSOFF_MASK) == BUSOFF_MASK)                     /* Bus-off */
      {
                  return (ERR_BOFF);
      }

      else
      {
            return (ERR_OK);
      }
}

/***************************************************************************
 * Function :   CAN_Reset
 *
 * Description: Puts msCAN module in soft reset state.
 * 
 * Returns:     ERR_OK if no error, otherwise error code.                   
 *
 * Notes:       If CMPTX argument is used, the fn could take the duration of
 *              a single message transmission to return 
 *
 **************************************************************************/
UINT8 CAN_Reset(UINT8 rmode)
{
      if (CANCTL1 & INITAK_MASK) return ERR_OK;                   
      if (CANCTL0 & INITRQ_MASK) return ERR_INIT;

      if (!((rmode == FAST) || (rmode == CMPTX)))
      {
            return (ERR_MODE);
      }

      DisableInterrupts;                                     /* disable interrupts */
      if (rmode == CMPTX)                                    /* wait for current transmission to complete */
      {
            CANRIER = 0;                                     /* clear all interrupt enables */
            CANTARQ  = (UINT8)(~CANTFLG);                    /* and abort any scheduled transmissions */

            EnableInterrupts;                                /* enable interrupts */
            
            while ((CANTFLG & 0x07) != 0x07)
            {                                                /* wait for all msCAN tx buffers not scheduled */
            }                                                /* this could take time of one message transmission */
      }

      CANCTL0  = INITRQ_MASK;                                     /* soft reset */

      EnableInterrupts;                                      /* enable interrupts */
      
      DriverFlags = 0;                                       /* clear flags */

      if (CANCTL1 & INITAK_MASK) return ERR_OK;
      else return ERR_INIT;
}

/***************************************************************************
 * Function :   CAN_Sleep
 *
 * Description: Puts msCAN module into low-power SLEEP mode.
 * 
 * Returns:     ERR_OK if no error, otherwise error code.                   
 *
 * Notes:       None 
 *
 **************************************************************************/
UINT8 CAN_Sleep(UINT8 rmode)
{
      if (!((rmode == FAST) || (rmode == CMPTX)))
      {
            return (ERR_MODE);
      }

      if (CANCTL1 & INITAK_MASK)                                  /* in initialization mode */
      {
            return (ERR_INIT);
      }

      CANRFLG = WUPIF_MASK;                                       /* clear wakeup interrupt flag */
                                                              
      if ((rmode == CMPTX) && (CANTIER & 0x07))              /* rmode = CMPTX and Tx interrupt enabled */       
      {
            DriverFlags |= SLEEP_MASK;                          /* TransmitISR_CAN will set SLPRQ */
      }
      else                                                   /* rmode = FAST */
      {
            CANCTL0 |= SLPRQ_MASK;                               /* sleep request */
            CANRIER |= WUPIE_MASK;                                /* enable wakeup interrupt */
      }

      return (ERR_OK);
}

/***************************************************************************
 * Function :   CAN_Wakeup
 *
 * Description: wakes up the msCAN module from SLEEP mode.
 * 
 * Returns:     ERR_OK if no error, otherwise error code.                   
 *
 * Notes:       None 
 *
 **************************************************************************/
UINT8 CAN_Wakeup(void)
{
      UINT8 currentCANCTL0;

      currentCANCTL0 = CANCTL0;

      if (CANCTL1 & INITAK_MASK)                           /* initialization mode */
      {
            return (ERR_INIT);
      }

      DriverFlags &= ~SLEEP_MASK;                                 /* clear driver sleep request flag */

      if ((currentCANCTL0 & SLPRQ_MASK) && !(CANCTL1 & SLPAK_MASK))    /* SLPRQ set but SLPAK clear */
      {                                                           
            return (ERR_NSLP);                               /* not yet in Sleep mode: cannot clear SLPRQ */
      }

      else                                                   /* in Sleep mode (or SLPRQ not set) */
      {
            DisableInterrupts;                               /* disable interrupts */
            
            CANRIER &= ~WUPIE_MASK;                               /* clear wakeup interrupt */
   
            EnableInterrupts;                                /* enable interrupts */; 
            
            CANCTL0 &= ~SLPRQ_MASK;                               /* wake-up msCAN */

            return (ERR_OK);
      }

}

/***************************************************************************
 * Function :   CAN_CheckStatus
 *
 * Description: obtains the status of the msCAN module.
 * 
 * Returns:     ERR_OK                    
 *
 * Notes:       None 
 *
 **************************************************************************/
UINT8 CAN_CheckStatus(UINT16 *statusPtr)
{
      *((UINT8*)(statusPtr))     = (UINT8)((CANCTL0 & (SYNCH_MASK | SLPRQ_MASK | INITRQ_MASK)) | ((CANCTL1 & (SLPAK_MASK | INITAK_MASK)) << 2));
      *((UINT8*)(statusPtr + 1)) = CANRFLG;

      return (ERR_OK);
}

/***************************************************************************
 * Function :   CAN_ClearStatus
 *
 * Description: Clears status flags except RXF.
 * 
 * Returns:     ERR_OK if no error, otherwise error code.                   
 *
 * Notes:       None 
 *
 **************************************************************************/
UINT8 CAN_ClearStatus(void)
{
      CANRFLG = 0xFE;                                        /* write 1's to clear all flags except RXF */

      return (ERR_OK);
}

/***************************************************************************
 * Function :   CAN_ConfigMB
 *
 * Description: configures a Message Buffer.
 * 
 * Returns:     ERR_OK if no error, otherwise error code.                   
 *
 * Notes:       None 
 *
 **************************************************************************/
UINT8 CAN_ConfigMB(UINT8 buffer, UINT8 cmode, UINT8 identifierRef)
{
      UINT8 currentCANRIER, 
            currentCANTIER;

      if (identifierRef >= NoOfID)
      {
            return (ERR_ID);
      }

      if (cmode >= INVALID)
      {
            return (ERR_MODE);
      }

      if (buffer >= NoOfMB)
      {
            return (ERR_MB);
      }

      DisableInterrupts;                                     /* disable interrupts */
      
      currentCANRIER = CANRIER;                              /* save interrupt enables */
      currentCANTIER = CANTIER;                            

      CANRIER = 0;                                           /* disable all interrupts */
      CANTIER = 0;

      EnableInterrupts;                                      /* enable interrupts */

      if(BufferStatus_CAN[buffer] == QUEUED2)                /* MB in msCAN buffer */            
      {
            if (MB_MSCAN[0] == buffer)                       /* MB in this MSCAN buffer */ 
            {
                  CANTARQ |= 0x01;                           /* set ABTRQi for msCAN buffer */
            }
            else
            {
                  if (MB_MSCAN[1] == buffer)                 /* MB in this MSCAN buffer */
                  {
                        CANTARQ |= 0x02;                     /* set ABTRQi for msCAN buffer */
                  }
                  else                                       /* must be MB_MSCAN[2] */
                  {
                        CANTARQ |= 0x04;                     /* set ABTRQi for msCAN buffer */
                  }
            }
#if (NO_OF_MSCAN_TXBUFFER != 3)
#error msCAN buffer not aborted
#endif
      }

      BufferMode_CAN[buffer] = cmode;
      BufferStatus_CAN[buffer] = NODATA;
      BufferID_Ref_CAN[buffer] = identifierRef;

      CANRIER = currentCANRIER;                              /* restore interrupts */
      CANTIER  = currentCANTIER;

      return (ERR_OK);
}

/***************************************************************************
 * Function :   CAN_CheckStatusMB
 *
 * Description: obtains the status and mode of the specified Message Buffer.
 * 
 * Returns:     ERR_OK if no error, otherwise error code.                   
 *
 * Notes:       None 
 *
 **************************************************************************/
UINT8 CAN_CheckStatusMB (UINT8 buffer, UINT8 *statusPtr)
{
      if (buffer >= NoOfMB)
      {
            return (ERR_MB);
      }

      DisableInterrupts; /* disable interrupts */
      
      *(statusPtr) = BufferStatus_CAN[buffer];
      *(statusPtr + 1) = BufferMode_CAN[buffer];
      
      EnableInterrupts; /* enable interrupts */

      return (ERR_OK);
}

/***************************************************************************
 * Function :   CAN_LoadMB
 *
 * Description: loads a Message Buffer with data.
 * 
 * Returns:     ERR_OK if no error, otherwise error code.                   
 *
 * Notes:       None 
 *
 **************************************************************************/
UINT8 CAN_LoadMB (UINT8 buffer, UINT8 *dataPtr)
{
      UINT8 *bufferPtr;             
      UINT8 dataLength, 
              i;

      if (buffer >= NoOfMB)
      {
            return (ERR_MB);
      }

      if (*dataPtr > 8)
      {
            return (ERR_DLC);
      }

      if ((BufferMode_CAN[buffer] != TXDF)
            && (BufferMode_CAN[buffer] != AUTOTXDF))
      {
            return (ERR_CONFIG);
      }

      if ((BufferStatus_CAN[buffer] == QUEUED)               /* Message Buffer already queued */
            || (BufferStatus_CAN[buffer] == QUEUED2) 
            || (BufferStatus_CAN[buffer] == ABORTREQ))
      {
            return (ERR_QED);
      }

      DisableInterrupts;                                     /* disable interrupts */
      
      dataLength = *dataPtr;
      BufferDLC_CAN[buffer] = dataLength;                    /* copy DLC */

      bufferPtr = BufferData_CAN[buffer];            
      dataPtr++;                                             /* increment to start of data */

      for (i = 0; i < dataLength; i++)
      {
            bufferPtr[i] = dataPtr[i];                       /* copy data */
      }

      EnableInterrupts;                                      /* enable interrupts */

      BufferStatus_CAN[buffer] = VALIDDATA;

      return (ERR_OK);
}

/***************************************************************************
 * Function :   CAN_TransmitMB
 *
 * Description: schedules a Message Buffer for transmission.
 * 
 * Returns:     ERR_OK if no error, otherwise error code.                   
 *
 * Notes:       None 
 *
 **************************************************************************/
UINT8 CAN_TransmitMB (UINT8 buffer)
{
      UINT8 bufferStatus, 
              bufferMode;

      if (CANCTL1 & INITAK_MASK)
      {
            return (ERR_INIT);
      }

      if ((CANRFLG & BUSOFF_MASK) == BUSOFF_MASK)
      {
            return (ERR_BOFF);
      }

      if (buffer >= NoOfMB)
      {
            return (ERR_MB);
      }

      DisableInterrupts;                                     /* disable interrupts */
      
      bufferMode = BufferMode_CAN[buffer];

      if (!((bufferMode == TXDF) || (bufferMode == TXRF)))
      {
            EnableInterrupts;                                /* enable interrupts */
                  
            return (ERR_CONFIG);
      }

      bufferStatus = BufferStatus_CAN[buffer];

      if ((bufferMode == TXDF)
            && (bufferStatus == NODATA))
      {
            EnableInterrupts;                                /* enable interrupts */
                  
            return (ERR_NODATA);
      }

      if ((bufferStatus == QUEUED)                           /* MB already queued */
            || (bufferStatus == QUEUED2)  
            || (bufferStatus == ABORTREQ))
      {
            EnableInterrupts;                                /* enable interrupts */
                  
            return (ERR_QED);
      }

      CAN_QueueMB (buffer);                                  /* queue buffer, copy to msCAN if buffer available,   */
                                                             /* abort msCAN buffer if all full and this buffer higher priority */

      EnableInterrupts;                                      /* enable interrupts */

      if (CANCTL1 & SLPAK_MASK)                                   /* msCAN in SLEEP mode */
      {                                                         
            CANCTL0 &= ~SLPRQ_MASK;                               /* wake-up msCAN */
      }

      DriverFlags &= ~SLEEP_MASK;                                 /* clear sleep request flag */


      if (CANCTL0 & SYNCH_MASK)
      {
            return (ERR_OK);
      }

      else
      {
            return (ERR_SYNCH);
      }

}

/***************************************************************************
 * Function :   CAN_QueueMB
 *
 * Description: Internal driver function
 * 
 * Returns:     None                   
 *
 * Notes:       All interrupts must be disabled when this function is called 
 *
 **************************************************************************/
void CAN_QueueMB (UINT8 buffer)
{
      UINT8 MB_ID_Ref = BufferID_Ref_CAN[buffer];           /* ID_Ref of this MB = priority of MO */

      /* Variables req. only if msCAN transmit buffer is not available */
      UINT8 lowestPriority, 
            lowestPriorityTXBuffer;

      /* If: All msCAN transmit buffers are already scheduled:
       * and if priority of Message Object in buffer is greater (ID_Ref is less) than any
       * Message Object already in msCAN transmit buffer, request abort transmission of the
       * msCAN tx buffer with the   lowest priority Message Object.
       */

      if (CANTIER == 0x07)                             /* msCAN transmit buffers all scheduled */
                                                             /* do not use CANTFLG: TXEn could be set but interrupt not serviced yet */
      {
            if (buffer < LowestQueuedBuffer)
            {
                  LowestQueuedBuffer = buffer;               /* lowest buffer # with QUEUED status, used in TxISR */
            }

            BufferStatus_CAN[buffer] = QUEUED; 

            if (  (MB_ID_Ref < MBP_MSCAN[0])                  /* queued MO higher priority than MO in msCAN ? */   
                ||(MB_ID_Ref <MBP_MSCAN[1])
                ||(MB_ID_Ref < MBP_MSCAN[2]))                  
            {
                  lowestPriority = MBP_MSCAN[0];              /* find msCAN buffer with lowest priority MO */
                  lowestPriorityTXBuffer = 0x01;
                  
                  if (MBP_MSCAN[1] > lowestPriority)
                  {
                        lowestPriority = MBP_MSCAN[1];
                        lowestPriorityTXBuffer = 0x02;
                  }
                  
                  if (MBP_MSCAN[2] > lowestPriority)
                  {
               //       lowestPriority = TBP_MSCAN[2];
                        lowestPriorityTXBuffer = 0x04;
                  }
                        
                  CANTARQ |= lowestPriorityTXBuffer;  /* set ABTRQ for msCAN buffer */

                  if((AbortMessage == 0xff) || (MB_ID_Ref < BufferID_Ref_CAN[AbortMessage]))         
                  {
                        AbortMessage = buffer;               /* highest priority queued MB, causing abort */ 
                  }
            }
      }

      /* Else: msCAN transmit buffer is available:
       * Find free tx buffer: copy identifier into buffer.
       * If data frame is to be transmitted, copy data and DLC into buffer, otherwise set RTR bit.
       * Write priority reg, schedule buffer and enable interrupt.
       */

      else
      {
            if (MB_MSCAN[0] == 0xff)                              /* TX_BUFF0 is empty */
            {
                  MBP_MSCAN[0] = BufferID_Ref_CAN[buffer];
                  MB_MSCAN[0] = buffer;                           /* write buffer # */
                  MSCAN_Load_Buff(TX_BUFF0, buffer, MB_ID_Ref);
                  CANTFLG = 0x01;                                 /* write 1 to TXE to schedule buffer */
                  CANTIER |= 0x01;                                /* enable interrupt for this buffer */
            }
            else
            {
                  if (MB_MSCAN[1] == 0xff)                        /* TX_BUFF1 is empty */
                  {
                        MBP_MSCAN[1] = BufferID_Ref_CAN[buffer];              
                        MB_MSCAN[1] = buffer;                           /* write buffer # */
                        MSCAN_Load_Buff(TX_BUFF1, buffer, MB_ID_Ref);
                        CANTFLG = 0x02;                                 /* write 1 to TXE to schedule buffer */
                        CANTIER |= 0x02;                                /* enable interrupt for this buffer */
                  }
                  else                                            /* TX_BUFF2 is empty */
                  {
                        MBP_MSCAN[2] = BufferID_Ref_CAN[buffer];
                        MB_MSCAN[2] = buffer;                           /* write buffer # */
                        MSCAN_Load_Buff(TX_BUFF2, buffer, MB_ID_Ref);
                        CANTFLG = 0x04;                                 /* write 1 to TXE to schedule buffer */
                        CANTIER |= 0x04;                                /* enable interrupt for this buffer */
                  }
#if (NO_OF_MSCAN_TXBUFFER != 3)
#error msCAN buffer not checked
#endif
            }
      }     /* end: If all msCAN transmit buffers are already scheduled */
}

/***************************************************************************
 * Function :   MSCAN_Load_Buff
 *
 * Description: Internal driver function
 * 
 * Returns:     None                   
 *
 * Notes:       All interrupts must be disabled when this function is called 
 *
 **************************************************************************/
void MSCAN_Load_Buff (UINT8 TX_Buff, UINT8 buffer, UINT8 MO_ID_Ref)
{
      UINT8 *bufferData;                        
      UINT8 dataLength, 
              i;
      UINT32 *TX_BUFF_IDR_Ptr = (UINT32)(& CANTIDR0);
      UINT8  *TX_BUFF_DSR_Ptr = & _CANTDSR0.Byte;
      
      CANTBSEL = TX_Buff;                                    /* select TX buffer into register space */
      
      *TX_BUFF_IDR_Ptr = M_Identifier_CAN[MO_ID_Ref];       /* copy 32-bit ID to msCAN buffer */
            
      if (BufferMode_CAN[buffer] == TXRF)                   /* remote frame to be transmitted */
      {
            if (M_IdentifierType_CAN[MO_ID_Ref] == STANDARD)
            {
                  TX_BUFF_IDR1 |= 0x10;                      /* set RTR in Standard Identifier */
            }
            else  /* extended ID */
            {
                  TX_BUFF_IDR3 |= 0x01;                      /* set RTR in Extended Identifier */
            }
      }  /* end: if remote frame */

      else            /* data frame to be transmitted */
      {
            dataLength = BufferDLC_CAN[buffer];
            TX_BUFF_DLR = dataLength;                        /* copy DLC to msCAN buffer */

            bufferData = BufferData_CAN[buffer];                 

            for (i = 0; i < dataLength; i++)
            {
                  TX_BUFF_DSR_Ptr[i] = bufferData[i];        /* copy data to msCAN buffer */
            }
      }               /* end: else data frame */


      TX_BUFF_TBPR = BufferID_Ref_CAN[buffer];          /* set priority = ID_Ref */

      BufferStatus_CAN[buffer] = QUEUED2;            
}

/***************************************************************************
 * Function :   CAN_AbortMB
 *
 * Description: requests that the transmission of a queued Message Buffer 
 *              is aborted.
 * 
 * Returns:     ERR_OK if no error, otherwise error code.                   
 *
 * Notes:       None 
 *
 **************************************************************************/
UINT8 CAN_AbortMB (UINT8 buffer)
{
      UINT8 currentCANTIER;

      if (CANCTL1 & INITAK_MASK)
      {
            return (ERR_INIT);
      }

      if (buffer >= NoOfMB)
      {
            return (ERR_MB);
      }

      if (BufferMode_CAN[buffer] == RXDF)
      {
            return (ERR_CONFIG);
      }

      DisableInterrupts;                                     /* disable interrupts */
      
      currentCANTIER = CANTIER;
      CANRIER &= ~RXFIE_MASK;                                     /* disable receive interrupt */
      CANTIER = 0;                                           /* disable transmit interrupts */
      
      EnableInterrupts;                                      /* enable interrupts */

      switch (BufferStatus_CAN[buffer])
      {
            case QUEUED:
                  BufferStatus_CAN[buffer] = ABORTED;
                  break;

            case QUEUED2:
                  BufferStatus_CAN[buffer] = ABORTREQ;

                  if (MB_MSCAN[0] == buffer)                 /* Message Buffer in msCAN buffer */
                  {
                        CANTARQ |= 0x01;              /* set ABTRQi for msCAN buffer */
                  }
                  
                  else
                  {
                        if (MB_MSCAN[1] == buffer)           /* Message Buffer in msCAN buffer */
                        {
                              CANTARQ |= 0x02;        /* set ABTRQi for msCAN buffer */
                        }
                        
                        else
                        {
                              CANTARQ |= 0x04;        /* set ABTRQi for msCAN buffer */
                        }

#if (NO_OF_MSCAN_TXBUFFER != 3)
#error msCAN buffer not checked
#endif
                  }
                  break;

            default:                                         /* not queued */
                  CANRIER |= RXFIE_MASK;                          /* restore receive interrupt */
                  CANTIER = currentCANTIER;                  /* restore transmit interrupts */
                  return (ERR_NOTQED);
      }

      CANRIER |= RXFIE_MASK;                                        /* restore receive interrupt */
      CANTIER = currentCANTIER;                              /* restore transmit interrupts */
      return (ERR_OK);
}

/***************************************************************************
 * Function :   CAN_ReadDataMB
 *
 * Description: reads the data in a Message Buffer.
 * 
 * Returns:     ERR_OK if no error, otherwise error code.                   
 *
 * Notes:       None 
 *
 **************************************************************************/
UINT8 CAN_ReadDataMB (UINT8 buffer, UINT8* dataPtr)
{
      UINT8 errorStatus;

      errorStatus = ReadDataMB (buffer, dataPtr);

      CANRIER |= RXFIE_MASK;                                        /* restore receive interrupt */
      return (errorStatus);   
}

/***************************************************************************
 * Function :   ReadDataMB
 *
 * Description: Internal driver function
 * 
 * Returns:     ERR_OK if no error, otherwise error code                   
 *
 * Notes:       None 
 *
 **************************************************************************/
UINT8 ReadDataMB (UINT8 buffer, UINT8* dataPtr)
{
      UINT8 *bufferPtr;       
      UINT8 dataLength, 
              i;

      if (buffer >= NoOfMB)
      {
            return (ERR_MB);
      }

      if (BufferMode_CAN[buffer] != RXDF)
      {
            return (ERR_CONFIG);
      }

      if (BufferStatus_CAN[buffer] == NODATA)
      {
            return (ERR_NODATA);
      }
      if (BufferStatus_CAN[buffer] == READDATA)
      {
    	  BufferStatus_CAN[buffer] = NODATA;
            return (ERR_NODATA);
      }

      DisableInterrupts;                                     /* disable interrupts */
      
      CANRIER &= ~RXFIE_MASK;                                     /* disable receive interrupt */
      
      EnableInterrupts;                                      /* enable interrupts */

      dataLength = BufferDLC_CAN[buffer];
      *dataPtr = dataLength;                                 /* copy DLC */
      dataPtr++;

      bufferPtr = BufferData_CAN[buffer];                   /* BufferData_CAN[buffer] is first dimension */

      for (i = 0; i < dataLength; i++)                       /* copy data */
      {
            dataPtr[i] = bufferPtr[i];
      }

      BufferStatus_CAN[buffer] = READDATA;

      return (ERR_OK);
}

/***************************************************************************
 * Function :   CAN_ReadTimeMB
 *
 * Description: reads the time stamp in a Message Buffer.
 * 
 * Returns:     ERR_OK if no error, otherwise error code.                   
 *
 * Notes:       None 
 *
 **************************************************************************/
UINT8 CAN_ReadTimeMB (UINT8 buffer, UINT16* dataPtr)
{
      if (!TimeStampOption)
      {
            return (ERR_TS);
      }

      if (buffer >= NoOfMB)
      {
            return (ERR_MB);
      }

      if (BufferMode_CAN[buffer] == CLOSED)
      {
            return (ERR_CONFIG);
      }

      if (BufferStatus_CAN[buffer] < TRANSMITTED)           /* NODATA, VALIDDATA, QUEUED, QUEUED2, ABORTREQ, ABORTED */
      {
            return (ERR_NODATA);
      }

      DisableInterrupts;                                    /* disable interrupts */
      
      *dataPtr = BufferTimeStamp_CAN[buffer];               /* copy time stamp */

      EnableInterrupts;                                     /* enable interrupts */ 

      return (ERR_OK);

}

/***************************************************************************
 * Interrupt:   CAN_TransmitISR
 *
 * Description: msCAN transmit interrupt handler.
 * 
 * Returns:     None.                   
 *
 * Notes:       None 
 *
 **************************************************************************/
interrupt void CAN_TransmitISR (void)
{
      UINT8  TX_Buff;
      UINT8  interruptBufferBit,
             interruptBufferNo, 
             buffer, 
             highestPriority = 0xff, 
             highestPriorityBuffer,
             nextBuffer = LowestQueuedBuffer;

#ifdef BENCHMARK
      PORTB |= 0x02;                                          /* set bit 1 */
#endif

      /* Find buffer that caused interrupt, and write new status */

      interruptBufferBit = (UINT8)(CANTFLG & CANTIER);                       /* position of set bit indicates buffer */

      if (interruptBufferBit & 0x01)
      {
            interruptBufferNo = 0;
            TX_Buff = TX_BUFF0;
            CANTIER &= ~0x01;                                    /* disable interrupt */
      }
      else
      {
            if (interruptBufferBit & 0x02)
            {
                  interruptBufferNo = 1;
                  TX_Buff = TX_BUFF1;
                  CANTIER &= ~0x02;                              /* disable interrupt */
            }
            else
            {
                  interruptBufferNo = 2;
                  TX_Buff = TX_BUFF2;
                  CANTIER &= ~0x04;                              /* disable interrupt */
            }
#if (NO_OF_MSCAN_TXBUFFER != 3)
#error msCAN buffer not checked
#endif
      }

      buffer = MB_MSCAN[interruptBufferNo];                     /* buffer number */
      MB_MSCAN[interruptBufferNo] = 0xff;                       /* clear Message Buffer from MB_MSCAN */
      MBP_MSCAN[interruptBufferNo] = 0xff;                      /* initialise message buffer priority */

      if (CANTAAK & interruptBufferBit)                  /* transmission aborted */
      {
            if (BufferStatus_CAN[buffer] == QUEUED2)         /* aborted due to higher priority queued MB */
            {                                                                 
                  BufferStatus_CAN[buffer] = QUEUED;         /* set MB Queued to put back in queue */
                  if (buffer < LowestQueuedBuffer)
                  {
                        nextBuffer = LowestQueuedBuffer = buffer;
                  }
            }
            
            else
            {                                                     
                  if (BufferStatus_CAN[buffer] == ABORTREQ)  /* aborted due to AbortMB_CAN */    
                  {                                                           
                        BufferStatus_CAN[buffer] = ABORTED;
                  }
            }                                                 /* otherwise aborted due to ConfigMB_CAN */
      }

      else        /* transmission completed */
      {
            if (BufferMode_CAN[buffer] == TXRF)              /* transmitted remote frame */
            {                                                  
                  BufferMode_CAN[buffer] = RXDF;             /* receive data frame */
                  BufferStatus_CAN[buffer] = NODATA;
            }

            else                                              /* transmitted Data TXDF or AUTOTXDF */
            {           
                  BufferStatus_CAN[buffer] = TRANSMITTED;
            }

            if (TimeStampOption)
            {
                  CANTBSEL =  interruptBufferBit;
                  BufferTimeStamp_CAN[buffer] = TX_BUFF_TSR;    
            }

      }              /* end: else transmission completed */

      /* source of interrupt now dealt with */

      /* find highest priority queued MB */

      if(AbortMessage == 0xff)                                /* No abort due to higher priority MO */
      {                                                                             
            for(buffer = LowestQueuedBuffer; buffer < NoOfMB; buffer++)             
            {                                                                       
                  if (BufferStatus_CAN[buffer] == QUEUED)  
                  {
                        if(nextBuffer == LowestQueuedBuffer)  /* Search for next queued MB */
                        {
                              nextBuffer = buffer;                      
                        }

                        if (BufferID_Ref_CAN[buffer] < highestPriority) /* Search for highest priority queued buffer */
                        {
                              highestPriority = BufferID_Ref_CAN[buffer];
                              highestPriorityBuffer = buffer;
                        }
                  }
            }

            if (highestPriorityBuffer == LowestQueuedBuffer)
            {
                  LowestQueuedBuffer = nextBuffer;            /* start of search for next TxISR */
            }
      }
                                                                        
      else        /* Highest priority MB known */
      {
                                                                           /* copy the message number that caused the abort */
            highestPriorityBuffer = AbortMessage;
            highestPriority = BufferID_Ref_CAN[highestPriorityBuffer];
            AbortMessage = 0xff;                                           /* show that this message has been handled */
      }


      if (highestPriority == 0xff)                            /* no MBs queued: assume < 256 Message buffers */
      {
            if (DriverFlags & SLEEP_MASK)
            {
                  if (CANTFLG == 0x07)                 /* msCAN transmit buffers all transmitted */
                  {
                        CANCTL0 |= SLPRQ_MASK;                       /* set msCAN SLEEP request */
                        CANRIER |= WUPIE_MASK;                       /* enable wakeup interrupt */
                        DriverFlags &= ~SLEEP_MASK;                /* clear sleep request */
                  }
            }
      }

      else
      /* Message queued: copy highest priority MO into msCAN buffer and schedule transmission   */
      {
            MSCAN_Load_Buff (TX_Buff, highestPriorityBuffer, BufferID_Ref_CAN[highestPriorityBuffer]);

            CANTFLG = interruptBufferBit;                    /* write 1 to TXE to schedule buffer */
            CANTIER |= interruptBufferBit;                   /* enable interrupt for this buffer */
            MB_MSCAN[interruptBufferNo] = highestPriorityBuffer;      /* write buffer # */
            MBP_MSCAN[interruptBufferNo] = highestPriority;           /* write priority of buffer */
      
      }     /* end else: MB queued */

#ifdef BENCHMARK
      PORTB &= ~0x02;                                         /* clear bit 1 */
#endif
      u8CANTxOK = 0xffU;
      return;
}

/***************************************************************************
 * Interrupt:   CAN_ReceiveISR
 *
 * Description: msCAN receive interrupt handler.
 * 
 * Returns:     None.                   
 *
 * Notes:       None 
 *
 **************************************************************************/
interrupt void CAN_ReceiveISR (void)
{
      UINT8 *MO_ID8_CAN_Ptr;
      UINT8 *bufferDataPtr;         
      UINT8 RX_IdentifierType, 
            RX_FrameType,
            MB_ID_Type, 
            MB_ID_Ref, 
            match,
            dataLength, 
            buffer, 
            i,
            bufferMode;
      UINT8  *RX_BUFF_DSR_Ptr = & _CANRDSR0.Byte;

#ifdef BENCHMARK
      PORTB |= 0x01;                                          /* set bit 0 */
#endif

      if (DriverFlags & SLEEP_MASK)
      {
            DriverFlags &= ~SLEEP_MASK;                            /* clear SLEEP request */
            CANRIER &= ~WUPIE_MASK;                                  /* clear wakeup interrupt enable */
      }

      /* Find received Identifier Type and Frame Type */

      RX_IdentifierType = (UINT8)(RX_BUFF_IDR1 & IDE_MASK);                 /* test IDE bit */

      if (RX_IdentifierType == STANDARD)                      /* IDE clear */
      {
            RX_FrameType = (UINT8)(RX_BUFF_IDR1 & SRTR_MASK);               /* test RTR bit */
      }

      else        /* Extended ID */
      {
            RX_FrameType = (UINT8)(RX_BUFF_IDR3 & ERTR_MASK);               /* test RTR bit */
      }

      match = FALSE;

      /* Find matching buffer (if any): for each buffer:
       * If Data Frame received and buffer mode = RXDF 
       * or Remote Frame received and buffer mode = AUTOTXDF,
       * and if received Identifier type = buffer identifier type,
       * then compare received Identifier with buffer identifier,
       * until match found or all buffers tested.
       *
       * Note: TXRF buffers are not tested although ReceiveISR_CAN may 
       * theoretically be executed before TransmitISR_CAN
       */

      for (buffer = 0; buffer < NoOfMB; buffer++)
      {
            bufferMode = BufferMode_CAN[buffer];

            if (   ((RX_FrameType == DATAFRAME) && (bufferMode == RXDF))
                  || ((RX_FrameType != DATAFRAME) && (bufferMode == AUTOTXDF)))
            {
                  MB_ID_Ref      = BufferID_Ref_CAN[buffer];
                  MB_ID_Type     = (UINT8)(M_IdentifierType_CAN[MB_ID_Ref] & 0x08);
                  MO_ID8_CAN_Ptr = (UINT8*)&(M_Identifier_CAN[MB_ID_Ref]);
  
                  if (RX_IdentifierType == STANDARD)
                  {
                        if (MB_ID_Type == STANDARD)                                  /* MB_ID_Type matches */
                        {
                              if (  (RX_BUFF_IDR0           == MO_ID8_CAN_Ptr[0])    /* ID10 to ID3 match */
                                    && ((RX_BUFF_IDR1 & 0xE0) == MO_ID8_CAN_Ptr[1])) /* ID2 to ID0 match */
                              {
                                    match = TRUE;                                    /* match found */
                                    break;
                              }
                        }
                  }           /* end: if identifier == STANDARD */

                  else        /* MO_IdentiferType == EXTENDED */
                  {
                        if (MB_ID_Type != STANDARD)                                  /* MB_ID_Type matches */
                        {
                              if (    (RX_BUFF_IDR0           == MO_ID8_CAN_Ptr[0])  /* ID28 to ID21 match */
                                    &&  (RX_BUFF_IDR1         == MO_ID8_CAN_Ptr[1])  /* ID20 to ID15 match */
                                    &&  (RX_BUFF_IDR2         == MO_ID8_CAN_Ptr[2])  /* ID14 to ID7 match */
                                    && ((RX_BUFF_IDR3 & 0xFE) == MO_ID8_CAN_Ptr[3])) /* ID6 to ID0 match */
                              {
                                    match = TRUE;             /* match found */
                                    break;
                              }
                        }           /* end: if identifier type match */
                  }                 /* end: else identifier = extended */
            }                       /* end: if buffer mode */
      }                             /* end: for buffer < NO_OF_MB */

      if (match)
      {
            if (bufferMode == AUTOTXDF)
            {
                  CAN_QueueMB (buffer);                     /* queue buffer, copy to msCAN if buffer available,   */
                                                              /* abort msCAN buffer if all full and this buffer higher priority */
            }           

            else        /* bufferMode == RXDF */
            {
                  dataLength = (UINT8)(RX_BUFF_DLR & 0x0F);            /* mask upper nibble - undefined */
         
                  if (dataLength > 8)
                  {
                        dataLength = 8;
                  }

                  BufferDLC_CAN[buffer] = dataLength;        /* copy DLC */

                  bufferDataPtr = BufferData_CAN[buffer];    /* calculate first dimension */

                  for (i = 0; i < dataLength; i++)
                  {
                        bufferDataPtr[i] = RX_BUFF_DSR_Ptr[i];   /* copy data */
                  }

                  if (TimeStampOption)                        /* copy timestamp to Message Buffer */
                  {
                        BufferTimeStamp_CAN[buffer] = RX_BUFF_TSR;
                  }


                  if (  (BufferStatus_CAN[buffer] == READDATA)            
                        ||(BufferStatus_CAN[buffer] == NODATA))
                  {
                        BufferStatus_CAN[buffer] = NEWDATA;
                  }

                  else
                  {
                        BufferStatus_CAN[buffer] = OVERRUN;  /* previous data not read */
                  }

            }           /* end: bufferMode == RXDF */
      }                 /* end: match == TRUE */

      CANRFLG = RXF_MASK;                                          /* write 1 to clear RXF */

#ifdef BENCHMARK
      PORTB &= ~0x01;                                         /* clear bit 0 */
#endif

      u8gvbuffer = buffer - 16U;
      return;
}

/***************************************************************************
 * Interrupt:   CAN_WakeupISR
 *
 * Description: msCAn wake up interrupt handler.
 * 
 * Returns:     None.                   
 *
 * Notes:       None 
 *
 **************************************************************************/
interrupt void CAN_WakeupISR(void)
{
      CANRFLG = WUPIF_MASK;                                          /* clear wakeup interrupt flag */
      CANRIER &= ~WUPIE_MASK;                                        /* clear wakeup interrupt enable */
}

/***************************************************************************
 * Function :   CAN_GetVersion
 *
 * Description: returns the driver version.
 * 
 * Returns:     Version number.                   
 *
 * Notes:       None 
 *
 **************************************************************************/
UINT8 CAN_GetVersion (void)
{
      return (VERSION);
}

/*****************************************************************************/
/* end msCANdrv.c */

