/*******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
****************************************************************************//*!
*
* @file     MC33905_routines.c
*
* @author   b06050
*
* @version  1.0.3.0
*
* @date     Aug-3-2012
*
* @brief    MC33905 System Basis Chip initialization function file,
*           with routines for setup and diagnostics.
*
*******************************************************************************/
#include "MC33905_routines.h"

// transfer status
bool_t trans_status;

// SBC MC33905 wake-up source
uint16_t sbcStat;

MC33905_REG_T sbc_reg;


/***************************************************************************//*!
@brief          MC33905 read SBC flags

@param[in,out]  *ptr    - MC33905 configuration structure pointer

@return         bool_t  - function status

@details        The function read flags from SBC device

*******************************************************************************/
bool_t mc33905ReadFlag(MC33905_REG_T *ptr)
{
    static vuint16_t sbcFlags;

    /* Read and Clear VREG flags */
    trans_status = dspiSendWord(&ptr->dspiSet, SBC_RD_FLAG_REG_H, &sbcFlags);
    ptr->flags.regFlg.R = ((sbcFlags & 0xff) << 8);

    trans_status = dspiSendWord(&ptr->dspiSet, SBC_RD_FLAG_REG_L, &sbcFlags);
    ptr->flags.regFlg.R = ptr->flags.regFlg.R | (sbcFlags & 0xff);

    /* Read and Clear CAN flags */
    trans_status = dspiSendWord(&ptr->dspiSet, SBC_RD_FLAG_CAN_L, &sbcFlags);
    ptr->flags.canFlg.R = sbcFlags & 0xff;

    trans_status = dspiSendWord(&ptr->dspiSet, SBC_RD_FLAG_CAN_H, &sbcFlags);
    ptr->flags.canFlg.R = ptr->flags.canFlg.R | ((sbcFlags & 0xff) << 8);

    /* Read and Clear IO flags */
    trans_status = dspiSendWord(&ptr->dspiSet, SBC_RD_FLAG_IO_L, &sbcFlags);
    ptr->flags.ioFlg.R = sbcFlags & 0xff;

    trans_status = dspiSendWord(&ptr->dspiSet, SBC_RD_FLAG_IO_H, &sbcFlags);
    ptr->flags.ioFlg.R = ptr->flags.ioFlg.R | ((sbcFlags & 0xff) << 8);

    /* Read and Clear INT flags */
    trans_status = dspiSendWord(&ptr->dspiSet, SBC_RD_FLAG_INT_L, &sbcFlags);
    ptr->flags.intFlg.R = (sbcFlags & 0xff);

    /* Read and Clear INT flags */
    trans_status = dspiSendWord(&ptr->dspiSet, SBC_RD_FLAG_INT_H, &sbcFlags);
    ptr->flags.intFlg.R = ptr->flags.intFlg.R | ((sbcFlags & 0xff) << 8);

    /* Read and Clear LIN1 flags */
    trans_status = dspiSendWord(&ptr->dspiSet, SBC_RD_FLAG_LIN1_L, &sbcFlags);
    ptr->flags.lin1Flg.R = (sbcFlags & 0xff);

    /* Read and Clear LIN2 flags */
    trans_status = dspiSendWord(&ptr->dspiSet, SBC_RD_FLAG_LIN2_L, &sbcFlags);
    ptr->flags.lin2Flg.R = (sbcFlags & 0xff);

    return(1);
}

/***************************************************************************//*!
@brief          The function read MC33905 SBC I/O real time and device
                identification bits.

@param[in,out]  *ptr    - MC33905 configuration structure pointer

@return         bool_t  - function status

@details        The function read I/O real time and device identification value
                from SBC device.

*******************************************************************************/
bool_t mc33905ReadRtDevId(MC33905_REG_T *ptr)
{
    static vuint16_t sbcFlags;

    /* Read CAN status register value */
    trans_status = dspiSendWord(&ptr->dspiSet, SBC_GET_CAN_STATUS, &sbcFlags);
    ptr->flags.canStat.R = (sbcFlags & 0xff);

    /* Read LIN1 status register value */
    trans_status = dspiSendWord(&ptr->dspiSet, SBC_GET_LIN1_STATUS, &sbcFlags);
    ptr->flags.lin1Stat.R = (sbcFlags & 0xff);

    /* Read LIN2 status register value */
    trans_status = dspiSendWord(&ptr->dspiSet, SBC_GET_LIN2_STATUS, &sbcFlags);
    ptr->flags.lin2Stat.R = (sbcFlags & 0xff);

    return(1);
}

/***************************************************************************//*!
@brief          Resets the SBC watchdog

@param[in,out]  *ptr    - DSPI configuration structure pointer

@return         void

@details        The function resets the SBC watchdog

*******************************************************************************/
void mc33905ClearWdt(DSPI_SET_T *ptr)
{
    static vuint16_t sbcFlags;

    /* Clear watchdog */
    trans_status = dspiSendWord(ptr, SBC_SET_WDG_REG_VALUE, &sbcFlags);
}

/***************************************************************************//*!
@brief          Sets the SBC watchdog period

@param[in,out]  *ptr    - DSPI configuration structure pointer

@param[in]      u8WDT_Period, Code for desired period as defined in
                MC33905_routines.h (SBC_WDOG_2_5ms, SBC_WDOG_3ms, ...,
                SBC_WDOG_512ms)

@return         void

@details        The function sets the SBC watchdog period

*******************************************************************************/
void mc33905SetWdtPeriod(MC33905_REG_T *ptr)
{
    static vuint16_t sbcFlags, dspiDataOut;

    dspiDataOut = ptr->timA.R;
    trans_status = dspiSendWord(&ptr->dspiSet, dspiDataOut, &sbcFlags);
}

/***************************************************************************//*!
@brief          SBC initial configuration with default values from MC33905.h

@param[in,out]  *ptr    - MC33905 configuration structure pointer

@return         bool_t - function status

@details        The function configures SBC
*******************************************************************************/
bool_t mc33905Config(MC33905_REG_T *ptr)
{
    static vuint16_t u16SbcStatus,dspiDataOut;
    static vuint32_t pit_mcr, pit_ldval, pit_tctrl;

    /* Read 16 bit flags */
    ptr->dspiSet.txCmd = (DSPI_ASSERT_CS0 | DSPI_USE_CTAR_1);

    /* Read and Clear all flags */
    mc33905ReadFlag(ptr);
    mc33905ReadRtDevId(ptr);

    /* Read and Clear SAFE flag */
    dspiDataOut = SBC_GET_MODE_REG_H_FLAGS;
    trans_status = dspiSendWord(&ptr->dspiSet, dspiDataOut, &u16SbcStatus);

    // Initialize device
    /* LIN1 and LIN2 */
    dspiDataOut = ptr->initLinIo.R;
    trans_status = dspiSendWord(&ptr->dspiSet, dspiDataOut, &u16SbcStatus);

    /* Watchdog settings */
    dspiDataOut = ptr->initWdg.R;
    trans_status = dspiSendWord(&ptr->dspiSet, dspiDataOut, &u16SbcStatus);

    /* VREG */
    dspiDataOut = ptr->initReg.R;
    trans_status = dspiSendWord(&ptr->dspiSet, dspiDataOut, &u16SbcStatus);

    /* MISC */
    dspiDataOut = ptr->initMisc.R;
    trans_status = dspiSendWord(&ptr->dspiSet, dspiDataOut, &u16SbcStatus);

    /* Wait at least 50 microseconds before writing to timers registers */
    // read original PIT channel setup
    pit_mcr   = ioctl(PIT_BASE,PIT_READ_VALUE_REG_MCR, NULL);
    pit_ldval = ioctl(PIT_CHNL0_BASE,PIT_READ_VALUE_REG_LDVAL, NULL);
    pit_tctrl = ioctl(PIT_CHNL0_BASE,PIT_READ_VALUE_REG_TCTRL, NULL);

    // wait defined time for power supply voltage stabilization
    delayNanoSec(50000);

    // write original PIT channel setup
    ioctl(PIT_BASE,PIT_WRITE_VALUE_REG_MCR, pit_mcr);
    ioctl(PIT_CHNL0_BASE,PIT_SET_TMR_RELOAD_VALUE, pit_ldval);
    ioctl(PIT_CHNL0_BASE,PIT_WRITE_VALUE_REG_TCTRL, pit_tctrl);

    /* Set watchdog default period */
    mc33905SetWdtPeriod(ptr);

    /* Transition to normal mode */
    mc33905ClearWdt(&ptr->dspiSet);

    /* Resources initial configuration */
    /* Interrupt sources */
    dspiDataOut = ptr->interrupt.R;
    trans_status = dspiSendWord(&ptr->dspiSet, dspiDataOut, &u16SbcStatus);

    /* Regulator */
    dspiDataOut = ptr->reg.R;
    trans_status = dspiSendWord(&ptr->dspiSet, dspiDataOut, &u16SbcStatus);

    /* I/O configuration */
    dspiDataOut = ptr->io.R;
    trans_status = dspiSendWord(&ptr->dspiSet, dspiDataOut, &u16SbcStatus);

    /* MUX output */
    dspiDataOut = ptr->mux.R;
    trans_status = dspiSendWord(&ptr->dspiSet, dspiDataOut, &u16SbcStatus);

    /* Read MODE flag register to clear SAFE flag */
    dspiDataOut = SBC_GET_MODE_REG_H_FLAGS;
    trans_status = dspiSendWord(&ptr->dspiSet, dspiDataOut, &u16SbcStatus);

    // wait 10ms for power supply voltage stabilization
    // read original PIT channel setup
    pit_mcr   = ioctl(PIT_BASE,PIT_READ_VALUE_REG_MCR, NULL);
    pit_ldval = ioctl(PIT_CHNL0_BASE,PIT_READ_VALUE_REG_LDVAL, NULL);
    pit_tctrl = ioctl(PIT_CHNL0_BASE,PIT_READ_VALUE_REG_TCTRL, NULL);

    // wait defined time for power supply voltage stabilization
    delayNanoSec(DELAY_10MS);

    // write original PIT channel setup
    ioctl(PIT_BASE,PIT_WRITE_VALUE_REG_MCR, pit_mcr);
    ioctl(PIT_CHNL0_BASE,PIT_SET_TMR_RELOAD_VALUE, pit_ldval);
    ioctl(PIT_CHNL0_BASE,PIT_WRITE_VALUE_REG_TCTRL, pit_tctrl);

    return(1);
}

/***************************************************************************//*!
@brief          CAN transceiver configuration

@param[in]      *ptr    - MC33905 configuration structure pointer

@return         void

@details        The function configures MC33905 CAN transceiver

*******************************************************************************/
void mc33905CanCfg(MC33905_REG_T *ptr)
{
    static  vuint16_t   u16SbcStatus, dspiDataOut;
    /* CAN configuration */
    dspiDataOut = ptr->can.R;
    trans_status = dspiSendWord(&ptr->dspiSet, dspiDataOut, &u16SbcStatus);
}