/*******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
****************************************************************************//*!
*
* @file     MC33937_routines.c
*
* @author   b06050
*
* @version  1.0.3.0
*
* @date     Aug-3-2012
*
* @brief    The MC33937A MOSFET pre-driver Init function file, with routines for
*           pre-driver setup and diagnostics.
*
*******************************************************************************/

/*******************************************************************************
| Includes
------------------------------------------------------------------------------*/
#include "MC33937_routines.h"

/*******************************************************************************
| Global variable definitions   (scope: module-local)
------------------------------------------------------------------------------*/
bool_t bFcnStatusMC33937;


/*******************************************************************************
| Function implementations      (scope: module-exported)
------------------------------------------------------------------------------*/

/***************************************************************************//*!
@brief          MC33937 pre-driver fault clear

@param[in]      *ptr    - MC33937A configuration structure pointer

@return         void

@details        The function clears a portion of the fault latch corresponding
                to MASK0 and MASK1 using lower four bits of commands. A 1 bit
                clears the interrupt latch for that flag. INT remains asserted
                if other unmasked faults are still present.

*******************************************************************************/
void mc33937ClearFault(MC33937_SET_T *ptr)
{
    vuint8_t help;

    // Disable MC33937 for operation
    ioctl(SIUL_BASE,SIUL_CLEAR_PAD, ptr->enPcrNb);

    bFcnStatusMC33937 = dspiSendByte(&ptr->dspiSet, CLINT0_CMD, &help);
    bFcnStatusMC33937 = dspiSendByte(&ptr->dspiSet, CLINT1_CMD, &help);
}


/***************************************************************************//*!
@brief          MC33937 pre-driver set Enable pin

@param[in]      *ptr    - MC33937A configuration structure pointer

@return         The MC33937_enable function returns nothing.

@details        The function sets enable inputs pin of MC33937 pre-driver
                (enable driver outputs).


@note           This function is used to enable outputs of the MC33937

@warning
*******************************************************************************/
void mc33937Enable(MC33937_SET_T *ptr)
{
    // Enable MC33937 for operation
    ioctl(SIUL_BASE,SIUL_SET_PAD, ptr->enPcrNb);
}


/***************************************************************************//*!
@brief          Clear Enable pin of the MC33937 pre-driver.

@param[in]      *ptr    - MC33937A configuration structure pointer

@return         void

@details        The function clear enable inputs of MC33937A pre-driver (disable
                driver outputs). The IC is in Standby mode.

@note           The IC is fully biased up and all functions are operating, the
                output drivers actively turn off all of the external FETs
                (after initialization). The SPI port is functional. Logic level
                 outputs are driven with low impedance. SO is high impedance
                unless CS is low. VDD, Charge Pump and VLS regulators are all
                operating. The IC is ready to move to Enable Mode.

@warning
*******************************************************************************/
void mc33937Disable(MC33937_SET_T *ptr)
{
    // Disable MC33937 for operation
    ioctl(SIUL_BASE,SIUL_CLEAR_PAD, ptr->enPcrNb);
}

/***************************************************************************//*!
@brief          Reset the MC33937 pre-driver.

@param[in,out]  *ptr    - MC33937A configuration structure pointer

@return         void

@details        The function resets MC33937 pre-driver

@note

@warning
*******************************************************************************/
void mc33937Reset(MC33937_SET_T *ptr)
{
    // Disable MC33937 for operation
    ioctl(SIUL_BASE,SIUL_CLEAR_PAD, ptr->rstPcrNb);
}

/***************************************************************************//*!
@brief          Read the status registers of the MC33937 pre-driver.

@param[in]      *ptr    - MC33937A configuration structure pointer

@return         bool_t  - function return status

@details        The function read status registers values of the MC33937A
                pre-driver. The actual values of the status registers are stored
                in ptr structure.
*******************************************************************************/
bool_t  mc33937ReadSr(MC33937_SET_T *ptr)
{
    vuint8_t help;

    // assert transmit command
    ptr->dspiSet.txCmd = DSPI_ASSERT_CS0 | DSPI_USE_CTAR_0;
    // send NULL0 command - ignore return value (previous command is unknown)
    bFcnStatusMC33937 = dspiSendByte(&ptr->dspiSet, STATUS0_CMD, &help);

    // send NULL0 command - return value is status register 0 value
    bFcnStatusMC33937 = dspiSendByte(&ptr->dspiSet, STATUS1_CMD, &help);
    ptr->status.sr0.R = help;
    // send NULL1 command - return value is status register 1 value
    bFcnStatusMC33937 = dspiSendByte(&ptr->dspiSet, STATUS2_CMD, &help);
    ptr->status.sr1.R = help;
    // send NULL2 command - return value is status register 2 value
    bFcnStatusMC33937 = dspiSendByte(&ptr->dspiSet, STATUS3_CMD, &help);
    ptr->status.sr2.R = help;
    // send NULL0 command - return value is status register 3 value
    bFcnStatusMC33937 = dspiSendByte(&ptr->dspiSet, STATUS0_CMD, &help);
    ptr->status.sr3 = help;

    return (1);
}


/***************************************************************************//*!
@brief          MC33937 pre-driver deadtime setup

@param[in]      *ptr    - MC33937A configuration structure pointer

@return         bool_t  - function return status

@details        The function setup deadtime between top and bottom transistor
                of MC33937A pre-driver. Required deadtime value is stored in
                structure item ptr->deadtime. The MC33937A real deadtime
                value is stored in the ptr->setDeadtime. The real deadtime
                value corresponds to the value of the status register 3
                multiplied by 50ns.


@note           This function is included in the MC33937_config function

@warning
*******************************************************************************/
bool_t  mc33937DeadtimeSetup(MC33937_SET_T *ptr)
{
    static  vuint32_t pit_mcr, pit_ldval, pit_tctrl;
    static  vuint32_t deadtime_reg, pcr_regOrigSetup;
    static  vuint16_t help1;
    static  vuint8_t help_sr0;

    // send command for setup deadtime
    bFcnStatusMC33937 = dspiSendByte(&ptr->dspiSet, DEADTIME_CMD, &help_sr0);

    // store original PIT channel setup
    pit_mcr = ioctl(PIT_BASE,PIT_READ_VALUE_REG_MCR, NULL);
    pit_ldval = ioctl(ptr->pitChnlBaseAdr,PIT_READ_VALUE_REG_LDVAL, NULL);
    pit_tctrl = ioctl(ptr->pitChnlBaseAdr,PIT_READ_VALUE_REG_TCTRL, NULL);

    delayNanoSec(20000);

    // configure pad as GPIO
    ioctl(SIUL_BASE,SIUL_CONFIGURE_PAD_GPIO, ptr->dspiSet.csPcrNb);

    // deadtime value is calculated (required_deadtime * 16) in [ns]
    // for the PIT timer
    deadtime_reg = (uint32_t)(ptr->deadtime * MC33937_CALIB_CLK);

    // pull-down CS0 for setup deadtime
    ioctl(SIUL_BASE,SIUL_CLEAR_PAD, ptr->dspiSet.csPcrNb);

    delayNanoSec(deadtime_reg);

    // pull-up CS0 at the end of deadtime setup
    ioctl(SIUL_BASE,SIUL_SET_PAD, ptr->dspiSet.csPcrNb);

    // restore original pin setup
    ioctl(SIUL_BASE,SIUL_CONFIGURE_PAD_ALT2, ptr->dspiSet.csPcrNb);

    // read deadtime register from MC33937
    bFcnStatusMC33937 = dspiSendByte(&ptr->dspiSet, STATUS3_CMD, &help_sr0);
    bFcnStatusMC33937 = dspiSendByte(&ptr->dspiSet, STATUS2_CMD, &help_sr0);

    help1 = (uint16_t)(help_sr0);

    // return deadtime in [ns]
    ptr->setDeadtime = ((uint16_t)(help_sr0) * 50);

    // return 1 if deadtime setup correctly
    return(1);
}


/***************************************************************************//*!
@brief          MC33937 pre-driver configuration function.

@param[in]      *ptr    - MC33937A configuration structure pointer

@return         bool_t  - deadtime calibration status

@details        The function provide configuration of MC33937A pre-driver and
                setup deadtime between top and bottom transistor. Function uses
                PIT module and DSPI module.

@note           This function is used to configure MC33937A

@warning
*******************************************************************************/
bool_t  mc33937Config(MC33937_SET_T *ptr)
{
    static  vuint32_t pit_mcr, pit_ldval, pit_tctrl;
    static  vuint16_t deltaDT;
    static  vuint16_t MC33937_deadError;
    static  vuint8_t help_sr0, error_cnt;
    bool_t dt_set_fault;

    // remove /RST, EN1 and EN2 are still low -> VDD rise, init interrupt
    // handler, enable SPI C[10] to high
    ioctl(SIUL_BASE,SIUL_SET_PAD, ptr->rstPcrNb);

    // store original PIT channel setup
    pit_mcr = ioctl(PIT_BASE,PIT_READ_VALUE_REG_MCR, NULL);
    pit_ldval = ioctl(PIT_CHNL0_BASE,PIT_READ_VALUE_REG_LDVAL, NULL);
    pit_tctrl = ioctl(PIT_CHNL0_BASE,PIT_READ_VALUE_REG_TCTRL, NULL);

    // wait defined time for power supply voltage stabilization
    delayNanoSec(MC33937_POWER_UP_TIME);

    // apply original PIT channel setup
    ioctl(PIT_BASE,PIT_WRITE_VALUE_REG_MCR, pit_mcr);
    ioctl(PIT_CHNL0_BASE,PIT_SET_TMR_RELOAD_VALUE, pit_ldval);
    ioctl(PIT_CHNL0_BASE,PIT_WRITE_VALUE_REG_TCTRL, pit_tctrl);

    // clear faults first
    mc33937ClearFault(ptr);
    // Initialize MASK register to mask unwanted interrupts
    // send MASK0 and MASK1 command to mask out unwanted interrupts
    bFcnStatusMC33937 = dspiSendByte(&ptr->dspiSet, ptr->mask0, &help_sr0);
    bFcnStatusMC33937 = dspiSendByte(&ptr->dspiSet, ptr->mask1, &help_sr0);

    error_cnt = 5;
    // setup desired deadtime of MC33937 value is in [ns]
    do {
        bFcnStatusMC33937 = mc33937DeadtimeSetup(ptr);
        deltaDT = ptr->deadtime - ptr->setDeadtime;
        dt_set_fault = (deltaDT > 150) ? 1 : 0;
        error_cnt--;
    // deadtime error is greater than 150ns, setup deadtime again
    }while(dt_set_fault && error_cnt);

    mc33937ClearFault(ptr);
    // send MODE command with desired bits, and also the LOCK bit to prevent
    // further mode changes
    bFcnStatusMC33937 = dspiSendByte(&ptr->dspiSet, ptr->mode, &help_sr0);
    return (!dt_set_fault);
}