/*******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
****************************************************************************//*!
*
* @file     MC33937_routines.h
*
* @author   b06050
*
* @version  1.0.3.0
*
* @date     Aug-3-2012
*
* @brief    MC33937A MOSFET pre-driver Init function header file, with routines
*           for setup and diagnostic.
*
*******************************************************************************/
#ifndef __MC33937_ROUTINES_H
#define __MC33937_ROUTINES_H

#include "typedefs.h"
#include "MPC5606B_LLD_qs.h"
#include "MPC5606B_DSPI_LLD.h"
#include "MPC5606B_PIT_LLD.h"
#include "MPC5606B_SIUL_LLD.h"

/*******************************************************************************
| Defines and macros            (scope: module-local)
------------------------------------------------------------------------------*/

//#define PICTUS
//#define LEOPARD
#define BOLERO
// deadtime value in [ns]
#define DEADTIME                    100
// define commands for MC33937A predriver

// MASK command
/*
MASK 0
-------------------------------------------------------------------------------
| 7 | 6 | 5 | 4 |     bit3     |       bit2       |     bit1     |    bit0    |
-------------------------------------------------------------------------------
| 0 | 0 | 1 | 0 | Over current | VLS undervoltage | Desaturation | Over Temp. |
-------------------------------------------------------------------------------
*/
#define MASK0_CMD                   0x20

/*
MASK 1
---------------------------------------------------------------------------
| 7 | 6 | 5 | 4 |     bit3    |     bit2    |     bit1      |    bit0     |
---------------------------------------------------------------------------
| 0 | 0 | 1 | 1 | Reset Event | Write Error | Framing Error | Phase Error |
---------------------------------------------------------------------------
*/
#define MASK1_CMD                   0x30

// CLINT command
// Clears portion of the fault latch corresponding to MASK0
#define CLINT0_CMD                  0x6F
// Clears portion of the fault latch corresponding to MASK1
#define CLINT1_CMD                  0x7F

/*
MODE command
---------------------------------------------------------
| 7 | 6 | 5 | 4 |     bit3    | 2 |  bit1   |    bit0   |
---------------------------------------------------------
| 0 | 1 | 0 | 0 | Desat Fault | 0 | FULLON  | Mode Lock |
|   |   |   |   |    mode     |   |  mode   |           |
---------------------------------------------------------
*/
#define MODE_CMD                    0x49

// DEADTIME command
#define DEADTIME_CMD                0x81

// STATUS register 0
#define STATUS0_CMD                 0x00
// STATUS register 0
#define STATUS1_CMD                 0x01
// STATUS register 0
#define STATUS2_CMD                 0x02
// STATUS register 0
#define STATUS3_CMD                 0x03

#define PIT_DELAY_2MS               (PIT_CLK * 2000)
#define PIT_DELAY_10US              (PIT_CLK * 10)

// MC33937A deadtime calibration clock
#define MC33937_CALIB_CLK           (16)
// timeout for stabilize power on MC33937A device = 2ms
#define MC33937_POWER_UP_TIME       2000000

#ifdef BOLERO
/*! Macro containing motor CS values for MPC560xB Motor Control board */
#define MOTOR1_CS                   0

/*! Macro containing number of the PCR register of the CS for the motor 1*/
#define MOTOR1_CSPCR                107

/*! Macro containing number of the PCR register of the Enable for motor 1 */
#define ENBLPCR                     14

/*! Macro containing number of the PCR register of the Reset for motor 1 */
#define RSTPCR                      90

#endif

#ifdef LEOPARD
/*! Macro containing motor CS values for MPC560xB Motor Control board */
#define MOTOR1_CS                   0
#define MOTOR2_CS                   1

/*! Macro containing number of the PCR register of the CS for motor 1 and 2 */
#define MOTOR1_CSPCR                36
#define MOTOR2_CSPCR                79

/*! Macro containing number of the PCR register of the Enable for motor 1 */
#define ENBLPCR                     11

/*! Macro containing number of the PCR register of the Reset for motor 1 */
#define RSTPCR                      10

#endif

#ifdef PICTUS
/*! Macro containing motor CS values for MPC560xB Motor Control board */
#define MOTOR1_CS                   0

/*! Macro containing number of the PCR register of the CS for motor 1 and 2 */
#define MOTOR1_CSPCR                83

#endif

/*! Macro containing default values for setup MPC560xB control board.*/
#define MPC560XB_LQFP144_CB_DEFAULT        {MOTOR1_CS,MOTOR1_CSPCR,(DSPI_ASSERT_CS0 | DSPI_USE_CTAR_0),DSPI4_BASE,0,0,0,0,0,DEADTIME,MODE_CMD,MASK0_CMD,MASK1_CMD,PIT_CHNL0_BASE,RSTPCR,ENBLPCR,3}

/*! Macro containing default value for setup MPC5643L dual motor control board.*/
#define MPC5643L_BGA257_CB_M1_DEFAULT      {MOTOR1_CS,MOTOR1_CSPCR,(ASSERT_CS0 | ASSERT_CTAR0),DSPI0_BASE,0,0,0,0,0,DEADTIME,MODE_CMD,MASK0_CMD,MASK1_CMD,PIT_CHNL0_BASE,RSTPCR,ENBLPCR,2}
//#define MPC5643L_BGA257_CB_M1_DEFAULT      {0,0,0,0,0,DEADTIME,MODE_CMD,MASK0_CMD,MASK1_CMD,3,MOTOR1_CS,0,MOTOR1_CSPCR,10,11,2}
#define MPC5643L_BGA257_CB_M2_DEFAULT      {0,0,0,0,0,DEADTIME,MODE_CMD,MASK0_CMD,MASK1_CMD,3,MOTOR2_CS,0,MOTOR2_CSPCR,10,11,2}

/*! Macro containing default values for setup MPC5604P control board.*/
#define MPC5604P_TQFP144_CB_DEFAULT        {0,0,0,0,0,DEADTIME,MODE_CMD,MASK0_CMD,MASK1_CMD,3,MOTOR1_CS,3,MOTOR1_CSPCR,42,96,1}



/*******************************************************************************
| Typedefs and structures       (scope: module-local)
------------------------------------------------------------------------------*/
/*-------------------------------------------------------------------------*//*!
@struct SR0_T "\MC33937_routines.h"

@brief  Structure containing status register 0 image of the MC33937A device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint8_t R;
    struct {
        uint8_t    reset_event  :1; /*!< reset event flag, is set upon exiting /RST */
        uint8_t    write_err    :1; /*!< Write Error After the Lock flag */
        uint8_t    framing_err  :1; /*!< Framing error flag */
        uint8_t    phase_err    :1; /*!< Phase error flag */
        uint8_t    overcurrent  :1; /*!< Over-current event flag */
        uint8_t    lowvls       :1; /*!< Low VLS voltage flag */
        uint8_t    desat        :1; /*!< DESAT flag detected on any channel */
        uint8_t    tlim         :1; /*!< TLIM flag detected on any channel */
    } B;
}SR0_T;

/*-------------------------------------------------------------------------*//*!
@struct SR1_T "\MC33937_routines.h"

@brief  Structure containing status register 1 image of the MC33937A device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint8_t R;
    struct {
        uint8_t    :1;                  /*!< reserve */
        uint8_t    desat_mode   :1;     /*!< current state of the Desaturation/Phase Error turn-off mode */
        uint8_t    zds          :1;     /*!< Zero deadtime is commanded */
        uint8_t    calib_overfl :1;     /*!< flag for a Deadtime Calibration Overflow */
        uint8_t    deadtime_cal :1;     /*!< Deadtime calibration occurred */
        uint8_t    :1;                  /*!< reserve */
        uint8_t    fullon       :1;     /*!< present status of FULLON MODE */
        uint8_t    lockbit      :1;     /*!< LockBit indicates the IC regs. are locked */
    } B;
}SR1_T;

/*-------------------------------------------------------------------------*//*!
@struct SR2_T "\MC33937_routines.h"

@brief  Structure containing status register 2 image of the MC33937A device
*//*--------------------------------------------------------------------------*/
typedef union {
    uint8_t R;
    struct {
        uint8_t    mask1_3      :1;     /*!< status of the MASK1.3 bit */
        uint8_t    mask1_2      :1;     /*!< status of the MASK1.2 bit */
        uint8_t    mask1_1      :1;     /*!< status of the MASK1.1 bit */
        uint8_t    mask1_0      :1;     /*!< status of the MASK1.0 bit */
        uint8_t    mask0_3      :1;     /*!< status of the MASK0.3 bit */
        uint8_t    mask0_2      :1;     /*!< status of the MASK0.2 bit */
        uint8_t    mask0_1      :1;     /*!< status of the MASK0.1 bit */
        uint8_t    mask0_0      :1;     /*!< status of the MASK0.0 bit */
    } B;
}SR2_T;

/*-------------------------------------------------------------------------*//*!
@struct MC33937_SR_T "\MC33937_routines.h"

@brief  Structure containing status registers image of the MC33937A device
*//*--------------------------------------------------------------------------*/
typedef struct
{
    SR0_T       sr0;    /*!< status register 0 */
    SR1_T       sr1;    /*!< status register 1 */
    SR2_T       sr2;    /*!< status register 2 */
    uint8_t     sr3;    /*!< status register 3 */
}MC33937_SR_T;

/*-------------------------------------------------------------------------*//*!
@struct MC33937_SET_T "\MC33937_routines.h"

@brief  Structure containing configuration setup for the MC33937A device
*//*--------------------------------------------------------------------------*/
typedef struct
{
    DSPI_SET_T      dspiSet;        /*!< DSPI settings */
    MC33937_SR_T    status;         /*!< image of the Status registers */
    uint16_t        setDeadtime;    /*!< real dead time of HS and LS transistors */
    uint16_t        deadtime;       /*!< define dead time of HS and LS transistors */
    uint8_t         mode;           /*!< define value of the MODE register */
    uint8_t         mask0;          /*!< define value of the MASK0 register */
    uint8_t         mask1;          /*!< define value of the MASK1 register */
    uint32_t        pitChnlBaseAdr; /*!< define number of the PIT channel used for generation delays */
    uint8_t         rstPcrNb;       /*!< define number of the PCR register used for /RESET signal on the CB */
    uint8_t         enPcrNb;        /*!< define number of the PCR register used for ENABLE signal on the CB */
    uint8_t         typeCb;         /*!< define type of the used CB */
}MC33937_SET_T;


extern bool_t  bFcnStatusMC33937;

// *****************************************************************************
extern  void        mc33937ClearFault(MC33937_SET_T *ptr);
extern  void        mc33937Enable(MC33937_SET_T *ptr);
extern  void        mc33937Disable(MC33937_SET_T *ptr);
extern  void        mc33937Reset(MC33937_SET_T *ptr);
extern  bool_t      mc33937ReadSr(MC33937_SET_T *ptr);
extern  bool_t      mc33937DeadtimeSetup(MC33937_SET_T *ptr);
extern  bool_t      mc33937Config(MC33937_SET_T *ptr);

#endif  /* __MC33937_ROUTINES_H */