/*******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
****************************************************************************//*!
*
* @file     MPC5606B_ADC_BSP.h
*
* @author   b06050
*
* @version  1.0.3.0
*
* @date     Aug-3-2012
*
* @brief    Header file for MPC5606B Controller Board Support Package - ADC
*           module
*
********************************************************************************
*
* This file includes MACRO and user type definitions of Board Support Package
* for MPC5606B Controller Board
*
*******************************************************************************/
#ifndef __MPC5606B_ADC_BSP_H
#define __MPC5606B_ADC_BSP_H

/*******************************************************************************
| Defines and macros            (scope: module-local)
------------------------------------------------------------------------------*/
// ADC channel numbers
#define ADC_CH0             0
#define ADC_CH1             1
#define ADC_CH2             2
#define ADC_CH3             3
#define ADC_CH4             4
#define ADC_CH5             5
#define ADC_CH6             6
#define ADC_CH7             7
#define ADC_CH8             8
#define ADC_CH9             9
#define ADC_CH10            10
#define ADC_CH11            11
#define ADC_CH12            12
#define ADC_CH13            13
#define ADC_CH14            14
#define ADC_CH15            15

// Phase current channels assignment
#define BEMFA_CURRENT_CHNL  ADC_CH4
#define BEMFB_CURRENT_CHNL  ADC_CH5
#define BEMFC_CURRENT_CHNL  ADC_CH6

#define UDCB_VOLTAGE_CHNL   ADC_CH7
#define IDCB_CURRENT_CHNL   ADC_CH8

#define TEMPERATURE_CHNL    ADC_CH9

#define ADC_RAW_RESULT_DEFAULT  {{0,0},{0,0},{0,0},{0,0},{0,0},{0,0}}
#define ADC_OFFSET_DEFAULT      {0,0,0,0,0,0}
#define ADC_PARAM_DEFAULT       {0}
#define ADC_FLAGS_DEFAULT       {0}

#define ADC_MODULE_DEFAULT  {ADC_RAW_RESULT_DEFAULT,\
                            ADC_OFFSET_DEFAULT,\
                            ADC_PARAM_DEFAULT,\
                            ADC_FLAGS_DEFAULT}

#define FILTER_MA2_DEFAULT {0,0}

/******************************************************************************
| Typedefs and structures       (scope: module-local)
------------------------------------------------------------------------------*/
/*-------------------------------------------------------------------------*//*!
@brief  Structure containing raw ADC results
*//*--------------------------------------------------------------------------*/
typedef struct
{
    volatile tFrac32 raw;       // raw value
    volatile tFrac32 filt;      // filtered value
}adcFilt32_t;

typedef struct
{
    volatile tFrac16 raw16;     // raw value
    volatile tFrac16 filt16;    // filtered value
}adcFilt16_t;

/*-------------------------------------------------------------------------*//*!
@brief  Structure containing raw ADC results
*//*--------------------------------------------------------------------------*/
typedef struct
{
    adcFilt32_t f32PhA;     // DC offset measured on phase A current
    adcFilt32_t f32PhB;     // DC offset measured on phase B current
    adcFilt32_t f32PhC;     // DC offset measured on phase C current
    adcFilt32_t f32Idcb;    // DC offset measured on DC bus current
    adcFilt32_t f32Udcb;    // DC offset measured on DC bus voltage
    adcFilt32_t f32Temp;    // DC offset measured on temperature
}adcRawResult_t;

typedef struct
{
    adcFilt16_t f16BemfA;   // DC offset measured on phase A current
    adcFilt16_t f16BemfB;   // DC offset measured on phase B current
    adcFilt16_t f16BemfC;   // DC offset measured on phase C current
    adcFilt16_t f16Idcb;    // DC offset measured on DC bus current
    adcFilt16_t f16Udcb;    // DC offset measured on DC bus voltage
    adcFilt16_t f16Temp;    // DC offset measured on temperature
}adcRawResult16_t;

/*-------------------------------------------------------------------------*//*!
@brief  Structure containing variables for software DC offset calibration.
*//*--------------------------------------------------------------------------*/
typedef struct
{
    volatile tFrac32 f32PhA;    // DC offset measured on phase A current
    volatile tFrac32 f32PhB;    // DC offset measured on phase B current
    volatile tFrac32 f32PhC;    // DC offset measured on phase C current
    volatile tFrac32 f32Idcb;   // DC offset measured on DC bus current
    volatile tFrac32 f32Udcb;   // DC offset measured on DC bus voltage
    volatile tFrac32 f32Temp;   // DC offset measured on temperature
}adcOffset_t;

typedef struct
{
    volatile tFrac16 f16BemfA;  // DC offset measured on phase A current
    volatile tFrac16 f16BemfB;  // DC offset measured on phase B current
    volatile tFrac16 f16BemfC;  // DC offset measured on phase C current
    volatile tFrac16 f16Idcb;   // DC offset measured on DC bus current
    volatile tFrac16 f16Udcb;   // DC offset measured on DC bus voltage
    volatile tFrac16 f16Temp;   // DC offset measured on temperature
}adcOffset16_t;

/*-------------------------------------------------------------------------*//*!
@brief  Structure containing all variables to configure ADC on the application
        level.
*//*--------------------------------------------------------------------------*/
typedef struct
{
    tU16 u16CalibSamples;       // Number of samples taken for ADC calibration

}adcParam_t;

/*-------------------------------------------------------------------------*//*!
@brief  Union containing ADC module operation flags.
*//*--------------------------------------------------------------------------*/
typedef union
{
    vuint16_t R;
    struct {
        vuint16_t reserved      :14;// RESERVED
        vuint16_t calibDone     :1; // DC offset calibration done
        vuint16_t calibInitDone :1; // initial setup for DC offset calibration done
    } B;
}adcFlags_t;

/*-------------------------------------------------------------------------*//*!
@brief  ADC module structure containing all ADC related variables.
*//*--------------------------------------------------------------------------*/
typedef struct
{
    volatile adcRawResult_t measured;
    volatile adcOffset_t offset[7];
    volatile adcParam_t param;
    volatile adcFlags_t flag;
}adcModule_t;

typedef struct
{
    volatile adcRawResult16_t measured;
    volatile adcOffset16_t offset[7];
    volatile adcParam_t param;
    volatile adcFlags_t flag;
}adcModule16_t;

typedef struct
{
    tFrac32 nAcc;
    tU16 nN;
}FILTER_MA2;

/*******************************************************************************
| Exported function prototypes
------------------------------------------------------------------------------*/
extern bool_t ADCModuleInit(adcModule_t *ptr);
extern bool_t ADCModuleInit16(adcModule16_t *ptr);
extern bool_t ADCModuleCalib(adcModule_t *ptr, tU32 svmSector);
extern bool_t ADCModuleCalib16(adcModule16_t *ptr, tU16 svmSector);
extern bool_t ADCMeasureBEMF(adcModule16_t *ptr, tU16 svmSector,tFrac16 *pIdcb);
/*******************************************************************************
| Inline functions
------------------------------------------------------------------------------*/

#endif /* __MPC5606B_ADC_BSP_H */