/*******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
****************************************************************************//*!
*
* @file     BLDCHallBased.c
*
* @author   RC574C
*
* @version  1.0.4.0
*
* @date     Jul-27-2012
*
* @brief    BLDC Motor Control Hall Based Application for MC9S12G128.
*
*******************************************************************************/
#include <hidef.h>      /* Common defines and macros */
#include "derivative.h" /* Derivative-specific definitions */
#include "sys.h"
#include "pim.h"
#include "spi.h"
#include "sci.h"
#include "tim.h"
#include "adc.h"
#include "pwm.h"
#include "can.h"
#include "can_cfg.h"
#include "MC33905_routines.h"
#include "MC33937_routines.h"
#include "math.h"
#include "S12G128_appconfig.h"
#include "freemaster.h"

#define APP_INIT                0   /* Application states */
#define APP_RUN                 1
#define APP_STOP                2
#define APP_FAULT               3

/*******************************************************************************
*
* Type definition
*
*******************************************************************************/
typedef void (*tPointerFcn)(void);  /* pointer to a function */

typedef union {
    uint16_t R;
    struct {
        uint16_t StallCheckReq:1;
        uint16_t CurrentLimiting:1;
        uint16_t Fault:1;
        uint16_t Reserved:13;
    }B;
}tDriveStatus;

typedef union {
    uint8_t R;
    struct {
        uint8_t OverDCBusCurrent:1;
        uint8_t OverDCBusVoltage:1;
        uint8_t UnderDCBusVoltage:1;
        uint8_t PreDriverError:1;
        uint8_t MotorStall:1;
        uint8_t HallPaternError:1;
        uint8_t Reserved:2;
    }B;
}tFaultStatus;

typedef struct {
    tFrac16 DCBVVoltage;
    tFrac16 DCBIVoltage;
    tFrac16 DCBIOffset;
}tADCresults;

typedef union
{
    uint8_t R;
    struct 
    {
        uint8_t H0:1;
        uint8_t H1:1;
        uint8_t H2:1; 
        uint8_t Reserved:5;     
    }B;
}tHallState; 


/*******************************************************************************
*
* Function prototypes
*
*******************************************************************************/
static void AppInit(void);
static void AppRun(void);
static void AppStop(void);
static void AppFault(void);


static void FaultDetection(void);
static void LED_Indication(void);
static void Encoder_RotCtrl(void);
static void Encoder_SwCtrl(void);

static void PWM_Vector_0(void);
static void PWM_Vector_1(void);
static void PWM_Vector_2(void);
static void PWM_Vector_3(void);
static void PWM_Vector_4(void);
static void PWM_Vector_5(void);

static void Control_Loop(void);

#pragma CODE_SEG __NEAR_SEG NON_BANKED
void interrupt ADC_SC_ISR(void);
void interrupt PP_ISR(void);
#pragma CODE_SEG DEFAULT


void main(void);

/*******************************************************************************
*
* Variables definition
*
*******************************************************************************/
Can_ReturnType ret_val;

static vuint16_t duty_cycle;
static uint16_t  speedLoopCounter = SPEED_LOOP_PRESCALER, PICounter = 20;

MC33937_SR_T        mc33937_Status;
static uint8_t      appState, appSwitchState, faultSwitchClear;
static tDriveStatus driveStatus;
static tFaultStatus faultStatus, faultStatusLatched;
static tADCresults  ADCResults;

static uint16_t period6ZC;
static tFrac16 torqueErr, torque, torque6ZC;

static uint16_t stallCheckCounter;

static tFrac16 speedErr, requiredSpeed = 3000;
static tFrac16 actualSpeed;
static tFrac16 speedPIOut, currentPIOut;
static CONTROLLER_PIAW_R_T_F16 speedPIPrms, currentPIPrms;
static FILTER_MA_T_F16 Udcb_filt;

static tFrac16 requiredSpeed;

static uint8_t  rotSwitchA, rotSwitch;
static uint8_t  LEDcounter;
static uint16_t switchCounter;

static tHallState hallPattern;

/* Array with pointers to the state machine functions */
static tPointerFcn AppStateMachine[] = \
{
    AppInit,
    AppRun,
    AppStop,
    AppFault
};

/* Array with pointers to the commutation functions */
static tPointerFcn PWMCommutationFcn[] = \
{
    PWM_Vector_0,
    PWM_Vector_2,
    PWM_Vector_1,
    PWM_Vector_4,
    PWM_Vector_5,
    PWM_Vector_3
};

/* FreeMASTER constants (need to be referenced in ENTRIES section)
   of the Project.prm to avoid deadstripping */
const tFrac16 fm_voltage = FM_U_SCALE;
const tFrac16 fm_current = FM_I_SCALE;

/* Array with Hall PortP interrupts setup, clockwise */
uint8_t hallIntrConfigCw[6] = {0x50, 0x44, 0x40, 0x14, 0x10, 0x04};

/* Array with Hall PortP interrupts flag clear masks, clockwise */
uint8_t hallIntrClrMaskCw[6] = {0x40, 0x04, 0x10, 0x10, 0x04, 0x40};

/* Array period */
uint16_t hallEdgePeriod[6] = {0, 0, 0, 0, 0, 0};

/* Array torque */
tFrac16 hallEdgeTorque[6] = {0, 0, 0, 0, 0, 0};

uint16_t timerValue = 0, timerLastValue = 0, timerValueLim, brakeCounter;
uint8_t brakeCounterFlag;

static tFrac16 DCBusVoltageThreshold;

int16_t test;

/*******************************************************************************
*
* Function: void main(void)
*
* Description: main function
*
*******************************************************************************/
void main(void)
{
    uint8_t ret;

    _DISABLE_COP(); /* Disable COP watchdog */
    DISABLE_INTERRUPTS();

    /* Init system clock to 50MHz, BUSclk=25MHz */
    PLL_Init();
    /* Initialize pin states */
    PIM_Init();
    /* Initialize SPI0 */
    SPI_0_Init();
    /* Initialize SCI0 */
    SCI_0_Init();
    /* initialize FreeMASTER */
    ret = FMSTR_Init();
    /* Initialize MC33905 System Basis Chip */
    MC33905_Config();
    /* Initialize MSCAN driver */
    ret_val = Can_Init(&Can_ModuleConfig[0]);
    /* Set MSCAN to RUN mode */
    ret_val = Can_SetMode(CAN_G128_CAN0,CAN_MODE_RUN);
    /* Initialize ADC */
    ADC_Init();
    /* Initialize PWM */
    PWM_Init();
    /* Configure MC33937A MOSFET pre-driver */
    do {
        ret = MC33937_Config();
    }while(ret == 0);

    /* Reset fault logic */
    FLT_RST = SET;
    FLT_RST = CLEAR;

    /* Enable MC33937A MOSFET Pre-driver */
    MC33937_Enable();

    /* Initialize moving average filter */
    FilterMAInit_F16(&Udcb_filt);
    Udcb_filt.u16NSamples = 2;

    /* Save actual encoder switch position */
    rotSwitchA = ROT_A;

    /* TEST */
    DDRA_DDRA7 = 1;
    DDRA_DDRA6 = 1;

    /* Wait 10ms for power stage 3.3V power supply voltage to settle down */
    API_Delay(API_5MS);
    API_Delay(API_5MS);
    
    /* Init state machine */
    appState = APP_INIT;
    
    /* Brake resistor control */
    brakeCounter = 0;
    brakeCounterFlag = 0;
    
    /* Load DC bus voltage */
    DCBusVoltageThreshold = FRAC16(U_DCB_TRIP/VOLT_RANGE_MAX);  
    
    ENABLE_INTERRUPTS();
    
    /* Initialize and start TIM counter:
       tTIM2 = 1ms (fTIM=BUSclk/32) */
    TIM_Init(TIM_PERIOD_1MS); 
    
 
    /* Main loop */
    for(;;)
    {
        /* Fault detection */
        FaultDetection();
        /* Speed, current and user input control loop */
        if(TFLG1_C2F == 1)
        {
            /* Clear Clear output compare event flag */
            TFLG1 = TFLG1_C2F_MASK;
            TC2 = TCNT + TIM_PERIOD_1MS;
            Control_Loop();
        }
        /* FreeMASTER polling function */
        FMSTR_Poll();
        /* Call BLDC application state machine function */
        AppStateMachine[appState]();
        /* Clear SBC watchdog */
        MC33905_ClearWDT();
    }
}

/******************************************************************************
*
* Function: void AppInit(void)
*
* Description: BLDC application init. function
*
*******************************************************************************/
static void AppInit(void)
{
    PWME = PWM_DISABLE;         /* Disable all PWMs */
    PORTA &= PWM_LS_RESET;      /* Disable low-side PWMs */

    duty_cycle = 0;
    PWMDTY23 = duty_cycle;
    PWMDTY45 = duty_cycle;
    PWMDTY67 = duty_cycle;
    PWMDTY01 = 1;
    
    PWME = PWM_TRIG_EN;         /* Enable PWM1 (ADC trigger signal) */
    
    stallCheckCounter = 0;
    
    /* PortP interrupt disable */
    PIEP = 0x00;
    
    /* Wait for DC-BUS current to settle down (after motor stops) */
    API_Delay(API_100US);

    /* DC-Bus current calibration */
    ADC_StartSingleConversion(ADC_INPUT_AN5);

    /* Wait until the DC-Bus current conversion is complete */
    while(ATDSTAT0_SCF == 0)
    {
    }

    ADCResults.DCBIOffset = (tFrac16)(ATDDR0 >> 1);
    
    /* ADC sequence AN5 - DCBI, AN6 - DCBV */
    ADC_EnableTrigSeq(ADC_INPUT_AN5,2);

    /* Enable PWM1 (ADC trigger signal) */
    PWME = PWM_TRIG_EN;         

    appState = APP_STOP;
}

/*******************************************************************************
*
* Function: void AppStop(void)
*
* Description: BLDC application Stop state function
*
*******************************************************************************/
static void AppStop(void)
{
    /* Speed PI controller initialization */
    speedPIPrms.f16UpperLimit = SPEED_PI_UPPER_LIMIT;
    speedPIPrms.f16LowerLimit = SPEED_PI_LOWER_LIMIT;
    speedPIPrms.f16CC1sc = SPEED_PI_CC1;
    speedPIPrms.f16CC2sc = SPEED_PI_CC2;
    speedPIPrms.u16NShift = 0;
    speedPIPrms.f32Acc = (((tFrac32)duty_cycle << 15)/(tFrac32)PWM_MODULO) << 16;
    speedPIPrms.f16InErrK1 = 0;  
    
    /* Current PI controller initialization */
    currentPIPrms.f16UpperLimit = CURRENT_PI_UPPER_LIMIT;
    currentPIPrms.f16LowerLimit = CURRENT_PI_LOWER_LIMIT;
    currentPIPrms.f16CC1sc = CURRENT_PI_CC1;
    currentPIPrms.f16CC2sc = CURRENT_PI_CC2;
    currentPIPrms.u16NShift = 0;
    currentPIPrms.f32Acc = speedPIPrms.f32Acc;
    currentPIPrms.f16InErrK1 = 0;
    
    
    if(appSwitchState)
    {
        /* PortP clear interrupt flags */
        PIFP = 0xFF;
        /* PortP PP2, PP4, PP6 interrupt enable */
        PIEP = 0x54;
        
        // Read Hall port
        hallPattern.B.H0 = PTIP_PTIP2; 
        hallPattern.B.H1 = PTIP_PTIP4;
        hallPattern.B.H2 = PTIP_PTIP6;
        
        // Check Hall pattern
        if ((hallPattern.R == 0) | (hallPattern.R == 7))
        {
            // Hall pattern fault
            faultStatus.B.HallPaternError = 1;
        }
        else
        {
            // Set next interrupt on next edge 
            PPSP = (PPSP & 0xAB) | (hallIntrConfigCw[(hallPattern.R) - 1]);  
            
            // Set minimal actual speed
            hallEdgePeriod[0] = 10900;
            hallEdgePeriod[1] = 10900;
            hallEdgePeriod[2] = 10900;
            hallEdgePeriod[3] = 10900;
            hallEdgePeriod[4] = 10900;
            hallEdgePeriod[5] = 10900;
               
            // Set PWM vector
            PWMCommutationFcn[(hallPattern.R) - 1]();
            
            // Initial speed:
            // ~ 1100rpm @ 24V DC-BUS
            // ~ 550rpm @ 12V DC-BUS
            duty_cycle = START_DUTY_CYCLE;
            PWMDTY23 = duty_cycle;
            PWMDTY45 = duty_cycle;
            PWMDTY67 = duty_cycle;
            
            appState = APP_RUN;
        }
    }
}

/*******************************************************************************
*
* Function: void AppRun(void)
*
* Description:  BLDC application Run state function
*
*******************************************************************************/
static void AppRun(void)
{   
    if(appSwitchState == 0)
    {
        appState = APP_INIT;
    }
}

/*******************************************************************************
*
* Function: void AppFault(void)
*
* Description: BLDC application Fault state function
*
*******************************************************************************/
static void AppFault(void)
{
    PWME = PWM_DISABLE;         /* Disable all PWMs */
    PORTA &= PWM_LS_RESET;      /* Disable low-side PWMs */

    PWME = PWM_TRIG_EN;         /* Enable PWM1 (ADC trigger signal) */
    
    /* Reset variable to not run the motor immediately when fault is cleared  */
    appSwitchState = 0;

    if(faultSwitchClear == 1)
    {
        /* Clear MC33937A status register */
        MC33937_ClearFaults();

        /* Reset fault logic */
        FLT_RST = SET;
        FLT_RST = CLEAR;

        /* Re-enable MC33937A */
        MC33937_Enable();
        
        /* Clear faults */
        faultStatus.R = 0;
        faultStatusLatched.R = 0;
        driveStatus.B.Fault = 0;
 
        faultSwitchClear = 0;
        appState = APP_INIT;
    }
}

/*******************************************************************************
*
* Function: void FaultDetection(void)
*
* Description:  BLDC application Fault detection function
*
*******************************************************************************/
static void FaultDetection(void)
{
    faultStatus.B.UnderDCBusVoltage = FAULT0;
    faultStatus.B.OverDCBusCurrent = FAULT1;
    
    faultStatus.B.PreDriverError = MC33937_INT;
    
    /* Reading faults from MC33937A pre-driver */
    if(faultStatus.B.PreDriverError)
    {
        mc33937_Status = MC33937_ReadStatus();
    }
    
    // Read Hall port
    hallPattern.B.H0 = PTIP_PTIP2; 
    hallPattern.B.H1 = PTIP_PTIP4;
    hallPattern.B.H2 = PTIP_PTIP6;
    
    // Check Hall pattern
    if ((hallPattern.R == 0) | (hallPattern.R == 7))
    {
        // Hall pattern fault
        faultStatus.B.HallPaternError = 1;
        
        PWME = PWM_DISABLE;         /* Disable all PWMs */
        PORTA &= PWM_LS_RESET;      /* Disable low-side PWMs */
    }

    faultStatusLatched.R |= faultStatus.R;

    if(faultStatusLatched.R != 0)
    {
        driveStatus.B.Fault = 1;
        appState = APP_FAULT;
    }
}

/*******************************************************************************
*
* Function: void LED_Indication(void)
*
* Description:  LED indication control function
*
*******************************************************************************/
static void LED_Indication(void)
{
    /* State LED indication */
    if(LEDcounter-- == 0)
    {
        /* Default value */
        LEDcounter = PERIOD_STOP;
        switch(appState)
        {
            case APP_RUN:
            {
                LED0 = CLEAR;
                break;
            }
            case APP_INIT:
            case APP_STOP:
            {
                LED0 = ~LED0;
                break;
            }
            case APP_FAULT:
            {
                LED0 = ~LED0;
                LEDcounter = PERIOD_FAULT;
                break;
            }
        }
    }
}

/*******************************************************************************
*
* Function: void Encoder_RotCtrl(void)
*
* Description:  Encoder speed control function
*
*******************************************************************************/
static void Encoder_RotCtrl(void)
{
    if(ROT_A != rotSwitchA)
    {
        rotSwitchA = ROT_A;
        if(rotSwitchA == ROT_B)
        {
            requiredSpeed -= SPEED_STEP;

            if(requiredSpeed < MIN_SPEED)
            {
                requiredSpeed = MIN_SPEED;
            }
        }
        else
        {
            requiredSpeed += SPEED_STEP;

            if(requiredSpeed > MAX_SPEED)
            {
                requiredSpeed = MAX_SPEED;
            }
        }
    }
}

/*******************************************************************************
*
* Function: void Encoder_SwCtrl(void)
*
* Description:  Encoder switch control function
*
*******************************************************************************/
static void Encoder_SwCtrl(void)
{
    if((ROT_SW == 0) && (ROT_SW == rotSwitch))
    {
        switchCounter++;

        if((switchCounter == SW_PRESS_LONG) && (appState == APP_FAULT))
        {
            faultSwitchClear = 1;
        }

        if(switchCounter > SW_PRESS_LONG)
        {
            switchCounter = SW_PRESS_LONG + 1;
        }
    }
    else
    {
        if((switchCounter > SW_PRESS_DEBOUNCE) && (switchCounter < SW_PRESS_SHORT_MAX))
        {
            appSwitchState = appSwitchState ^ 0x01;
        }

        switchCounter = 0;
    }

    rotSwitch = ROT_SW;
}

/*******************************************************************************
*
* Function: void PWM_Vector_0(void)
*
* Description:  Generates 6-channel complementary bipolar PWM output for
*               the six-step commutation sector 0.
*
* Notes:        Phase A complementary PWM (zero dead time)
*               Phase B TOP off, BOTTOM on
*               Phase C TOP off, BOTTOM off
*
*******************************************************************************/
static void PWM_Vector_0(void)
{
    PWME = PWM_DISABLE;         /* Disable all high-side PWMs (driven logic high
                                   by PIM) */
    PORTA &= PWM_LS_RESET;      /* Disable low-side PWMs */

    PWMCNT1 = 0x0000;           /* Reset PWM1 counter (ADC trigger signal) */
    PWMCNT3 = 0x0000;           /* Reset PWM3 counter (phase A high-side PWM) */

    PORTA |= PWM_LS_SECT0;      /* Enable phase A & B low-side switches to be
                                   driven by high-side PWM signals */
    PWME |= PWM_HS_SECT0;       /* Enable phase A high-side PWM and ADC trigger
                                   PWM */
}

/*******************************************************************************
*
* Function: void PWM_Vector_1(void)
*
* Description:  Generates 6-channel complementary bipolar PWM output for
*               the six-step commutation sector 1.
*
* Notes:        Phase A complementary PWM (zero dead time)
*               Phase B TOP off, BOTTOM off
*               Phase C TOP off, BOTTOM on
*
*******************************************************************************/
static void PWM_Vector_1(void)
{
    PWME = PWM_DISABLE;         /* Disable all high-side PWMs (driven logic high
                                   by PIM) */
    PORTA &= PWM_LS_RESET;      /* Disable low-side PWMs */

    PWMCNT1 = 0x0000;           /* Reset PWM1 counter (ADC trigger signal) */
    PWMCNT3 = 0x0000;           /* Reset PWM3 counter (phase A high-side PWM) */

    PORTA |= PWM_LS_SECT1;      /* Enable phase A & C low-side switches to be
                                   driven by high-side PWM signals */
    PWME |= PWM_HS_SECT1;       /* Enable phase A high-side PWM and ADC trigger
                                   PWM */
}

/*******************************************************************************
*
* Function: void PWM_Vector_2(void)
*
* Description:  Generates 6-channel complementary bipolar PWM output for
*               the six-step commutation sector 2.
*
* Notes:        Phase A TOP off, BOTTOM off
*               Phase B complementary PWM (zero dead time)
*               Phase C TOP off, BOTTOM on
*
*******************************************************************************/
static void PWM_Vector_2(void)
{
    PWME = PWM_DISABLE;         /* Disable all high-side PWMs (driven logic high
                                   by PIM) */
    PORTA &= PWM_LS_RESET;      /* Disable low-side PWMs */

    PWMCNT1 = 0x0000;           /* Reset PWM1 counter (ADC trigger signal) */
    PWMCNT5 = 0x0000;           /* Reset PWM5 counter (phase B high-side PWM) */

    PORTA |= PWM_LS_SECT2;      /* Enable phase B & C low-side switches to be
                                   driven by high-side PWM signals */
    PWME |= PWM_HS_SECT2;       /* Enable phase B high-side PWM and ADC trigger
                                   PWM */
}

/*******************************************************************************
*
* Function: void PWM_Vector_3(void)
*
* Description:  Generates 6-channel complementary bipolar PWM output for
*               the six-step commutation sector 3.
*
* Notes:        Phase A TOP off, BOTTOM on
*               Phase B complementary PWM (zero dead time)
*               Phase C TOP off, BOTTOM off
*
*******************************************************************************/
void PWM_Vector_3(void)
{
    PWME = PWM_DISABLE;         /* Disable all high-side PWMs (driven logic high
                                   by PIM) */
    PORTA &= PWM_LS_RESET;      /* Disable low-side PWMs */

    PWMCNT1 = 0x0000;           /* Reset PWM1 counter (ADC trigger signal) */
    PWMCNT5 = 0x0000;           /* Reset PWM5 counter (phase B high-side PWM) */

    PORTA |= PWM_LS_SECT3;      /* Enable phase A & B low-side switches to be
                                   driven by high-side PWM signals */
    PWME |= PWM_HS_SECT3;       /* Enable phase B high-side PWM and ADC trigger
                                   PWM */
}

/*******************************************************************************
*
* Function: void PWM_Vector_4(void)
*
* Description:  Generates 6-channel complementary bipolar PWM output for
*               the six-step commutation sector 4.
*
* Notes:        Phase A TOP off, BOTTOM on
*               Phase B TOP off, BOTTOM off
*               Phase C complementary PWM (zero dead time)
*
*******************************************************************************/
static void PWM_Vector_4(void)
{
    PWME = PWM_DISABLE;         /* Disable all high-side PWMs (driven logic high
                                   by PIM) */
    PORTA &= PWM_LS_RESET;      /* Disable low-side PWMs */

    PWMCNT1 = 0x0000;           /* Reset PWM1 counter (ADC trigger signal) */
    PWMCNT7 = 0x0000;           /* Reset PWM7 counter (phase C high-side PWM) */

    PORTA |= PWM_LS_SECT4;      /* Enable phase A & C low-side switches to be
                                   driven by high-side PWM signals */
    PWME |= PWM_HS_SECT4;       /* Enable phase C high-side PWM and ADC trigger
                                   PWM */
}

/*******************************************************************************
*
* Function: void PWM_Vector_5(void)
*
* Description:  Generates 6-channel complementary bipolar PWM output for
*               the six-step commutation sector 5.
*
* Notes:        Phase A TOP off, BOTTOM off
*               Phase B TOP off, BOTTOM on
*               Phase C complementary PWM (zero dead time)
*
*******************************************************************************/
static void PWM_Vector_5(void)
{
    PWME = PWM_DISABLE;         /* Disable all high-side PWMs (driven logic high
                                   by PIM) */
    PORTA &= PWM_LS_RESET;      /* Disable low-side PWMs */

    PWMCNT1 = 0x0000;           /* Reset PWM1 counter (ADC trigger signal) */
    PWMCNT7 = 0x0000;           /* Reset PWM7 counter (phase C high-side PWM) */

    PORTA |= PWM_LS_SECT5;      /* Enable phase B & C low-side switches to be
                                   driven by high-side PWM signals */
    PWME |= PWM_HS_SECT5;       /* Enable phase C high-side PWM and ADC trigger
                                   PWM */
}

/*******************************************************************************
*
* Function: void Control_Loop(void)
*
* Description:  Speed control, current limitation and user control loop
*               function.
*
*******************************************************************************/
void Control_Loop(void)
{    
    /* Evaluate motor stall */
    if (driveStatus.B.StallCheckReq == 1)
    {
        stallCheckCounter++;
        
        /* If reached max. limit, proceed fault */
        if (stallCheckCounter > MOTOR_STALL_LIMIT)  
        {
            driveStatus.B.StallCheckReq = 0;
            stallCheckCounter = 0;
            faultStatus.B.MotorStall = 1;
        }
    }
    else
    {
        stallCheckCounter = 0;
    }

    /* DC bus current evaluation */
    if(duty_cycle > DC_THRESHOLD)
    {
        /* Current limitation calculation (all currents shifted >> 3) */
        torque6ZC = hallEdgeTorque[0] + hallEdgeTorque[1] + hallEdgeTorque[2] + \
                    hallEdgeTorque[3] + hallEdgeTorque[4] + hallEdgeTorque[5];
    }
    else
    {
        /* Ignore current measurement if duty cycle is too small to measure
           the DC-Bus current properly */
        torque6ZC = 0;
    }
    /* Shift left to scale back to <-1,1) */
    torque = (Mul_F16(torque6ZC,TORQUE_FILT_CONST) << 3);
    torqueErr = SubSat_F16(MAX_TORQUE, torque);
    currentPIOut = ControllerPIrAW_F16(torqueErr, &currentPIPrms);

    /* Control PWM duty cycle */
    if(--speedLoopCounter == 0)
    {   
        speedLoopCounter = SPEED_LOOP_PRESCALER;

        /* Speed control */
        period6ZC = hallEdgePeriod[0] + hallEdgePeriod[1] + hallEdgePeriod[2] + \
                    hallEdgePeriod[3] + hallEdgePeriod[4] + hallEdgePeriod[5];
        actualSpeed = (tFrac16)Div_U32U16U16(SPEED_CALC_NUMERATOR, period6ZC);
        speedErr = requiredSpeed - (tFrac16)actualSpeed;
        speedPIOut = ControllerPIrAW_F16(speedErr, &speedPIPrms);

        if(currentPIOut >= speedPIOut)
        {
            /* If max torque not achieved, use speed PI output */
            currentPIPrms.f32Acc = speedPIPrms.f32Acc;
            /* PWM duty cycle update <- speed PI */
            duty_cycle = Mul_F16(speedPIOut, PWM_MODULO);
            driveStatus.B.CurrentLimiting = 0;
        }
        else
        {
            /* Limit speed PI output by current PI if max. torque achieved */
            speedPIPrms.f32Acc = currentPIPrms.f32Acc;
            /* PWM duty cycle update <- current PI */
            duty_cycle = Mul_F16(currentPIOut, PWM_MODULO);
            driveStatus.B.CurrentLimiting = 1;
        }

        /* Duty cycle 0-1 -> 0-PWM_MODULO */
        DISABLE_INTERRUPTS();
        PWMDTY23 = duty_cycle;
        PWMDTY45 = duty_cycle;
        PWMDTY67 = duty_cycle;
        PWMDTY01 = Mul_F16(duty_cycle,SENSING_POINT_FRAC) + DELAY;
        ENABLE_INTERRUPTS();
    }

    /* User input control and LED indication */
    LED_Indication();
    Encoder_SwCtrl();
    Encoder_RotCtrl();
}

#pragma CODE_SEG __NEAR_SEG NON_BANKED
/*******************************************************************************
*
* Function: void interrupt ADC_SC_ISR(void)
*
* Description:  ADC conversion sequence complete interrupt service function.
*               Function execution takes ~3us.  
*
*******************************************************************************/
void interrupt ADC_SC_ISR(void)
{
    //PORTA_PA6 = 1;
    
    /* Save conversion result of DC-BUS current */
    ADCResults.DCBIVoltage = ((tFrac16)(ADC_DCBI_RES >> 1) - ADCResults.DCBIOffset);
    
    /* Load torque array */
    hallEdgeTorque[(hallPattern.R) - 1] = ADCResults.DCBIVoltage >> 3;
    
    /* Save conversion result of DC-BUS voltage */
    ADCResults.DCBVVoltage = (tFrac16)(ADC_DCBV_RES >> 1);
    
    /* Vdcbus > DCBUS_OVERVOLT_THRESH (e.g. brake resistor in not capable to
       drop down the DC bus voltage) */
    if(ADCResults.DCBVVoltage > FRAC16(DCBUS_OVERVOLT_THRESH/VOLT_RANGE_MAX))
    {
        faultStatus.B.OverDCBusVoltage = 1;
    }
    
    /* Brake resistor control. ON if Vdcbus > DCBusVoltageThreshold */
    if ((ADCResults.DCBVVoltage > DCBusVoltageThreshold) && (brakeCounterFlag) && (appState != APP_FAULT))   
    {
        /* Brake resistor ON */
        UNI3_BRAKE = SET;
        
        /* Set DC bus voltage threshold with hysteresis */
        DCBusVoltageThreshold = FRAC16((U_DCB_TRIP - U_DCB_TRIP_HYST)/VOLT_RANGE_MAX);
        
        /* Resistor overheating protection only in case of continuous DC bus overvoltage */
        if (brakeCounter < BRAKE_ON_COUNT_MAX)
        {
            brakeCounter++;
        }
        else
        {
            /* DC BUS overvoltage fault */
            brakeCounterFlag = 0;
            faultStatus.B.OverDCBusVoltage = 1;
        }
    }
    else
    {
        /* Break resistor OFF */
        UNI3_BRAKE = CLEAR;
        
        /* Load DC bus voltage */
        DCBusVoltageThreshold = FRAC16(U_DCB_TRIP/VOLT_RANGE_MAX); 
        
        /* Wait for user fault clear */
        if (appState != APP_FAULT)
        {
            /* Reset variables */
            brakeCounterFlag = 1;
            brakeCounter = 0;
        }
    }
    
    /* Clear conversion complete flag */
    ATDSTAT0_SCF = 1;

    //PORTA_PA6 = 0;
}

/*******************************************************************************
*
* Function: void interrupt PP_ISR(void)
*
* Description:  Hall interrupt processing, ~6us
*
*******************************************************************************/
void interrupt PP_ISR(void)
{    
    //PORTA_PA6 = 1;
          
    // Load timer value 
    timerValue = TCNT;
    
    // Read Hall port
    hallPattern.B.H0 = PTIP_PTIP2; 
    hallPattern.B.H1 = PTIP_PTIP4;
    hallPattern.B.H2 = PTIP_PTIP6;
    
    // Check Hall pattern
    if ((hallPattern.R == 0) | (hallPattern.R == 7))
    {
        // Hall pattern fault
        faultStatus.B.HallPaternError = 1;
        
        PWME = PWM_DISABLE;         /* Disable all PWMs */
        PORTA &= PWM_LS_RESET;      /* Disable low-side PWMs */
    }
    else
    {
        // Set next interrupt on next edge 
        PPSP = (PPSP & 0xAB) | (hallIntrConfigCw[(hallPattern.R) - 1]);  
        
        // Set PWM vector
        PWMCommutationFcn[(hallPattern.R) - 1]();
      
        // Calculate period between two hall edges
        timerValueLim = timerValue - timerLastValue;
        
        // Limit the value to do not exceed 16bit value
        if (timerValueLim > 10900)
        {
            timerValueLim = 10900;
            driveStatus.B.StallCheckReq = 1;
        }
        else
        {
            driveStatus.B.StallCheckReq = 0;
        }
        
        // Save hall signal period (measured between two following edges)
        hallEdgePeriod[(hallPattern.R) - 1] = timerValueLim;
        
        // Load timer value
        timerLastValue = timerValue; 
        
        // Clear interrupt flag
        PIFP = PIFP | hallIntrClrMaskCw[(hallPattern.R) - 1];
    }
    
    //PORTA_PA6 = 0;
}


#pragma CODE_SEG DEFAULT

/*******************************************************************************
*
* Function: void Can_RxNotification
*
* Description:  CAN receive notification function
*
*******************************************************************************/
void Can_RxNotification
(
    Can_ModuleIDType    Can_ModuleID,
    Can_IDType          Can_ID,
    Can_DataLengthType  Can_DataLength,
    uint8               *Can_Data
)
{
    asm
    {
        nop
    }
}