/*******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
****************************************************************************//*!
*
* @file     math.h
*
* @author   B06050
* 
* @version  1.0.5.0
* 
* @date     Mar-28-2012
* 
* @brief    Mathematical function header file.
*
*******************************************************************************/
#include "typedefs.h"

#ifndef _MATH_H
#define _MATH_H

/**************************** Defines and macros ******************************/
/* Constant representing the maximal positive value of a signed 32-bit fixed
   point integer number equal to (2^(32-1))-1 = 0x7fff ffff */
#define INT32_MAX           ((int32_t) 0x7fffffff)  
/* Constant representing the maximal negative value of a signed 32-bit fixed
   point integer number equal to -2^(32-1) = 0x8000 0000 */
#define INT32_MIN           ((int32_t) 0x80000000)

/* Constant representing the maximal negative value of a signed 16-bit fixed
   point fractional number equal to -1.0 */
#define SFRACT_MIN          (-1.0)
/* Constant representing the maximal positive value of a signed 16-bit fixed
   point fractional number equal to 0.999969482421875 */
#define SFRACT_MAX          (0.999969482421875)

/* Macro converting a signed fractional [-1,1) number into a 16-bit fixed point 
   number in format Q1.15 */
#define FRAC16(x)           ((tFrac16) ((x) < (SFRACT_MAX) ? ((x) >= SFRACT_MIN ? (x)*0x8000 : 0x8000) : 0x7fff))

/* Constant representing the maximal negative value of signed 32-bit fixed
   point fractional number equal to -1.0 */
#define FRACT_MIN           (-1.0)
/* Constant representing the maximal positive value of a signed 32-bit fixed
   point fractional number equal to 0.9999999995343387126922607421875 */
#define FRACT_MAX           (0.9999999995343387126922607421875)

/* Macro converting a signed fractional [-1,1) number into a 32-bit fixed point
   number in format Q1.31 */
#define FRAC32(x)           ((tFrac32) ((x) < (FRACT_MAX) ? ((x) >= FRACT_MIN ? (x)*0x80000000 : 0x80000000) : 0x7fffffff))

/* Moving average filter default parameters */
#define FILTER_MA_DEFAULT_F16   {0,0}

/************************** Typedefs and structures ***************************/
typedef int16_t tFrac16;    /* 16-bit signed fractional Q1.15 type */
typedef int32_t tFrac32;    /* 32-bit signed fractional Q1.31 type */

/* Structure containing parameters and states of the recurrent form PI
   controller with anti-windup */
typedef struct{
    tFrac16 f16CC1sc;      /* CC1 coefficient, fractional format normalized to fit into [-2^15,(2^15)-1) */
    tFrac16 f16CC2sc;      /* CC2 coefficient, fractional format normalized to fit into [-2^15,(2^15)-1) */
    tFrac32 f32Acc;        /* Internal controller accumulator */
    tFrac16 f16InErrK1;    /* Controller input from the previous calculation step */
    tFrac16 f16UpperLimit; /* Upper Limit of the controller, fractional format normalized to fit into [-2^15,(2^15)-1) */
    tFrac16 f16LowerLimit; /* Lower Limit of the controller, fractional format normalized to fit into [-2^15,(2^15)-1) */
    uint16_t u16NShift;    /* Scaling factor for the controller coefficients, integer format [0,15] */
}CONTROLLER_PIAW_R_T_F16;

/* Structure containing MA filter buffer and coefficients */
typedef struct{
    tFrac32   f32Acc;         /* filter accumulator */
    uint16_t  u16NSamples;    /* number of samples for averaging, filter sample window [0,15] */
}FILTER_MA_T_F16;

/**************************** Function prototypes *****************************/
tFrac16 AddSat_F16(tFrac16 f16In1, tFrac16 f16In2);
tFrac16 SubSat_F16(tFrac16 f16In1, tFrac16 f16In2);
tFrac32 AddSat_F32(tFrac32 f32In1, tFrac32 f32In2);
tFrac32 SubSat_F32(tFrac32 f32In1, tFrac32 f32In2);
tFrac16 Mul_F16(tFrac16 f16In1,tFrac16 f16In2);
tFrac32 MulSat_F32F16F16(tFrac16 f16In1,tFrac16 f16In2);
tFrac32 MacSat_F32F16F16(tFrac32 f32In1,tFrac16 f16In2,tFrac16 f16In3);
tFrac16 ShL_F16(tFrac16 f16In1,uint16_t u16In2);
tFrac32 ShL_F32(tFrac32 f32In1,uint16_t u16In2);
tFrac32 ShR_F32(tFrac32 f32In1,uint16_t u16In2);
tFrac16 ControllerPIrAW_F16(tFrac16 f16InErr,CONTROLLER_PIAW_R_T_F16 * const pParam);
void FilterMAInit_F16(FILTER_MA_T_F16 *pParam);
tFrac16 FilterMA_F16(tFrac16 f16In, FILTER_MA_T_F16 *pParam);
tFrac16 Div_F16(tFrac16 f16In1, tFrac16 f16In2);
uint16_t Div_U32U16U16(uint32_t u32In1,uint16_t u16In2);

#endif /* _MATH_H */