/*****************************************************************************
 * (c) Copyright 2012, Freescale Semiconductor Inc.
 * ALL RIGHTS RESERVED.
 ***************************************************************************//*!
 * @file      main.c
 * @author    R55013
 * @version   1.0.1.0
 * @date      Jun-4-2012
 * @brief     MC9RS08KA1 Safety Watchdog.
 ******************************************************************************/
#include <hidef.h>              /* for EnableInterrupts macro                 */
#include "derivative.h"         /* include peripheral declarations            */

#pragma MESSAGE DISABLE C4000

/******************************************************************************
 * Application setting                                                        *
 ******************************************************************************/
#define tPO                 1.0 /* Power off timeout period                   */
#define tWD                 0.5 /* Watchdog timeout period                    */

/******************************************************************************
 * Module macros/constants definitions                                        *
 ******************************************************************************/
/* MTIM modulo value for given number of seconds                              */
#define MTIM_CLKSRC             (float)31.25e3  /* factory trimmed            */   
#define MTIM_DIVIDE             (float)128.0
#define MTIM_MOD(sec)           (byte)((float)sec/(2.0/MTIM_CLKSRC*MTIM_DIVIDE))
/* MTIM reset - resets counter register and TOF is cleared.                   */
#define MTIM_RST()              { MTIMSC_TRST = 1;                  } 

/* WDI & /MR signal macros                                                    */
#define GET_WDI_STATE()         PTAD_PTAD4
#define GET_MR_STATE()          PTAD_PTAD5
#define SET_MR_AS_INP()         { PTADD_PTADD5=0;                   }
#define SET_MR_AS_OUT(state)    { PTAD_PTAD5=state; PTADD_PTADD5=1; }

/* Macros for enable/disable output power (controls PMOS and NMOS transistor) */
#define POWER_EN()              { PTADD_PTADD1=0; PTADD_PTADD0=1;   }
#define POWER_DI()              { PTADD_PTADD0=0; PTADD_PTADD1=1;   }

/******************************************************************************
 * External function declarations                                             *  
 ******************************************************************************/  
extern void MCU_init(void);     /* Device initialization function declaration */

/******************************************************************************
 * Main function definition                                                   *
 ******************************************************************************/  
void main(void) 
{
  register unsigned char tmp = TRUE; 
  
 /*****************************************************************************
  * Power-up sequence                                                         *
  *****************************************************************************/  
  MCU_init  ();           /* function generated by Device Initialization tool */ 
  POWER_DI  ();           /* power off external MCU                           */

  /* An external MCU will be left powered off until /MR signal is pulled high */
  while (!GET_MR_STATE())         { __RESET_WATCHDOG(); }  
     
  /* Power up sequence - count down tPO (Power off Timeout Period) with       */
  /* subsequent power up of an external MCU                                   */
  MTIM_RST  ();           /* initialize timer                                 */
  while (MTIMCNT < MTIM_MOD(tPO)) { __RESET_WATCHDOG(); }    
  POWER_EN  ();           /* power up an external MCU                         */
  MTIM_RST  ();           /* initialize timer                                 */

 /*****************************************************************************
  * Periodic sequence - ~5.12 ms period                                       *
  *****************************************************************************/  
  while (TRUE) 
  {               
    /* WDI tristated mode check - the routine toggles /MR pin and response    */
    /* of the WDI signal state is measured. If WDI signal state follows /MR   */
    /* signal states the WDI signal is tristated and timer is initialzied     */ 
    SET_MR_AS_OUT (FALSE);
    if (!GET_WDI_STATE ()) 
    { 
      SET_MR_AS_OUT (TRUE);
      if (GET_WDI_STATE ()) 
      {
        SET_MR_AS_OUT (FALSE);
        if (!GET_WDI_STATE ()) 
        { 
          MTIM_RST (); 
        }
      }
    }
    SET_MR_AS_INP ();
    
    /* /MR signal processing - power off external MCU and reset device if     */
    /* signal pulled low                                                      */
    if (!GET_MR_STATE ())         { POWER_DI (); while (TRUE);            } 
    
    /* WDI signal processing - reinitialize timer if signal change detected   */
    if (GET_WDI_STATE () != tmp)  { tmp = GET_WDI_STATE (); MTIM_RST ();  } 
         
    /* Timer processing - if timer >= tWD then power off an external MCU and  */
    /* perform power up sequence to recover external MCU from latch-up state  */
    if (MTIMCNT > MTIM_MOD (tWD)) { POWER_DI (); while (TRUE);            } 
            
    /* Watchdog timer must be refreshed within 32 ms period                   */
    __RESET_WATCHDOG();
  }
}
/******************************************************************************
 * End of module                                                              *
 ******************************************************************************/