# This configuration script is compatible with PXR40xx devices and
# performs minimal intialization like ECC SRAM, IVORs.
#
# MMU: single 4GB addrss space, VLE
# Init SRAM ECC
# Disable Watchdog Timer
# Init IPVR and IVORx

######################################
# Initialize target variables
######################################
# var booke_vle: BOOKE = booke, VLE = vle
set booke_vle vle

# SRAM memory sizes for supported devices.
# PXR40xx, where x  = 3, 4; below key = x
# Supported memory DPM/LSM configurations:
# 256K 0x40000
# 192K 0x30000
set mem_size(3) 0x30000
set mem_size(4) 0x40000

# device names 
set pxdev(3) PXR4030
set pxdev(4) PXR4040

set current_cfg 4

######################################
## Register group definitions
######################################

# GPR registrer group
set GPR_GROUP "General Purpose Registers/"
# special purpose register group
set SPR_GROUP "e200z760n3 Special Purpose Registers/"
#TLB1 registers group
set TLB1_GROUP "regPPCTLB1/"

################################################
# Initialize the core registers: reset timers
# set SPE support and set IVORs.
################################################
proc init_z7 {} {
	global GPR_GROUP
	global SPR_GROUP

   	# reset watch dog timer
	reg ${SPR_GROUP}TCR = 0x0

	# Set up all interrupt vectors
	reg ${SPR_GROUP}IVPR = 0x40000000
	
	for {set i 0} {$i < 15} {incr i} {
		reg ${SPR_GROUP}IVOR${i} %d = [expr $i * 0x10]
	}
	reg ${SPR_GROUP}IVOR32 = 0x100
	reg ${SPR_GROUP}IVOR33 = 0x110
	reg ${SPR_GROUP}IVOR34 = 0x120
	
    # set SP to an unaligned (4bytes) to avoid creating
	# invalid stack frames
    reg ${GPR_GROUP}SP = 0x3

    # enable SPE and Debug interrupts
	reg ${SPR_GROUP}MSR = 0x2002000
}

################################################
## Configures MMU for Z7 core.
################################################
proc init_z7_MMU {} {
	global booke_vle
	global TLB1_GROUP
	# Setup MMU for entire 4GB address space
	# Base address = 0x0000_0000
	# TLB0, 4 GByte Memory Space, Not Guarded, Cache inhibited, All Access
	if {$booke_vle == "vle"} {
		# VLE page
		reg ${TLB1_GROUP}MMU_CAM0 = 0xB0000008FE0800000000000000000001
	} else {
		# BOOKE page
		reg ${TLB1_GROUP}MMU_CAM0 = 0xB0000008FC0800000000000000000001
	}
}
#################################################
# Initialize a RAM 'range' from 'start' address,
# downloading the init program at 0x4000_0000.
#################################################
proc init_ram {start range} {
  	global GPR_GROUP
  	global booke_vle

    puts "init ECC SRAM $start:$range"
  	# default SRAM address
  	set pstart 0x40000000
  	# the offset from the given start address
  	# at which the init prgram starts to init SRAM
	set offset 0x0
    # stmw write page size = 128 = 4bytes * 32 GPRS
    set psize 0x80

	if {$start == $pstart} {
		# init first 4 bytes (mem access) x 128 = 512
		# bytes to avoid reading the memory around PC
		# after stopping the core
    	mem $start 256 = 0x0
		# base init address
		set offset 0x80
	}
	
	# address to start initialization
    set start [expr {$start + $offset}]
    
	# load add into GPR
    reg ${GPR_GROUP}GPR11 %d = $start

    # compute actual times stmw is called
    # and set counter
    set c [expr ((($range - $offset)/$psize))]
    reg ${GPR_GROUP}GPR12 %d = $c

    # execute init ram code
    if {$booke_vle == "vle"} {
	    #mtctr r12
	    mem $pstart = 0x7D8903A6
	    #stmw r0,0(r11)
	    mem [format %x [expr $pstart + 0x4]] = 0x180B0900
	    #addi r11,r11,128
	    mem [format %x [expr $pstart + 0x8]] = 0x1D6B0080
	    #bdnz -8
	    mem [format %x [expr $pstart + 0xc]] = 0x7A20FFF8
	    # infinte loop
	    #se_b *+0
	    mem [format %x [expr $pstart + 0x10]] = 0xE8000000
    } else {
	    #mtctr r12
	    mem $pstart = 0x7D8903A6
	    #stmw r0,0(r11)
	    mem [format %x [expr $pstart + 0x4]] = 0xBC0B0000
	    #addi r11,r11,128
	    mem [format %x [expr $pstart + 0x8]] = 0x396B0080
	    #bdnz -8
	    mem [format %x [expr $pstart + 0xc]] = 0x4200FFF8
	    # infinte loop
	    #se_b *+0
	    mem [format %x [expr $pstart + 0x10]] = 0x48000000    
    }
    
    # set PC to the first init instruction
    reg ${GPR_GROUP}PC = $pstart
    # execute init ram code
    # timeout 1 second to allow the code to execute
    go 1
    stop
}
#######################################
## Detects the current PXR40xx device
#######################################
proc detect_pxr_device {} {
	global current_cfg
	global mem_size
	global pxdev
    
	set SIUL_MIDR1 0x50000
	puts "reading SIUL.MIDR1 ..."
	set r [catch {set SIUL_MIDR1 [mem 0xC3F90004]}]
	if {$r == 0} {
		# get PXR device
		set r [expr ($SIUL_MIDR1 & 0xF0000) / 0x10000]
		if {[info exists mem_size($r)]} {
			puts [format "Current device found as %s" $pxdev($r)]
			set current_cfg $r
		} else {
			puts [format "Unable to detect device. Unknown processor code %s" $SIUL_MIDR1]
		}
	}
}

proc pxr40xx_init {} {
  	global GPR_GROUP
  	global booke_vle
  	global current_cfg
  	global mem_size
  	
  	puts "Start initializing the device ..."
	reset hard
			
	# Explicitly stop core
	stop
	
	puts "initialize core"
	init_z7
	puts "initialize MMU"
	init_z7_MMU
	
	# Disable Watchdog Timer
	mem 0xFFF38010 = 0x0000C520
	mem 0xFFF38010 = 0x0000D928
	mem 0xFFF38000 = 0x8000010A

	# detects the current device
	detect_pxr_device
	
	# initialize ECC SRAM for LSM
	init_ram 0x40000000 $mem_size($current_cfg)
	puts "Finsh initializing the device."
}

########################################
## Environment Setup: 32bit meme access
## or numbers start with hex notation.
########################################
proc envsetup {} {
	radix x 
	config hexprefix 0x
	config MemIdentifier v 
	config MemWidth 32 
	config MemAccess 32 
	config MemSwap off
}
 
# env setup
envsetup

# main entry
pxr40xx_init
