/*
 * File:        k60_Camera.c
 *              Copyright (c) 2013 Freescale
 * Version:     1.0
 * Date:        May 2013 
 * Purpose:     This software is set to configure a CMOS sensor (camera) by K60 through I2C (K60 is the master). 
 *              The camera is configured to generate a QQVGA image at a 30fps and send each 8bit of data via
 *              DMA to the internal SRAM. DMA is triggered by hardware from the camera clock which is connected 
 *              to the K60 through GPIO. The other camera control signals - VSYNC and HREF - are also connected 
 *              to the K60 through GPIO. The VSYNC and HREF signals are configured in software as GPIO interrupts 
 *              to determine the way the image data is transferred to the K60 internal SRAM.
 *              8 data lines representing 1Byte of data sent rapidly to the K60 SRAM are connected as GPIOs.
 *              The main loop of transfer consists of 2 QQVGA frames. After transmittion of 2 frames is completed 
 *              another 2 images are starting transfer running over the previous data transferred.
*/

#include "common.h"

#ifdef TWR_K60N512
  #include "k60_Camera.h"
#endif

unsigned char MasterTransmission;
unsigned char SlaveID;

#pragma location="RAM_Low_Scratch_region"           
uint32 rows,frames;

//Function declarations
void Init_Gpio(void);
void Set_FTM0(void);
void Set_Dma(void); 
void Set_Dma_Mux(void);
void vsync_isr(void);
void href_isr(void);
void Set_Register(unsigned char reg, unsigned char val);
void Set_Camera(void);
void Set_Pit0(void);
void Delay_us( int delay );
void Delay_ms( int delay );
void init_I2C(void);      
void IIC_StartTransmission (unsigned char SlaveID, unsigned char Mode);
void OV7675WriteRegister(unsigned char u8RegisterAddress, unsigned char u8Data);

/********************************************************************/

void main (void)
{
  SIM_SCGC5 = SIM_SCGC5_PORTB_MASK | SIM_SCGC5_PORTC_MASK | SIM_SCGC5_PORTD_MASK | SIM_SCGC5_PORTE_MASK;   // Ports B,C,D,E clocks
    SIM_SCGC6 |= SIM_SCGC6_DMAMUX_MASK;   // DMA Mux clock
    
  rows = QQVGA_rows;    // Counter for number of rows per frame (in QQVGA resolution)
  frames=frame_count;   // Counter for number of frames in this stage of demo (according to RAM size limitation)
  init_I2C();           // Initialize I2C
  Set_FTM0();           // Initialize FTM for external clock
  Init_Gpio();          // Initialize GPIO for data transfer 
  Set_Dma();            // configuring DMA behavior for the data transfer
  Set_Dma_Mux();        // configuring DMA Mux to transfer data with HW request
  Set_Pit0();           // Initialize PIT clock for timing the delay before data transfer commence. (not necessary in the future)
  Set_Camera();         // Defining the camera configuration for desired working mode 
  
  Delay_ms(500);        // Reason for delay: in order to receive a frame when brightness and contrast are already stabilized
     
  PORTE_PCR12 |= PORT_PCR_IRQC(0x09);   // Enabling interrupt on rising edge for VSYNC signal
   
  while(1);
}

/***************************************/            
/* I2C Initialization                  */
/* Set Baud Rate and turn on I2C0      */
/***************************************/
void init_I2C(void)
{
    SIM_SCGC4 |= SIM_SCGC4_I2C0_MASK;   // Turn on clock to I2C0 module
    
    PORTD_PCR9 = PORT_PCR_MUX(2);       // Configure pin functionality for I2C0
    PORTD_PCR8 = PORT_PCR_MUX(2);

    I2C0_F  = 0x17;                     // SLC divider is 128 for 50MHz clock (I2C works in ~390KHz). 
                                        // For XCLK=24MHz, SCCB(camera) capable of 400KHz   
    I2C0_C1 = I2C_C1_IICEN_MASK;        // enable IIC
}

/****************************************************/
/* Start I2C Transmision                            */
/* @param SlaveID is the 7 bit Slave Address        */
/* @param Mode sets Read or Write Mode              */
/****************************************************/
void IIC_StartTransmission (unsigned char SlaveID, unsigned char Mode)
{
  if(Mode == MWSR)
  {
    /* set transmission mode */
    MasterTransmission = MWSR;
  }
  else
  {
    /* set transmission mode */
    MasterTransmission = MRSW;
  }

  /* shift ID in right possition */
  SlaveID = (unsigned char) OV7675_I2C_ADDRESS << 1;

  /* Set R/W bit at end of Slave Address */
  SlaveID |= (unsigned char)MasterTransmission;

  /* send start signal */
  i2c_Start();

  /* send ID with W/R bit */
  I2C0_D = SlaveID;
}

/****************************************/
/*           Pause Routine              */
/****************************************/
void Pause(void){
    int n;
    for(n=1;n<50;n++) {
      asm("nop");
    }
}

/*******************************************************************/
/* Write a byte of Data to specified register on OV7675            */
/* @param u8RegisterAddress is Register Address                    */
/* @param u8Data is Data to write                                  */
/*******************************************************************/
void OV7675WriteRegister(unsigned char u8RegisterAddress, unsigned char u8Data)
{
  // send data to slave 
  IIC_StartTransmission(SlaveID,MWSR);
  i2c_Wait();

  I2C0_D = u8RegisterAddress;
  i2c_Wait();

  I2C0_D = u8Data;
  i2c_Wait();

  i2c_Stop();

  Pause();
}


/**************************************************************/
/*   Creates 24MHz putput clock for the OV7675 through FTM    */
/**************************************************************/
void Set_FTM0(void)
{
	SIM_SCGC6 |= SIM_SCGC6_FTM0_MASK;                       // Enable clock for FTM0 
	PORTC_PCR1 = PORT_PCR_MUX(0x04)|PORT_PCR_DSE_MASK;      // Define pin for FTM0    
        FTM0_MOD = FTM_24MHZ_PERIOD;                            // PWM period 
        // Configure timers for edge aligned PWM High True Pulses 
	FTM0_C0SC = 0x28;                                       // No Interrupts; High True pulses on Edge Aligned PWM
	FTM0_C0V = FTM_24MHZ_DUTY;                              // 50% pulse width   
	FTM0_SC = 0x08;                                         // Edge Aligned PWM running from BUSCLK / 1 
}


/*************************************************************************************/
/* Initialize all 8 GPIO of data lines, VSYNC, HREF and  PCLK signals of the camera. */
/* Defines VSYNC and HREF to initialize  interrupts.                                 */
/*************************************************************************************/
void Init_Gpio(void){

/* GPIO related isr's: */
  
  __VECTOR_RAM[VSYNC_VEC] = (uint32)vsync_isr;  // plug isr into vector table in case not there already
  enable_irq(VSYNC_NO) ;                        // ready for this interrupt 
  
  __VECTOR_RAM[HREF_VEC] = (uint32)href_isr;    // plug isr into vector table in case not there already
  enable_irq(HREF_NO) ;                         // ready for this interrupt  
    
/* control signals */
  
//PORTE12 
  PORTE_PCR12 |= PORT_PCR_IRQC(0x00)|PORT_PCR_MUX(0x01);         // selec GPIO function for VSYNC signal 
  GPIOE_PDDR  |= 0x00 << 12 ;                                     // input enable
    
// PORTC15 
  PORTC_PCR15 |= PORT_PCR_IRQC(0x00)|PORT_PCR_MUX(0x01);         // selec GPIO function for HREF signal
  GPIOC_PDDR  |= 0x00 << 15 ;                                    // input enable
 
//PORTB18 
  PORTB_PCR18 |= PORT_PCR_IRQC(0x00)|PORT_PCR_MUX(0x01);        // selec GPIO function for HREF signal
  GPIOB_PDDR  |= 0x00 << 18 ;                                    // input enable  
  
  /* data signals - Port E bits [0:7] */
  
//PORTE0 
  PORTE_PCR0 = PORT_PCR_MUX(1) ;         // selec GPIO function
  GPIOE_PDDR |= 0x00 << 0 ;              // input enable
//PORTE1 
  PORTE_PCR1 = PORT_PCR_MUX(1) ;         // selec GPIO function
  GPIOE_PDDR |= 0x00 << 1 ;              // input enable
//PORTE2 
  PORTE_PCR2 = PORT_PCR_MUX(1) ;         // selec GPIO function
  GPIOE_PDDR |= 0x00 << 2 ;              // input enable
//PORTE3 
  PORTE_PCR3 = PORT_PCR_MUX(1) ;         // selec GPIO function
  GPIOE_PDDR |= 0x00 << 3 ;              // input enable
//PORTE4 
  PORTE_PCR4 = PORT_PCR_MUX(1) ;         // selec GPIO function
  GPIOE_PDDR |= 0x00 << 4 ;              // input enable
//PORTE5 
  PORTE_PCR5 = PORT_PCR_MUX(1) ;         // selec GPIO function
  GPIOE_PDDR |= 0x00 << 5 ;              // input enable
//PORTE6 
  PORTE_PCR6 = PORT_PCR_MUX(1) ;         // selec GPIO function
  GPIOE_PDDR |= 0x00 << 6 ;              // input enable
//PORTE7 
  PORTE_PCR7 = PORT_PCR_MUX(1) ;         // selec GPIO function
  GPIOE_PDDR |= 0x00 << 7 ;              // input enable
  
}

/*****************************************************************************/
/* DMA is configured to be HW triggered and have the highest channel         */
/* priority. DMA transfer is constructed to read 1 byte at a time from       */
/* same address of the PORT and write to a consecutive address in RAM        */
/*****************************************************************************/
void Set_Dma(void)  
{
  DMA_CR = 0;                                                            // no minor loop mapping
  DMA_DCHPRI15 = 15;                                                    // cannot be pre-empeted, can pre-empt, highest priority , same as reset value  
    
  DMA_TCD15_SADDR          = DMA_SOURCE_ADDRESS ;                       // Address of first 8 bits of PORT E
  DMA_TCD15_SOFF           = 0;                                         // No need of offset, transferring only 1 Byte of the same address
  DMA_TCD15_ATTR           = DMA_ATTR_SMOD(0) 
                             | DMA_ATTR_SSIZE(0) 
                             | DMA_ATTR_DMOD(0) 
                             | DMA_ATTR_DSIZE(0);                       // 8 bit S&D, no modulo 
  DMA_TCD15_NBYTES_MLNO    = 1;                                         // Transferring 1Byte in each itteration
  DMA_TCD15_SLAST          = 0;   
  DMA_TCD15_DADDR          = DMA_FRAME_DESTINATION ;
  DMA_TCD15_DOFF           = 1;                                         // destination address progresses while source address remains the same
  DMA_TCD15_CITER_ELINKNO  = DMA_CITER_ELINKNO_CITER(QQVGA_columns*2);  // not major loop counter is 320 (160 bits of a row x2 for each pixel)
  DMA_TCD15_DLASTSGA       = -(QQVGA_columns*2); 
  DMA_TCD15_CSR            = 0;                                         // working without an interrupt
  DMA_TCD15_BITER_ELINKNO  = DMA_BITER_ELINKNO_BITER(QQVGA_columns*2);  // BITER value = CITER value 
  DMA_ERQ = DMA_ERQ_ERQ15_MASK;                                         // Enable DMA trigger in HW request 
}

/***************************************************/
/* Configure DMA Mux for Channel 15                */
/***************************************************/
void Set_Dma_Mux(void)  
{
// Enable routing of DMA request from source PORT B, no triggering
  DMAMUX_CHCFG15 |= DMAMUX_CHCFG_ENBL_MASK | DMAMUX_CHCFG_SOURCE(50);    
}

/************************************************************************/
/*  Once a VSYNC signal is detected it initiates the following routine: */
/*  isr flag is cleared, enabling the HREF isr for falling egde and     */
/*  enabling the DMA HW request by the PCLK pin.                        */
/************************************************************************/
void vsync_isr(void)
{ 
  PORTE_ISFR=0xFFFFFFFF;                                 // Clear Port E ISR flags
    if ( frames == 0 )                                   // when 2 frames are fully transferred and a new frame arrives
  {
    frames = frame_count;                                // Starting frame count again
    DMA_TCD15_DADDR = DMA_FRAME_DESTINATION ;            // Resetting the destination address of DMA
  }
  PORTC_PCR15 = PORT_PCR_IRQC(0x0A)|PORT_PCR_MUX(0x01);  // Enabling interrupt on falling edge for HREF signal
  PORTB_PCR18 |= PORT_PCR_IRQC(0x02)
              | PORT_PCR_PE_MASK 
              | PORT_PCR_PS_MASK 
              | PORT_PCR_DSE_MASK ;                      // DMA request enabled with PCLK signal rise
  frames-- ;                                             // Counting down number of frames
  
}
/***********************************************************************/
/*  Once a HREF signal is detected it initiates the following routine: */
/*  Counting down the number of rows in a QQVGA frame and setting      */    
/*  it back when countdown is exhausted.                               */
/*  Everytime HREF indicates an end of a row the DMA destination       */
/*  address updated in the number of pixel per line x2.                */
/***********************************************************************/

void href_isr(void)
{
  PORTC_ISFR=0xFFFFFFFF;                                    //Clear Port C ISR flags
  rows-- ;                                                  // Counting down the rows for a frame
  if (rows==0)                                              // When a full frame is fully transferred
    rows=QQVGA_rows;                                        // Starting Rows count again
  DMA_TCD15_DADDR = DMA_TCD15_DADDR + (QQVGA_columns*2);    // DMA dest address update after every row transfer
}

/******************************************************/
/* This function write new values to camera registers */
/* in order to configure to desired working mode.     */
/******************************************************/
void Set_Register(unsigned char reg, unsigned char val)
{
  OV7675WriteRegister(reg, val);    // Writing a new value to OV7675 register through I2C
  Delay_us(7);                      // Minimal waiting period for the writing acknowledge
}

/****************************************************************************************/
/* Configuring camera settings for desired work mode.                                   */
/*                                                                                      */
/* The following  set of instructions for the OV7675 was provided by OmniVision and     */
/* has gone through some minor changes in order to meet the requirements of the K60.    */
/* For more information on how to configure the camera, please refer to the data sheet. */
/****************************************************************************************/
void Set_Camera(void)
{
  Set_Register(0x12 , 0x80);
  Set_Register(0x09 , 0x10);
  Set_Register(0xc1 , 0x7f);
  Set_Register(0x11 , 0x87);   
  Set_Register(0x3a , 0x0c);   
  Set_Register(0x3d , 0xc0);
  Set_Register(0x12 , 0x00);
  Set_Register(0x15 , 0x60);  
  Set_Register(0x17 , 0x13);
  Set_Register(0x18 , 0x01);
  Set_Register(0x32 , 0xbf);
  Set_Register(0x19 , 0x02);
  Set_Register(0x1a , 0x7a);
  Set_Register(0x03 , 0x0a);
  Set_Register(0x0c , 0x00);
  Set_Register(0x3e , 0x00);
  Set_Register(0x70 , 0x3a);
  Set_Register(0x71 , 0x35);
  Set_Register(0x72 , 0x11);
  Set_Register(0x73 , 0xf0);
  Set_Register(0xa2 , 0x02);
  Set_Register(0x7a , 0x20);
  Set_Register(0x7b , 0x03);
  Set_Register(0x7c , 0x0a);
  Set_Register(0x7d , 0x1a);
  Set_Register(0x7e , 0x3f);
  Set_Register(0x7f , 0x4e);
  Set_Register(0x80 , 0x5b);
  Set_Register(0x81 , 0x68);
  Set_Register(0x82 , 0x75);
  Set_Register(0x83 , 0x7f);
  Set_Register(0x84 , 0x89);
  Set_Register(0x85 , 0x9a);
  Set_Register(0x86 , 0xa6);
  Set_Register(0x87 , 0xbd);
  Set_Register(0x88 , 0xd3);
  Set_Register(0x89 , 0xe8);
  Set_Register(0x13 , 0xe0);
  Set_Register(0x00 , 0x00);
  Set_Register(0x10 , 0x00);
  Set_Register(0x0d , 0x40);
  Set_Register(0x14 , 0x28);
  Set_Register(0xa5 , 0x02);
  Set_Register(0xab , 0x02);
  Set_Register(0x24 , 0x68);
  Set_Register(0x25 , 0x58);
  Set_Register(0x26 , 0xc2);
  Set_Register(0x9f , 0x78);
  Set_Register(0xa0 , 0x68);
  Set_Register(0xa1 , 0x03);
  Set_Register(0xa6 , 0xd8);
  Set_Register(0xa7 , 0xd8);
  Set_Register(0xa8 , 0xf0);
  Set_Register(0xa9 , 0x90);
  Set_Register(0xaa , 0x14);
  Set_Register(0x13 , 0xe5);
  Set_Register(0x0e , 0x61);
  Set_Register(0x0f , 0x4b);
  Set_Register(0x16 , 0x02);
  Set_Register(0x1e , 0x37);
  Set_Register(0x21 , 0x02);
  Set_Register(0x22 , 0x91);
  Set_Register(0x29 , 0x07);
  Set_Register(0x33 , 0x0b);
  Set_Register(0x35 , 0x0b);
  Set_Register(0x37 , 0x1d);
  Set_Register(0x38 , 0x71);
  Set_Register(0x39 , 0x2a);
  Set_Register(0x3c , 0x78);
  Set_Register(0x4d , 0x40);
  Set_Register(0x4e , 0x20);
  Set_Register(0x69 , 0x00);
  Set_Register(0x6b , 0x0a);
  Set_Register(0x74 , 0x10);
  Set_Register(0x8d , 0x4f);
  Set_Register(0x8e , 0x00);
  Set_Register(0x8f , 0x00);
  Set_Register(0x90 , 0x00);
  Set_Register(0x91 , 0x00);
  Set_Register(0x96 , 0x00);
  Set_Register(0x9a , 0x80);
  Set_Register(0xb0 , 0x84);
  Set_Register(0xb1 , 0x0c);
  Set_Register(0xb2 , 0x0e);
  Set_Register(0xb3 , 0x82);
  Set_Register(0xb8 , 0x0a);
  Set_Register(0x43 , 0x0a);
  Set_Register(0x44 , 0xf2);
  Set_Register(0x45 , 0x39);
  Set_Register(0x46 , 0x62);
  Set_Register(0x47 , 0x3d);
  Set_Register(0x48 , 0x55);
  Set_Register(0x59 , 0x83);
  Set_Register(0x5a , 0x0d);
  Set_Register(0x5b , 0xcd);
  Set_Register(0x5c , 0x8c);
  Set_Register(0x5d , 0x77);
  Set_Register(0x5e , 0x16);
  Set_Register(0x6c , 0x0a);
  Set_Register(0x6d , 0x65);
  Set_Register(0x6e , 0x11);
  Set_Register(0x6a , 0x40);
  Set_Register(0x01 , 0x56);
  Set_Register(0x02 , 0x44);
  Set_Register(0x13 , 0xe7);
  Set_Register(0x4f , 0x88);
  Set_Register(0x50 , 0x8b);
  Set_Register(0x51 , 0x04);
  Set_Register(0x52 , 0x11);
  Set_Register(0x53 , 0x8c);
  Set_Register(0x54 , 0x9d);
  Set_Register(0x55 , 0x00);
  Set_Register(0x56 , 0x40);
  Set_Register(0x57 , 0x80);
  Set_Register(0x58 , 0x9a);
  Set_Register(0x41 , 0x08);
  Set_Register(0x3f , 0x00);
  Set_Register(0x75 , 0x04);
  Set_Register(0x76 , 0x60);
  Set_Register(0x4c , 0x00);
  Set_Register(0x77 , 0x01);
  Set_Register(0x3d , 0xc2);
  Set_Register(0x4b , 0x09);
  Set_Register(0xc9 , 0x30);
  Set_Register(0x41 , 0x38);
  Set_Register(0x56 , 0x40);
  Set_Register(0x34 , 0x11);
  Set_Register(0x3b , 0x12);
  Set_Register(0xa4 , 0x88);
  Set_Register(0x96 , 0x00);
  Set_Register(0x97 , 0x30);
  Set_Register(0x98 , 0x20);
  Set_Register(0x99 , 0x30);
  Set_Register(0x9a , 0x84);
  Set_Register(0x9b , 0x29);
  Set_Register(0x9c , 0x03);
  Set_Register(0x9d , 0x99);
  Set_Register(0x9e , 0x7f);
  Set_Register(0x78 , 0x04);
  Set_Register(0x79 , 0x01);
  Set_Register(0xc8 , 0xf0);
  Set_Register(0x79 , 0x0f);
  Set_Register(0xc8 , 0x00);
  Set_Register(0x79 , 0x10);
  Set_Register(0xc8 , 0x7e);
  Set_Register(0x79 , 0x0a);
  Set_Register(0xc8 , 0x80);
  Set_Register(0x79 , 0x0b);
  Set_Register(0xc8 , 0x01);
  Set_Register(0x79 , 0x0c);
  Set_Register(0xc8 , 0x0f);
  Set_Register(0x79 , 0x0d);
  Set_Register(0xc8 , 0x20);
  Set_Register(0x79 , 0x09);
  Set_Register(0xc8 , 0x80);
  Set_Register(0x79 , 0x02);
  Set_Register(0xc8 , 0xc0);
  Set_Register(0x79 , 0x03);
  Set_Register(0xc8 , 0x40);
  Set_Register(0x79 , 0x05);
  Set_Register(0xc8 , 0x30);
  Set_Register(0x79 , 0x26);
  Set_Register(0x62 , 0x00);
  Set_Register(0x63 , 0x00);
  Set_Register(0x64 , 0x06);
  Set_Register(0x65 , 0x00);
  Set_Register(0x66 , 0x05);
  Set_Register(0x94 , 0x05);
  Set_Register(0x95 , 0x09);
  Set_Register(0x2a , 0x10);
  Set_Register(0x2b , 0xc2);
  Set_Register(0x15 , 0x20);    
  Set_Register(0x3a , 0x04);
  Set_Register(0x3d , 0xc3);
  Set_Register(0x19 , 0x03);
  Set_Register(0x1a , 0x7b);
  Set_Register(0x2a , 0x00);
  Set_Register(0x2b , 0x00);
  Set_Register(0x18 , 0x01);
  Set_Register(0x19 , 0x03);
  Set_Register(0x1a , 0x21);
  Set_Register(0x03 , 0x05);
  Set_Register(0x17 , 0x13);
  Set_Register(0x18 , 0x27);
  Set_Register(0x32 , 0x24);
  Set_Register(0xe6 , 0x80);
  Set_Register(0xe1 , 0x40);
  Set_Register(0xe4 , 0xb8);
  Set_Register(0xe5 , 0x33);
  Set_Register(0xbf , 0xf0);
  Set_Register(0x66 , 0x05);
  Set_Register(0x62 , 0x10);
  Set_Register(0x63 , 0x0b);
  Set_Register(0x65 , 0x07);
  Set_Register(0x64 , 0x0f);
  Set_Register(0x94 , 0x0e);
  Set_Register(0x95 , 0x12);
  Set_Register(0x4f , 0x74);
  Set_Register(0x50 , 0x59);
  Set_Register(0x51 , 0x1a);
  Set_Register(0x52 , 0x12);
  Set_Register(0x53 , 0x6a);
  Set_Register(0x54 , 0x7c);
  Set_Register(0x58 , 0x1e);
  Set_Register(0x41 , 0x38);
  Set_Register(0x75 , 0x05);
  Set_Register(0x76 , 0xe0);
  Set_Register(0x77 , 0x07);
  Set_Register(0x24 , 0x38);
  Set_Register(0x25 , 0x28);
  Set_Register(0x26 , 0x80);
  Set_Register(0x7a , 0x10);
  Set_Register(0x7b , 0x0c);
  Set_Register(0x7c , 0x17);
  Set_Register(0x7d , 0x2c);
  Set_Register(0x7e , 0x50);
  Set_Register(0x7f , 0x60);
  Set_Register(0x80 , 0x6e);
  Set_Register(0x81 , 0x7b);
  Set_Register(0x82 , 0x87);
  Set_Register(0x83 , 0x92);
  Set_Register(0x84 , 0x9c);
  Set_Register(0x85 , 0xaf);
  Set_Register(0x86 , 0xbf);
  Set_Register(0x87 , 0xd7);
  Set_Register(0x88 , 0xe8);
  Set_Register(0x89 , 0xf4);
  Set_Register(0x43 , 0x0a);
  Set_Register(0x44 , 0xf2);
  Set_Register(0x45 , 0x46);
  Set_Register(0x46 , 0x5f);
  Set_Register(0x47 , 0x2e);
  Set_Register(0x48 , 0x42);
  Set_Register(0x59 , 0xb1);
  Set_Register(0x5a , 0xb5);
  Set_Register(0x5b , 0xdd);
  Set_Register(0x5c , 0x7b);
  Set_Register(0x5d , 0x57);
  Set_Register(0x5e , 0x14);
  Set_Register(0x6c , 0x0e);
  Set_Register(0x6d , 0x65);
  Set_Register(0x6e , 0x11);
  Set_Register(0x6f , 0x9e);
  Set_Register(0x09 , 0x00); 
}

/******************************************/
/* Clock enabled to measure delay in work */
/******************************************/
void Set_Pit0(void)
{
  SIM_SCGC6 |= SIM_SCGC6_PIT_MASK;   // turn on PIT cloccks
  PIT_MCR    = 0;                    // reset MDIS -> enable the module
  PIT_LDVAL0 = 0xFFFFFFFF;           // free running on 32 bits 
  PIT_TCTRL0 = PIT_TCTRL_TEN_MASK;   // start the timer  
}  

/****************************************************/
/* Time delay in usec units. In use by Set_Register */
/****************************************************/
void Delay_us( int delay )
{
int start,end,i;

end = delay*50;   // PIT is running at 20nsec/tick  

start = PIT_CVAL0;   // initial time

do i= start-PIT_CVAL0;
  while(i < end) ;
}

/**********************************************/
/* Time delay in msec units. In use to create */
/* a delay before capturing the first frame.  */
/**********************************************/
void Delay_ms( int delay )
{
int start,end,i;

end = delay*50000;   // PIT is running at 20nsec/tick  

start = PIT_CVAL0;   // initial time

do i= start-PIT_CVAL0;
  while(i < end) ;
}